/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* calendar_list_cb.c -- Calendar list window */

#include <schedwi.h>

#include <schedwi_interface.h>

#include <lib_functions.h>
#include <sql_common.h>
#include <sql_calendar.h>
#include <references_cb.h>
#include <sql_where_used.h>
#include <message_windows.h>
#include <cursor.h>
#include <schedwi_g_utils.h>
#include <calendar_canvas.h>
#include <calendar_folder_cb.h>
#include <calendar_cal_cb.h>
#include <calendar_list_cb.h>

#define MIN_YEAR 1971

/*
 * Calendar description from the database
 */
struct calendar {
	guint64 id, parent_id;
	gint type;
	gchar *name, *description, *formula;
};
typedef struct calendar calendar_t; 


/*
 * Free a calendar structure
 */
static void
destroy_calendar (calendar_t *ptr)
{
	if (ptr != NULL) {
		g_free (ptr->name);
		g_free (ptr->description);
		g_free (ptr->formula);
		g_free (ptr);
	}
}


/*
 * Allocate and initialise a new calendar structure
 *
 * Return:
 *   The new structure (to be freed by destroy_calendar())
 */
static calendar_t *
new_calendar (	const gchar *id, const gchar *parent_id,
		const gchar *type, const gchar *name,
		const gchar *description, const gchar *formula)
{
	calendar_t *ptr;

	ptr = (calendar_t *) g_malloc (sizeof (calendar_t));
	if (id != NULL && id[0] != '\0') {
		ptr->id = g_ascii_strtoull (id, NULL, 0);
	}
	else {
		ptr->id = 0;
	}
	if (parent_id != NULL && parent_id[0] != '\0') {
		ptr->parent_id = g_ascii_strtoull (parent_id, NULL, 0);
	}
	else {
		ptr->parent_id = 0;
	}
	ptr->type = (type == NULL || type[0] == '\0' || type[0] == '0') ? 0: 1;
	ptr->name = g_strdup (name);
	ptr->description = g_strdup (description);
	ptr->formula = g_strdup (formula);
	return ptr;
}


/*
 * Free a calendar structure.  This function is used by g_slist_foreach()
 * to free all the elements of a list
 */
static void
remove_calendar_from_list (gpointer data, gpointer user_data)
{
	destroy_calendar ((calendar_t *)data);
}


/*
 * Free a list of calendar structures
 */
static void
destroy_calendar_list (gpointer data)
{
	GSList *lst = data;

	g_slist_foreach (lst, remove_calendar_from_list, NULL);
	g_slist_free (lst);
}


/*
 * Display the Edit popup for the provided item (calendar or folder)
 */
static void
popup_edit_window (GtkTreeView *view, GtkTreeModel *model, GtkTreeIter *iter)
{
	guint64 id;
	gint type;
	gchar *id_str;
	int workload_date;

	/* A folder or a calendar must be selected (not the `(Empty Item))' */
	gtk_tree_model_get (model, iter, 1, &id, 2, &type, -1);
	if (type != 0 && type != 1) {
		return;
	}

	/* Retrieve the workload date */
	workload_date = (int) GPOINTER_TO_INT (g_object_get_data (
				G_OBJECT (lookup_widget (GTK_WIDGET (view),
							"dialog_cal_list")),
				"workload_date"));

	cursor_busy (GTK_WIDGET (view));
	id_str = schedwi_ulltostr (id);

	if (type == 1) {
		/* Edit the selected folder */
		new_dialog_cal_folder (view, id_str, NULL, workload_date);
		g_free (id_str);
	}
	else {
		/* Edit the selected calendar */
		new_dialog_cal_calendar (view, id_str, NULL, workload_date);
		g_free (id_str);
	}
	cursor_normal (GTK_WIDGET (view));
}
	

/*
 * Callback for the `Edit' button
 */
void
calendar_list_edit_clicked (GtkButton *button)
{
	GtkTreeView *view;
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter iter;

	view = (GtkTreeView *)lookup_widget (	GTK_WIDGET (button),
						"treeview_cal");

	/* Retrieve the current selected object */
	selection = gtk_tree_view_get_selection (view);
	if (gtk_tree_selection_get_selected (selection, &model, &iter) == FALSE)
	{
		/* Nothing selected */
		return;
	}

	popup_edit_window (view, model, &iter);
}


/*
 * Callback for the Edit item in the popup menu
 */
void
calendar_list_edit_menu (GtkMenuItem *item)
{
	GtkWidget *menu, *view;
	GtkButton *bt;

	menu = gtk_widget_get_ancestor (GTK_WIDGET (item), GTK_TYPE_MENU);
	view = (GtkWidget *)g_object_get_data (G_OBJECT (menu), "tree_view");
	bt = (GtkButton *)lookup_widget (view, "button_cal_edit");
	calendar_list_edit_clicked (bt);
}


/*
 * Double-click on a row
 */
static void
view_onRowActivated (	GtkTreeView *view,
			GtkTreePath *path,
			GtkTreeViewColumn *col,
			gpointer userdata)
{
	GtkTreeModel *model;
	GtkTreeIter iter;
	gpointer dialog_type;
	GtkWidget *dialog, *button;
	gint type;


	/* Retrieve the dialog type */
	dialog = lookup_widget (GTK_WIDGET (view), "dialog_cal_list");
	dialog_type = g_object_get_data (G_OBJECT (dialog), "type");

	/* Retrieve the model */
	model = gtk_tree_view_get_model (view);

	if (gtk_tree_model_get_iter (model, &iter, path) == TRUE) {

		/* Calendar list window */
		if (GPOINTER_TO_INT (dialog_type) == 0) {
			popup_edit_window (view, model, &iter);
		}
		else {
			/* Retrieve the details of the selected row */
			gtk_tree_model_get (model, &iter, 2, &type, -1);

			/* If the selected item is a calendar */
			if (type == 0) {
				button = lookup_widget (dialog,
					       		"button_cal_ok");
				calendar_list_ok_clicked (GTK_BUTTON (button));
				gtk_dialog_response (	GTK_DIALOG (dialog),
					       		GTK_RESPONSE_OK);
			}
		}
	}
}


/*
 * COPY/CUT/PASTE
 */

/*
 * Recursively load the calendars and sub-folders details from the database
 *
 * Return:
 *   0 --> No error (*lst contains the items details)
 *  -1 --> Database error.  An error message has been displayed
 */
static gint
build_calendar_list_recursive (	int workload_date, GSList **lst,
				const gchar *parent_id)
{
	lwc_LL *sql_result;
	char **row;
	calendar_t *cal;

	g_assert (lst != NULL && parent_id != NULL);

	/*
	 * Load all the children (one level only) of this folder
	 * (ie. calendars and sub-folders) from the database
	 */
	if (sql_cal_get_parent (workload_date, parent_id, &sql_result,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error")) != 0)
	{
		return -1;
	}

	/* Store the loaded items in the list */
	lwc_rewindLL (sql_result);
	while ((row = (char **)lwc_nextLL (sql_result)) != NULL) {
		cal = new_calendar (	row[0],	/* ID */ 
					row[1],	/* Parent ID */
					row[3],	/* Type: 0=Calendar 1=Folder */
					row[2],	/* Name */
					row[4],	/* Description */
					row[5]);	/* Formula */
		*lst = g_slist_prepend (*lst, cal);
	}

	/* Call this function recursively on all the sub-folders */
	lwc_rewindLL (sql_result);
	while ((row = (char **)lwc_nextLL (sql_result)) != NULL) {
		if (	   row[3] != NULL
			&& row[3][0] != '\0'
			&& row[3][0] != '0'
			&& build_calendar_list_recursive (workload_date, lst,
							row[0]) != 0)
		{
			lwc_delLL (	sql_result,
					(void (*)(const void *)) sql_free_row);
			return -1;
		}
	}
	lwc_delLL (sql_result, (void (*)(const void *)) sql_free_row);
	return 0;
}


/*
 * Load the calendar details from the database.  If `id' is the ID of a
 * calendar, the details of this calendar are stored in the list which is
 * then a singled-element list.  If `id' is the ID of a folder, the list
 * contains its details and the details of all its sub-folders and calendars.
 *
 * Return:
 *   0 --> No error.  *lst is set and must be freed by the caller with
 *         something like that:
 *               g_slist_foreach (*lst, remove_calendar_from_list, NULL);
 *               g_slist_free (*lst);
 *  -1 --> Database error.  An error message has been displayed
 */
static gint
build_calendar_list (int workload_date, GSList **lst, const gchar *id)
{
	lwc_LL *sql_result;
	char **row;
	calendar_t *cal;

	g_assert (lst != NULL && id != NULL);

	/* First, get the calendar/folder details from the database */
	if (sql_cal_get_main (	workload_date, id, &sql_result,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error")) != 0)
	{
		return -1;
	}

	/* Store the details of the first item in the empty list */
	*lst = NULL;
	lwc_rewindLL (sql_result);
	row = (char **) lwc_nextLL (sql_result);
	if (row == NULL) {
		return 0;
	}
	cal = new_calendar (	row[0],		/* ID */ 
				row[1],		/* Parent ID */
				row[3],		/* Type: 0=Calendar 1=Folder */
				row[2],		/* Name */
				row[4],		/* Description */
				row[5]);	/* Formula */
	*lst = g_slist_prepend (NULL, cal);

	/*
	 * If the item is a folder, recursively add all its sub-folders and
	 * calendars
	 */ 
	if (	   cal->type == 1
		&& build_calendar_list_recursive (	workload_date,
							lst, row[0]) != 0)
	{
		lwc_delLL (sql_result, (void (*)(const void *)) sql_free_row);
		destroy_calendar_list (*lst);
		*lst = NULL;
		return -1;

	}
	lwc_delLL (sql_result, (void (*)(const void *)) sql_free_row);

	/*
	 * For speed, g_slist_prepend has been used to add the items
	 * to the list.  So the list need to be reversed
	 */
	*lst = g_slist_reverse (*lst);
	return 0;
}


/*
 * Mark the days corresponding to the formula provided in value
 */
static void
draw_calendar (const char *value, void *data)
{
	calendar_canvas_t *cal_canvas = data;

	switch (set_calendar (cal_canvas, value, NULL)) {

		case CAL_NOERROR:
			break;
		case CAL_MALLOC:
			error_window (NULL, _("Memory allocation error"));
			break;
		case CAL_EMPTYFIELD:
			error_window (	_("Calendar syntax error"),
					_("A required field is empty"));
			break;
		case CAL_BADMONTHNAME:
			error_window (	_("Calendar syntax error"),
					_("Invalid month name"));
			break;
		case CAL_BADDAYNAME:
			error_window (	_("Calendar syntax error"),
					_("Invalid day name"));
			break;
		case CAL_MONTHOOR:
			error_window (	_("Calendar syntax error"),
				_("Month number must be between 1 and 12"));
			break;
		case CAL_DAYOOR:
			error_window (	_("Calendar syntax error"),
				_("Day number must be between 1 and 31"));
			break;
		case CAL_BADOPTION:
			error_window (	_("Calendar syntax error"),
					_("Invalid option"));
			break;
		default:
			error_window (	_("Calendar syntax error"),
					_("Syntax error"));
		break;
	}
}


/*
 * Selection change callback
 */
static void
tree_selection_changed_cb (GtkTreeSelection *selection, gpointer data)
{
	GtkWidget *widget = data;
	GtkTreeIter iter;
	GtkWidget *edit, *ref, *canvas;
	GtkWidget *dialog, *ok_button;
	GtkTreeModel *model;
	guint64 id;
	gint type;
	gchar *id_str;
	calendar_canvas_t *cal_canvas;
	gpointer dialog_type;
	int workload_date;

	/* Retrieve the dialog type */
	dialog = lookup_widget (widget, "dialog_cal_list");
	dialog_type = g_object_get_data (G_OBJECT(dialog), "type");

	/* Retrieve the workload date */
	workload_date = (int) GPOINTER_TO_INT (g_object_get_data (
				G_OBJECT (dialog), "workload_date"));

	/* Retrieve the required objects */
	edit = lookup_widget (widget, "button_cal_edit");
	ref = lookup_widget (widget, "button_cal_where_used");
	canvas = lookup_widget (widget, "canvas_cal");
	cal_canvas = (calendar_canvas_t *)g_object_get_data (G_OBJECT (canvas),
								"calendar");

	/* Clear all the marks from the calendar canvas */
	clear_calendar (cal_canvas);

	/* If a row is selected */
	if (gtk_tree_selection_get_selected (selection, &model, &iter) == TRUE)
	{
		/* Retrieve the details of the selected row */
		gtk_tree_model_get (model, &iter, 1, &id, 2, &type, -1);

		switch (type) {
			/* If the selected item is a calendar */
			case 0:
				/*
				 * Retrieve the formula from the database and
				 * update the calendar canvas accordingly
				 */
				cursor_busy (ref);
				id_str = schedwi_ulltostr (id);
				sql_cal_get_details (	workload_date,
							id_str,
							NULL, NULL, NULL,
							NULL, NULL,
							draw_calendar,
							cal_canvas,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
							_("Database error"));
				g_free (id_str);
				cursor_normal (ref);

				/* If it is a choose dialog */
				if (GPOINTER_TO_INT (dialog_type) != 0) {
					/* Enable the OK button */
					ok_button = lookup_widget (widget,
							"button_cal_ok");
					gtk_widget_set_sensitive (ok_button,
									TRUE);
					break;
				}

				/* No break.  Continue to enable buttons */

			/* If the selected item is a folder */
			case 1:

				gtk_widget_set_sensitive (edit, TRUE);
				gtk_widget_set_sensitive (ref, TRUE);

				/* If it is a choose dialog */
				if (GPOINTER_TO_INT (dialog_type) != 0) {
					/* Disable the OK button */
					ok_button = lookup_widget (widget,
							"button_cal_ok");
					gtk_widget_set_sensitive (ok_button,
									FALSE);
				}
				break;

			/* If the selected item is `(Empty)' */
			default:
				gtk_widget_set_sensitive (edit, FALSE);
				gtk_widget_set_sensitive (ref, FALSE);

				/* If it is a choose dialog */
				if (GPOINTER_TO_INT (dialog_type) != 0) {
					/* Disable the OK button */
					ok_button = lookup_widget (widget,
							"button_cal_ok");
					gtk_widget_set_sensitive (ok_button,
									FALSE);
				}
				break;
		}
	}
	else {
		/* Nothing selected */
		gtk_widget_set_sensitive (edit, FALSE);
		gtk_widget_set_sensitive (ref, FALSE);

		/* If it is a choose dialog */
		if (GPOINTER_TO_INT (dialog_type) != 0) {
			/* Disable the OK button */
			ok_button = lookup_widget (widget, "button_cal_ok");
			gtk_widget_set_sensitive (ok_button, FALSE);
		}
	}
}


/*
 * INITIALISATION FUNCTIONS
 */


/*
 * Recursively fill the model
 */
static GtkTreeRowReference *
fill_model_recursive (	GtkTreeStore *store, GSList *lst,
			guint64 parent_id, GtkTreeIter *parent_iter,
			guint64 selected_id,
			guint64 id_to_not_display)
{
	GtkTreeIter iter;
	GSList *item;
	calendar_t *ptr;
	gchar flag_added;
	GtkTreeRowReference *ref = NULL, *ref_tmp;
	GtkTreePath *path;

	flag_added = 0;
	for (item = lst; item != NULL; item = g_slist_next (item)) {
		ptr = (calendar_t *)(item->data);

		/*
		 * Only the items which direct parent is PARENT_ID are
		 * taken into account (and the item to not display is
		 * ignored)
		 */
		if (	   ptr->parent_id == parent_id
			&& ptr->id != id_to_not_display)
		{
			gtk_tree_store_append (store, &iter, parent_iter);
			gtk_tree_store_set (	store, &iter,
						0, ptr->name,
						1, ptr->id,
						2, ptr->type,
						-1);

			/*
			 * If the item is the one that must be selected,
			 * create a new GtkTreeRowReference
			 */
			if (ptr->id == selected_id) {
				path = gtk_tree_model_get_path (
							GTK_TREE_MODEL (store),
							&iter);
				ref = gtk_tree_row_reference_new (
							GTK_TREE_MODEL (store),
							path);
				gtk_tree_path_free (path);
			}

			/*
			 * The item is a folder.  Recursively call this
			 * function
			 */ 
			if (ptr->type == 1) {
				ref_tmp = fill_model_recursive (store, lst,
							ptr->id, &iter,
							selected_id,
							id_to_not_display);
				if (ref_tmp != NULL) {
					gtk_tree_row_reference_free (ref);
					ref = ref_tmp;
				}
			}
			flag_added = 1;
		}
	}

	/* If the folder is empty, add an `(Empty)' tag */
	if (flag_added == 0) {
		gtk_tree_store_append (store, &iter, parent_iter);
		gtk_tree_store_set (	store, &iter,
					0, NULL,
					1, (guint64)0,
					2, 2,
					-1);
	}
	return ref;
}


/*
 * Fill the model
 * selected_id is the item ID for which a GtkTreeRowReference must be created.
 * This is used when the tree is refreshed (destroyed and recreated) to
 * re-select the item that was previously selected before the refresh.
 *
 * Return:
 *    The GtkTreeRowReference corresponding to `selected_id' or
 *    NULL if `selected_id' does not exist in the new tree
 *
 *    The returned GtkTreeRowReference must be freed by the caller using
 *    gtk_tree_row_reference_free()
 */
static GtkTreeRowReference *
fill_model (	GtkTreeStore *store, lwc_LL *cals,
		guint64 selected_id, guint64 id_to_not_display)
{
	char **row;
	GSList *lst = NULL;
	GtkTreeRowReference *ref;

	/* Build a GSList with the result of the SQL result */
	lwc_rewindLL (cals);
	while ((row = (char **) lwc_nextLL (cals)) != NULL) {
		lst = g_slist_prepend (lst, new_calendar (
							row[0],	/* ID */ 
							row[1],	/* Parent ID */
							row[3],	/* Type */
							row[2],	/* Name */
							NULL, NULL));
	}
	lst = g_slist_reverse (lst);

	/* Fill the model */
	ref = fill_model_recursive (	store, lst, 0, NULL,
					selected_id, id_to_not_display);

	/* Destroy the list */
	g_slist_foreach (lst, remove_calendar_from_list, NULL);
	g_slist_free (lst);	
	return ref;
}


/*
 * Retrieve the calendar list from the database and fill the provided
 * GtkTreeStore model.  
 * If ref is not NULL it is set with the GtkTreeRowReference corresponding
 * to SELECTED_ID item ID.  It must be freed by the caller with 
 * gtk_tree_row_reference_free()
 *
 * Return:
 *       0 --> No error (ref is set if not NULL)
 *   other --> SQL error (a message has been displayed to the user)
 */
static int
calendar_retrieve_and_fill_model (	GtkTreeStore *store,
					guint64 selected_id,
					GtkTreeRowReference **ref,
					guint64 id_to_not_display,
					int workload_date)
{
	lwc_LL *cals;
	unsigned int ret;
	GtkTreeRowReference *ref_tmp;

	ret = sql_cal_list (	workload_date, &cals,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error"));
	if (ret != 0) {
		return -1;
	}

	/* Fill the GtkTreeStore model */
	ref_tmp = fill_model (store, cals, selected_id, id_to_not_display);
	if (ref != NULL) {
		*ref = ref_tmp;
	}
	else {
		gtk_tree_row_reference_free (ref_tmp);
	}

	lwc_delLL (cals, (void (*)(const void *)) sql_free_row);
	return 0;
}


/*
 * Cell renderer function
 */
static void
cell_renderer_func (	GtkTreeViewColumn *tree_column,
			GtkCellRenderer *cell,
			GtkTreeModel *tree_model,
			GtkTreeIter *iter,
			gpointer data)
{
	gchar *name;
	gint type;

	gtk_tree_model_get (tree_model, iter, 0, &name, 2, &type, -1);
	if (type == 0 || type == 1) {
		/* Folder or calendar */
		g_object_set (cell, "text", name, NULL);
	}
	else {
		/* Empty folder */
		g_object_set (cell, "markup", _("    <i>(Empty)</i>"), NULL);
	}
	g_free (name);
}


/*
 * Add the column and cell to a GtkTreeView
 */
static void
build_view (GtkTreeView *view)
{
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;
	GtkTreeSelection *select;

	/* Calendar name column */
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_title (column, _("Calendar"));
	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_set_cell_data_func (
				column, renderer,
				cell_renderer_func, NULL, NULL);
	gtk_tree_view_append_column (view, column);

	/*
	 * Selection callbacks
	 */
	select = gtk_tree_view_get_selection (view);
	gtk_tree_selection_set_mode (select, GTK_SELECTION_SINGLE);
	/* Selection change */
	g_signal_connect (	G_OBJECT (select), "changed",
				G_CALLBACK (tree_selection_changed_cb), view);
	/* Double click */
	g_signal_connect(	G_OBJECT (view), "row-activated",
				G_CALLBACK (view_onRowActivated), NULL);
}


/*
 * Build and fill the calendar model
 *
 * Return:
 *       0 --> No error.  model is set and must be freed by the caller by
 *             g_object_unref()
 *   other --> SQL error (a message has been displayed to the user)
 */
int
calendar_list_build_model (GtkTreeStore **model, int workload_date)
{
	GtkTreeStore *store;
	int ret;

	/* Create the GtkTreeStore model */
	store = gtk_tree_store_new (3,
			G_TYPE_STRING,	/* Calendar name */
			G_TYPE_UINT64,	/* Calendar id (not displayed) */
			G_TYPE_INT);	/* Type: folder(1) or calendar(0) */

	/* Retrieve the calendars and fill the model */
	ret = calendar_retrieve_and_fill_model (store, 0, NULL, 0,
						workload_date);
	if (ret < 0) {
		g_object_unref (store);
		return ret;
	}
	*model = store;
	return 0;
}


/*
 * Initialize the calendar list window
 *
 * Return:
 *       0 --> No error
 *   other --> SQL error (a message has been displayed to the user)
 */
static int
dialog_calendar_list_init (GtkWidget *dialog_cal, int workload_date)
{
	GtkWidget *view, *label_year;
	GtkTreeStore *store;
	GtkTreePath *root_path;
	int ret;
	GnomeCanvas *canvas;
	calendar_canvas_t *cal_canvas;
	GDate *current_date;
	gchar *year_str;

	/*
	 * Save the dialog type in the widget
	 */
	g_object_set_data (G_OBJECT (dialog_cal), "type", GINT_TO_POINTER (0));

	gtk_window_set_type_hint (	GTK_WINDOW (dialog_cal),
					GDK_WINDOW_TYPE_HINT_NORMAL);

	/* Save the workload date in the widget */
	g_object_set_data (     G_OBJECT (dialog_cal),
				"workload_date",
				GINT_TO_POINTER (workload_date));

	/*
	 * Initialize the calendar list
	 */

	view = lookup_widget (dialog_cal, "treeview_cal");

	/* Create and fill the GtkTreeStore model */
	ret = calendar_list_build_model (&store, workload_date);
	if (ret < 0) {
		return ret;
	}

	/* Associate the model with the view */
	gtk_tree_view_set_model (GTK_TREE_VIEW (view), GTK_TREE_MODEL (store));
	g_object_unref (store);
	gtk_tree_view_set_search_column (GTK_TREE_VIEW (view), 0);

	/* Add the columns, cells and selection callbacks */
	build_view (GTK_TREE_VIEW (view));

	/* Expand the first level */
	root_path = gtk_tree_path_new_first ();
	gtk_tree_view_expand_to_path (GTK_TREE_VIEW (view), root_path);
	gtk_tree_path_free (root_path);

	/*
	 * Initialize the calendar canvas
	 */

	canvas = (GnomeCanvas *)lookup_widget (dialog_cal, "canvas_cal");
	cal_canvas = new_calendar_canvas (canvas, READONLY);
	current_date = g_date_new ();
	g_date_set_time (current_date, time (NULL));
	draw_year_canvas (cal_canvas, g_date_get_year (current_date));
	/* Change the year label */
	label_year = lookup_widget (dialog_cal, "label_cal_year");
	year_str = g_strdup_printf (	"<big><b>%d</b></big>",
					g_date_get_year (current_date));
	gtk_label_set_markup (GTK_LABEL (label_year), year_str);
	g_free (year_str);
	g_date_free (current_date);
	g_object_set_data (G_OBJECT (canvas), "calendar", cal_canvas);

	return 0;
}


/*
 * Create a calendar list dialog
 *
 * Return:
 *   The new GtkWidget (`show'ed by this function) or
 *   NULL in case of error (an error message has been displayed for the user)
 */
GtkWidget *
new_dialog_calendar_list (	void (*hide_cb)(gpointer), gpointer user_data,
				int workload_date)
{
	GtkWidget *widget;

	widget = create_dialog_cal_list ();
	if (dialog_calendar_list_init (widget, workload_date) != 0) {
		gtk_widget_destroy (widget);
		return NULL;
	}
	else {
		g_object_set_data (G_OBJECT (widget), "hide_cb",
						hide_cb);
		g_object_set_data (G_OBJECT (widget), "hide_cb_data",
						user_data);
		gtk_widget_show (widget);
		return widget;
	}
}


/*
 * Set or change the workload date
 */
void
dialog_calendar_list_set_workload_date (GtkWidget *dialog_cal_list,
					int workload_date)
{
	g_object_set_data (     G_OBJECT (dialog_cal_list),
				"workload_date",
				GINT_TO_POINTER (workload_date));
	calendar_list_refresh_widget (dialog_cal_list);
}


/*
 * delete-event callback (when the user closes the window)
 */
gboolean
calendar_list_delete_event (GtkWidget *widget)
{
	GtkWidget *window, *canvas;
	void (*hide_cb)(gpointer);
	gpointer type, user_data;
	calendar_canvas_t *cal_canvas;

	window = lookup_widget (widget, "dialog_cal_list");
	type = g_object_get_data (G_OBJECT(window), "type");

	/* Calendar list window */
	if (GPOINTER_TO_INT (type) == 0) {
		/*
		 * Call the hide_cb function provided when the
		 * new_dialog_calendar_list() was called
		 */
		hide_cb = (void (*)(gpointer))g_object_get_data (
							G_OBJECT(window),
							"hide_cb");
		user_data = g_object_get_data (	G_OBJECT (window),
						"hide_cb_data");
		if (hide_cb != NULL) {
			hide_cb (user_data);
		}
		else {
			/* Hide the Calendar list window */
			gtk_widget_hide (window);
		}

		/*
		 * Return TRUE to stop other handlers from being invoked for
		 * the event. Otherwise, the window will be destroyed
		 */
		return TRUE;
	}
	else {
		/* Retrieve and free the calendar_canvas_t object */
		canvas = lookup_widget (window, "canvas_cal");
		cal_canvas = (calendar_canvas_t *)g_object_get_data (
							G_OBJECT (canvas),
							"calendar");
		destroy_calendar_canvas (cal_canvas);
	}
	return FALSE;
}


/*
 * CLOSE button callback
 */
void
calendar_list_close_clicked (GtkButton *button)
{
	GtkWidget *window, *canvas;
	void (*hide_cb)(gpointer);
	gpointer type, user_data;
	calendar_canvas_t *cal_canvas;

	window = lookup_widget (GTK_WIDGET (button), "dialog_cal_list");
	type = g_object_get_data (G_OBJECT(window), "type");

	/* Calendar list window */
	if (GPOINTER_TO_INT (type) == 0) {
		/*
		 * Call the hide_cb function provided when the
		 * new_dialog_calendar_list() was called
		 */
		hide_cb = (void (*)(gpointer))g_object_get_data (
							G_OBJECT(window),
							"hide_cb");
		user_data = g_object_get_data (	G_OBJECT (window),
						"hide_cb_data");
		if (hide_cb != NULL) {
			hide_cb (user_data);
		}
		else {
			/* Hide the Calendar list window */
			gtk_widget_hide (window);
		}
	}
	else {
		/* Retrieve and free the calendar_canvas_t object */
		canvas = lookup_widget (window, "canvas_cal");
		cal_canvas = (calendar_canvas_t *)g_object_get_data (
							G_OBJECT (canvas),
							"calendar");
		destroy_calendar_canvas (cal_canvas);
	}
}


/*
 * OK button callback
 */
void
calendar_list_ok_clicked (GtkButton *button)
{
	GtkWidget *canvas;
	calendar_canvas_t *cal_canvas;

	/* Retrieve and free the calendar_canvas_t object */
	canvas = lookup_widget (GTK_WIDGET (button), "canvas_cal");
	cal_canvas = (calendar_canvas_t *)g_object_get_data (G_OBJECT (canvas),
								"calendar");
	destroy_calendar_canvas (cal_canvas);
}


/*
 * Build a calendar path.  This function is used by g_slist_foreach() in
 * calendar_list_get_selected().
 */
static void
calendar_list_concat_path (gpointer data, gpointer user_data)
{
	gchar *calendar_name = (gchar *)data;
	gchar **s = (gchar **)user_data;
	gchar *tmp;

	tmp = g_strconcat (*s, "/", calendar_name, NULL);
	g_free (*s);
	*s = tmp;

	/*
	 * The calendar name is also freed from the list.  After the
	 * g_slist_foreach() which calls this function, the list cannot be
	 * used anymore and must be freed (by g_slist_free())
	 */
	g_free (calendar_name);
}


/*
 * Build the path string for the item provided by the GtkTreeIter object
 *
 * Return:
 *   The path (to be freed by the caller by g_free())
 */
static gchar *
calendar_list_build_path (GtkTreeModel *model, GtkTreeIter *iter)
{
	GSList *l = NULL;
	gchar *calendar_name, *s;
	GtkTreeIter child;

	/* Build the path in a list */
	do {
		gtk_tree_model_get (model, iter, 0, &calendar_name, -1);
		l = g_slist_prepend (l, calendar_name);
		child = *iter;
	} while (gtk_tree_model_iter_parent (model, iter, &child) == TRUE);

	/* Convert the list to a string */
	s = g_strdup ("");
	g_slist_foreach (l, calendar_list_concat_path, &s);
	g_slist_free (l);
	return s;
}


/*
 * Get the selected calendar
 *
 * Return:
 *   TRUE --> id and path are set (if no NULL) to the selected calendar ID and
 *            path.  id and path must be freed by the caller by g_free().
 *  FALSE --> There is no calendar selected
 */
gboolean
calendar_list_get_selected (GtkWidget *dialog, gchar **id, gchar **path)
{
	GtkWidget *view;
	GtkTreeModel *model;
	GtkTreeSelection *selection;
	GtkTreeIter iter;
	guint64 calendar_id;

	/* Retrieve the current selected object */
	view = lookup_widget (dialog, "treeview_cal");
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
	if (gtk_tree_selection_get_selected (selection, &model, &iter) != TRUE)
	{
		return FALSE;
	}

	/* Retrieve the calendar ID */
	if (id != NULL) {
		gtk_tree_model_get (model, &iter, 1, &calendar_id, -1);
		*id = schedwi_ulltostr (calendar_id);
	}

	/* Build the path */
	if (path != NULL) {
		*path = calendar_list_build_path (model, &iter);
	}

	return TRUE;
}


/*
 * Exchange structure for gtk_tree_model_foreach() in calendar_list_get_path()
 */
struct _calendar_list_get_path_str {
	GtkTreeIter iter;
	gboolean found;
	guint64 id;
};


/*
 * Comparison function called by gtk_tree_model_foreach() in
 * calendar_list_get_path()
 *
 * Return:
 *   TRUE --> Calendar ID matches
 *   FALSE --> Calendar ID does not match
 */
static gboolean
calendar_list_search_id (	GtkTreeModel *model,
				GtkTreePath *path,
				GtkTreeIter *iter,
				gpointer data)
{
	struct _calendar_list_get_path_str *ptr =
			(struct _calendar_list_get_path_str *) data;
	guint64 calendar_id;

	gtk_tree_model_get (model, iter, 1, &calendar_id, -1);
	if (calendar_id == ptr->id) {
		ptr->iter = *iter;
		ptr->found = TRUE;
		return TRUE;
	}
	return FALSE;
}


/*
 * Get the calendar path for the provided calendar ID
 *
 * Return:
 *   The path (to be freed by the caller by g_free()) or NULL if not found 
 */
gchar *
calendar_list_get_path (GtkTreeStore *model, const gchar *id)
{
	gchar *path;
	struct _calendar_list_get_path_str *ptr;

	/* Build the exchange structure */
	ptr = g_new (struct _calendar_list_get_path_str, 1);
	ptr->id = g_ascii_strtoull (id, NULL, 0);
	ptr->found = FALSE;

	/* Search for the calendar ID */
	gtk_tree_model_foreach (GTK_TREE_MODEL (model),
				calendar_list_search_id, ptr);

	/* Not found */
	if (ptr->found == FALSE) {
		g_free (ptr);
		return NULL;
	}

	/* Build the path */
	path = calendar_list_build_path (GTK_TREE_MODEL (model), &(ptr->iter));
	g_free (ptr);
	return path;
}


/*
 * Callback from the `View previous year button'
 */
void
calendar_list_year_prev_clicked (GtkButton *button)
{
	GtkWidget *canvas, *label_year;
	gchar *year_str;
	calendar_canvas_t *cal_canvas;

	canvas = lookup_widget (GTK_WIDGET (button), "canvas_cal");
	cal_canvas = (calendar_canvas_t *)g_object_get_data (G_OBJECT (canvas),
							"calendar");

	if (cal_canvas->year <= MIN_YEAR) {
		return;
	}

	/* Draw the days for the new year */
	draw_year_canvas (cal_canvas, cal_canvas->year - 1);

	/* Change the year label */
	label_year = lookup_widget (GTK_WIDGET (button), "label_cal_year");
	year_str = g_strdup_printf ("<big><b>%d</b></big>", cal_canvas->year);
	gtk_label_set_markup (GTK_LABEL (label_year), year_str);
	g_free (year_str);
}


/*
 * Callback from the `View next year button'
 */
void
calendar_list_year_next_clicked (GtkButton *button)
{
	GtkWidget *canvas, *label_year;
	gchar *year_str;
	calendar_canvas_t *cal_canvas;

	canvas = lookup_widget (GTK_WIDGET (button), "canvas_cal");
	cal_canvas = (calendar_canvas_t *)g_object_get_data (G_OBJECT (canvas),
							"calendar");

	/* Draw the days for the new year */
	draw_year_canvas (cal_canvas, cal_canvas->year + 1);

	/* Change the year label */
	label_year = lookup_widget (GTK_WIDGET (button), "label_cal_year");
	year_str = g_strdup_printf ("<big><b>%d</b></big>", cal_canvas->year);
	gtk_label_set_markup (GTK_LABEL (label_year), year_str);
	g_free (year_str);
}



/*
 * REFRESH
 */


/*
 * Structure used to pass data to the for `foreach' functions
 *  gtk_tree_view_map_expanded_rows() and gtk_tree_model_foreach()
 */
struct expanded_items {
	GArray *array;
	GtkTreeModel *model;
	GtkTreeView *view;
};


/*
 * Compare two guint64.  This function is used by g_array_sort to sort the
 * array of expanded items in the view
 */
static gint
compare_guint64 (gconstpointer a, gconstpointer b)
{
	guint64 i = *((guint64 *)a), j = *((guint64 *)b);

	return (gint)(i - j);
}


/*
 * Function called by gtk_tree_view_map_expanded_rows() to build the
 * array of expanded item ID
 */
static void
build_array_expanded (GtkTreeView *tree_view, GtkTreePath *path, gpointer data)
{
	struct expanded_items *ptr = data;
	GtkTreeIter iter;
	guint64 id;

	gtk_tree_model_get_iter (ptr->model, &iter, path);
	gtk_tree_model_get (ptr->model, &iter, 1, &id, -1);
	ptr->array = g_array_append_val (ptr->array, id);
}


/*
 * Function called by gtk_tree_model_foreach() to expand the items which IDs
 * are in the array of items previously expanded before the refresh
 */
static gboolean
expand_item_in_array (	GtkTreeModel *model,
			GtkTreePath *path,
			GtkTreeIter *iter,
			gpointer data)
{
	struct expanded_items *ptr = data;
	guint64 id, tmp_id;
	guint i;

	gtk_tree_model_get (model, iter, 1, &id, -1);

	i = 0;
	do {
		tmp_id = g_array_index (ptr->array, guint64, i);
		i++;
	} while (tmp_id != 0 && tmp_id < id);

	if (tmp_id == id) {
		gtk_tree_view_expand_to_path (ptr->view, path);	
	}
	return FALSE;
}


/*
 * Refresh the calendar list
 */
void
calendar_list_refresh (GtkTreeView *view)
{
	GtkTreeModel *model;
	GtkTreeSelection *selection;
	GtkTreeIter iter;
	guint64 selected_id, cut_id;
	GtkTreeRowReference *ref = NULL;
	GtkTreePath *path;
	struct expanded_items expanded;
	gpointer int_ptr;
	GSList *lst;
	int workload_date;

	cursor_busy (GTK_WIDGET (view));

	/* Retrieve the current selected object */
	selection = gtk_tree_view_get_selection (view);
	if (gtk_tree_selection_get_selected (selection, &model, &iter) == TRUE)
	{
		gtk_tree_model_get (model, &iter, 1, &selected_id, -1);
	}
	else {
		selected_id = 0;
	}

	/* Retrieve the expanded rows */
	expanded.array = g_array_new (TRUE, FALSE, sizeof (guint64));
	expanded.model = model;
	expanded.view = view;
	gtk_tree_view_map_expanded_rows (	view, build_array_expanded,
						&expanded);
	g_array_sort (expanded.array, compare_guint64);

	/*
	 * Retrieve the ID of the item in the cut buffer (as it must not
	 * be displayed after the refresh)
	 */
	lst = g_object_get_data (G_OBJECT (view), "copied_cal");
	int_ptr = g_object_get_data (G_OBJECT (view), "copied_op");
	if (lst != NULL && int_ptr != NULL && GPOINTER_TO_INT (int_ptr) == 1) {
		cut_id = ((calendar_t *)(lst->data))->id;
	}
	else {
		cut_id = 0;
	}

	/* Clear the model */
	gtk_tree_store_clear (GTK_TREE_STORE (model));

	/* Retrieve the workload date */
	workload_date = (int) GPOINTER_TO_INT (g_object_get_data (
				G_OBJECT (lookup_widget (GTK_WIDGET (view),
							"dialog_cal_list")),
				"workload_date"));

	/* Retrieve the calendar list and fill the model */
	calendar_retrieve_and_fill_model (	GTK_TREE_STORE (model),
						selected_id, &ref, cut_id,
						workload_date);

	/* Expand the previously expanded items */
	gtk_tree_model_foreach (model, expand_item_in_array, &expanded);
	g_array_free (expanded.array, TRUE);

	if (gtk_tree_row_reference_valid (ref) == TRUE) {
		/* Select the item */
		path = gtk_tree_row_reference_get_path (ref);
		gtk_tree_selection_select_path (selection, path);
		gtk_tree_path_free (path);
	}
	else {
		/* Expand the first level */
		path = gtk_tree_path_new_first ();
		gtk_tree_view_expand_to_path (view, path);
		gtk_tree_path_free (path);
	}
	gtk_tree_row_reference_free (ref);
	cursor_normal (GTK_WIDGET (view));
}


/*
 * Callback of the `Refresh' button
 */
void
calendar_list_refresh_clicked (GtkButton *button)
{
	GtkWidget *view;

	view = lookup_widget (GTK_WIDGET (button), "treeview_cal");
	calendar_list_refresh (GTK_TREE_VIEW (view));
}


/*
 * Refresh the list
 */
void
calendar_list_refresh_widget (GtkWidget *widget)
{
	GtkWidget *view;

	view = lookup_widget (widget, "treeview_cal");
	calendar_list_refresh (GTK_TREE_VIEW (view));
}


/*
 * POPUP MENU
 */

struct view_and_path {
	GtkTreeView *view;
	GtkTreePath *path;
};


/*
 * Menu positioning function used when the popup menu is displayed by
 * Shift+F10
 */
static void
menu_position (	GtkMenu *menu, gint *x, gint *y,
		gboolean *push_in, gpointer user_data)
{
	struct view_and_path *ptr = user_data;
	GdkRectangle rect;

	gtk_tree_view_get_cell_area (ptr->view, ptr->path, NULL, &rect);
	gdk_window_get_origin ((GTK_WIDGET (ptr->view))->window, x, y);
	*x += 10;
	*y += rect.y + rect.height + 1;
	*push_in = TRUE;
}


/*
 * Popup the calendar menu
 */
static GtkWidget *
view_popup_menu_calendar (	GdkEventButton *event,
				GtkTreeView *view, GtkTreePath *path)
{
	GtkWidget *menu;
	struct view_and_path ptr;

	menu = create_menu_cal_list ();
	gtk_widget_show_all (menu);

	/*
	 * Store the GtkTreeView widget in the menu.  This will be
	 * used by the callbacks to access the view
	 */
	g_object_set_data (G_OBJECT (menu), "tree_view", view);

	if (view != NULL && path != NULL) {
		ptr.view = view;
		ptr.path = path;
	}
	/* Note: event can be NULL here when called from view_onPopupMenu;
	 *  gdk_event_get_time() accepts a NULL argument
	 */
	gtk_menu_popup (GTK_MENU (menu), NULL, NULL,
			(view != NULL && path != NULL)? menu_position: NULL,
			&ptr,
			(event != NULL) ? event->button : 0,
			gdk_event_get_time((GdkEvent*)event));
	return menu;
}


/*
 * Callback for the "button-press-event" event.  The clicked item
 * is retrieved and selected and the corresponding popup menu is displayed
 * (calendar or folder menu)
 *
 * Return:
 *    TRUE --> The event has been taken into account
 *   FALSE --> Event no handled (to propagate the event further)
 */
gboolean
calendar_on_treeview_cal_button_press_event (	GtkWidget *treeview,
						GdkEventButton *event)
{
        GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter iter;
	guint64 id;
	gint type;
	GtkTreePath *path;
	GtkWidget *menu, *dialog;
	GtkWidget *edit, *ref;
	gpointer dialog_type;

	/* Retrieve the dialog type */
	dialog = lookup_widget (treeview, "dialog_cal_list");
	dialog_type = g_object_get_data (G_OBJECT (dialog), "type");

	/* List dialog and single click with the right mouse button? */
	if (	   GPOINTER_TO_INT (dialog_type) != 0
		|| event->type != GDK_BUTTON_PRESS
		||  event->button != 3)
       	{
		return FALSE;
	}

	/* Get the path of the item that was clicked */
	if (gtk_tree_view_get_path_at_pos (	GTK_TREE_VIEW (treeview),
						(gint) event->x,
						(gint) event->y,
						&path,
						NULL, NULL, NULL) == FALSE)
	{
		return FALSE;
	}

	/* Select this row */
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
	gtk_tree_selection_select_path (selection, path);

	/* Retrieve the row */
	model = gtk_tree_view_get_model (GTK_TREE_VIEW (treeview));
	gtk_tree_model_get_iter (model, &iter, path);
	gtk_tree_path_free (path);
	gtk_tree_model_get (model, &iter, 1, &id, 2, &type, -1);

	/* Popup the menu */
	menu = view_popup_menu_calendar (event, GTK_TREE_VIEW (treeview), NULL);

	/* Retrieve the item widgets */
	edit = lookup_widget (menu, "cal_menu_edit");
	ref = lookup_widget (menu, "cal_menu_references");

	/* Set the sensitive flags of the menu items */
	if (type == 0 || type == 1) {
		gtk_widget_set_sensitive (edit, TRUE);
		gtk_widget_set_sensitive (ref, TRUE);
	}
	else {
		gtk_widget_set_sensitive (edit, FALSE);
		gtk_widget_set_sensitive (ref, FALSE);
	}

	return TRUE;
}


/*
 * Callback for the "popup-menu" event (Shift-F10).
 *
 * Return:
 *    TRUE --> A popup menu has been displayed
 *   FALSE --> No popup menu displayed because no item is selected or
 *             an "(Empty)" item is selected or the dialog is a choose dialog
 */
gboolean
calendar_on_treeview_cal_popup_menu (GtkWidget *treeview)
{
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter iter;
	GtkTreePath *path;
	guint64 id;
	gint type;
	GtkWidget *menu, *dialog;
	GtkWidget *edit, *ref;
	gpointer dialog_type;

	/* Retrieve the dialog type */
	dialog = lookup_widget (treeview, "dialog_cal_list");
	dialog_type = g_object_get_data (G_OBJECT (dialog), "type");

	/* Do nothing if it is not a list dialog */
	if (GPOINTER_TO_INT (dialog_type) != 0) {
		return FALSE;
	}

	/* Retrieve the selected item */
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
	if (gtk_tree_selection_get_selected (selection, &model, &iter) == FALSE)
	{
		return FALSE;
	}
	gtk_tree_model_get (model, &iter, 1, &id, 2, &type, -1);

	/* Popup the menu */
	path = gtk_tree_model_get_path (model, &iter);
	menu = view_popup_menu_calendar (NULL, GTK_TREE_VIEW (treeview), path);
	gtk_tree_path_free (path);

	/* Retrieve the item widgets */
	edit = lookup_widget (menu, "cal_menu_edit");
	ref = lookup_widget (menu, "cal_menu_references");

	/* Set the sensitive flags of the menu items */
	if (type == 0 || type == 1) {
		gtk_widget_set_sensitive (edit, TRUE);
		gtk_widget_set_sensitive (ref, TRUE);
	}
	else {
		gtk_widget_set_sensitive (edit, FALSE);
		gtk_widget_set_sensitive (ref, FALSE);
	}

	return TRUE;
}

/*
 * Callback for the `References' button
 */
void
calendar_list_where_used_clicked (GtkButton *button)
{
	GtkWidget *view;
	GtkTreeSelection *selection;
	GtkTreeIter iter;
	GtkTreeModel *model;
	guint64 id;
	gint type;
	gchar *id_str, *name;
	GSList *lst;
	gchar **id_array;
	guint len, i;
	calendar_t *ptr;
	int ret;
	lwc_LL *list_job;
	int workload_date;

	view = lookup_widget (GTK_WIDGET (button), "treeview_cal");
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));

	/* Nothing to do if nothing is selected */
	if (gtk_tree_selection_get_selected (selection, &model, &iter) == FALSE)
	{
		return;
	}

	/* Check the type of the selected item */
	gtk_tree_model_get (model, &iter, 0, &name, 1, &id, 2, &type, -1);
	if (type != 0 && type != 1) {
		/* An "(Empty)" item is selected */
		g_free (name);
		return;
	}

	/* Retrieve the workload date */
	workload_date = (int) GPOINTER_TO_INT (g_object_get_data (
				G_OBJECT (lookup_widget (GTK_WIDGET (button),
							"dialog_cal_list")),
				"workload_date"));

	/* Build the list of the selected item and all its children */
	cursor_busy (view);
	id_str = schedwi_ulltostr (id);
	if (build_calendar_list (workload_date, &lst, id_str) != 0) {
		g_free (id_str);
		g_free (name);
		cursor_normal (view);
		return;
	}
	g_free (id_str);

	/* Build the array of IDs */
	len = g_slist_length (lst);
	id_array = (gchar **) g_malloc (sizeof (gchar *) * len);
	for (i = 0; i < len; i++) {
		ptr = (calendar_t *) g_slist_nth_data (lst, i);
		id_array[i] = schedwi_ulltostr (ptr->id);
	}
	g_slist_foreach (lst, remove_calendar_from_list, NULL);
	g_slist_free (lst);	

	/* Retrieve the jobs/jobsets associated with this item */
	ret = sql_cal_job_list (workload_date, id_array, len, &list_job,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error"));
	for (i = 0; i < len; i++) {
		g_free (id_array[i]);
	}
	g_free (id_array);

	if (ret != 0) {
		g_free (name);
		cursor_normal (view);
		return;
	}

	/*
	 * The selected calendar/folder is not associated with any jobs/jobsets
	 */
	if (lwc_getNumNode (list_job) <= 0) {
		info_window (NULL, (type == 0)
			? _("This calendar is not used by any other object")
			: _("This folder is not used by any other object"));
		lwc_delLL (list_job, (void (*)(const void *))
						destroy_referenced_object);
		g_free (name);
		cursor_normal (view);
		return;
	}

	new_dialog_cal_references (name, list_job);
	g_free (name);
	lwc_delLL (list_job, (void (*)(const void *))
						destroy_referenced_object);
	cursor_normal (view);
}


/*
 * Callback for the `References' item in the popup menu
 */
void
calendar_list_where_used_menu (GtkMenuItem *item)
{
	GtkWidget *menu, *view;
	GtkButton *bt;

	menu = gtk_widget_get_ancestor (GTK_WIDGET (item), GTK_TYPE_MENU);
	view = (GtkWidget *)g_object_get_data (G_OBJECT (menu), "tree_view");
	bt = (GtkButton *)lookup_widget (view, "button_cal_where_used");
	calendar_list_where_used_clicked (bt);
}

/*------------------------======= End Of File =======------------------------*/
