/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/*
 * calendar.h -- Manage calendars
 *
 * The calcomp function is used to compile a calendar string to an internal
 * object.
 * The calmatch function check the provided date with compiled calendar
 *
 * The calendar string is inspired by several tools such as calendar(1) or
 * crontab(5).
 * Its format is one of the following:
 *
 *    + month/month_day/option
 *      with month
 *         - An asterisk (*), which stands for all month
 *         - A month name.  The first three characters (case insensitive) are
 *           scanned so the following month names are all valid: Jan, february,
 *           DeCemBer, ocTob
 *         - A month number between 1 and 12
 *         - A range of month.  Ranges are two month name or number separated
 *           with a hyphen.  The specified range is inclusive. For example,
 *           6-10 specifies June, July, August, September and October.  The
 *           range Oct-Fev (or 10-2) specifies October, November, December,
 *           January and February.
 *           Other examples: JAN-MAR (or 1-3 or Jan-3 or 1-march),
 *           April-August, sep-sep
 *         - A list.  A list is a set of month (or ranges) separated by commas.
 *           Examples:
 *               Jan,sep-nov
 *               1,2,3,5-8,december
 *               Jan-4,Nov
 *      with month_day
 *         - An asterisk (*), which stands for all the days of the month
 *         - A day number between 1 and 31
 *         - A negative day number from the end if the month.  -1 stands for
 *           the last day of the month (30, 31, 28 or 29 depending of the
 *           month), -2 for the day before the last day of the month, ...
 *         - A range of days.  Ranges are two day numbers separated
 *           with a hyphen.  The specified range is inclusive. For example,
 *           6-10 specifies the 6th, 7th, 8th, 9th and 10th.  The range
 *           15--1 specifies all the days between the 15th and the last day
 *           of the month included. -1-7 specifies all the days between the
 *           last day of the month and the 7th (therefore from the 1st to the
 *           7th, plus the last day of the month)
 *         - A list.  A list is a set of days (or ranges) separated by commas.
 *           For example: 1-15,17,20--1
 *      with option (optional) used to specify that if a day fall on a
 *      specific week day, it must be moved.  Option is the list of week day
 *      names and increment (or decrement).  Items in the list are separated
 *      by commas.  An item is composed of a week day name followed by + (or
 *      -) and a number of days.
 *      Examples:
 *           Sat+2         All days that fall on Saturday are moved to the next
 *                         Monday (Saturday + 2 days = Monday)
 *           Sat-1         All days that fall on Saturday are moved to the
 *                         previous Friday
 *           Sat+2,Sun+1   All the days that fall the weekend are moved to
 *                         the next Monday
 *      Warning, the option string is parsed from left to right.  For instance,
 *           Sat+1,Sun-2   means that all Saturdays and Sundays will be moved
 *                         to the previous Friday (first, Saturdays are moved
 *                         to Sundays and then Sundays are moved to Fridays)
 *      The correct order for this list is then
 *           Sun-2,Sat+1   All days that fall on Sundays are moved to Fridays
 *                         and all days the fall on Saturdays are moved to the
 *                         next Monday
 *
 *    + month/week_day/option
 *      with month the same as the previous format
 *      with week_day
 *         - A day name.  The first three characters (case insensitive) are
 *           scanned so the following day names are all valid: mon, Friday,
 *           Saturdays, sUnDAy
 *         - A range of days.  Ranges are two day names separated
 *           with a hyphen.  The specified range is inclusive.  For example,
 *           thu-saturday specifies Thusday, Friday and Saturday.  Sun-tue
 *           specifies Sunday, Monday and Tuesday.  Saturday-Tuesday
 *           specifies Saturday, Sunday, Monday and Tuesday.
 *         - A list.  A list is a set of day names (or ranges) separated by
 *           commas.  Example: MON,wed,Fri-sun
 *      with option used to specifies the week in the month (first, second,
 *      last, ...).  If option is not set, all the days of the `week_day'
 *      field applies to all the weeks of the month.  Valid values for
 *      option are: First, Second, Third, Fourth, Fifth, Last, 1, 2, 3, 4, 5,
 *      -1 (for last week of the month), -2, -3, -4, -5.  The option field is
 *      case insensitive.
 *
 *    + month/Open/option
 *      with month the same as the previous formats.
 *      Open is a special keyword which stands for the days between
 *      Monday and Friday (included)
 *      Option used to specifies the day in the month (first, second,
 *      last, ...).  If option is not set, all the open days of the month
 *      are selected.  Valid values for option are: First, Second, Third,
 *      Fourth, Fifth, Last, 1, 2, 3, 4, 5, 6, 7, 8, ..., -1 (for last open
 *      day of the month), -2, -3, -4, -5, ....  The option field is
 *      case insensitive.  For instance 1 (or first) means the first open
 *      day of the month, -1 (or last) stands for the last open day of the
 *      month.
 *
 *    + Easter or Easter+n or Easter-n
 *      for Easter plus or less a number of days. For example, Easter-2 stands
 *      for 2 days before Easter.
 *
 *    + Paskha or Paskha+n or Paskha-n
 *      for Paskha (Orthodox Easter) plus or less a number of days. For
 *      example, Paskha+21 stands for 21 days after Paskha.
 *
 *  Not or ! at the beginning of the calendar string means `Every day except
 *  the ones defined by the following calendar string'
 *  
 *  Full examples:
 *      1/Thur
 *            Every Thursday of January
 *      Jan-March/open
 *            Every open days (Monday to Friday) in January, February and March
 *      1-12/Saturday,Sunday
 *            Every week-end
 *      7/Fri/First
 *            First Friday in July
 *      05/01
 *            1st May
 *      Jan/1/Sat+2,Sun+1
 *            1st January.  If the 1st January occurs a week-end, moves it to
 *            the following Monday (Sat+2 = Monday and Sun+1 = Monday)
 *      Easter+49
 *            49 days after Easter (Pentecost-Whitsunday)
 *      11/Sun/-1
 *            Last Sunday of November
 *      12/open/last
 *            Last open day of the year
 *      Jan/OPEN/1
 *            First open day of the year
 *      2/2,3
 *            2nd and 3rd of February
 *      not 1/sat,sun
 *            Every day except the Saturdays and Sundays in January
 *      ! 1-6/open/last
 *            Every day except the last open days in January to June
 */

#ifndef _CALENDAR_H
#define _CALENDAR_H 1

#include <schedwi.h>

/* Allow the use in C++ code.  */
#ifdef __cplusplus
extern "C" {
#endif


struct cal_str {
	unsigned long int month;
	unsigned long int year[12];
	int cal_year;
};
typedef struct cal_str cal_t;

typedef enum
{
	CAL_MALLOC = -1,   /* Memory allocation error */
	CAL_NOERROR = 0,   /* Success */
	CAL_NOMATCH,	   /* The day doesn't match the calendar (calmatch) */
	CAL_BADMONTHNAME,  /* Unknown month name (must be Jan...Dec) */
	CAL_BADDAYNAME,    /* Unknown day name (must be Sun...Sat) */
	CAL_MONTHOOR,      /* Month number out of range (1...12) */
	CAL_DAYOOR,        /* Day out of range (1...31) */
	CAL_BADOPTION,     /* Unknown option (must be Last, first, -1, ...) */
	CAL_EMPTYFIELD     /* A part is empty (1-6//Last) */
} cal_errcode_t;


extern cal_errcode_t str2cal OF((cal_t *cal, const char *calendar, int year,
					int *error_idx_in_str));
extern cal_errcode_t calcomp OF((cal_t *cal, const char *calendar, int year,
					int *error_idx_in_str));
extern cal_errcode_t calmatch OF((const cal_t *cal, int day, int month,
					int year));
extern int get_last_day OF((int year, int month));
extern int day_of_week OF((int day, int month, int year));

#ifdef __cplusplus
}
#endif  /* C++ */

#endif /* calendar.h */

/*-----------------============== End Of File ==============-----------------*/

