/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/*
 * net_utils_sock.c -- Useful network functions (basic socket connection
 * without SSL)
 */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif

#if HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif

#if HAVE_NETDB_H
#include <netdb.h>
#endif

#if HAVE_SYS_SELECT_H
#include <sys/select.h>
#else
#if HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_TIME_H
#include <time.h>
#endif

#if HAVE_CTYPE_H
#include <ctype.h>
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#ifndef HAVE_RECV
#define recv(a,b,c,d) read(a,b,c)
#endif

#ifndef HAVE_SEND
#define send(a,b,c,d) write(a,b,c)
#endif

#ifndef PF_INET
#define PF_INET AF_INET
#endif

#ifndef HAVE_CLOSESOCKET
#define closesocket(x) close(x)
#endif


#include <pthread.h>

#include <lib_functions.h>
#include <lwc_log.h>
#include <net_utils_sock.h>


#define BUFFER_LEN 1024
#define ENF_OF_REQUEST "\nbYe"

/* Protocol number for TCP */
static int protocol = -1;

/* Lock for the getservbyname function */
static pthread_mutex_t getservbyname_lock = PTHREAD_MUTEX_INITIALIZER;

/* Lock for the gethostby* functions */
static pthread_mutex_t gethostby_lock = PTHREAD_MUTEX_INITIALIZER;

/* Lock for the getprotobyname function */
static pthread_mutex_t getprotobyname_lock = PTHREAD_MUTEX_INITIALIZER;


/*
 * Compare two hostnames
 *
 * Return:
 *   0 --> The two hosts are the same
 *   1 --> The two hosts are not the same
 *  -1 --> Error (a message is displayed using lwc_writeLog())
 */
int
hostname_cmp (const char *h1, const char *h2)
{
	int in_num_dot, i, try_no;
	struct hostent *he;
	struct in_addr in;
#if HAVE_NANOSLEEP
	struct timespec req;

	req.tv_sec = SCHEDWI_HOSTLOOKUP_SLEEP;
	req.tv_nsec = 0;
#endif

	if (h1 == NULL || h2 == NULL) {
		return 1;
	}

	if (schedwi_strcasecmp (h1, h2) == 0) {
		return 0;
	}

	/* If the hostname is in number-dot notation, convert it */
	in_num_dot = inet_aton (h1, &in);

	/* Lock the mutex as we are going to use non thread safe functions */
	pthread_mutex_lock (&gethostby_lock);

	try_no = 0;
	do {
		if (in_num_dot == 0) {
			he = gethostbyname (h1);
		}
		else {
			he = gethostbyaddr (&in, sizeof (in), AF_INET);
		}
		if (he == NULL) {
			if (h_errno == TRY_AGAIN) {
				try_no++;
				if (try_no < SCHEDWI_HOSTLOOKUP_RETRIES) {
#if HAVE_NANOSLEEP
					nanosleep (&req, NULL);
#elif HAVE_SLEEP
					sleep (SCHEDWI_HOSTLOOKUP_SLEEP);
#elif HAVE_USLEEP
					usleep (  SCHEDWI_HOSTLOOKUP_SLEEP
						* 1000000);
#endif
				}
			}
			else {
			if (h_errno == NO_RECOVERY) {
				pthread_mutex_unlock (&gethostby_lock);
				lwc_writeLog (	LOG_ERR,
						_("Name server error"));
				return -1;
			}
			else {
				pthread_mutex_unlock (&gethostby_lock);
				return 1;
			}}
		}
	} while (he == NULL && try_no < SCHEDWI_HOSTLOOKUP_RETRIES);
	if (he == NULL) {
		pthread_mutex_unlock (&gethostby_lock);
		lwc_writeLog (LOG_WARNING, _("Name server unavailable"));
		return -1;
	}

	if (schedwi_strcasecmp (he->h_name, h2) == 0) {
		pthread_mutex_unlock (&gethostby_lock);
		return 0;
	}

	for (i = 0; (he->h_aliases)[i] != NULL; i++) {
		if (schedwi_strcasecmp ((he->h_aliases)[i], h2) == 0) {
			pthread_mutex_unlock (&gethostby_lock);
			return 0;
		}
	}
	pthread_mutex_unlock (&gethostby_lock);
	return 1;
}


/*
 *  Convert a port number/name in an unsigned short int suitable for the
 *  member sin_port of the sockaddr_in structure
 *
 *  Return :
 *    The port number or 0 in case of error
 */
short int
get_port_number (const char *port)
{
	unsigned short int ret;
	int i;
#if HAVE_GETSERVBYNAME
	struct servent *entry;
#endif

#if HAVE_ASSERT_H
	assert (port != NULL);
#endif

	/* Skip the spaces */
	while (*port != '\0' && isspace (*port) != 0) {
		port++;
	}

	/* Empty string ! */
	if (*port == '\0') {
		return 0;
	}

	/* Check if the string contains only digits */
	for (i = 0; port[i] != '\0' && isdigit (port[i]) != 0; i++);

	/* Only digits */
	if (port[i] == '\0') {
		return htons ((unsigned short int) atoi (port));
	}

#if HAVE_GETSERVBYNAME
	/*
	 * The string is a port name.  This name will be searched in
	 * /etc/services (or equivalent)
	 */
	/* Lock the mutex as we are going to use non thread safe functions */
	pthread_mutex_lock (&getservbyname_lock);
	entry = getservbyname (port, "tcp");
	if (entry != NULL) {
		ret = (unsigned short int) entry->s_port;
		pthread_mutex_unlock (&getservbyname_lock);
		return ret;
	}
	pthread_mutex_unlock (&getservbyname_lock);
#endif
	return  0;
}


/*
 * Create a socket ready to accept requests
 *
 * port_number is the port number to listen to (a number or a service name
 *             in /etc/services)
 *       iface is the network interface to listen to (NULL or empty means all
 *             the interfaces)
 *
 * Return:
 *   The socket (to be closed by the caller by net_close_sock()) or
 *   -1 in case of error (an error message has been logged by lwc_writeLog())
 */
int
net_server (const char *port_number, const char *iface)
{
	struct protoent *prot;
	int sock, opt, try_no;
	struct  sockaddr_in sad;
	unsigned long int iface_addr;
	unsigned short int port;
	struct hostent *he;
#if HAVE_NANOSLEEP
	struct timespec req;

	req.tv_sec = SCHEDWI_HOSTLOOKUP_SLEEP;
	req.tv_nsec = 0;
#endif

#if HAVE_ASSERT_H
	assert (port_number != NULL);
#endif

	/* Get the port number */
	port = get_port_number (port_number);
	if (port == 0) {
		lwc_writeLog (LOG_ERR,
		 	_("Cannot get the port number for `%s'"),
			port_number);
		return -1;
	}

	/* Lock the mutex as we are going to use non thread safe functions */
	pthread_mutex_lock (&gethostby_lock);

	/* Get the IP address of the network interface to listen to */
	if (iface == NULL || iface[0] == '\0') {
		iface_addr = htonl (INADDR_ANY);
	}
	else {
		try_no = 0;
		do {
			he = gethostbyname (iface);
			if (he == NULL) {
				if (h_errno == TRY_AGAIN) {
					try_no++;
					if (try_no < SCHEDWI_HOSTLOOKUP_RETRIES)
					{
#if HAVE_NANOSLEEP
						nanosleep (&req, NULL);
#elif HAVE_SLEEP
						sleep (
						    SCHEDWI_HOSTLOOKUP_SLEEP);
#elif HAVE_USLEEP
						usleep (
						    SCHEDWI_HOSTLOOKUP_SLEEP
						  * 1000000);
#endif
					}
				}
				else {
				if (h_errno == NO_RECOVERY) {
					pthread_mutex_unlock (&gethostby_lock);
					lwc_writeLog (	LOG_ERR,
							_("Name server error"));
					return -1;
				}
				else {
					pthread_mutex_unlock (&gethostby_lock);
					lwc_writeLog (	LOG_ERR,
						_("Cannot get address for %s"),
						iface);
					return -1;
				}}
			}
		} while (he == NULL && try_no < SCHEDWI_HOSTLOOKUP_RETRIES);

		if (he == NULL) {
			pthread_mutex_unlock (&gethostby_lock);
			lwc_writeLog (	LOG_WARNING,
					_("Name server unavailable"));
			return -1;
		}

		iface_addr = *((unsigned long *)(he->h_addr_list[0]));
	}
	pthread_mutex_unlock (&gethostby_lock);

	/* Get the protocol number for tcp */
	if (protocol == -1) {
#if HAVE_GETPROTOBYNAME
		/* Lock the mutex as we are going to use getprotobyname */
		pthread_mutex_lock (&getprotobyname_lock);
		prot = getprotobyname ("tcp");
		if (prot == NULL) {
			pthread_mutex_unlock (&getprotobyname_lock);
			lwc_writeLog (LOG_ERR,
		    _("getprotobyname: cannot get protocol number for tcp"));
			return -1;
		}
		protocol = prot->p_proto;
		pthread_mutex_unlock (&getprotobyname_lock);
#else
		protocol = 0;
#endif
	}

	/* Create the socket */
	sock = socket (PF_INET, SOCK_STREAM, protocol);
	if (sock < 0) {
		lwc_writeLog (LOG_ERR, _("socket: %s"), strerror (errno));
		return -1;
	}

	/* Set the REUSEADDR flag */
	opt = -1;
	if (setsockopt(	sock, SOL_SOCKET, SO_REUSEADDR,
			&opt, sizeof (opt)) != 0)
	{
		/* Error but continue anyway */
		lwc_writeLog (LOG_ERR, _("setsockopt: %s"), strerror (errno));
	}

	/* Compose the address */
	schedwi_memset (&sad, 0, sizeof(sad));
        sad.sin_family = AF_INET;
        sad.sin_addr.s_addr = iface_addr;
	sad.sin_port = port;

	/* Bind to the address */
	if (bind (sock, (struct sockaddr *)&sad, sizeof(sad)) != 0) {
		lwc_writeLog (LOG_ERR, _("bind: %s"), strerror (errno));
		closesocket (sock);
		return -1;
	}

	/* Prepare to accept requests */
	if (listen (sock, SCHEDWI_LISTEN_BACKLOG) != 0) {
		lwc_writeLog (LOG_ERR, _("listen: %s"), strerror (errno));
		closesocket (sock);
		return -1;
	}

	return sock;
}


/*
 * From the provided in_addr structure build a list of all the associated
 * names (dot notation, name and aliases from the DNS).
 *
 * Return:
 *   0 --> No error (or may be some DNS availability errors).  names is
 *         set and must be freed by the caller by the destroy_client_names()
 *         function.  names is a NULL terminated array.
 *  -1 --> Error.  An error message has been logged by lwc_writeLog()
 */
int
build_client_names_from_addr (struct in_addr in, char ***names)
{
	char **lst;
	char *addr;
	int try_no;
	struct hostent *he;
	unsigned int i, j;
#if HAVE_NANOSLEEP
        struct timespec req;

        req.tv_sec = SCHEDWI_HOSTLOOKUP_SLEEP;
        req.tv_nsec = 0;
#endif

#if HAVE_ASSERT_H
	assert (names != NULL);
#endif

	/* Lock the mutex as we are going to use non thread safe functions */
	pthread_mutex_lock (&gethostby_lock);

	/* Retrieve the address in dot notation */
	addr = inet_ntoa (in);
	
	try_no = 0;
	do {
		/* Retrieve the name and aliases from the DNS */
		he = gethostbyaddr (&in, sizeof (in), AF_INET);
		if (he == NULL) {
			if (h_errno == TRY_AGAIN) {
				try_no++;
				if (try_no < SCHEDWI_HOSTLOOKUP_RETRIES) {
#if HAVE_NANOSLEEP
					nanosleep (&req, NULL);
#elif HAVE_SLEEP
					sleep (SCHEDWI_HOSTLOOKUP_SLEEP);
#elif HAVE_USLEEP
					usleep (  SCHEDWI_HOSTLOOKUP_SLEEP
						* 1000000);
#endif
				}
			}
			else {
			if (h_errno == NO_RECOVERY) {
				/* Log an error message but continue anyway */
				lwc_writeLog (LOG_ERR, _("Name server error"));
				break;
			}
			else {
				/* Not in DNS, NIS, /etc/hosts */
				break;
			}}
		}
	} while (he == NULL && try_no < SCHEDWI_HOSTLOOKUP_RETRIES);

	/* Failed to get a reply from the DNS.  Log an error but continue */
	if (try_no >= SCHEDWI_HOSTLOOKUP_RETRIES) {
		lwc_writeLog (LOG_WARNING, _("Name server unavailable"));
		he = NULL;
	}

	if (he == NULL) {
		/* Only add the IP adress in dot notation to the array */
		lst = (char **) malloc (2 * sizeof (char *));
	}
	else {
		for (i = 0; (he->h_aliases)[i] != NULL; i++);
		/*
		 * Number of array elements:
		 * 	num aliases + dot notation + DNS name + NULL
		 */
		lst = (char **) malloc ((i + 3) * sizeof (char *));
	}
	if (lst == NULL) {
		pthread_mutex_unlock (&gethostby_lock);
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}

	/* Add the IP address in dot notation in the array */
	i = 0;
	if (addr != NULL && addr[0] != '\0') {
		lst[i] = (char *)malloc (schedwi_strlen (addr) + 1);
		if (lst[i] == NULL) {
			pthread_mutex_unlock (&gethostby_lock);
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			for (j = 0; j < i; j++) {
				free (lst[j]);
			}
			free (lst);
			return -1;
		}
		strcpy (lst[i++], addr);
	}

	if (he != NULL) {
		/* Add the hostname in the array */
		if (he->h_name != NULL && (he->h_name)[0] != '\0') {
			lst[i] = (char *)malloc (schedwi_strlen (he->h_name)
							+ 1);
			if (lst[i] == NULL) {
				pthread_mutex_unlock (&gethostby_lock);
				lwc_writeLog (	LOG_CRIT,
						_("Memory allocation error"));
				for (j = 0; j < i; j++) {
					free (lst[j]);
				}
				free (lst);
				return -1;
			}
			strcpy (lst[i++], he->h_name);
		}

		/* Add all the aliases in the array */
		for (j = 0; (he->h_aliases)[j] != NULL; j++) {
			lst[i] = (char *)malloc (
				schedwi_strlen ((he->h_aliases)[j]) + 1);
			if (lst[i] == NULL) {
				pthread_mutex_unlock (&gethostby_lock);
				lwc_writeLog (	LOG_CRIT,
						_("Memory allocation error"));
				for (j = 0; j < i; j++) {
					free (lst[j]);
				}
				free (lst);
				return -1;
			}
			strcpy (lst[i++], (he->h_aliases)[j]);
		}
	}
	pthread_mutex_unlock (&gethostby_lock);
	lst[i] = NULL;
	*names = lst;
	return 0;
}


/*
 * From the provided name build a list of all the associated
 * names (dot notation, name and aliases from the DNS).
 *
 * Return:
 *   0 --> No error (or may be some DNS availability errors).  names is
 *         set and must be freed by the caller by the destroy_client_names()
 *         function.  names is a NULL terminated array.
 *  -1 --> Error.  An error message has been logged by lwc_writeLog()
 */
int
build_client_names_from_name (const char *host_name, char ***names)
{
	char **lst;
	char *addr;
	int try_no;
	struct hostent *he;
	struct in_addr in;
	unsigned int i, j;
	int in_num_dot;
#if HAVE_NANOSLEEP
        struct timespec req;

        req.tv_sec = SCHEDWI_HOSTLOOKUP_SLEEP;
        req.tv_nsec = 0;
#endif

#if HAVE_ASSERT_H
	assert (host_name != NULL && names != NULL);
#endif

	/* If the hostname is in number-dot notation, convert it */
	in_num_dot = inet_aton (host_name, &in);

	/* Lock the mutex as we are going to use non thread safe functions */
	pthread_mutex_lock (&gethostby_lock);

	try_no = 0;
	do {
		/* Retrieve the name and aliases from the DNS */
		if (in_num_dot == 0) {
			he = gethostbyname (host_name);
		}
		else {
			he = gethostbyaddr (&in, sizeof (in), AF_INET);
		}
		if (he == NULL) {
			if (h_errno == TRY_AGAIN) {
				try_no++;
				if (try_no < SCHEDWI_HOSTLOOKUP_RETRIES) {
#if HAVE_NANOSLEEP
					nanosleep (&req, NULL);
#elif HAVE_SLEEP
					sleep (SCHEDWI_HOSTLOOKUP_SLEEP);
#elif HAVE_USLEEP
					usleep (  SCHEDWI_HOSTLOOKUP_SLEEP
						* 1000000);
#endif
				}
			}
			else {
			if (h_errno == NO_RECOVERY) {
				/* Log an error message but continue anyway */
				lwc_writeLog (LOG_ERR, _("Name server error"));
				break;
			}
			else {
				/* Not in DNS, NIS, /etc/hosts */
				break;
			}}
		}
	} while (he == NULL && try_no < SCHEDWI_HOSTLOOKUP_RETRIES);

	/* Failed to get a reply from the DNS.  Log an error but continue */
	if (try_no >= SCHEDWI_HOSTLOOKUP_RETRIES) {
		lwc_writeLog (LOG_WARNING, _("Name server unavailable"));
		he = NULL;
	}

	if (he == NULL) {
		/* Only add the name to the array */
		lst = (char **) malloc (2 * sizeof (char *));
	}
	else {
		for (i = 0; (he->h_aliases)[i] != NULL; i++);
		for (j = 0; (he->h_addr_list)[j] != NULL; j++);
		/*
		 * Number of array elements:
		 * 	  num aliases + num addresses dot notation
		 * 	+ host name from parameter + host name + NULL
		 */
		lst = (char **) malloc ((i + j + 3) * sizeof (char *));
	}
	if (lst == NULL) {
		pthread_mutex_unlock (&gethostby_lock);
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}

	/* Add the name in the array */
	i = 0;
	lst[i] = (char *)malloc (schedwi_strlen (host_name) + 1);
	if (lst[i] == NULL) {
		pthread_mutex_unlock (&gethostby_lock);
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		for (j = 0; j < i; j++) {
			free (lst[j]);
		}
		free (lst);
		return -1;
	}
	strcpy (lst[i++], host_name);

	if (he != NULL) {
		/* Add the hostname in the array */
		if (he->h_name != NULL && (he->h_name)[0] != '\0') {
			lst[i] = (char *)malloc (schedwi_strlen (he->h_name)
							+ 1);
			if (lst[i] == NULL) {
				pthread_mutex_unlock (&gethostby_lock);
				lwc_writeLog (	LOG_CRIT,
						_("Memory allocation error"));
				for (j = 0; j < i; j++) {
					free (lst[j]);
				}
				free (lst);
				return -1;
			}
			strcpy (lst[i++], he->h_name);
		}

		/* Add all the aliases in the array */
		for (j = 0; (he->h_aliases)[j] != NULL; j++) {
			lst[i] = (char *)malloc (
				schedwi_strlen ((he->h_aliases)[j]) + 1);
			if (lst[i] == NULL) {
				pthread_mutex_unlock (&gethostby_lock);
				lwc_writeLog (	LOG_CRIT,
						_("Memory allocation error"));
				for (j = 0; j < i; j++) {
					free (lst[j]);
				}
				free (lst);
				return -1;
			}
			strcpy (lst[i++], (he->h_aliases)[j]);
		}

		/* Add all the addresses in dot notation */
		for (j = 0; (he->h_addr_list)[j] != NULL; j++) {
			in.s_addr = *((unsigned long *)(he->h_addr_list[j]));
			addr = inet_ntoa (in);
			if (addr != NULL) {
				lst[i] = (char *)malloc (
						schedwi_strlen (addr) + 1);
				if (lst[i] == NULL) {
					pthread_mutex_unlock (&gethostby_lock);
					lwc_writeLog (	LOG_CRIT,
						_("Memory allocation error"));
					for (j = 0; j < i; j++) {
						free (lst[j]);
					}
					free (lst);
					return -1;
				}
				strcpy (lst[i++], addr);
			}
		}
	}
	pthread_mutex_unlock (&gethostby_lock);
	lst[i] = NULL;
	*names = lst;
	return 0;
}


/*
 * Free the provided array
 */
void
destroy_client_names (char **names)
{
	unsigned int i;

	if (names != NULL) {
		for (i = 0; names[i] != NULL; i++) {
			free (names[i]);
		}
		free (names);
	}
}


/*
 * Create a socket connected to a server
 * port_number is the port number to connect to (a number or a service name
 *             in /etc/services)
 *
 * Return:
 *   The socket or
 *   -1 in case of error (an error message is displayed using lwc_writeLog())
 */
int
net_client_sock (const char *port_number, const char *hostname)
{
	struct protoent *prot;
	int sock, try_no;
	struct  sockaddr_in sad;
	unsigned long int iface_addr;
	unsigned short int port;
	struct hostent *he;
#if HAVE_NANOSLEEP
	struct timespec req;

	req.tv_sec = SCHEDWI_HOSTLOOKUP_SLEEP;
	req.tv_nsec = 0;
#endif

#if HAVE_ASSERT_H
	assert (port_number != NULL && hostname != NULL);
#endif

	/* Get the port number */
	port = get_port_number (port_number);
	if (port == 0) {
		lwc_writeLog (LOG_ERR,
		 	_("Cannot get the port number for `%s'"),
			port_number);
		return -1;
	}

	/* Get the IP address of the server */
	try_no = 0;
	do {
		he = gethostbyname (hostname);
		if (he == NULL) {
			if (h_errno == TRY_AGAIN) {
				try_no++;
				if (try_no < SCHEDWI_HOSTLOOKUP_RETRIES)
				{
#if HAVE_NANOSLEEP
					nanosleep (&req, NULL);
#elif HAVE_SLEEP
					sleep (SCHEDWI_HOSTLOOKUP_SLEEP);
#elif HAVE_USLEEP
					usleep (  SCHEDWI_HOSTLOOKUP_SLEEP
						* 1000000);
#endif
				}
			}
			else {
			if (h_errno == NO_RECOVERY) {
				lwc_writeLog (	LOG_ERR,
						_("Name server error"));
				return -1;
			}
			else {
				lwc_writeLog (	LOG_ERR,
					_("Cannot get address for %s"),
					hostname);
				return -1;
			}}
		}
	} while (he == NULL && try_no < SCHEDWI_HOSTLOOKUP_RETRIES);

	if (he == NULL) {
		lwc_writeLog (	LOG_WARNING,
				_("Name server unavailable"));
		return -1;
	}

	iface_addr = *((unsigned long *)(he->h_addr_list[0]));

	/* Get the protocol number for tcp */
	if (protocol == -1) {
#if HAVE_GETPROTOBYNAME
		prot = getprotobyname ("tcp");
		if (prot == NULL) {
			lwc_writeLog (LOG_ERR,
		    _("getprotobyname: cannot get protocol number for tcp"));
			return -1;
		}
		protocol = prot->p_proto;
#else
		protocol = 0;
#endif
	}

	/* Create the socket */
	sock = socket (PF_INET, SOCK_STREAM, protocol);
	if (sock < 0) {
		lwc_writeLog (LOG_ERR, _("socket: %s"), strerror (errno));
		return -1;
	}

	/* Compose the remote address */
	schedwi_memset(&sad, 0, sizeof(sad));
        sad.sin_family = AF_INET;
        sad.sin_addr.s_addr = iface_addr;
	sad.sin_port = port;

	/* Connect to the address */
	if (connect (sock, (struct sockaddr *)&sad, sizeof(sad)) != 0) {
		lwc_writeLog (LOG_ERR, _("Failed to connect to %s: %s"),
				hostname, strerror (errno));
		closesocket (sock);
		return -1;
	}

	return sock;
}


/*
 * Read data from the provided socket handler.
 * The data are store in *buff (of size *len). If buffer is NULL or not big
 * enough, it is reallocated to a bigger size. It must be freed by the caller
 *
 * Return:
 *    The total number of characters read and stored in *buff or
 *    -1 on error (a message is displayed using lwc_writeLog())
 *    -2 if too many data are received (a message is displayed using
 *       lwc_writeLog())
 */
ssize_t
net_read_sock (int sock, char **buff, unsigned int *len)
{
	ssize_t nb_read, total;
	size_t to_read;
	char *b, *tmp;
	fd_set rfds;
	struct timeval tv;
	int ret;

#if HAVE_ASSERT_H
	assert (buff != NULL && len != NULL);
#endif

	if (*buff == NULL || *len == 0) {
		*buff = (char *) malloc (BUFFER_LEN);
		if (*buff == NULL) {
			*len = 0;
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			return -1;
		}
		*len = BUFFER_LEN;
	}

	total = 0;
	to_read = *len;
	b = *buff;
	do {
		do {
			FD_ZERO (&rfds);
			FD_SET (sock, &rfds);
			tv.tv_sec = 5;
			tv.tv_usec = 0;
			ret = select (sock + 1, &rfds, NULL, NULL, &tv);
		} while (ret < 0 && (errno == EINTR || errno == EAGAIN));

		if (ret < 0) {
			lwc_writeLog (LOG_ERR,
					_("Error while reading data: %s"),
					strerror (errno));
			return -1;
		}

		if (ret == 0) {
			lwc_writeLog (LOG_NOTICE,
			_("Timeout while waiting to read network data"));
			return -1;
		}

		do {
			nb_read = recv (sock, b, to_read, 0);
		} while (nb_read < 0 && (errno == EINTR || errno == EAGAIN));

		if (nb_read == 0) {
			return total;
		}

		if (nb_read < 0) {
			lwc_writeLog (	LOG_ERR,
					_("Error while reading data: %s"),
					strerror (errno));
			return -1;
		}

		total += nb_read;

		/* Find the EOF tag */
		tmp = schedwi_strnstr (*buff, ENF_OF_REQUEST, total);
		if (tmp != NULL) {
			return (ssize_t)(tmp - *buff);
		}

		if (nb_read < to_read) {
			to_read -= nb_read;
			b += nb_read;
		}
		else {
			if (*len + BUFFER_LEN > SCHEDWI_MAX_BUFFER_LEN) {
				/* Too many data */
				lwc_writeLog (LOG_NOTICE,
		_("Failed to read the network request: buffer overflow"));
				return -2;
			}

			tmp = (char *) realloc (*buff, *len + BUFFER_LEN);
			if (tmp == NULL) {
				lwc_writeLog (	LOG_CRIT,
						_("Memory allocation error"));
				return -1;
			}
			*buff = tmp;
			*len += BUFFER_LEN;
			b = *buff + total;
			to_read = BUFFER_LEN;
		}
	} while (1);

	return total;
}


/*
 * Write data to the provided socket handler.
 *
 * Return:
 *   0 --> No error
 *  -1 --> Write error (a message is displayed using lwc_writeLog())
 */
int
net_write_sock (int sock, const char *buff, size_t len)
{
	ssize_t nb_write;

	if (buff == NULL || len == 0) {
		return 0;
	}

	do {
		nb_write = send (sock, buff, len, 0);
		if (nb_write == len) {
			return 0;
		}

		if (nb_write < 0) {
			if (errno != EINTR && errno != EAGAIN) {
				lwc_writeLog (LOG_ERR,
					_("Error while sending data: %s"),
					strerror (errno));
				return -1;
			}
		}
		else {
			buff += nb_write;
			len -= nb_write;
		}

	} while (1);

	return 0;
}


/*
 * Close the socket
 */
int
net_close_sock (int sock)
{
	if (sock >= 0) {
#if HAVE_SHUTDOWN
		shutdown (sock, SHUT_RDWR);
#endif
		return closesocket (sock);
	}
	return 0;
}

/*------------------------======= End Of File =======------------------------*/
