/* Schedwi
   Copyright (C) 2007 Herve Quatremain
     
   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/*
 * send_process.c -- Management functions for the process which is responsible
 *                   for the send of the job results to the server
 */

#include <schedwi.h>

#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_TIME_H
#include <time.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_SIGNAL_H
#include <signal.h>
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif


#include <conf_clnt.h>
#include <conf.h>
#include <net_utils_init.h>
#include <lwc_log.h>
#include <signal_utils.h>
#include <lib_functions.h>
#include <send_result.h>
#include <send_process.h>


#if HAVE_PID_T
static pid_t send_process_pid = 0;
#else
static int send_process_pid = 0;
#endif


/*
 * Stop the sending process
 */
void
send_process_stop ()
{
#if HAVE_NANOSLEEP
	struct timespec req;

	req.tv_sec = SCHEDWI_SLEEP_BETWEEN_TERM_KILL;
	req.tv_nsec = 0;
#endif

	if (send_process_pid > 0) {

		kill (send_process_pid, SIGTERM);

#if HAVE_NANOSLEEP
		nanosleep (&req, 0);
#elif HAVE_SLEEP
		sleep (SCHEDWI_SLEEP_BETWEEN_TERM_KILL);
#elif HAVE_USLEEP
		usleep (SCHEDWI_SLEEP_BETWEEN_TERM_KILL * 1000000);
#endif

		kill (send_process_pid, SIGKILL);
		send_process_pid = 0;
	}
}


/*
 * Wake up the sending process (results to send)
 */
void
send_process_wakeup ()
{
	if (send_process_pid > 0) {
		kill (send_process_pid, SIGHUP);
	}
}


/*
 * Signal handler for SIGTERM
 */
static void
sigterm_action (int sig)
{
	conf_destroy_clnt ();
	net_free ();
	exit (0);
}


/*
 * Signal handler for SIGHUP
 */
static void
sighup_action (int sig)
{
	return;
}


/*
 * Start the sending process
 *
 * Return:
 *   0 --> No error (the process is running)
 *  -1 --> Error (a message is logged by lwc_writeLog())
 */
int
send_process_start ()
{
	struct sigaction sa;
	sigset_t mask;
	const char *result_dir, *prefix, *suffix;
	const char *log_file, *date_format;
	long int facility;

#if HAVE_NANOSLEEP
	struct timespec req;

	req.tv_sec = SCHEDWI_SLEEP_BETWEEN_ALARM;
	req.tv_nsec = 0;
#endif

	/* Already running */
	if (send_process_pid > 0) {
		return 0;
	}

	send_process_pid = fork ();
	if (send_process_pid < 0) {
		lwc_writeLog (	LOG_CRIT, _("Cannot create a sub-process: %s"),
				strerror (errno));
		return -1;
	}
	

	/*
	 * Father
	 */
	if (send_process_pid > 0) {
		return 0;
	}


	/*
	 * Child
	 */
	if (	   conf_get_param_string ("RESULT_DIR", &result_dir) != 0
		|| conf_get_param_string ("RESULT_PREFIX", &prefix) != 0
		|| conf_get_param_string ("RESULT_SUFFIX", &suffix) != 0
		|| conf_get_param_string ("LOG_FILE", &log_file) != 0
		|| conf_get_param_syslog_facility ("SYSLOG", &facility) != 0
		|| conf_get_param_string ("DATE_FORMAT", &date_format) != 0)
	{
		/* Internal error. Unknown parameter name or wrong type */
		lwc_writeLog (	LOG_CRIT,
				_("Cannot load configuration parameters"));
		exit (0);
	}

	lwc_newLog (	PACKAGE_NAME,
			(facility != -1) ? 1 : 0,
			(facility != -1) ? (int)facility : 0,
			log_file,
			date_format);

	/* Mask all the signals */
	signal_unmask_all ();
	sigfillset (&mask);
	sigdelset (&mask, SIGKILL);
	sigdelset (&mask, SIGSTOP);
	sigdelset (&mask, SIGCONT);
	sigdelset (&mask, SIGALRM);
	sigdelset (&mask, SIGTERM);
	sigdelset (&mask, SIGHUP);
	sigprocmask (SIG_SETMASK, &mask, 0);

	/* Install the signal handlers */
	schedwi_memset (&sa, 0, sizeof (struct sigaction));
	sa.sa_handler = sigterm_action;
	sigemptyset (&(sa.sa_mask));
	sa.sa_flags = 0;
	sigaction (SIGTERM, &sa, 0);

	schedwi_memset (&sa, 0, sizeof (struct sigaction));
	sa.sa_handler = sighup_action;
	sigemptyset (&(sa.sa_mask));
	sa.sa_flags = 0;
	sigaction (SIGHUP, &sa, 0);

	/* Main loop */
	while (1) {

		signal_mask_all ();
		send_result (result_dir, prefix, suffix);
		signal_unmask ();

#if HAVE_NANOSLEEP
		nanosleep (&req, 0);
#elif HAVE_SLEEP
		sleep (SCHEDWI_SLEEP_BETWEEN_ALARM);
#elif HAVE_USLEEP
		usleep (SCHEDWI_SLEEP_BETWEEN_ALARM * 1000000);
#endif
	}
	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
