/* Schedwi
   Copyright (C) 2007 Herve Quatremain
     
   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* result_mgnt.c -- Read/write job result status from/to file */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_CTYPE_H
#include <ctype.h>
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lwc_linkedlist.h>
#include <lib_functions.h>
#include <utils.h>
#include <result_mgnt.h>


/*
 * Initialize the provided structure
 */
static void
init_result (result_t *r)
{
	if (r != NULL) {
		r->job_id     = NULL;
		r->job_id_len = 0;
		r->duration   = 0;
		r->was_killed = 0;
		r->exit_code  = 0;
	}
}


/*
 * Create a new result structure
 *
 * Return:
 *   The new structure (to free by destroy_result) or
 *   NULL on memory allocation error
 */
result_t *
new_result ()
{
	result_t *r;

	r = (result_t *) malloc (sizeof (result_t));
	if (r != NULL) {
		init_result (r);
	}
	return r;
}


/*
 * Free the provided structure
 */
void
destroy_result (result_t *r)
{
	if (r != NULL) {
		if (r->job_id != NULL) {
			free (r->job_id);
		}
		free (r);
	}
}


/*
 * Copy a result_t structure
 *
 * Return:
 *   0 --> No error (dst->job_id must be freed by the caller by free())
 *  -1 --> Memory allocation error
 */
int
result_copy (const result_t *src, result_t *dst)
{
#if HAVE_ASSERT_H
	assert (src != NULL && dst != NULL);
#endif

	dst ->job_id = (char *) malloc (src->job_id_len + 1);
	if (dst ->job_id == NULL) {
		return -1;
	}
	strncpy (dst->job_id, src->job_id, src->job_id_len);
	(dst->job_id)[src->job_id_len] = '\0';
	dst->job_id_len = src->job_id_len;
	dst->duration   = src->duration;
	dst->was_killed = src->was_killed;
	dst->exit_code  = src->exit_code;

	return 0;
}


/*
 * Write the provided structure to the provided file handler
 *
 * Return:
 *   0 --> No error
 *  -1 --> Write error (errno is probably set)
 */
static int
result_fwrite (const result_t *r, const char *job_id, FILE *f)
{
	if (r == NULL || f == NULL) {
		return 0;
	}
	if (fwrite (r, sizeof (result_t), 1, f) != 1) {
		return -1;
	}
	if (	   job_id != NULL && r->job_id_len > 0
		&& fwrite (job_id, 1, r->job_id_len, f) != r->job_id_len)
	{
		return -1;
	}
	return 0;
}


/*
 * Read a structure from the provided file handler
 *
 * Return:
 *   0 --> No error (if  r is nor NULL, r->job_id is allocated by this function
 *         and must be freed by free() by the caller)
 *  -1 --> Read error (errno is probably set)
 *   1 --> End of file reached (the result structure is invalid)
 *   2 --> Memory allocation error
 */
static int
result_fread (result_t *r, FILE *f)
{
#if HAVE_ASSERT_H
	assert (f != NULL);
#endif

	if (r == NULL) {
		return 0;
	}

	/* Read the structure */
	if (fread (r, sizeof (result_t), 1, f) != 1) {
		if (ferror (f) != 0) {
			return -1;
		}
		return 1;
	}

	/* Allocate memory for the job ID string */
	r->job_id = (char *) malloc (r->job_id_len + 1);
	if (r->job_id == NULL) {
		return 2;
	}

	/* Read the job ID */
	if (fread (r->job_id, 1, r->job_id_len, f) != r->job_id_len) {
		free (r->job_id);
		r->job_id = NULL;
		if (ferror (f) != 0) {
			return -1;
		}
		return 1;
	}
	(r->job_id)[r->job_id_len] = '\0';
	return 0;
}


/*
 * Write the provided structure to the provided file name.  The file is
 * truncated if it exists.
 *
 * Return:
 *   0 --> No error
 *  -1 --> Write error (errno is probably set)
 */
static int
result_to_file (const result_t *r, const char *job_id, const char *file_name)
{
	FILE *f;
	int save_errno;

	if (r == NULL || file_name == NULL || file_name[0] == '\0') {
		return 0;
	}

	f = fopen (file_name, "wb");
	if (f == NULL) {
		return -1;
	}

	if (result_fwrite (r, job_id, f) != 0) {
		save_errno = errno;
		fclose (f);
		my_unlink (file_name);
		errno = save_errno;
		return -1;
	}

	if (fclose (f) != 0) {
		save_errno = errno;
		my_unlink (file_name);
		errno = save_errno;
		return -1;
	}

	return 0; 
}


/*
 * Read a structure from the provided file name.  If the file contains more
 * than one structure, only the first one is read.
 *
 * Return:
 *   0 --> No error (r->job_id is allocated by this function and must be
 *         freed by free() by the caller)
 *  -1 --> File operation error (errno is probably set)
 *   1 --> The file is empty (the result structure is not set)
 *   2 --> Memory allocation error
 */
static int
result_from_file (result_t *r, const char *file_name)
{
	FILE *f;
	int ret;
	int save_errno;

#if HAVE_ASSERT_H
	assert (file_name != NULL);
#endif

	if (r == NULL) {
		return 0;
	}

	f = fopen (file_name, "rb");
	if (f == NULL) {
		return -1;
	}

	ret = result_fread (r, f);
	if (ret < 0) {
		save_errno = errno;
		fclose (f);
		errno = save_errno;
		return -1;
	}

	fclose (f);
	return ret;
}


/*
 * Convert a result_t structure to a string
 *
 * Return:
 *   0 --> No error (buffer contains the string. It is allocated if NULL or
 *         not big enough and must be freed by the caller)
 *  -1 --> Memory allocation error
 *  -3 --> Internal error (buffer must be freed by the caller anyway)
 */
int
result_to_str (result_t *r, char **buffer, size_t *buffer_len)
{
	size_t len;
	int ret;

#if HAVE_ASSERT_H
	assert (   r != NULL && r->job_id != NULL
		&& buffer != NULL && buffer_len != NULL);
#endif

	len = 5 * 4 + r->job_id_len + 55; 
	if (len > *buffer_len || *buffer == NULL) {
		if (*buffer != NULL) {
			free (*buffer);
		}
		*buffer = (char *) malloc (len);
		if (*buffer == NULL) {
			return -1;
		}
		*buffer_len = len;
	}

	ret = snprintf (*buffer, *buffer_len,
			"JID:%s\nDUR:%ld\nKIL:%c\nCOD:%d\n",
			r->job_id,
			r->duration,
			(r->was_killed == 0) ? '0': '1',
			r->exit_code);
	if (ret >= *buffer_len || ret < 0) {
		return -3;
	}
	return 0;
}


/*
 * Convert the string to a list of result_t structure
 *
 * Return:
 *   0 --> No error (list is set and must be freed by the caller)
 *  -1 --> Memory allocation error
 */
int
str_to_results (const char *buffer, size_t buffer_size, lwc_LL **list)
{
	lwc_LL *tmp_list;
	char *buff_cpy, *str;
	result_t *ptr;

#if HAVE_ASSERT_H
	assert (list != NULL);
#endif

	tmp_list = lwc_newLL ();
	if (tmp_list == NULL) {
		return -1;
	}

	if (buffer == NULL || buffer_size <= 0) {
		*list = tmp_list;
		return 0;
	}

	/* Copy the input buffer to add a '\0' at the end (for strtok below) */
	buff_cpy = (char *) malloc (buffer_size + 1);
	if (buff_cpy == NULL) {
		lwc_delLL (tmp_list, NULL);
		return -1;
	}
	strncpy (buff_cpy, buffer, buffer_size);
	buff_cpy [buffer_size] = '\0';

	/* Parse each line */
	ptr = NULL;
	for (	str = strtok (buff_cpy, "\n");
		str != NULL;
		str = strtok (NULL, "\n"))
	{
		/* Skip the spaces at the begining of the line */
		while (*str != '\0' && isspace (*str) != 0) {
			str++;
		}

		/* Job ID */
		if (schedwi_strncasecmp (str, "JID:", 4) == 0) {
			/* A new result */
			ptr = new_result ();
			if (ptr == NULL || lwc_addEndLL (tmp_list, ptr) != 0) {
				destroy_result (ptr);
				lwc_delLL (tmp_list,
					(void (*)(const void *))destroy_result);
				free (buff_cpy);
				return -1;
			}
			ptr->job_id_len = schedwi_strlen (str + 4);
			ptr->job_id = (char *) malloc (ptr->job_id_len + 1);
			if (ptr->job_id == NULL) {
				lwc_delLL (tmp_list,
					(void (*)(const void *))destroy_result);
				free (buff_cpy);
				return -1;
			}
			strcpy (ptr->job_id, str + 4);
			continue;
		}

		/* Duration */
		if (schedwi_strncasecmp (str, "DUR:", 4) == 0) {
			if (ptr != NULL) {
				ptr->duration = (time_t)
						strtol (str + 4, NULL, 0);
			}
			continue;
		}

		/* Has the job been killed? */
		if (schedwi_strncasecmp (str, "KIL:", 4) == 0) {
			if (ptr != NULL) {
				ptr->was_killed = (str[4] == '0') ? 0: 1;
			}
			continue;
		}

		/* Return code or signal */
		if (schedwi_strncasecmp (str, "COD:", 4) == 0) {
			if (ptr != NULL) {
				ptr->exit_code = atoi (str + 4);
			}
			continue;
		}
	}

	free (buff_cpy);
	*list = tmp_list;
	return 0;
}


/*
 * Write the status of ended job to the provided file name.  The file is
 * truncated if it exists.
 *
 * Return:
 *   0 --> No error
 *  -1 --> Write error (errno is probably set)
 */
int
write_job_ended_to_file (	const char *file_name,
				const char *job_id,
				int exit_code,
				time_t duration)
{
	result_t r;

	init_result (&r);
	r.duration   = duration;
	r.was_killed = 0;
	r.exit_code  = exit_code;
	r.job_id_len = schedwi_strlen (job_id);
	return result_to_file (&r, job_id, file_name);
}


/*
 * Write the status of the killed job to the provided file name.  The file is
 * truncated if it exists.
 *
 * Return:
 *   0 --> No error
 *  -1 --> Write error (errno is probably set)
 */
int
write_job_killed_to_file (	const char *file_name,
				const char *job_id,
				int signal,
				time_t duration)
{
	result_t r;

	init_result (&r);
	r.duration   = duration;
	r.was_killed = 1;
	r.exit_code  = signal;
	r.job_id_len = schedwi_strlen (job_id);
	return result_to_file (&r, job_id, file_name);
}


/*
 * Create a file_result_t structure with the provided result_t structure
 *
 * Return:
 *   The new structure (to be freed by the caller) or
 *   NULL in case of memory allocation error
 */
file_result_t *
new_file_result_job (result_t *r)
{
	file_result_t *fr;

	fr = (file_result_t *) malloc (sizeof (file_result_t));
	if (fr == NULL) {
		return NULL;
	}
	fr->file_name = NULL;
	fr->r = r;
	return fr;
}


/*
 * Create and fill a file_result_t structure (in ret_ptr)
 *
 * Return:
 *   0 --> No error. ret_ptr contains the new structure (or NULL if the
 *         file was empty) and must be freed by the caller using
 *         destroy_file_result()
 *  -1 --> Memory allocation error
 *  -2 --> File operation error (errno id probably set)
 */
int
new_file_result (	const char *directory, const char *file_name,
			file_result_t **ret_ptr)
{
	file_result_t *fr;
	int ret;

#if HAVE_ASSERT_H
	assert (directory != NULL && file_name != NULL && ret_ptr != NULL);
#endif

	fr = (file_result_t *) malloc (sizeof (file_result_t));
	if (fr == NULL) {
		return -1;
	}

	fr->file_name = (char *) malloc (	  schedwi_strlen (directory)
						+ schedwi_strlen (DIR_SEP)
						+ schedwi_strlen (file_name)
						+ 1);
	if (fr->file_name == NULL) {
		free (fr);
		return -1;
	}

	fr->r = new_result ();
	if (fr->r == NULL) {
		free (fr->file_name);
		free (fr);
		return -1;
	}

	strcpy (fr->file_name, directory);
	strcat (fr->file_name, DIR_SEP);
	strcat (fr->file_name, file_name);

	ret = result_from_file (fr->r, fr->file_name);
	switch (ret) {
		case -1:
			destroy_result (fr->r);
			free (fr->file_name);
			free (fr);
			return -2;

		case 1:
			destroy_result (fr->r);
			free (fr->file_name);
			free (fr);
			*ret_ptr = NULL;
			return 0;

		case 2:
			destroy_result (fr->r);
			free (fr->file_name);
			free (fr);
			return -1;
	}
	*ret_ptr = fr;
	return 0;
}


/*
 * Free the provided structure
 */
void
destroy_file_result (file_result_t *ret_ptr)
{
	if (ret_ptr != NULL) {
		destroy_result (ret_ptr->r);
		if (ret_ptr->file_name != NULL) {
			free (ret_ptr->file_name);
		}
		free (ret_ptr);
	}
}

/*------------------------======= End Of File =======------------------------*/
