/* Schedwi
   Copyright (C) 2007 Herve Quatremain
     
   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* net_utils_glob.c -- Useful network functions */

#include <schedwi.h>

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#if HAVE_LIBGNUTLS
#include <net_utils_ssl.h>
#endif

#include <net_utils_sock.h>
#include <net_utils_init.h>
#include <net_utils_glob.h>


/*
 * Create a socket ready to accept requests
 *
 *          id is set by this function.  It is the network file descriptor
 *             to use by the other network functions
 * port_number is the port number to listen to (a number or a service name
 *             in /etc/services)
 *       iface is the network interface to listen to (NULL or empty means all
 *             the interfaces)
 *
 * Return:
 *   0 --> No error (the network descriptor in id is set)
 *  -1 --> Error (an error message is displayed using lwc_writeLog())
 */
int
net_server (net_id *id, const char *port_number, const char *iface)
{
#if HAVE_ASSERT_H
	assert (id != (net_id *)0);
#endif

	id->sock = net_server_sock (port_number, iface);
	if (id->sock < 0) {
		return -1;
	}

	return 0;
}


/*
 * Wait for a network request and return the socket (in new_id) associated with
 * the accepted connection.  The client host is check to be sure that it's the
 * Schedwi server (if schedwi_server_name is not NULL or empty)
 *
 * Return:
 *   0 --> No error (the network descriptor of the new connection is set
 *         in new_id)
 *   1 --> Authentication error (an error message is displayed
 *         using lwc_writeLog())
 *  -1 --> Error (a message is displayed using lwc_writeLog())
 */
int
net_accept (net_id *id, net_id *new_id, const char *schedwi_server_name)
{
#if HAVE_ASSERT_H
	assert (id != (net_id *)0 && new_id != (net_id *)0);
#endif

	new_id->sock = net_accept_sock (id->sock, schedwi_server_name);
	if (new_id->sock < 0) {
		return -1;
	}

#if HAVE_LIBGNUTLS
	if (ssl_enabled () != 0) {
		int ret;

		ret = net_accept_ssl (	new_id->sock, &(new_id->session),
					schedwi_server_name);
		if (ret != 0) {
			net_close_sock (new_id->sock);
		}
		return ret;
	}
#endif
	return 0;
}


/*
 * Create a socket connected to a server
 *
 *          id is set by this function.  It is the network file descriptor
 *             to use by the other network functions
 * port_number is the port number to connect to (a number or a service name
 *             in /etc/services)
 *
 * Return:
 *   0 --> No error (the network descriptor in id is set)
 *  -1 --> Error (an error message is displayed using lwc_writeLog())
 */
int
net_client (net_id *id, const char *port_number, const char *hostname)
{
#if HAVE_ASSERT_H
	assert (id != (net_id *)0);
#endif

	id->sock = net_client_sock (port_number, hostname);
	if (id->sock < 0) {
		return -1;
	}

#if HAVE_LIBGNUTLS
	if (ssl_enabled () != 0) {
		if (net_client_ssl (id->sock, &(id->session), hostname) != 0) {
			net_close_sock (id->sock);
			return -1;
		}
	}
#endif
	return 0;
}


/*
 * Read data from the provided socket handler.
 * The data are store in *buff (of size *len). If buffer is NULL or not big
 * enough, it is reallocated to a bigger size. It must be freed by the caller
 *
 * Return:
 *    The total number of characters read and stored in *buff or
 *    -1 on error (a message is displayed using lwc_writeLog())
 *    -2 if too many data are received (a message is displayed using
 *       lwc_writeLog())
 */
ssize_t
net_read (net_id *id, char **buff, size_t *len)
{
#if HAVE_ASSERT_H
	assert (id != (net_id *)0);
#endif

	if (ssl_enabled () == 0) {
		return net_read_sock (id->sock, buff, len);
	}

#if HAVE_LIBGNUTLS
	return net_read_ssl (id->session, buff, len);
#else
	return 0;
#endif
}


/*
 * Read the reply
 *
 * The format of the reply must be as follow:
 *    The first character: '0' means success, an other character means failure
 *    The next characters: An optional explanation message (which is returned
 *                         in result_msg)
 * No reply (nothing to read) means failure.
 *
 * Return:
 *   0 --> The reply means the remote action was successfull. If result_msg is
 *         not NULL, it is set with the message sent by the remote socket. It
 *         must be freed by the caller.
 *   1 --> The reply means the remote action failed. If result_msg is not NULL,
 *         it is set with the message sent by the remote socket. It may be NULL
 *         if no message was sent. It must be freed by the caller.
 *  -1 --> Error (a message is diplayed using lwc_writeLog())
 */
int
net_read_result (net_id *id, char **result_msg)
{
#if HAVE_ASSERT_H
	assert (id != (net_id *)0);
#endif

	if (ssl_enabled () == 0) {
		return net_read_result_sock (id->sock, result_msg);
	}

#if HAVE_LIBGNUTLS
	return net_read_result_ssl (id->session, result_msg);
#else
	return 1;
#endif
}


/*
 * Write data to the provided socket handler.
 *
 * Return:
 *   0 --> No error
 *  -1 --> Write error (a message is displayed using lwc_writeLog())
 */
int
net_write (net_id *id, const char *buff, size_t len)
{
#if HAVE_ASSERT_H
	assert (id != (net_id *)0);
#endif

	if (ssl_enabled () == 0) {
		return net_write_sock (id->sock, buff, len);
	}

#if HAVE_LIBGNUTLS
	return net_write_ssl (id->session, buff, len);
#else
	return 0;
#endif
}


/*
 * Write the End of Request tag
 *
 * Return:
 *   0 --> No error
 *  -1 --> Write error (a message is displayed using lwc_writeLog())
 */
int
net_write_eof (net_id *id)
{
	if (id != (net_id *)0) {
		if (ssl_enabled () == 0) {
			return net_write_eof_sock (id->sock);
		}

#if HAVE_LIBGNUTLS
		return net_write_eof_ssl (id->session);
#endif
	}
	return 0;
}


/*
 * Close the network descriptor
 */
int
net_close (net_id *id)
{
	if (id == (net_id *)0) {
		return 0;
	}

	if (ssl_enabled () == 0) {
		return net_close_sock (id->sock);
	}

#if HAVE_LIBGNUTLS
	return net_close_ssl (id->session, id->sock);
#else
	return 0;
#endif
}

/*------------------------======= End Of File =======------------------------*/
