/* Schedwi
   Copyright (C) 2007 Herve Quatremain
     
   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/*
 * job_run.h -- Running parameters for a job
 *
 * The parameters (members of the job_run structure) are the
 * following:
 *
 *        command --> The command to be run
 *    file_stdout --> File name of the command stdout redirection 
 *    file_stderr --> File name of the command stderr redirection 
 *           work --> Working directory (a chdir(2) will be done in this
 *                    directory before running (exec) the user command)
 *          shell --> User default shell
 *            uid --> User ID
 *            gid --> User group ID
 *           mask --> umask
 *            lim --> ulimit
 *           nice --> Scheduling priority (nice) for this user
 *      arguments --> Arguments list (included argv[0], the command name)
 *        environ --> Environement to load
 */

#ifndef _JOB_RUN_H
#define _JOB_RUN_H

#include <schedwi.h>

#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#if HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#if HAVE_SYS_RESOURCE_H
#include <sys/resource.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <env_utils.h>
#include <args_utils.h>

/* Allow the use in C++ code.  */
#ifdef __cplusplus
extern "C" {
#endif

#ifndef HAVE_MODE_T
typedef int mode_t;
#endif

#ifndef HAVE_RLIM_T
typedef unsigned long rlim_t;
#endif

struct job_run {
	char *command;
	char *file_stderr;
	char *file_stdout;
	char *work;
	char *shell;

#if HAVE_UID_T
	uid_t uid;
#else
	int uid;
#endif

#if HAVE_GID_T
	gid_t gid;
#else
	int gid;
#endif

	mode_t mask;
	rlim_t lim;
	int nice;
	argument_t arguments;
	environment_t environ;
};
typedef struct job_run job_run_t;


extern void init_job_run OF((job_run_t *j));
extern void destroy_job_run OF((job_run_t *j));
extern int add_username_job_run OF((job_run_t *j, const char *username));
extern int add_outfile_job_run OF((	job_run_t *j, const char *file_stdout,
					const char *file_stderr));
extern int add_command_job_run OF((	job_run_t *j, const char *command,
					const argument_t *args, int load_env,
					int *idx));
extern int add_env_job_run OF((job_run_t *j, const environment_t *e, int *idx));

#ifdef __cplusplus
}
#endif  /* C++ */

#endif /* job_run.h */

/*-----------------============== End Of File ==============-----------------*/
