/* Schedwi
   Copyright (C) 2007 Herve Quatremain
     
   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/*
 * args_utils.h -- Arguments array management functions
 *
 * The arguments array is an array of strings.  This array is embedded in the
 * argument_t structure but can be used directly by system functions like exec.
 */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#include <lib_functions.h>
#include <args_utils.h>

/*
 * Init the argument structure
 */
void
init_argument (argument_t *e)
{
	if (e != NULL) {
		e->arg = NULL;
		e->length = 0;
		e->idx = 0;
	}
}


/*
 * Free the argument structure
 */
void
destroy_argument (argument_t *e)
{
	int i;

	if (e != NULL) {
		if (e->arg != NULL && e->length > 0) {
			for (i = 0; i < e->idx; i++) {
				if ((e->arg)[i] != NULL) {
					free ((e->arg)[i]);
				}
			}
			free (e->arg);
		}
		e->arg = NULL;
		e->length = 0;
		e->idx = 0;
	}
}


/*
 * Add an argument at the end of the array
 *
 * Return:
 *    0 --> No error
 *   -1 --> Memory allocation error
 */
int
add_arg_s (	argument_t *e,
		const char *val, size_t val_len)
{
	char **tmp;

	if (e == NULL || val == NULL || val_len == 0) {
		return 0;
	}

	if (e->idx + 1 >= e->length) {
		tmp = (char **) realloc (e->arg,
					(e->length + 10) * sizeof (char **));
		if (tmp == NULL) {
			return -1;
		}
		e->arg = tmp;
		e->length += 10;
	}

	(e->arg)[e->idx] = (char *) malloc (val_len + 1);
	if ((e->arg)[e->idx] == NULL) {
		return -1;
	}
	strncpy ((e->arg)[e->idx], val, val_len);
	(e->arg)[e->idx][val_len] = '\0';
	(e->arg)[++e->idx] = NULL;
	return 0;
}


/*
 * Add an argument at the end of the array
 *
 * Return:
 *    0 --> No error
 *   -1 --> Memory allocation error
 */
int
add_arg (argument_t *e, const char *val)
{
	return add_arg_s (e, val, schedwi_strlen (val));
}


/*
 * Return the variable at the idx position in the array structure (or NULL
 * if idx is out of range or e is NULL)
 */
const char *
getvar_argument (argument_t *e, int idx)
{
	if (e == NULL || idx < 0 || idx >= e->idx) {
		return NULL;
	}
	return (e->arg)[idx];
}


/*
 * Convert an argument array to a string.  In this strings, `"' are added
 * around the arguments and `"' in arguments are escaped by \.  A space
 * is the argument separator.
 *
 * Return:
 *    The argument string (to be freed by the caller) or
 *    NULL on malloc error
 */
char *
args_array_to_string (char **a)
{
	int i, j, nb;
	size_t str_len;
	char *s, *out, *ptr;

	if (a == NULL) {
		out = (char *)malloc (1);
		if (out == NULL) {
			return NULL;
		}
		out[0]='\0';
		return out;
	}

	/* Compute the length of the output sring */
	str_len = 0;
	for (i = 0; a[i] != NULL; i++) {
		/* Count the number of " (they'll be preceded by \) */
		nb = 0;
		s = strchr (a[i], '"');
		while (s != NULL) {
			nb++;
			s++;
			s = strchr (s, '"');
		}

		str_len += schedwi_strlen (a[i]) + nb + 3;
	}

	/* Buffer allocation */
	out = (char *)malloc (str_len + 1);
	if (out == NULL) {
		return NULL;
	}
	out[0] = '\0';
	j = 0;

	/* Concatenate the arguments */
	for (i = 0; a[i] != NULL; i++) {
		strcpy (out + j, "\"");
		j++;

		ptr = a[i];
		s = strchr (ptr, '"');
		while (s != NULL) {
			strncpy (out + j, ptr, s - ptr);
			j += s - ptr;
			out[j++] = '\\';
			out[j++] = '"';
			ptr = s + 1;
			s = strchr (ptr, '"');
		}
		strcpy (out + j, ptr);
		j += schedwi_strlen (ptr);
		strcpy (out + j, "\" ");
		j += 2;
	}
	return out;
}


/*
 * Convert an argument array to a string.  In this strings, `"' are added
 * around the arguments and `"' in arguments are escaped by \.  A space
 * is the argument separator.
 *
 * Return:
 *    The argument string (to be freed by the caller) or
 *    NULL on malloc error
 */
char *
args_to_string (argument_t *e)
{
	char *out;

	if (e == NULL) {
		out = (char *)malloc (1);
		if (out == NULL) {
			return NULL;
		}
		out[0]='\0';
		return out;
	}
	return args_array_to_string (e->arg);
}


/*
 * Append the src arguments to the end of dst
 *
 * Return:
 *    0 --> No error
 *   -1 --> Memory allocation error
 */
int
args_concat (argument_t *dst, argument_t *src)
{
	int i, save_idx;

	if (dst == NULL || src == NULL) {
		return 0;
	}

	save_idx = dst->idx;
	for (i = 0; i < src->idx; i++) {
		if (add_arg (dst, (src->arg)[i]) != 0) {
			dst->idx = save_idx;
			(dst->arg)[save_idx] = NULL;
			return -1;
		}
	}
	return 0;
}


/*
 * Set the dst argument list with the src one
 */
void
set_arg (argument_t *dst, argument_t *src)
{
	destroy_argument (dst);

	if (src != NULL) {
		dst->arg = src->arg;
		dst->length = src->length;
		dst->idx = src->idx;
	}
}

/*-----------------============== End Of File ==============-----------------*/
