/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* utils.c -- Several functions */

#include <schedwi.h>

#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_STDIO_H
#include <stdio.h>
#endif

#if HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_SIGNAL_H
#include <signal.h>
#endif

#if HAVE_FCNTL_H
#include <fcntl.h>
#endif

#if HAVE_CTYPE_H
#include <ctype.h>
#endif

#if HAVE_PWD_H
#include <pwd.h>
#endif

#if HAVE_GRP_H
#include <grp.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#include <pthread.h>

#include <lib_functions.h>
#include <utils.h>

#if _GNU_SOURCE && !HAVE_CLEARENV
extern char **environ;
#endif

#if !HAVE_UID_T
typedef int uid_t;
#endif

#if !HAVE_GID_T
typedef int gid_t;
#endif


/*
 * Convert a unsigned long long int to a string
 *
 * Return:
 *   The number of characters copied in str
 */
unsigned int
copy_ulltostr (unsigned long long int v, char *str)
{
	char c;
	unsigned int len, i, j;

	if (str == NULL) {
	    return 0;
	}

	i = 0;
	do {
		str[i++] = '0' + v % 10;
		v /= 10;
	} while (v > 0);
	str[i] = '\0';
	len = i;

	/* Reverse the string */
	j = 0;
	i--;
	while (j < i) {
	    c = str[i];
	    str[i--] = str[j];
	    str[j++] = c;
	}

	return len;
}


/*
 * Return the filename part of a path
 *
 * Return:
 *   A pointer to the filename part in the provided path
 */
const char *
base_name (const char *path)
{
	const char * s, * prev;
	size_t sep_length;

#ifdef HAVE_ASSERT_H
	assert (path != 0);
#endif
	prev = 0;
	s    = path;
	sep_length = schedwi_strlen (DIR_SEP);
	while ((s = strstr (s, DIR_SEP)) != 0) {
		s += sep_length;
		prev = s;
	}
	if (prev == 0) {
		return path;
	}
	return prev;
}


/*
 * Print copyright and version information to stdout
 */
void
version (const char *prog_name)
{
	printf ("%s (Schedwi) %s\n", prog_name, PACKAGE_VERSION);
	puts("Copyright (C) 2007 Herve Quatremain");
	puts("Schedwi is free software, covered by the GNU");
	puts("General Public License version 3 or later.");
	puts("You may redistribute copies of Schedwi under");
	puts("the terms of this license.");
}


/*
 * Remove a file. If the remove fails, then try to truncate (size=0) the
 * file
 *
 * Return:
 *   0 --> File removed or truncated
 *  -1 --> Error (errno is set)
 */
int
my_unlink (const char *file_name)
{
	FILE *f;
	int save_errno;

	if (file_name == NULL || file_name[0] == '\0') {
		return 0;
	}

	/* Remove the file */
	if (unlink (file_name) != 0) {

		/* File does not exist */
		if (errno == ENOENT) {
			return 0;
		}

		/* Can't remove the file.  Try to truncate it */
		save_errno = errno;
		f = fopen (file_name, "wb");
		if (f == NULL) {
			/* Fail to truncate the file.  Return an error */
			errno = save_errno;
			return -2;
		}
		fclose (f);
	}
	return 0;
}


/*
 * RAZ the process structures:
 *    - Close all open files (except the provided file descriptor)
 *    - Empty the environment
 *    - Set the default action for all the signals
 *    - Remove the signal mask
 *
 * Failure of one of this step is not reported
 *
 * Note:
 *    This function is called in two places by the Schedwi agent (schedwiclnt):
 *       - At the start of the agent, when it goes into background
 *       - Every time a job is started
 *
 *    In the Schedwi server (schedwisrc) it is just called once at the
 *    beginning, when the program goes into background.
 *
 *    Clear the environment may cause issues.  In particular, some variables
 *    may be required by the job subprocess.  For instance, using Cygwin, the
 *    PATH environment variable contains the path to the Cygwin DLL
 *    (cygwin1.dll) which is required to fork/exec the job command.  This is
 *    why the environment is not cleared for Cygwin only.
 *    On other systems, other variables may be required (LD_LIBRARY_PATH for
 *    instance for SunOS and Linux) but no special care is taken and they
 *    will be erased by this function.  If they are required they can be
 *    defined as an environment for the job in the Schedwi server.
 */
void
sanitise (int fd)
{
	int i;
	struct sigaction sa;
	sigset_t mask;

	/* Close all files (except fd) */
	for (i = getdtablesize () - 1; i >= 0; i--) {
		if (i != fd) {
			close (i);
		}
	}

	/* Clear the environment */
#ifndef CYGWIN_IN_USE
#if HAVE_CLEARENV
	clearenv ();
#elif _GNU_SOURCE
	environ = NULL;
#endif
#endif

	/* Clear the signal actions and mask */
	schedwi_memset (&sa, 0, sizeof (struct sigaction));
	sa.sa_handler = SIG_DFL;
	sigemptyset (&sa.sa_mask);
	for (i = 0; i < 128; i++) {
		sigaction (i, &sa, NULL);
	}
	sigfillset (&mask);
	sigprocmask (SIG_UNBLOCK, &mask, NULL);
}


/*
 * Clean the process
 */
void
clean_process ()
{
#if HAVE_CHDIR
	char *tmp;
#endif

	/* Change to a temporary directory */
	tmp = NULL;
#if HAVE_CHDIR
#if HAVE_GETENV
	tmp = getenv ("TMPDIR");
	if (tmp != NULL) {
		if (chdir (tmp) != 0) {
			tmp = NULL;
		}
	}
#endif /* HAVE_GETENV */
	if (tmp == NULL) {
#ifdef P_tmpdir
		chdir (P_tmpdir);
#else
		chdir ("/tmp");
#endif
	}
#endif /* HAVE_CHDIR */

	/* Clean the process */
	sanitise (-1);

#if HAVE_UMASK
	umask (0027);
#endif
}


/*
 * Start the daemon
 *
 * Return:
 *   0 --> No error
 *  -1 --> fork error (errno is set)
 */
int
go_daemon()
{
#if HAVE_PID_T
	pid_t p;
#else
	int p;
#endif

#if HAVE_WORKING_FORK
	p = fork ();
	if (p < 0) {
		return -1;
	}
	if (p > 0) {
		exit (0);
	}
#endif

#if HAVE_SETSID
	setsid();
#endif

#if HAVE_WORKING_FORK
	p = fork ();
	if (p < 0) {
		return -1;
	}
	if (p > 0) {
		exit (0);
	}
#endif

	clean_process ();
	return 0;
}


static int fd_pid_file = -1;

/*
 * Write the PID of the current process to the provided file.
 *
 * Return:
 *   0 --> No error
 *  -1 --> Fail to open the file (errno is set)
 *   1 --> An other daemon is already running
 */
int
write_pid_file (const char *file_name)
{
	char pid_str[26];
	size_t l;

	if (file_name == NULL) {
		return 0;
	}

	/* Open the PID file */
	fd_pid_file = open (file_name, O_RDWR|O_CREAT, 00644);
	if (fd_pid_file < 0) {
		return -1;
	}

	/*
	 * Try to lock it.  If it fails it's probably because an other
	 * daemon is already running
	 */
#if HAVE_LOCKF && F_TLOCK
	if (lockf (fd_pid_file, F_TLOCK, 0) < 0) {
		return 1;
	}
#elif HAVE_FLOCK && LOCK_EX && LOCK_NB
	if (flock (fd_pid_file, LOCK_EX | LOCK_NB) < 0) {
		return 1;
	}
#endif

	/* Write the PID to the file */
	l = snprintf (pid_str, 26, "%ld\n", (long int)getpid ());
	pid_str [26 - 1] = '\0';
	write (fd_pid_file, pid_str, schedwi_strlen (pid_str));
	return 0;
}


/*
 * Close and remove the file containing the PID of the daemon
 */
void
close_pid_file (const char *file_name)
{
	if (file_name != NULL) {
		my_unlink (file_name);
	}

	if (fd_pid_file >= 0) {
		close (fd_pid_file);
	}
	fd_pid_file = -1;
}


/*
 * Create a temporary file and write in it the provided string (of length
 * string_len).  The temporary file name is returned in tmp_path and must
 * be freed by the caller by using string_to_file_destroy() (the associated
 * file is also removed from the file system).
 * The temporary file name returned in tmp_path is built by concatenating:
 *     - the temporary directory path (defined in the environment variable
 *     - TMPDIR if set or by `P_tmpdir' in stdio.h or "/tmp")
 *     - the prefix defined by SCHEDWI_TMP_PREFIX (configure.in)
 *     - a number (incremented in each call)
 * For instance: /tmp/schedwisrv234
 *
 * Return:
 *   0 --> No error.  tmp_path is set and must be freed by the caller by
 *	   string_to_file_destroy()
 *  -1 --> Memory allocation error
 *  -2 --> File error. errno is set
 */
int
string_to_file (const char *string, unsigned int string_len, char **tmp_path)
{
	static unsigned long long int nb = 0;
	static char *tmp_base = NULL;
	static unsigned int tmp_base_len;
	static pthread_mutex_t tmp_file_lock = PTHREAD_MUTEX_INITIALIZER;

	char *tmp_dir, *tmp_file;
	unsigned int tmp_dir_len, dir_sep_len, tmp_prefix_len, i;
	ssize_t nb_write;
	int fd, save_errno;

	if (tmp_path == NULL) {
		return 0;
	}

	/* Lock the access to the nb variable */
	pthread_mutex_lock (&tmp_file_lock);

	/* The first time, build the path name */
	if (tmp_base == 0) {

#if HAVE_GETENV
		tmp_dir = getenv ("TMPDIR");
#else
		tmp_dir = NULL;
#endif

		if (tmp_dir == NULL) {
#ifdef P_tmpdir
			tmp_dir = P_tmpdir;
#else
			tmp_dir = "/tmp";
#endif
		}
	
		tmp_dir_len    = schedwi_strlen (tmp_dir);
		dir_sep_len    = schedwi_strlen (DIR_SEP);
		tmp_prefix_len = schedwi_strlen (SCHEDWI_TMP_PREFIX);
		tmp_base_len   = tmp_dir_len + dir_sep_len + tmp_prefix_len;
	
		tmp_base = (char *) malloc (tmp_base_len + 1);
		if (tmp_base == NULL) {
			pthread_mutex_unlock (&tmp_file_lock);
			return -1;
		}

		strcpy (tmp_base, tmp_dir);
		strcpy (tmp_base + tmp_dir_len, DIR_SEP);
		strcpy (tmp_base + tmp_dir_len + dir_sep_len,
			SCHEDWI_TMP_PREFIX);
	}
	nb++;
	pthread_mutex_unlock (&tmp_file_lock);

	/* Build the temporary file name */
	tmp_file = (char *) malloc (tmp_base_len + 30);
	if (tmp_file == NULL) {
		return -1;
	}

	strcpy (tmp_file, tmp_base);
	copy_ulltostr (nb, tmp_file + tmp_base_len);

	/* Create the new file */
	fd = creat (tmp_file, S_IRUSR | S_IWUSR);
	if (fd < 0) {
		save_errno = errno;
		free (tmp_file);
		errno = save_errno;
		return -2;
	}

	/* Write the provided string in the file */
	if (string != NULL && string_len > 0) {
		i = 0;
		do {
			nb_write = write (fd, string + i, string_len);
	
			/* Error */
			if (nb_write < 0) {
				/* Real error */
				if (errno != EINTR && errno != EAGAIN) {
					save_errno = errno;
					close (fd);
					my_unlink (tmp_file);
					free (tmp_file);
					errno = save_errno;
					return -2;
				}
				/* Retry */
				continue;
			}
	
			/* Everithing has been written */
			if (nb_write == string_len) {
				break;
			}
	
			/* Partial write */
			string_len -= nb_write;
			i += nb_write;
			
		} while (nb_write > 0);
	}

	/* Close the file */
	close (fd);

	*tmp_path = tmp_file;
	return 0;
}


/*
 * Remove the provided file and free the path
 */
void
string_to_file_destroy (char *path)
{
	if (path != NULL) {
		my_unlink (path);
		free (path);
	}
}


/*
 * Change the user id of the current process
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error (errno is set)
 */
static int
change_user_id (uid_t user)
{
#if HAVE_SETREUID
	if (setreuid (user, user) != 0) {
		return -1;
	}
#else

#if HAVE_SETUID
	if (setuid (user) != 0) {
		return -1;
	}
#endif /* HAVE_SETUID */

#if HAVE_SETEUID
	if (seteuid (user) != 0) {
		return -1;
	}
#endif /* HAVE_SETEUID */

#endif /* HAVE_SETREUID */

	return 0;
}


/*
 * Change the group id of the current process
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error (errno is set)
 */
static int
change_group_id (gid_t group)
{
#if HAVE_SETREGID
	if (setregid (group, group) != 0) {
		return -1;
	}
#else

#if HAVE_SETGID
	if (setgid (group) != 0) {
		return -1;
	}
#endif /* HAVE_SETGID */

#if HAVE_SETEGID
	if (setegid (group) != 0) {
		return -1;
	}
#endif /* HAVE_SETEGID */

#endif /* HAVE_SETREGID */

	return 0;
}


/*
 * Change the user id of the current process.  name may contain a number or
 * a name in the password file
 *
 * Return:
 *   0 --> No error
 *  -1 --> System error (errno is set)
 *  -2 --> User name not found in the password file
 */
int
change_user (const char *name)
{
	unsigned int i;
#if HAVE_GETPWNAM || HAVE_GETPWENT
	struct passwd *pass;
#endif

	if (name == NULL) {
		return 0;
	}

	for (i = 0; name[i] != '\0' && isspace (name[i]) != 0; i++);
	if (name[i] == '\0') {
		return 0;
	}
	if (isdigit (name[i]) != 0) {
		return change_user_id ((uid_t) atoi (name + i));
	}

#if HAVE_GETPWNAM

	pass = getpwnam (name + i);
	if (pass == NULL) {
		return -2;
	}
	return change_user_id (pass->pw_uid);

#elif HAVE_GETPWENT && HAVE_ENDPWENT

	while ((pass = getpwent()) != NULL) {
		if (strcmp (pass->pw_name, name + i) == 0) {
			int ret;
			ret = change_user_id (pass->pw_uid);
			endpwent ();
			return ret;
		}
	}
	endpwent ();
	return -2;

#else
	return 0;
#endif
}


/*
 * Change the group id of the current process.  name may contain a number or
 * a name in the group file
 *
 * Return:
 *   0 --> No error
 *  -1 --> System error (errno is set)
 *  -2 --> Group name not found in the group file
 */
int
change_group (const char *name)
{
	unsigned int i;
#if HAVE_GETGRNAM || HAVE_GETGRENT
	struct group *group_str;
#endif

	if (name == NULL) {
		return 0;
	}

	for (i = 0; name[i] != '\0' && isspace (name[i]) != 0; i++);
	if (name[i] == '\0') {
		return 0;
	}
	if (isdigit (name[i]) != 0) {
		return change_group_id ((uid_t) atoi (name + i));
	}

#if HAVE_GETGRNAM

	group_str = getgrnam (name + i);
	if (group_str == NULL) {
		return -2;
	}
	return change_group_id (group_str->gr_gid);

#elif HAVE_GETGRENT && HAVE_ENDGRENT

	while ((group_str = getgrent()) != NULL) {
		if (strcmp (group_str->gr_name, name + i) == 0) {
			int ret;
			ret = change_group_id (group_str->gr_gid);
			endgrent ();
			return ret;
		}
	}
	endgrent ();
	return -2;

#else
	return 0;
#endif
}

/*------------------------======= End Of File =======------------------------*/
