/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* sql_job_hierarchy.c -- Retrieve a job/jobset parameters */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lib_functions.h>
#include <sql_common.h>
#include <sql_hierarchy.h>
#include <sql_job_hierarchy.h>


#define SQL_GET_JOB_PARAMS "SELECT %s FROM %s WHERE job_id=\"%s\""
#define SQL_GET_JOB_PARAMS_TIME "SELECT %s FROM %s WHERE id=\"%s\""
#define SQL_GET_JOB_PARAMS_SORTED "SELECT %s FROM %s WHERE job_id=\"%s\" ORDER BY %s"
#define SQL_DEL_JOB_PARAMS "DELETE FROM %s WHERE job_id=\"%s\""
#define SQL_DEL_JOB_MAIN "DELETE FROM job_main WHERE id=\"%s\""
#define SQL_GET_JOB_ENV "SELECT position,id,name FROM job_environment,environments WHERE job_id=\"%s\" AND env_id=id ORDER BY position"
#define SQL_GET_JOB_ENV_UNDEF "SELECT position FROM job_environment WHERE job_id=\"%s\" AND position=-1" 
#define SQL_REPLACE_JOB_PARAM "REPLACE INTO %s (job_id,%s) VALUES (\"%s\",\"%s\")"
#define SQL_JOB_ADD_ARRAY "REPLACE INTO %s (job_id,position,%s) VALUES "
#define SQL_JOB_ADD_ARRAY_VALUES "(\"%s\",%ld,\"%s\")"
#define SQL_JOB_ADD_CONSTRAINT "REPLACE INTO constraint_file (job_id,host_id,filename,exist) VALUES "
#define SQL_JOB_ADD_CONSTRAINT_VALUES "(\"%s\",\"%s\",\"%s\",%d)"
#define SQL_JOB_UPDATE "UPDATE job_main SET name=\"%s\",enabled=%d,description=\"%s\",cal_id=\"%s\",start_time=\"%s\" WHERE id=\"%s\""
#define SQL_JOB_SAVE "INSERT INTO job_main (parent,name,type,enabled,description,x,y,cal_id,start_time) VALUES (\"%s\",\"%s\",%d,%d,\"%s\",%ld,%ld,\"%s\",\"%s\")"
#define SQL_JOB_MOVE "UPDATE job_main SET parent=\"%s\",x=%ld,y=%ld WHERE id=\"%s\""

/*
 * Create a new sql_job_hierarchy object
 *
 * Return:
 *   The new object to be freed by the caller by sql_job_hierarchy_destroy() or
 *   NULL in case of error (if error_func() is not NULL, it is called with
 *        user_data_error_func as its first parameter and the error message
 *        as the second parameter)
 */
sql_job_hierarchy_ptr
sql_job_hierarchy_new (	const char *job_id, char only_hierarchy,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	sql_job_hierarchy_ptr ptr;
	char *err_msg = NULL;

#if HAVE_ASSERT_H
	assert (job_id != NULL);
#endif

	ptr = (sql_job_hierarchy_ptr) malloc (sizeof (sql_job_hierarchy));
	if (ptr == NULL) {
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Memory allocation error"), 0);
		}
		return NULL;
	}

	/* Retrieve the job hierarchy */
	ptr->parent_hierarchy_list = get_hierarchy_list (job_id, &err_msg);
	if (ptr->parent_hierarchy_list == NULL) {
		free (ptr);
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, 0);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return NULL;
	}

	if (only_hierarchy == 0) {
		/*
		 * Extract the first item from the list.
		 *  This is the current job ID
		 */
		ptr->job_id = (char *)lwc_delStartLL (
						ptr->parent_hierarchy_list);
	}
	else {
		ptr->job_id = NULL;
	}

	/* Store the error function */
	ptr->error_func = error_func;
	ptr->user_data_error_func = user_data_error_func;

	return ptr;
}


/*
 * Free the provided sql_job_hierarchy_ptr object
 */
void
sql_job_hierarchy_destroy (sql_job_hierarchy_ptr ptr)
{
	if (ptr != NULL) {
		if (ptr->job_id != NULL) {
			free (ptr->job_id);
		}
		lwc_delLL (	ptr->parent_hierarchy_list,
				(void (*)(const void *))free);
		free (ptr);
	}
}


/*
 * Retrieve parameters for a job and its hierarchy.  The provided 
 * sql_job_hierarchy_ptr object must have been built by sql_job_hierarchy_new()
 * prior to calling this function.  The error function which has been provided
 * to the sql_job_hierarchy_new() function is also used here in case of error.
 *
 * Return:
 *     0 --> No error.  values contains the retrieved parameters for the job
 *           and must be freed by the caller by sql_free_row().
 *           values_parent contains the retrieved parameters for the parent
 *           jobset of the job and must be freed by the caller by
 *           sql_free_row().  values_len and values_parent_len are set if not
 *           NULL to the length of the values.  They must be freed by the
 *           caller by free().
 *           If the job/jobset parameters have not been found in the database
 *           (ie. the job/jobset uses the parameters of its jobset), values and
 *           values_len are NULL.
 *     1 --> Memory allocation error (the error function - if not NULL - has
 *           been called)
 * other --> SQL error (the error function - if not NULL - has been called)
 */
unsigned int
sql_job_hierarchy_get (	sql_job_hierarchy_ptr ptr,
			const char *tablename, const char *colnames,
			char ***values, unsigned long int **values_len,
			char ***values_parent,
			unsigned long int **values_parent_len)
{
	MYSQL *sql;
	char *err_msg = NULL;
	unsigned int ret;
	lwc_LL *rows, *rows_len;
	char *buf;
	unsigned int buf_len;
	char **v_parent;
	unsigned long int *v_parent_len;

#if HAVE_ASSERT_H
	assert (   ptr != NULL && tablename != NULL && colnames != NULL
		&& values != NULL && values_parent != NULL);
#endif

	/* Connect to the database */
	sql = begin_mysql (&err_msg);
	if (sql == NULL) {
		if (ptr->error_func != NULL) {
			ptr->error_func (	ptr->user_data_error_func,
						err_msg, 0);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return 2;
	}

	/* Retrieve the parent parameters */
	err_msg = NULL;
	ret = get_job_parameters (	ptr->parent_hierarchy_list,
					tablename, colnames,
					&v_parent, &v_parent_len,
					&err_msg);
	if (ret != 0) {
		end_mysql ();
		if (ptr->error_func != NULL) {
			ptr->error_func (	ptr->user_data_error_func,
						err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	/* Retrieve the job/jobset parameters */
	if (ptr->job_id != NULL) {
		err_msg = NULL;
		buf = NULL;
		buf_len = 0;
		ret = sql_select (	&buf, &buf_len, &err_msg, NULL,
					&rows, &rows_len,
					SQL_GET_JOB_PARAMS,
					SQL_STRING_NON_ESCAPE, colnames,
					SQL_STRING_NON_ESCAPE, tablename,
					SQL_STRING, ptr->job_id,
					SQL_END);
		if (ret != 0) {
			end_mysql ();
			if (buf != NULL) {
				free (buf);
			}
			if (v_parent_len != NULL) {
				free (v_parent_len);
			}
			sql_free_row (v_parent);
			if (ptr->error_func != NULL) {
				ptr->error_func (ptr->user_data_error_func,
						err_msg, ret);
			}
			if (err_msg != NULL) {
				free (err_msg);
			}
			return ret;
		}

		/* Only keep the first row */
		*values = (char **) lwc_delStartLL (rows);
		if (values_len != NULL) {
			*values_len = (unsigned long int *)
						lwc_delStartLL (rows_len);
		}
		lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
		lwc_delLL (rows_len, (void (*)(const void *))free);

		/* Free the working buffer */
		if (buf != NULL) {
			free (buf);
		}
	}
	else {
		*values = NULL;
		if (values_len != NULL) {
			*values_len = NULL;
		}
	}

	/* Close the database */
	end_mysql ();

	*values_parent = v_parent;
	if (values_parent_len != NULL) {
		*values_parent_len = v_parent_len;
	}

	return 0;
}


/*
 * Retrieve parameters for a job and its hierarchy.  The provided 
 * sql_job_hierarchy_ptr object must have been built by sql_job_hierarchy_new()
 * prior to calling this function.  The error function which has been provided
 * to the sql_job_hierarchy_new() function is also used here in case of error.
 *
 * Return:
 *     0 --> No error.  values contains the retrieved parameters for the job
 *           and must be freed by the caller by 
 *              lwc_delLL (values, (void (*)(const void *))sql_free_row);
 *           values_parent contains the retrieved parameters for the parent
 *           jobset of the job and must be freed by the caller by
 *              lwc_delLL (values, (void (*)(const void *))sql_free_row);
 *           values_len and values_parent_len are set if not
 *           NULL to the length of the values.  They must be freed by the
 *           caller by
 *              lwc_delLL (values_len, (void (*)(const void *))free);
 *              lwc_delLL (values_parent_len, (void (*)(const void *))free);
 *           If the job/jobset parameters have not been found in the database
 *           (ie. the job/jobset uses the parameters of its jobset), values and
 *           values_len are NULL.
 *     1 --> Memory allocation error (the error function - if not NULL - has
 *           been called)
 * other --> SQL error (the error function - if not NULL - has been called)
 */
unsigned int
sql_job_hierarchy_get_multi (sql_job_hierarchy_ptr ptr,
			const char *tablename, const char *colnames,
			const char *sort_colname,
			lwc_LL **values, lwc_LL **values_len,
			lwc_LL **values_parent, lwc_LL **values_parent_len)
{
	MYSQL *sql;
	char *err_msg = NULL;
	unsigned int ret;
	char *buf;
	unsigned int buf_len;
	lwc_LL *v, *v_len, *v_parent, *v_parent_len;

#if HAVE_ASSERT_H
	assert (   ptr != NULL && tablename != NULL && colnames != NULL
		&& values != NULL);
#endif

	/* Connect to the database */
	sql = begin_mysql (&err_msg);
	if (sql == NULL) {
		if (ptr->error_func != NULL) {
			ptr->error_func (	ptr->user_data_error_func,
						err_msg, 0);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return 2;
	}

	/* Retrieve the parent parameters */
	if (values_parent != NULL) {
		err_msg = NULL;
		ret = get_job_parameters_multi_row (
					ptr->parent_hierarchy_list,
					tablename, colnames, sort_colname,
					&v_parent, &v_parent_len,
					&err_msg);
		if (ret != 0) {
			end_mysql ();
			if (ptr->error_func != NULL) {
				ptr->error_func (ptr->user_data_error_func,
						err_msg, ret);
			}
			if (err_msg != NULL) {
				free (err_msg);
			}
			return ret;
		}
	}
	else {
		v_parent = NULL;
		v_parent_len = NULL;
	}

	/* Retrieve the job/jobset parameters */
	if (ptr->job_id != NULL) {
		err_msg = NULL;
		buf = NULL;
		buf_len = 0;
		if (sort_colname == NULL) {
			ret = sql_select (&buf, &buf_len, &err_msg, NULL,
					&v, &v_len,
					SQL_GET_JOB_PARAMS,
					SQL_STRING_NON_ESCAPE, colnames,
					SQL_STRING_NON_ESCAPE, tablename,
					SQL_STRING, ptr->job_id,
					SQL_END);
		}
		else {
			ret = sql_select (&buf, &buf_len, &err_msg, NULL,
					&v, &v_len,
					SQL_GET_JOB_PARAMS_SORTED,
					SQL_STRING_NON_ESCAPE, colnames,
					SQL_STRING_NON_ESCAPE, tablename,
					SQL_STRING, ptr->job_id,
					SQL_STRING_NON_ESCAPE, sort_colname,
					SQL_END);
		}
		if (ret != 0) {
			end_mysql ();
			if (buf != NULL) {
				free (buf);
			}
			lwc_delLL (v_parent,
					(void (*)(const void *))sql_free_row);
			lwc_delLL (v_parent_len, (void (*)(const void *))free);
			if (ptr->error_func != NULL) {
				ptr->error_func (ptr->user_data_error_func,
						err_msg, ret);
			}
			if (err_msg != NULL) {
				free (err_msg);
			}
			return ret;
		}

		/* Free the working buffer */
		if (buf != NULL) {
			free (buf);
		}

		/* Not defined in the database */
		if (lwc_getNumNode (v) <= 0) {
			lwc_delLL (v, (void (*)(const void *))sql_free_row);
			lwc_delLL (v_len, (void (*)(const void *))free);
			*values = NULL;
			if (values_len != NULL) {
				*values_len = NULL;
			}
		}
		else {
			*values = v;
			if (values_len != NULL) {
				*values_len = v_len;
			}
			else {
				lwc_delLL (v_len, (void (*)(const void *))free);
			}
		}
	}
	else {
		*values = NULL;
		if (values_len != NULL) {
			*values_len = NULL;
		}
	}

	/* Close the database */
	end_mysql ();

	if (values_parent != NULL) {
		*values_parent = v_parent;
		if (values_parent_len != NULL) {
			*values_parent_len = v_parent_len;
		}
		else {
			lwc_delLL (v_parent_len, (void (*)(const void *))free);
		}
	}
	
	return 0;
}


/*
 * Retrieve from the database the environments associated with a job/jobset.
 * The provided sql_job_hierarchy_ptr object must have been built by
 * sql_job_hierarchy_new() prior to calling this function.
 * The error function which has been provided to the sql_job_hierarchy_new()
 * function is also used here in case of error.
 *
 * Return:
 *     0 --> No error.  values contains the retrieved environments for the job
 *           and must be freed by the caller by
 *              lwc_delLL (values, (void (*)(const void *))sql_free_row);
 *           values_len is set if not NULL to the length of the values.  It must
 *           be freed by the caller by
 *              lwc_delLL (values_len, (void (*)(const void *))free);
 *           If the job/jobset parameters have not been found in the database,
 *           values and values_len are set to NULL.
 *     1 --> Memory allocation error (the error function - if not NULL - has
 *           been called)
 * other --> SQL error (the error function - if not NULL - has been called)

 */
unsigned int
sql_job_hierarchy_env_list (	sql_job_hierarchy_ptr ptr,
				lwc_LL **values, lwc_LL **values_len)
{
	char *err_msg = NULL;
	unsigned int ret;
	lwc_LL *v, *v_len;

#if HAVE_ASSERT_H
	assert (ptr != NULL && values != NULL);
#endif

	if (ptr->job_id == NULL) {
		*values = NULL;
		if (values_len != NULL) {
			*values_len = NULL;
		}
		return 0;
	}

	/* First check if there is environments associated with the job */
	ret = sql_select (	NULL, NULL, &err_msg, NULL, &v, &v_len,
				SQL_GET_JOB_ENV_UNDEF,
				SQL_STRING, ptr->job_id,
				SQL_END);
	if (ret != 0) {
		if (ptr->error_func != NULL) {
			ptr->error_func (	ptr->user_data_error_func,
						err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	/* Do not use parent environments (position is set to -1) */
	if (lwc_getNumNode (v) > 0) {
		lwc_emptyLL (v, (void (*)(const void *))sql_free_row);
		lwc_emptyLL (v_len, (void (*)(const void *))free);
		*values = v;
		if (values_len != NULL) {
			*values_len = v_len;
		}
		else {
			lwc_delLL (v_len, (void (*)(const void *))free);
		}
		return 0;
	}
	else {
		lwc_delLL (v, (void (*)(const void *))sql_free_row);
		lwc_delLL (v_len, (void (*)(const void *))free);
	}

	/* Retrieve the environments */
	ret = sql_select (	NULL, NULL, &err_msg, NULL,
				&v, &v_len,
				SQL_GET_JOB_ENV,
				SQL_STRING, ptr->job_id,
				SQL_END);
	if (ret != 0) {
		if (ptr->error_func != NULL) {
			ptr->error_func (	ptr->user_data_error_func,
						err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	/* Not defined in the database */
	if (lwc_getNumNode (v) <= 0) {
		lwc_delLL (v, (void (*)(const void *))sql_free_row);
		lwc_delLL (v_len, (void (*)(const void *))free);
		*values = NULL;
		if (values_len != NULL) {
			*values_len = NULL;
		}
	}
	else {
		*values = v;
		if (values_len != NULL) {
			*values_len = v_len;
		}
		else {
			lwc_delLL (v_len, (void (*)(const void *))free);
		}
	}

	return 0;
}


/*
 * Retrieve parameters for a job and its hierarchy.  The provided 
 * sql_job_hierarchy_ptr object must have been built by sql_job_hierarchy_new()
 * prior to calling this function.  The error function which has been provided
 * to the sql_job_hierarchy_new() function is also used here in case of error.
 *
 * Return:
 *     0 --> No error.  values contains the retrieved parameters for the job
 *           and must be freed by the caller by sql_free_row().
 *           values_parent contains the retrieved parameters for the parent
 *           jobset of the job and must be freed by the caller by
 *           sql_free_row().  values_len and values_parent_len are set if not
 *           NULL to the length of the values.  They must be freed by the
 *           caller by free().
 *           If the job/jobset parameters are not defined in the database
 *           (ie. the first column is -1), values and values_len are NULL.
 *     1 --> Memory allocation error (the error function - if not NULL - has
 *           been called)
 * other --> SQL error (the error function - if not NULL - has been called)
 */
unsigned int
sql_job_hierarchy_get_by_id (
			sql_job_hierarchy_ptr ptr,
			const char *tablename, const char *colnames,
			const char *not_defined,
			char ***values, unsigned long int **values_len,
			char ***values_parent,
			unsigned long int **values_parent_len)
{
	MYSQL *sql;
	char *err_msg = NULL;
	unsigned int ret;
	lwc_LL *rows, *rows_len;
	char *buf;
	unsigned int buf_len;
	char **v_parent, **row;
	unsigned long int *v_parent_len;

#if HAVE_ASSERT_H
	assert (   ptr != NULL && not_defined != NULL
		&& tablename != NULL && colnames != NULL
		&& values != NULL && values_parent != NULL);
#endif

	/* Connect to the database */
	sql = begin_mysql (&err_msg);
	if (sql == NULL) {
		if (ptr->error_func != NULL) {
			ptr->error_func (	ptr->user_data_error_func,
						err_msg, 0);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return 2;
	}

	/* Retrieve the parent parameters */
	err_msg = NULL;
	ret = get_job_parameters_by_id (ptr->parent_hierarchy_list,
					tablename, colnames,
					not_defined,
					&v_parent, &v_parent_len,
					&err_msg);
	if (ret != 0) {
		end_mysql ();
		if (ptr->error_func != NULL) {
			ptr->error_func (	ptr->user_data_error_func,
						err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	/* Retrieve the job/jobset parameters */
	if (ptr->job_id != NULL) {
		err_msg = NULL;
		buf = NULL;
		buf_len = 0;
		ret = sql_select (	&buf, &buf_len, &err_msg, NULL,
					&rows, &rows_len,
					SQL_GET_JOB_PARAMS_TIME,
					SQL_STRING_NON_ESCAPE, colnames,
					SQL_STRING_NON_ESCAPE, tablename,
					SQL_STRING, ptr->job_id,
					SQL_END);
		if (ret != 0) {
			end_mysql ();
			if (buf != NULL) {
				free (buf);
			}
			if (v_parent_len != NULL) {
				free (v_parent_len);
			}
			sql_free_row (v_parent);
			if (ptr->error_func != NULL) {
				ptr->error_func (ptr->user_data_error_func,
						err_msg, ret);
			}
			if (err_msg != NULL) {
				free (err_msg);
			}
			return ret;
		}

		/* Only keep the first row */
		row = (char **) lwc_delStartLL (rows);
		if (row == NULL || strcmp (not_defined, row[0]) == 0) {
			sql_free_row (row);
			*values = NULL;
			if (values_len != NULL) {
				*values_len = NULL;
			}
		}
		else {
			*values = row;
			if (values_len != NULL) {
				*values_len = (unsigned long int *)
						lwc_delStartLL (rows_len);
			}
		}
		lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
		lwc_delLL (rows_len, (void (*)(const void *))free);

		/* Free the working buffer */
		if (buf != NULL) {
			free (buf);
		}
	}
	else {
		*values = NULL;
		if (values_len != NULL) {
			*values_len = NULL;
		}
	}

	/* Close the database */
	end_mysql ();

	*values_parent = v_parent;
	if (values_parent_len != NULL) {
		*values_parent_len = v_parent_len;
	}
	
	return 0;
}


/*
 * Remove the provided item from the provided database table
 *
 * Return:
 *     0 --> No error
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_job_hierarchy_del (	const char *job_id,
			const char *tablename,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret;

#if HAVE_ASSERT_H
	assert (job_id != NULL && tablename != NULL);
#endif

	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, NULL,
				SQL_DEL_JOB_PARAMS,
				SQL_STRING_NON_ESCAPE, tablename,
				SQL_STRING, job_id,
				SQL_END);
	if (ret != 0) { 
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Remove the provided job/jobset from the database
 *
 * Return:
 *     0 --> No error
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_job_hierarchy_del_job (const char *job_id,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret;

#if HAVE_ASSERT_H
	assert (job_id != NULL);
#endif

	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, NULL,
				SQL_DEL_JOB_MAIN,
				SQL_STRING, job_id,
				SQL_END);
	if (ret != 0) { 
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Add (or replace) a parameter in the provided table
 *
 * Return:
 *     0 --> No error
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_job_hierarchy_add (	const char *job_id,
			const char *tablename,
			const char *colname,
			const char *value,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret;

#if HAVE_ASSERT_H
	assert (job_id != NULL && tablename != NULL && colname != NULL);
#endif

	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, NULL,
				SQL_REPLACE_JOB_PARAM,
				SQL_STRING_NON_ESCAPE, tablename,
				SQL_STRING_NON_ESCAPE, colname,
				SQL_STRING, job_id,
				SQL_STRING, (value != NULL)? value: "",
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Add (or replace) an image  in the provided table
 *
 * Return:
 *     0 --> No error
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_job_hierarchy_add_icon (const char *job_id,
			const char *tablename,
			const char *colname,
			const char *image, unsigned long int image_len,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret;

#if HAVE_ASSERT_H
	assert (job_id != NULL && tablename != NULL && colname != NULL);
#endif

	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, NULL,
				SQL_REPLACE_JOB_PARAM,
				SQL_STRING_NON_ESCAPE, tablename,
				SQL_STRING_NON_ESCAPE, colname,
				SQL_STRING, job_id,
				SQL_BIN, image, image_len,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Associate the provided job with the values provided in the list
 *
 * Return:
 *      0 --> No error
 *      1 --> Memory allocation error (if error_func() is not NULL, it is
 *            called with user_data_error_func as its first parameter and the
 *            error message as the second parameter)
 *      2 --> Internal error (if error_func() is not NULL, it is called with
 *            user_data_error_func as its first parameter and the error message
 *            as the second parameter)
 *  other --> SQL error (if error_func() is not NULL, it is called with
 *            user_data_error_func as its first parameter and the error message
 *            as the second parameter)
 */
unsigned int
sql_job_hierarchy_add_array (const char *job_id,
			const char *tablename,
			const char *colname,
			lwc_LL *list,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	MYSQL *sql;
	char *err_msg = NULL;
	unsigned int ret, req_len, idx;
	char *esc_jobid, *esc_value, *req, *array_item;
	int num;
	long int pos;


#if HAVE_ASSERT_H
	assert (job_id != NULL && tablename != NULL && colname != NULL);
#endif

	/* No values.  Add and empty entry (position set to -1) */
	if (list == NULL || lwc_getNumNode (list) <= 0) {
		ret = sql_non_select (NULL, NULL, &err_msg, NULL, NULL,
				SQL_REPLACE_JOB_PARAM,
				SQL_STRING_NON_ESCAPE, tablename,
				SQL_STRING_NON_ESCAPE, "position",
				SQL_STRING, job_id,
				SQL_STRING_NON_ESCAPE, "-1",
				SQL_END);
		if (ret != 0) {
			if (error_func != NULL) {
				error_func (	user_data_error_func, err_msg,
						ret);
			}
			if (err_msg != NULL) {
				free (err_msg);
			}
			return ret;
		}
		return 0;
	}

	/* Connect to the database */
	sql = begin_mysql (&err_msg);
	if (sql == NULL) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, 0);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return 3;
	}

	/* Escape the job id string */
	esc_jobid = sql_escape (sql, job_id);
	if (esc_jobid == NULL) {
		end_mysql ();
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Memory allocation error"), 0);
		}
		return 1;
	}

	/* Compute the lenght of the SQL request string */
	req_len = 0;
	lwc_rewindLL (list);
	while ((array_item = (char *) lwc_nextLL (list)) != NULL) {
		/* The length of the escaped value */ 
		req_len += 2 * schedwi_strlen (array_item);
	}

	req_len +=	  schedwi_strlen (SQL_JOB_ADD_ARRAY)
			+ schedwi_strlen (tablename)
			+ schedwi_strlen (colname)
			+ (	  schedwi_strlen (SQL_JOB_ADD_ARRAY_VALUES)
				+ 25
				+ schedwi_strlen (esc_jobid))
					* lwc_getNumNode (list);

	/* Build the request */
	req = (char *) malloc (req_len);
	if (req == NULL) {
		end_mysql ();
		free (esc_jobid);
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Memory allocation error"), 0);
		}
		return 1;
	}

	num = snprintf (req, req_len, SQL_JOB_ADD_ARRAY, tablename, colname);
	if (num >= req_len || num < 0) {
		end_mysql ();
		free (req);
		free (esc_jobid);
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Internal error: buffer too small"),
					0);
		}
		return 2;
	}

	idx = schedwi_strlen (req);
	pos = 1;
	lwc_rewindLL (list);
	while ((array_item = (char *) lwc_nextLL (list)) != NULL) {

		/* Escape the value */
		esc_value = sql_escape (sql, array_item);
		if (esc_value == NULL) {
			end_mysql ();
			free (req);
			free (esc_jobid);
			if (error_func != NULL) {
				error_func (	user_data_error_func,
						_("Memory allocation error"),
						0);
			}
			return 1;
		}

		/* Buils the value part of the SQL request */
		num = snprintf (req + idx, req_len - idx,
				SQL_JOB_ADD_ARRAY_VALUES,
				esc_jobid, pos, esc_value);
		free (esc_value);

		if (num >= req_len - idx || num < 0) {
			end_mysql ();
			free (req);
			free (esc_jobid);
			if (error_func != NULL) {
				error_func (user_data_error_func,
					_("Internal error: buffer too small"),
					0);
			}
			return 2;
		}

		idx = schedwi_strlen (req);
		req[idx++] = ',';
		pos++;
	}
	req[idx -1] = '\0';
	free (esc_jobid);

	/* Run the SQL request */
	if (mysql_query (sql, req) != 0) {
		free (req);
		ret = mysql_errno (sql);
		compose_error_message (_("REPLACE"), &err_msg);
		end_mysql ();
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	end_mysql ();
	free (req);
	return 0;
}


/*
 * Save the provided environment list in the database
 *
 * Return:
 *      0 --> No error
 *      1 --> Memory allocation error (if error_func() is not NULL, it is
 *            called with user_data_error_func as its first parameter and the
 *            error message as the second parameter)
 *      2 --> Internal error (if error_func() is not NULL, it is called with
 *            user_data_error_func as its first parameter and the error message
 *            as the second parameter)
 *  other --> SQL error (if error_func() is not NULL, it is called with
 *            user_data_error_func as its first parameter and the error message
 *            as the second parameter)
 */
int
sql_job_hierarchy_add_env (const char *job_id,
			lwc_LL *list,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	return (int)sql_job_hierarchy_add_array (job_id,
						"job_environment", "env_id",
						list,
						error_func,
					       	user_data_error_func);
}


/*
 * Associate the provided job with the constraint files provided in the list.
 * Each item in the list is an array of strings with:
 *    a[0] --> Host ID
 *    a[1] --> File name
 *    a[2] --> Does the file must exist (0: No  and   1: Yes)
 *
 * Return:
 *      0 --> No error
 *      1 --> Memory allocation error (if error_func() is not NULL, it is
 *            called with user_data_error_func as its first parameter and the
 *            error message as the second parameter)
 *      2 --> Internal error (if error_func() is not NULL, it is called with
 *            user_data_error_func as its first parameter and the error message
 *            as the second parameter)
 *  other --> SQL error (if error_func() is not NULL, it is called with
 *            user_data_error_func as its first parameter and the error message
 *            as the second parameter)
 */
unsigned int
sql_job_hierarchy_add_constraint_files (
			const char *job_id,
			lwc_LL *list,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	MYSQL *sql;
	char *err_msg = NULL;
	unsigned int ret, req_len, idx;
	char *esc_jobid, *esc_hostid, *esc_filename, *req;
	char **array_item;
	int num;


#if HAVE_ASSERT_H
	assert (job_id != NULL);
#endif

	/* No constraint files */
	if (list == NULL || lwc_getNumNode (list) <= 0) {
		return 0;
	}

	/* Connect to the database */
	sql = begin_mysql (&err_msg);
	if (sql == NULL) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, 0);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return 3;
	}

	/* Escape the job id string */
	esc_jobid = sql_escape (sql, job_id);
	if (esc_jobid == NULL) {
		end_mysql ();
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Memory allocation error"), 0);
		}
		return 1;
	}

	/* Compute the lenght of the SQL request string */
	req_len = 0;
	lwc_rewindLL (list);
	while ((array_item = (char **) lwc_nextLL (list)) != NULL) {
		/* The length of the escaped value */ 
		req_len += 2 * (  schedwi_strlen (array_item[0])
				+ schedwi_strlen (array_item[1]));
	}

	req_len +=	  schedwi_strlen (SQL_JOB_ADD_CONSTRAINT)
			+ (	 schedwi_strlen (SQL_JOB_ADD_CONSTRAINT_VALUES)
				+ 25
				+ schedwi_strlen (esc_jobid))
					* lwc_getNumNode (list);

	/* Build the request */
	req = (char *) malloc (req_len);
	if (req == NULL) {
		end_mysql ();
		free (esc_jobid);
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Memory allocation error"), 0);
		}
		return 1;
	}

	strcpy (req, SQL_JOB_ADD_CONSTRAINT);

	idx = schedwi_strlen (req);
	lwc_rewindLL (list);
	while ((array_item = (char **) lwc_nextLL (list)) != NULL) {

		/* Escape the host ID */
		esc_hostid = sql_escape (sql, array_item[0]);
		if (esc_hostid == NULL) {
			end_mysql ();
			free (req);
			free (esc_jobid);
			if (error_func != NULL) {
				error_func (	user_data_error_func,
						_("Memory allocation error"),
						0);
			}
			return 1;
		}

		/* Escape the file name */
		esc_filename = sql_escape (sql, array_item[1]);
		if (esc_filename == NULL) {
			end_mysql ();
			free (esc_hostid);
			free (req);
			free (esc_jobid);
			if (error_func != NULL) {
				error_func (	user_data_error_func,
						_("Memory allocation error"),
						0);
			}
			return 1;
		}

		/* Buils the value part of the SQL request */
		num = snprintf (req + idx, req_len - idx,
				SQL_JOB_ADD_CONSTRAINT_VALUES,
				esc_jobid, esc_hostid, esc_filename,
				(array_item[2][0] == '0') ? 0: 1);
		free (esc_filename);
		free (esc_hostid);

		if (num >= req_len - idx || num < 0) {
			end_mysql ();
			free (req);
			free (esc_jobid);
			if (error_func != NULL) {
				error_func (user_data_error_func,
					_("Internal error: buffer too small"),
					0);
			}
			return 2;
		}

		idx = schedwi_strlen (req);
		req[idx++] = ',';
	}
	req[idx -1] = '\0';
	free (esc_jobid);

	/* Run the SQL request */
	if (mysql_query (sql, req) != 0) {
		free (req);
		ret = mysql_errno (sql);
		compose_error_message (_("REPLACE"), &err_msg);
		end_mysql ();
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	end_mysql ();
	free (req);
	return 0;
}


/*
 * Update the main parameters for the provided job/jobset
 *
 * Return:
 *      0 --> No error
 *      1 --> Memory allocation error (if error_func() is not NULL, it is
 *            called with user_data_error_func as its first parameter and the
 *            error message as the second parameter)
 *      2 --> Database connection error (if error_func() is not NULL, it is
 *            called with user_data_error_func as its first parameter and the
 *            error message as the second parameter)
 *  other --> SQL error (if error_func() is not NULL, it is called with
 *            user_data_error_func as its first parameter and the error message
 *            as the second parameter)
 */
unsigned int
sql_job_hierarchy_update_job (
			const char *job_id,
			const char *name,
			char enabled,
			const char *description,
			const char *cal_id,
			const char *start_time,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret;

#if HAVE_ASSERT_H
	assert (job_id != NULL && name != NULL);
#endif

	ret = sql_non_select (NULL, NULL, &err_msg, NULL, NULL,
			SQL_JOB_UPDATE,
			SQL_STRING, name,
			SQL_BOOL, (int)enabled,
			SQL_STRING, (description != NULL)? description: "",
			SQL_STRING, (cal_id != NULL)? cal_id: "0",
			SQL_STRING, (start_time != NULL)? start_time: "-1",
			SQL_STRING, job_id,
			SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Update the main parameters for the provided job/jobset
 *
 * Return:
 *      0 --> No error
 *      1 --> Memory allocation error (if error_func() is not NULL, it is
 *            called with user_data_error_func as its first parameter and the
 *            error message as the second parameter)
 *      2 --> Database connection error (if error_func() is not NULL, it is
 *            called with user_data_error_func as its first parameter and the
 *            error message as the second parameter)
 *  other --> SQL error (if error_func() is not NULL, it is called with
 *            user_data_error_func as its first parameter and the error message
 *            as the second parameter)
 */
unsigned int
sql_job_hierarchy_save_job (
			const char *parent_id,
			const char *name,
			char type,
			char enabled,
			const char *description,
			long int x,
			long int y,
			const char *cal_id,
			const char *start_time,
			unsigned long int *added_id,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret;

#if HAVE_ASSERT_H
	assert (parent_id != NULL && name != NULL);
#endif

	ret = sql_non_select (NULL, NULL, &err_msg, added_id, NULL,
			SQL_JOB_SAVE,
			SQL_STRING, parent_id,
			SQL_STRING, name,
			SQL_BOOL, (int)type,
			SQL_BOOL, (int)enabled,
			SQL_STRING, (description != NULL)? description: "",
			SQL_INT, x,
			SQL_INT, y,
			SQL_STRING, (cal_id != NULL)? cal_id: "0",
			SQL_STRING, (start_time != NULL)? start_time: "-1",
			SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Change the parent ID of the provided job
 *
 * Return:
 *     0 --> No error
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_job_hierarchy_move (const char *job_id,
			const char *new_jobset_id,
			long int x,
			long int y,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret;

#if HAVE_ASSERT_H
	assert (job_id != NULL && new_jobset_id != NULL);
#endif

	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, NULL,
				SQL_JOB_MOVE,
				SQL_STRING, new_jobset_id,
				SQL_INT, x,
				SQL_INT, y,
				SQL_STRING, job_id,
				SQL_END);
	if (ret != 0) { 
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
