/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* sql_children_job.c -- functions to retrieve job details from the database */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lib_functions.h>
#include <sql_common.h>
#include <sql_hierarchy.h>
#include <sql_children_job.h>

#define SQL_CHILDREN_JOB_LIST "SELECT id,name,type,x,y FROM job_main WHERE parent=\"%s\" ORDER BY id DESC"
#define SQL_CHILDREN_GET_ICON "SELECT icon FROM %s WHERE job_id=\"%s\""
#define SQL_CHILDREN_JOB_GET_DETAILS "SELECT id,parent,name,type,enabled,description,x,y,cal_id,start_time FROM job_main WHERE id=\"%s\""
#define SQL_CHILDREN_UPDATE_COORD "UPDATE job_main SET x=\"%s\",y=\"%s\" WHERE id=\"%s\""


/*
 * Get the children of the provided jobset
 *
 * For each child, the provided add_row() function is called with the
 * following parameters:
 *     1) the provided user_data_add_row parameter
 *     2) the child job/jobset id
 *     3) the name
 *     4) the type (1:job or 0:jobset)
 *     5) the X coordinate
 *     6) the Y coordinate
 * add_row() must return 0 if successful or an other code in case of error.
 *
 * Return:
 *   0 --> No error
 *  -1 --> The add_row() function returned with a code different from 0
 *  -2 --> SQL error (if error_func() is not NULL, it is called with
 *         user_data_error_func as its first parameter and the error message
 *         as the second parameter)
 */
int
sql_children_job_list (
		const char *parent_id,
		int (*add_row)(	void * ,const char *, const char *,
				const char *, const char *, const char *),
		void *user_data_add_row,
		void (*error_func)(void *, const char *, unsigned int),
		void *user_data_error_func)
{
	char *err_msg = NULL;
	lwc_LL *rows;
	unsigned int ret;
	char **row;

#if HAVE_ASSERT_H
	assert (parent_id != NULL && add_row != NULL);
#endif

	ret = sql_select (	NULL, NULL, &err_msg, NULL, &rows, NULL,
				SQL_CHILDREN_JOB_LIST,
				SQL_STRING, parent_id,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return -2;
	}

	while ((row = (char **) lwc_delStartLL (rows)) != NULL) {
		if (add_row (	user_data_add_row,
				row[0],		/* job ID */
				row[1],		/* job name */
				row[2],		/* type */
				row[3],		/* X */
				row[4]) != 0)	/* Y */
		{
			sql_free_row (row);
			lwc_delLL (rows, (void (*)(const void *))sql_free_row);
			return -1;
		}
		sql_free_row (row);
	}
	lwc_delLL (rows, NULL);
	return 0;
}


/*
 * Get icon associated with the provided ID in the provided database table
 *
 * Return:
 *     0 --> No error.  icon_stream contains the pixdata stream and must be
 *           freed by the caller by free().  icon_stream_len contains the
 *           length of the stream copied in icon_stream.
 *           If there is no icon associated with ID, *icon_stream is set to
 *           NULL and *icon_stream to 0
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_children_get_icon (	const char *id, const char *tablename,
			char **icon_stream, unsigned long int *icon_stream_len,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret;
	lwc_LL *rows;
	lwc_LL *rows_len;
	char **row, *s;
	unsigned long int *row_len;

#if HAVE_ASSERT_H
	assert (   id != NULL && tablename != NULL
		&& icon_stream != NULL && icon_stream_len != NULL);
#endif

	ret = sql_select (	NULL, NULL, &err_msg, NULL, &rows, &rows_len,
				SQL_CHILDREN_GET_ICON,
				SQL_STRING_NON_ESCAPE, tablename,
				SQL_STRING, id,
				SQL_END);

	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	row = (char **) lwc_delStartLL (rows);
	row_len = (unsigned long int *) lwc_delStartLL (rows_len);
	/* No icon found */
	if (row == NULL || row_len == NULL) {
		lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
		lwc_delLL (rows_len, (void (*)(const void *)) free);
		if (row != NULL) {
			sql_free_row (row);
		}
		if (row_len != NULL) {
			free (row_len);
		}
		*icon_stream = NULL;
		*icon_stream_len = 0;
		return 0;
	}
	lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
	lwc_delLL (rows_len, (void (*)(const void *)) free);

	s = (char *) malloc (row_len[0]);
	if (s == NULL) {
		sql_free_row (row);
		free (row_len);
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Memory allocation error"), 1);
		}
		return 1;
	}

	schedwi_memcpy (s, row[0], row_len[0]);
	*icon_stream = s;
	*icon_stream_len = row_len[0];
	sql_free_row (row);
	free (row_len);
	return 0;
}


/*
 * Get the job/jobset details from the database
 *
 * Return:
 *     0 --> No error
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_children_job_get_main (const char *id,
			void (*set_job_id)(const char *, void *),
			void (*set_job_parent)(const char *, void *),
			void (*set_job_name)(const char *, void *),
			void (*set_job_type)(const char *, void *),
			void (*set_job_enable)(const char *, void *),
			void (*set_job_descr)(const char *, void *),
			void (*set_job_x)(const char *, void *),
			void (*set_job_y)(const char *, void *),
			void (*set_job_calendar_id)(const char *, void *),
			void (*set_job_start_time)(const char *, void *),
			void *user_data_set_job,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret;
	lwc_LL *rows;
	char **row;

#if HAVE_ASSERT_H
	assert (id != NULL);
#endif

	ret = sql_select (	NULL, NULL, &err_msg, NULL, &rows, NULL,
				SQL_CHILDREN_JOB_GET_DETAILS,
				SQL_STRING, id, SQL_END);

	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	row = (char **) lwc_delStartLL (rows);
	if (row == NULL) {
		lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Unknown job id in the database"),
					ret);
		}
		return 2;
	}
	lwc_delLL (rows, (void (*)(const void *)) sql_free_row);

	if (set_job_id != NULL) {
		set_job_id (row[0], user_data_set_job);
	}
	if (set_job_parent != NULL) {
		set_job_parent (row[1], user_data_set_job);
	}
	if (set_job_name != NULL) {
		set_job_name (row[2], user_data_set_job);
	}
	if (set_job_type != NULL) {
		set_job_type (row[3], user_data_set_job);
	}
	if (set_job_enable != NULL) {
		set_job_enable (row[4], user_data_set_job);
	}
	if (set_job_descr != NULL) {
		set_job_descr (row[5], user_data_set_job);
	}
	if (set_job_x != NULL) {
		set_job_x (row[6], user_data_set_job);
	}
	if (set_job_y != NULL) {
		set_job_y (row[7], user_data_set_job);
	}
	if (set_job_calendar_id != NULL) {
		set_job_calendar_id (row[8], user_data_set_job);
	}
	if (set_job_start_time != NULL) {
		set_job_start_time (row[9], user_data_set_job);
	}

	sql_free_row (row);

	return 0;
}


/*
 * Update the job coordinates in the database
 *
 * Return:
 *     0 --> No error
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_children_job_update_coord (const char *id,
			const char *x_str, const char *y_str,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret;

#if HAVE_ASSERT_H
	assert (id != NULL && x_str != NULL && y_str != NULL);
#endif

	ret = sql_non_select (  NULL, NULL, &err_msg, NULL, NULL,
				SQL_CHILDREN_UPDATE_COORD,
				SQL_STRING, x_str,
				SQL_STRING, y_str,
				SQL_STRING, id,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Retrieve a job parameter from the job hierarchy. The provided hierarchy list
 * (in lst) must have been built by get_hierarchy_list() prior to calling
 * this function.
 *
 * Return:
 *     0 --> No error.  value contains the retrieved parameter if the top job
 *           defines it or NULL otherwise.  value_hist contains the parameter
 *           retrieved from the hierarchy if the top job does not defines it
 *           (value is NULL in this case).  value and value_hist must be
 *           freed by the caller by free().
 *           If value_len is not NULL it contains the length of the value
 *           stored in value or value_hist (without the added trailing '\0').
 *           If the parameter has not been found in the hierarchy (ie. no
 *           ancestor defines this parameter), value and value_hist are set to
 *           NULL and value_len (if not NULL) is set to 0.
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *           as the second parameter)
 *     2 --> Database connection error (if error_func() is not NULL, it is
 *           called with user_data_error_func as its first parameter and the
 *           error message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_children_job_get_parameter (lwc_LL *lst,
			const char *tablename,
			const char *colname,
			char **value, char **value_hist,
			unsigned long int *value_len,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	char *cols, **row, *s, *job_id;
	unsigned long int *row_len;
	unsigned int ret;

#if HAVE_ASSERT_H
	assert (   lst != NULL && tablename != NULL && colname != NULL
		&& value != NULL && value_hist != NULL);
#endif

	cols = (char *)malloc (	  schedwi_strlen ("job_id")
				+ schedwi_strlen (colname) + 2);
	if (cols == NULL) {
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Memory allocation error"), 1);
		}
		return 1;
	} 

	strcpy (cols, "job_id");
	strcat (cols, ",");
	strcat (cols, colname);

	ret = get_job_parameters (	lst, tablename, cols,
					&row, &row_len,
					&err_msg);
	if (ret != 0) {
		free (cols);
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	free (cols);

	/* Parameter not found */
	if (row == NULL || row_len == NULL) {
		*value = *value_hist = NULL;
		if (value_len != NULL) {
			*value_len = 0;
		}
		return 0;
	}

	/* Copy the returned value */
	s = (char *) malloc (row_len[1] + 1);
	if (s == NULL) {
		sql_free_row (row);
		free (row_len);
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Memory allocation error"), 1);
		}
		return 1;
	}
	schedwi_memcpy (s, row[1], row_len[1]);
	s[row_len[1]] = '\0';

	/* Copy the value length */
	if (value_len != NULL) {
		*value_len = row_len[1];
	}

	/* Check if the provided item ID defines the parameter */
	lwc_rewindLL (lst);
	job_id = (char *)lwc_nextLL (lst);
#if HAVE_ASSERT_H
	assert (job_id != NULL);
#endif
	if (strcmp (job_id, row[0]) == 0) {
		*value = s;
		*value_hist = NULL;
	}
	else {
		*value = NULL;
		*value_hist = s;
	}
	sql_free_row (row);
	free (row_len);

	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
