/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* schedwi_main.c -- Global functions */

#include <schedwi.h>

#include <schedwi_interface.h>

#include <hosts_cb.h>
#include <env_list_cb.h>
#include <calendar_list_cb.h>
#include <main_cb.h>
#include <find_cb.h>
#include <pref_cb.h>
#include <cursor.h>
#include <cache_pixbuf.h>
#include <schedwi_gconf.h>
#include <druid_cb.h>
#include <login_cb.h>
#include <sql_common.h>
#include <schedwi_main.h>

enum _menu_widgets {
	SCHEDWI_MAIN_MENU_CUT = 0,
	SCHEDWI_MAIN_MENU_COPY,
	SCHEDWI_MAIN_MENU_PASTE,
	SCHEDWI_MAIN_MENU_DELETE,
	SCHEDWI_MAIN_MENU_FIND,
	SCHEDWI_MAIN_MENU_ROOT_JOBSET,

	SCHEDWI_MAIN_MENU_REFRESH,
	SCHEDWI_MAIN_MENU_ZOOMIN,
	SCHEDWI_MAIN_MENU_ZOOMOUT,
	SCHEDWI_MAIN_MENU_ZOOMNORMAL,
	SCHEDWI_MAIN_MENU_SHOWGRID,
	SCHEDWI_MAIN_MENU_SNAPGRID,

	SCHEDWI_MAIN_MENU_SHOWHOST,
	SCHEDWI_MAIN_MENU_SHOWENV,
	SCHEDWI_MAIN_MENU_SHOWCAL,

	SCHEDWI_MAIN_TOOL_REFRESH,
	SCHEDWI_MAIN_TOOL_CUT,
	SCHEDWI_MAIN_TOOL_COPY,
	SCHEDWI_MAIN_TOOL_PASTE,
	SCHEDWI_MAIN_TOOL_DELETE,
	SCHEDWI_MAIN_TOOL_FIND,

	SCHEDWI_MAIN_LOCATION_BAR,
	SCHEDWI_MAIN_LOCATION_ENTRY,
	SCHEDWI_MAIN_LOCATION_ZOOMIN,
	SCHEDWI_MAIN_LOCATION_ZOOMOUT,

	SCHEDWI_MAIN_CANVAS_NAVIGATOR,

	SCHEDWI_MAIN_NB
};
typedef enum _menu_widgets menu_widgets;


static GtkWidget *dialog_host_list = NULL;
static GtkWidget *dialog_env_list = NULL;
static GtkWidget *dialog_cal_list = NULL;
static GtkWidget *dialog_find = NULL;
static GtkWidget *dialog_pref = NULL;

GtkWidget *application_main = NULL;

static GtkWidget *menus[SCHEDWI_MAIN_NB];


/*
 * Retrieve the widgets and store them in the menus array
 */
void
store_application_main (GtkWidget *w)
{
	application_main = w;

	menus[SCHEDWI_MAIN_MENU_CUT] = lookup_widget (w, "cut1");
	menus[SCHEDWI_MAIN_MENU_COPY] = lookup_widget (w, "copy1");
	menus[SCHEDWI_MAIN_MENU_PASTE] = lookup_widget (w, "paste1");
	menus[SCHEDWI_MAIN_MENU_DELETE] = lookup_widget (w, "delete1");
	menus[SCHEDWI_MAIN_MENU_FIND] = lookup_widget (w, "find");
	menus[SCHEDWI_MAIN_MENU_ROOT_JOBSET] = lookup_widget (w,
						"default_job_parameters1");

	menus[SCHEDWI_MAIN_MENU_REFRESH] = lookup_widget (w, "refresh");
	menus[SCHEDWI_MAIN_MENU_ZOOMIN] = lookup_widget (w, "zoomin");
	menus[SCHEDWI_MAIN_MENU_ZOOMOUT] = lookup_widget (w, "zoomout");
	menus[SCHEDWI_MAIN_MENU_ZOOMNORMAL] = lookup_widget (w, "normal_size");
	menus[SCHEDWI_MAIN_MENU_SHOWGRID] = lookup_widget (w, "show_grid");
	menus[SCHEDWI_MAIN_MENU_SNAPGRID] = lookup_widget (w, "snap_to_grid");

	menus[SCHEDWI_MAIN_MENU_SHOWHOST] = lookup_widget (w, "show_hosts");
	menus[SCHEDWI_MAIN_MENU_SHOWENV] = lookup_widget (w,
							"show_environments");
	menus[SCHEDWI_MAIN_MENU_SHOWCAL] = lookup_widget (w, "show_calendars");

	menus[SCHEDWI_MAIN_TOOL_REFRESH] = lookup_widget (w,
							"toolbutton_refresh");
	menus[SCHEDWI_MAIN_TOOL_CUT] = lookup_widget (w, "toolbutton_cut");
	menus[SCHEDWI_MAIN_TOOL_COPY] = lookup_widget (w, "toolbutton_copy");
	menus[SCHEDWI_MAIN_TOOL_PASTE] = lookup_widget (w, "toolbutton_paste");
	menus[SCHEDWI_MAIN_TOOL_DELETE] = lookup_widget (w,
							"toolbutton_delete");
	menus[SCHEDWI_MAIN_TOOL_FIND] = lookup_widget (w, "toolbutton_find");

	menus[SCHEDWI_MAIN_LOCATION_BAR] = gtk_widget_get_ancestor (
				lookup_widget (w, "hbox_main_location"),
				BONOBO_TYPE_DOCK_ITEM);
	menus[SCHEDWI_MAIN_LOCATION_ENTRY] = lookup_widget (w,
						"entry_main_location");
	menus[SCHEDWI_MAIN_LOCATION_ZOOMIN] = lookup_widget (w,
						"button_main_zoom_in");
	menus[SCHEDWI_MAIN_LOCATION_ZOOMOUT] = lookup_widget (w,
						"button_main_zoom_out");
	menus[SCHEDWI_MAIN_CANVAS_NAVIGATOR] = lookup_widget (w,
						"eventbox_main_navigator");
}


/*
 * Set sensitive all the widget of the main window (after a successful login)
 */
void
main_activate ()
{
	gint i;

	for (i = 0; i < SCHEDWI_MAIN_NB; i++) {
		if (menus[i] != NULL) {
			gtk_widget_set_sensitive (menus[i], TRUE);
		}
	}
	gtk_widget_set_sensitive (menus[SCHEDWI_MAIN_MENU_CUT], FALSE);
	gtk_widget_set_sensitive (menus[SCHEDWI_MAIN_MENU_COPY], FALSE);
	gtk_widget_set_sensitive (menus[SCHEDWI_MAIN_MENU_PASTE], FALSE);
	gtk_widget_set_sensitive (menus[SCHEDWI_MAIN_MENU_DELETE], FALSE);
	gtk_widget_set_sensitive (menus[SCHEDWI_MAIN_TOOL_CUT], FALSE);
	gtk_widget_set_sensitive (menus[SCHEDWI_MAIN_TOOL_COPY], FALSE);
	gtk_widget_set_sensitive (menus[SCHEDWI_MAIN_TOOL_PASTE], FALSE);
	gtk_widget_set_sensitive (menus[SCHEDWI_MAIN_TOOL_DELETE], FALSE);
}


/*
 * Set insensitive all the widget of the main window (after logout)
 */
void
main_deactivate ()
{
	gint i;

	for (i = 0; i < SCHEDWI_MAIN_NB; i++) {
		if (menus[i] != NULL) {
			gtk_widget_set_sensitive (menus[i], FALSE);
		}
	}

	if (dialog_host_list != NULL) {
		gtk_check_menu_item_set_active (
			GTK_CHECK_MENU_ITEM (menus[SCHEDWI_MAIN_MENU_SHOWHOST]),
			FALSE);
		gtk_widget_destroy (dialog_host_list);
		dialog_host_list = NULL;
	}
	if (dialog_env_list != NULL) {
		gtk_check_menu_item_set_active (
			GTK_CHECK_MENU_ITEM (menus[SCHEDWI_MAIN_MENU_SHOWENV]),
			FALSE);
		gtk_widget_destroy (dialog_env_list);
		dialog_env_list = NULL;
	}
	if (dialog_cal_list != NULL) {
		gtk_check_menu_item_set_active (
			GTK_CHECK_MENU_ITEM (menus[SCHEDWI_MAIN_MENU_SHOWCAL]),
			FALSE);
		gtk_widget_destroy (dialog_cal_list);
		dialog_cal_list = NULL;
	}
	if (dialog_find != NULL) {
		gtk_widget_destroy (dialog_find);
		dialog_find = NULL;
	}
}


/*
 * Remove the check mark in the Tools menu when the associated dialog is
 * closed
 */
static void
main_close_list (gpointer data)
{
	gtk_check_menu_item_set_active  (GTK_CHECK_MENU_ITEM (data), FALSE);
}


/*
 * CALLBACKS fot the main menu items
 */

void
main_on_show_hosts_activate (GtkMenuItem *menuitem)
{
	if (gtk_check_menu_item_get_active (
				GTK_CHECK_MENU_ITEM (menuitem)) == TRUE)
	{
		if (dialog_host_list == NULL) {
			dialog_host_list = new_dialog_host_list (
							main_close_list,
							menuitem);
		}
		else {
			hosts_refresh_widget (dialog_host_list);
			gtk_widget_show (dialog_host_list);
		}
	}
	else {
		gtk_widget_hide (dialog_host_list);
	}
}

void
main_on_show_environments_activate (GtkMenuItem *menuitem)
{
	if (gtk_check_menu_item_get_active (
				GTK_CHECK_MENU_ITEM (menuitem)) == TRUE)
	{
		if (dialog_env_list == NULL) {
			dialog_env_list = new_dialog_env_list (
							main_close_list,
							menuitem);
		}
		else {
			env_list_refresh_widget (dialog_env_list);
			gtk_widget_show (dialog_env_list);
		}
	}
	else {
		gtk_widget_hide (dialog_env_list);
	}
}

void
main_on_show_calendars_activate (GtkMenuItem *menuitem)
{
	if (gtk_check_menu_item_get_active (
			GTK_CHECK_MENU_ITEM (menuitem)) == TRUE)
	{
		if (dialog_cal_list == NULL) {
			dialog_cal_list = new_dialog_calendar_list (
							main_close_list,
							menuitem);
		}
		else {
			calendar_list_refresh_widget (dialog_cal_list);
			gtk_widget_show (dialog_cal_list);
		}
	}
	else {
		gtk_widget_hide (dialog_cal_list);
	}
}

void
main_on_find_clicked (GtkToolButton *button)
{
	if (dialog_find == NULL) {
		dialog_find = new_dialog_find (application_main);
	}
	else {
		gtk_window_present (GTK_WINDOW (dialog_find));
	}
}

void
main_on_find_activated (GtkMenuItem *menuitem)
{
	main_on_find_clicked (NULL);
}

void
main_on_delete_clicked (GtkToolButton *button)
{
	main_cb_delete (application_main);
}

void
main_on_delete_activated (GtkMenuItem *menuitem)
{
	main_on_delete_clicked (NULL);
}

void
main_on_cut_clicked (GtkToolButton *button)
{
	main_cb_cut (application_main);
}

void
main_on_cut_activated (GtkMenuItem *menuitem)
{
	main_on_cut_clicked (NULL);
}

void
main_on_copy_clicked (GtkToolButton *button)
{
	main_cb_copy (application_main);
}

void
main_on_copy_activated (GtkMenuItem *menuitem)
{
	main_on_copy_clicked (NULL);
}

void
main_on_paste_clicked (GtkToolButton *button)
{
	main_cb_paste (application_main);
}

void
main_on_paste_activated (GtkMenuItem *menuitem)
{
	main_on_paste_clicked (NULL);
}


static void
main_destroy_prefences (gpointer user_data)
{
	if (dialog_pref != NULL) {
		gtk_widget_destroy (dialog_pref);
		dialog_pref = NULL;
	}
}

void
main_on_preferences_activated (GtkMenuItem *menuitem)
{
	if (dialog_pref == NULL) {
		dialog_pref = new_dialog_pref (main_destroy_prefences, NULL);
	}
	else {
		gtk_window_present (GTK_WINDOW (dialog_pref));
	}
}

void
status_bar_activate (GtkMenuItem *menuitem)
{
	GtkWidget *bar;

	if (application_main != NULL) {
		bar = lookup_widget (application_main, "appbar_main");
		if (gtk_check_menu_item_get_active (
				GTK_CHECK_MENU_ITEM (menuitem)) == TRUE)
		{
			gtk_widget_show (bar);
		}
		else {
			gtk_widget_hide (bar);
		}
	}
}

void
location_bar_activate (GtkMenuItem *menuitem)
{
	if (menus[SCHEDWI_MAIN_LOCATION_BAR] != NULL) {
		if (gtk_check_menu_item_get_active (
				GTK_CHECK_MENU_ITEM (menuitem)) == TRUE)
		{
			gtk_widget_show (menus[SCHEDWI_MAIN_LOCATION_BAR]);
		}
		else {
			gtk_widget_hide (menus[SCHEDWI_MAIN_LOCATION_BAR]);
		}
	}
}

void
zoomin_activate ()
{
	if (application_main != NULL) {
		zoom_in (application_main);
	}
}

void
zoomout_activate ()
{
	if (application_main != NULL) {
		zoom_out (application_main);
	}
}

void
normal_size_activate ()
{
	if (application_main != NULL) {
		zoom_normal (application_main);
	}
}

void
show_grid_activate (GtkMenuItem *menuitem)
{
	if (application_main != NULL) {
		if (gtk_check_menu_item_get_active (
				GTK_CHECK_MENU_ITEM (menuitem)) == TRUE)
		{
			show_hide_grid (application_main, TRUE);
		}
		else {
			show_hide_grid (application_main, FALSE);
		}
	}
}

void
snap_to_grid_activate (GtkMenuItem *menuitem)
{
	if (application_main != NULL) {
		if (gtk_check_menu_item_get_active (
				GTK_CHECK_MENU_ITEM (menuitem)) == TRUE)
		{
			snap_grid (application_main, TRUE);
		}
		else {
			snap_grid (application_main, FALSE);
		}
	}
}

gboolean
show_grid_get_active ()
{
	if (menus[SCHEDWI_MAIN_MENU_SHOWGRID] != NULL) {
		return gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (
					menus[SCHEDWI_MAIN_MENU_SHOWGRID]));
	}
	else {
		return FALSE;
	}
}

gboolean
snap_to_grid_get_active ()
{
	if (menus[SCHEDWI_MAIN_MENU_SNAPGRID] != NULL) {
		return gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (
					menus[SCHEDWI_MAIN_MENU_SNAPGRID]));
	}
	else {
		return FALSE;
	}
}

void
reload_activate (GtkMenuItem *menuitem)
{
	if (application_main != NULL) {
		jobset_list_refresh (application_main);
	}
}

void
reload_tool_activate (GtkToolButton *button)
{
	if (application_main != NULL) {
		jobset_list_refresh (application_main);
	}
}

void
location_activate (GtkEntry *entry)
{
	if (application_main != NULL) {
		main_location_changed (	gtk_entry_get_text (entry),
					application_main);
	}
}

void
login_activate (GtkMenuItem *menuitem)
{
	if (application_main != NULL) {
		new_dialog_login (application_main);
	}
}

void
logout_activate (GtkMenuItem *menuitem)
{
	if (application_main != NULL) {
		login_logout (application_main);
	}
}

void
database_assistant_activate (GtkMenuItem *menuitem)
{
	new_dialog_druid (NULL, NULL);
}

void
main_copy_cut_set_sensitive (gboolean status)

{
	gtk_widget_set_sensitive (menus[SCHEDWI_MAIN_MENU_CUT], status);
	gtk_widget_set_sensitive (menus[SCHEDWI_MAIN_MENU_COPY], status);
	gtk_widget_set_sensitive (menus[SCHEDWI_MAIN_TOOL_CUT], status);
	gtk_widget_set_sensitive (menus[SCHEDWI_MAIN_TOOL_COPY], status);
}

void
main_paste_set_sensitive (gboolean status)
{
	gtk_widget_set_sensitive (menus[SCHEDWI_MAIN_MENU_PASTE], status);
	gtk_widget_set_sensitive (menus[SCHEDWI_MAIN_TOOL_PASTE], status);
}


void
main_delete_set_sensitive (gboolean status)

{
	gtk_widget_set_sensitive (menus[SCHEDWI_MAIN_MENU_DELETE], status);
	gtk_widget_set_sensitive (menus[SCHEDWI_MAIN_TOOL_DELETE], status);
}

/*
 * Quit
 *
 * Return:
 *    TRUE --> Quit cancelled
 *   FALSE --> The application must exit
 */
gboolean
schedwi_main_quit ()
{
	GtkWidget *dialog;
	gint ret;

	if (application_main != NULL) {
		dialog = gtk_message_dialog_new_with_markup (
					GTK_WINDOW (application_main),
					GTK_DIALOG_DESTROY_WITH_PARENT,
					GTK_MESSAGE_QUESTION,
					GTK_BUTTONS_NONE,
_("<span weight=\"bold\" size=\"larger\">Are you sure you want to quit?</span>"));
		gtk_dialog_add_buttons (GTK_DIALOG (dialog),
					GTK_STOCK_CANCEL,
					GTK_RESPONSE_NO,
					GTK_STOCK_QUIT,
					GTK_RESPONSE_YES,
					NULL);
		ret = gtk_dialog_run (GTK_DIALOG (dialog));
		gtk_widget_destroy (dialog);
		if (ret == GTK_RESPONSE_YES) {
			/* Cleanup */
			destroy_db_connection ();
			cursor_cleanup ();
			empty_cache_pixbuf ();
			schedwi_gconf_destroy ();
			return FALSE;
		}
	}
	return TRUE;
}

void
schedwi_main_destroy ()
{
	if (application_main != NULL) {
		gtk_widget_destroy (application_main);
	}
}

void
schedwi_main_login ()
{
	GConfValue *v;

	v = gconf_client_get (	schedwi_gconf,
				GCONF_PATH "/database/host",
				NULL);
	if (v != NULL) {
		gconf_value_free (v);
		new_dialog_login (application_main);
	}
	else {
		/* First time, launch the Database assistant */
		new_dialog_druid (	(void (*)(gpointer))new_dialog_login,
					application_main);
	}
}

void
schedwi_main_about ()
{
	char *license_text;
	GdkPixbuf *logo;

	const char *authors[] = {
		"Herve Quatremain <rv4m@yahoo.co.uk>",
		NULL
	};

	const char *documenters [] = {
		"Herve Quatremain <rv4m@yahoo.co.uk>", 
		NULL
	};

	const char *license[] = {
		"Schedwi is free software; you can redistribute it and/or modify "
		"it under the terms of the GNU General Public License as published by "
		"the Free Software Foundation; either version 3 of the License, or "
		"(at your option) any later version.",
		"Schedwi is distributed in the hope that it will be useful, "
		"but WITHOUT ANY WARRANTY; without even the implied warranty of "
		"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the "
		"GNU General Public License for more details.",
		"You should have received a copy of the GNU General Public License "
		"along with this program.  If not, see "
		"<http://www.gnu.org/licenses/>."
	};

	license_text = g_strconcat (	license[0], "\n\n", license[1], "\n\n",
					license[2], "\n\n", NULL);

	logo = create_pixbuf ("gschedwi/logo.png");

	gtk_show_about_dialog (	GTK_WINDOW (application_main),
				"version", VERSION,
				"copyright", "Copyright (C) 2007 Herve Quatremain",
				"comments", _("Schedwi is a command scheduler."),
				"authors", authors,
				"documenters", documenters,
				"translator-credits", "Herve Quatremain <rv4m@yahoo.co.uk>",
				"logo", logo,
				"license", license_text,
				"wrap-license", TRUE,
				"website", "http://www.nongnu.org/schedwi",
				NULL);

	g_object_unref (logo);
	g_free (license_text);
}

/*------------------------======= End Of File =======------------------------*/
