/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* canvas_utils.c -- Utility functions for the job/jobset canvas */

#include <schedwi.h>
#include <canvas_utils.h>

/* Zoom levels */
static const guint levels[] = {0, 9, 12, 18, 25, 33, 50, 66,
			100,
			150, 200, 300, 400, 550, 800, 1100, 1600, 2300,
			0};
/* Index of the 100% zoom in the previous levels array */
static const gint normal_level_id = 8;


/*
 * Return the default text size in points to use with CanvasItemText items
 */
gdouble
canvas_utils_get_default_text_size (GnomeCanvas *canvas)
{
	gint level;
	gpointer *p;

	p = g_object_get_data (G_OBJECT (canvas), "zoom");
	if (p == NULL) {
		level = normal_level_id;
	}
	else {
		level = GPOINTER_TO_INT (p);
	}
	return 10.0 * levels[level] / 100.0;
}


/*
 * Change the font size of all the texts in the canvas according to the
 * given zoom ratio.  This function is inspired from
 * gnome_canvas_text_set_zoom_size() (in gxkcanvas.c) from the Beast project
 * (http://beast.gtk.org/)
 */  
static void
text_zoom (GnomeCanvasGroup *group, guint old_zoom, guint new_zoom)
{
	GList *list;
	GnomeCanvasItem *item;
	gdouble size_points;
	gdouble zoom_ratio;

	zoom_ratio = (gdouble)new_zoom / (gdouble)old_zoom;

	for (list = group->item_list; list != NULL; list = g_list_next (list))
	{
		item = list->data;
		if (GNOME_IS_CANVAS_TEXT (item) == TRUE) {
			g_object_get (item, "size-points", &size_points, NULL);
			size_points *= zoom_ratio;
			g_object_set (item, "size-points", size_points, NULL);
		}
		else {
			if (GNOME_IS_CANVAS_GROUP (item) == TRUE) {
				text_zoom (GNOME_CANVAS_GROUP (item),
						old_zoom, new_zoom);
			}
		}
	}
}


/*
 * Zoom in the canvas and return the new zoom percentage level
 */
guint
canvas_utils_zoom_in (GnomeCanvas *canvas)
{
	gint level;
	guint old_zoom, new_zoom;
	gpointer *p;

	p = g_object_get_data (G_OBJECT (canvas), "zoom");
	if (p == NULL) {
		level = normal_level_id;
	}
	else {
		level = GPOINTER_TO_INT (p);
	}
	old_zoom = levels[level];
	new_zoom = levels[++level];
	if (new_zoom == 0) {
		return old_zoom;
	}

	text_zoom (gnome_canvas_root (canvas), old_zoom, new_zoom);
	gnome_canvas_set_pixels_per_unit (canvas, new_zoom / 100.0);
	g_object_set_data (	G_OBJECT (canvas), "zoom",
				GINT_TO_POINTER (level));
	return new_zoom;
}


/*
 * Zoom out the canvas and return the new zoom percentage level
 */
guint
canvas_utils_zoom_out (GnomeCanvas *canvas)
{
	gint level;
	guint old_zoom, new_zoom;
	gpointer *p;

	p = g_object_get_data (G_OBJECT (canvas), "zoom");
	if (p == NULL) {
		level = normal_level_id;
	}
	else {
		level = GPOINTER_TO_INT (p);
	}
	old_zoom = levels[level];
	if (level <= 1) {
		return old_zoom;
	}
	new_zoom = levels[--level];

	text_zoom (gnome_canvas_root (canvas), old_zoom, new_zoom);
	gnome_canvas_set_pixels_per_unit (canvas, new_zoom / 100.0);
	g_object_set_data (	G_OBJECT (canvas), "zoom",
				GINT_TO_POINTER (level));
	return new_zoom;
}


/*
 * Zoom to the normal size of the canvas and return the new zoom percentage
 * level (100)
 */
guint
canvas_utils_zoom_normal (GnomeCanvas *canvas)
{
	gint level;
	guint old_zoom, new_zoom;
	gpointer *p;

	p = g_object_get_data (G_OBJECT (canvas), "zoom");
	if (p == NULL) {
		level = normal_level_id;
	}
	else {
		level = GPOINTER_TO_INT (p);
	}
	old_zoom = levels[level];
	new_zoom = levels[normal_level_id];

	text_zoom (gnome_canvas_root (canvas), old_zoom, new_zoom);
	gnome_canvas_set_pixels_per_unit (canvas, new_zoom / 100.0);
	g_object_set_data (	G_OBJECT (canvas), "zoom",
				GINT_TO_POINTER (normal_level_id));
	return new_zoom;
}


/*
 * Retrieve the current zoom level (an index in the levels array)
 */
gint
canvas_utils_zoom_save (GnomeCanvas *canvas)
{
	gpointer *p;

	p = g_object_get_data (G_OBJECT (canvas), "zoom");
	if (p == NULL) {
		return normal_level_id;
	}
	else {
		return GPOINTER_TO_INT (p);
	}
}


/*
 * Set the zoom to the provided level (an index in the levels array) and
 * return the new zoom percentage
 */
guint
canvas_utils_zoom_restore (GnomeCanvas *canvas, gint new_level)
{
	gint level;
	guint old_zoom, new_zoom;
	gpointer *p;

	p = g_object_get_data (G_OBJECT (canvas), "zoom");
	if (p == NULL) {
		level = normal_level_id;
	}
	else {
		level = GPOINTER_TO_INT (p);
	}
	old_zoom = levels[level];
	new_zoom = levels[new_level];

	text_zoom (gnome_canvas_root (canvas), old_zoom, new_zoom);
	gnome_canvas_set_pixels_per_unit (canvas, new_zoom / 100.0);
	g_object_set_data (	G_OBJECT (canvas), "zoom",
				GINT_TO_POINTER (new_level));
	return new_zoom;
}


/*
 * Create and return a GnomeCanvasGroup (created in the provided dst group)
 * which position is the same as the src group with a resize ratio applied
 */
static GnomeCanvasGroup *
canvas_utils_new_group (GnomeCanvasGroup *dst, GnomeCanvasGroup *src,
			gdouble ratio)
{
	gdouble x, y;
	
	g_object_get (src, "x", &x, "y", &y, NULL);
	return (GnomeCanvasGroup *)gnome_canvas_item_new (
						dst,
						gnome_canvas_group_get_type (),
						"x", x * ratio,
						"y", y * ratio,
						NULL);
}


/*
 * Create a new line in the dst canvas group.  The new line is a copy of the
 * provided one but resized and positionned using ratio
 */
static void
canvas_utils_copy_line (GnomeCanvasGroup *dst, GnomeCanvasLine *line,
			gdouble ratio)
{
	GnomeCanvasPoints *points;
	GdkColor *color;
	gint i;

	g_object_get (line, "points", &points, "fill-color-gdk", &color, NULL);
	for (i = 0; i < points->num_points; i++) {
		points->coords[i * 2] *= ratio;
		points->coords[i * 2 + 1] *= ratio;
	}
	gnome_canvas_item_new (	dst,
				gnome_canvas_line_get_type (),
				"points", points,
				"fill-color-gdk", color,
				NULL);
	gnome_canvas_points_free (points);
}


/*
 * Create a new pixbuf canvas item in the dst canvas group.  The new pixbuf
 * item is a copy of the provided one but resized and positionned according
 * the provided ratio
 */
static void
canvas_utils_copy_pixbuf (	GnomeCanvasGroup *dst,
				GnomeCanvasPixbuf *pixbuf_item,
				gdouble ratio)
{
	gdouble x, y;
	gint width, height;
	GdkPixbuf *pixbuf, *new_pixbuf;
	
	g_object_get (	pixbuf_item,
			"x", &x, "y", &y,
			"pixbuf", &pixbuf,
			NULL);
	width = gdk_pixbuf_get_width (pixbuf);
	height = gdk_pixbuf_get_height (pixbuf);
	new_pixbuf = gdk_pixbuf_scale_simple (pixbuf,
					(int)width * ratio,
					(int)height * ratio,
					GDK_INTERP_BILINEAR);
	gnome_canvas_item_new (	dst,
				gnome_canvas_pixbuf_get_type (),
				"pixbuf", new_pixbuf,
				"x", x * ratio,
				"y", y * ratio,
				NULL);
	g_object_unref ((gpointer) new_pixbuf);
}


/*
 * Create a new rectangle canvas item in the dst canvas group.  The new item
 * is a copy of the provided one but resized and positionned according
 * the provided ratio
 */
static void
canvas_utils_copy_rect (GnomeCanvasGroup *dst, GnomeCanvasRect *rect,
			gdouble ratio)
{
	gdouble x1, y1, x2, y2;
	GdkColor *color;
	
	g_object_get (	rect,
			"x1", &x1, "y1", &y1,
			"x2", &x2, "y2", &y2,
			"fill-color-gdk", &color,
			NULL);
	gnome_canvas_item_new (	dst,
				gnome_canvas_rect_get_type (),
				"x1", x1 * ratio,
				"y1", y1 * ratio,
				"x2", x2 * ratio,
				"y2", y2 * ratio,
				"fill-color-gdk", color,
				NULL);
}


/*
 * Copy the content of the provided src canvas group in dst (the items are
 * resized and positionned according to ratio)
 */
static void
canvas_utils_copy_group (	GnomeCanvasGroup *dst, GnomeCanvasGroup *src,
				gdouble ratio)
{
	GList *list;
	GnomeCanvasItem *item;
	GnomeCanvasGroup *new_group;

	for (list = src->item_list; list != NULL; list = g_list_next (list))
	{
		item = list->data;
		if (GNOME_IS_CANVAS_LINE (item) == TRUE) {
			canvas_utils_copy_line (dst,
						GNOME_CANVAS_LINE (item),
						ratio);
		}
		else {
		if (GNOME_IS_CANVAS_PIXBUF (item) == TRUE) {
			canvas_utils_copy_pixbuf (dst,
						GNOME_CANVAS_PIXBUF (item),
						ratio);
		}
		else {
		if (GNOME_IS_CANVAS_RECT (item) == TRUE) {
			canvas_utils_copy_rect (dst,
						GNOME_CANVAS_RECT (item),
						ratio);
		}
		else {
		if (GNOME_IS_CANVAS_GROUP (item) == TRUE) {
			new_group = canvas_utils_new_group (dst,
						GNOME_CANVAS_GROUP (item),
						ratio); 
			canvas_utils_copy_group (new_group,
						GNOME_CANVAS_GROUP (item),
						ratio);
		}}}}
	}
}


/*
 * Copy the src canvas in dst.  All the items are resized in dst
 * according to ratio.
 * Only lines, pixbuf, rectangles an groups are copied (and only with
 * basics attributes).  The other item types are ignored
 */
void
canvas_utils_copy (GnomeCanvas *dst, GnomeCanvas *src, gdouble ratio)
{
	canvas_utils_copy_group (	gnome_canvas_root (dst),
					gnome_canvas_root (src), ratio);
} 


/*
 * Remove all the items from the provided canvas
 */
void
canvas_utils_clean (GnomeCanvas *canvas)
{
	GList *list;
	GSList *new_list = NULL, *tmp_list;
	GnomeCanvasGroup *root_group;

	root_group = gnome_canvas_root (canvas);
	for (	list = root_group->item_list;
		list != NULL;
		list = g_list_next (list))
	{
		new_list = g_slist_prepend (new_list, list->data);
	}
	for (	tmp_list = new_list;
		tmp_list != NULL;
		tmp_list = g_slist_next (tmp_list))
	{
		gtk_object_destroy (GTK_OBJECT (tmp_list->data));
	}
	g_slist_free (new_list);
}


/*
 * Get the scroll positions
 */
void
canvas_utils_get_position (GnomeCanvas *canvas, gdouble *v, gdouble *h)
{
	int x, y;

	gnome_canvas_get_scroll_offsets (canvas, &x, &y);
	*v = (gdouble) x;
	*h = (gdouble) y;
}


/*
 * Set the scroll position
 */
void
canvas_utils_set_position (GnomeCanvas *canvas, gdouble v, gdouble h)
{
	gnome_canvas_scroll_to (canvas, (int)v, (int)h);
}


/*
 * Store the size of the canvas
 */
void
canvas_utils_set_size (GnomeCanvas *canvas, gint width, gint height)
{
	gnome_canvas_set_scroll_region (canvas,
					0.0, 0.0,
					(gdouble)width, (gdouble)height);
	gnome_canvas_set_center_scroll_region (canvas, FALSE);

	g_object_set_data (	G_OBJECT (canvas),
				"width", GINT_TO_POINTER (width));
	g_object_set_data (	G_OBJECT (canvas),
				"height", GINT_TO_POINTER (height));

	/* Scroll to (0,0) */
	gnome_canvas_scroll_to (canvas, 0, 0);
}


/*
 * Get the size of the canvas
 */
void
canvas_utils_get_size (GnomeCanvas *canvas, gint *width, gint *height)
{
	gpointer *p;

	p = g_object_get_data (G_OBJECT (canvas), "width");
	if (p == NULL) {
		*width = GTK_WIDGET (canvas)->allocation.width;
	}
	*width = GPOINTER_TO_INT (p);

	p = g_object_get_data (G_OBJECT (canvas), "height");
	if (p == NULL) {
		*height = GTK_WIDGET (canvas)->allocation.height;
	}
	*height = GPOINTER_TO_INT (p);
}

/*-----------------============== End Of File ==============-----------------*/
