/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* calendar_cal_cb.c -- GUI functions for the calendar window */

#include <schedwi.h>

#include <schedwi_interface.h>

#include <message_windows.h>
#include <cursor.h>
#include <schedwi_g_utils.h>
#include <sql_calendar.h>
#include <calendar.h>
#include <calendar_canvas.h>
#include <calendar_list_cb.h>
#include <calendar_cal_cb.h>

#define MIN_YEAR 1971
#define ERROR_COLOR "red"
#define COMMENT_COLOR "blue"

/*
 * Get the error message string for the provided cal_errcode_t code
 */
static const gchar *
cal_errcode_to_str (cal_errcode_t err_code)
{
	switch (err_code) {
		case CAL_MALLOC:
			return _("Memory allocation error");

		case CAL_NOMATCH:
			return _("The provided day does not match the calendar");
		case CAL_BADMONTHNAME:
			return _("Invalid month name");

		case CAL_BADDAYNAME:
			return _("Invalid day name");

		case CAL_MONTHOOR:
			return _("Month number must be between 1 and 12");

		case CAL_DAYOOR:
			return _("Day number must be between 1 and 31");

		case CAL_BADOPTION:
			return _("Invalid option");

		case CAL_EMPTYFIELD:
			return _("A required field is empty");

		default:
			return _("No error");
	}
}


/*
 * Remove all our tags from the provided text buffer
 */
static void
clear_tags (GtkTextBuffer *buffer)
{
	GtkTextIter start, end;

	gtk_text_buffer_get_start_iter (buffer, &start);
	gtk_text_buffer_get_end_iter (buffer, &end);
	gtk_text_buffer_remove_tag_by_name (buffer, "error", &start, &end);
	gtk_text_buffer_remove_tag_by_name (buffer, "comment", &start, &end);
}


/*
 * Highlight all comments.  Formula must be the text from the buffer (obtained
 * by gtk_text_buffer_get_text())
 */
static void
highlight_comments (GtkTextBuffer *buffer, const gchar *formula)
{
	gint i;
	GtkTextIter start, end;

	if (buffer == NULL || formula == NULL) {
		return;
	}

	i = 0;
	while (formula[i] != '\0') {
		if (formula[i] == '#') {
			gtk_text_buffer_get_iter_at_offset (buffer, &start,
				(gint)g_utf8_pointer_to_offset (formula,
								formula + i));
			while (formula[i] != '\0' && formula[i] != '\n') {
				i++;
			}
			gtk_text_buffer_get_iter_at_offset (buffer, &end,
				(gint)g_utf8_pointer_to_offset (formula,
								formula + i));
			gtk_text_buffer_apply_tag_by_name (	buffer,
								"comment",
								&start, &end);
		}
		else {
			i++;
		}
	}
}


/*
 * Mark the days corresponding to the formula in the calendar_canvas_t object
 * and `syntax highlight' the formula in the text view.
 * The provided formula must be the text from the text buffer in the text view
 * (obtain by gtk_text_buffer_get_text())
 * If there is a syntax error in the formula, the Formula text view is updated
 * to show the error and an error popup window is displayed (only if
 * popup_error is TRUE).  If focus_on_error is TRUE, the text view cursor is
 * moved on the error and the text view grabs the focus
 *
 * Return:
 *   0 --> No error
 *  -1 --> Memory allocation error (a message has been displayed)
 *  -2 --> Syntax error in formula (a popup error message has been
 *         displayed if popup_error is TRUE)
 */
static gint
set_formula_canvas (	GtkWidget *tview, calendar_canvas_t *cal_canvas,
			const char *formula,
			gboolean popup_error, gboolean focus_on_error)
{
	GtkTextBuffer *buffer;
	GtkTextIter start, end;
	cal_errcode_t ret;
	int error_idx_in_formula;
	GtkTextMark *mark;

	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (tview));
	clear_tags (buffer);
	highlight_comments (buffer, formula);

	/* Mark the days corresponding to the formula */
	ret = set_calendar (cal_canvas, formula, &error_idx_in_formula);
	if (ret == CAL_NOERROR) {
		return 0;
	}

	/* Memory allocation error */
	if (ret == CAL_MALLOC) {
		error_window (_("Memory allocation error"), NULL);
		return -1;
	}


	/*
	 * Syntax error in the calendar
	 */

	/* Show the error in the text view */
	gtk_text_buffer_get_iter_at_offset (	buffer, &end,
						error_idx_in_formula);
	start = end;
	gtk_text_iter_backward_chars (  &start,
					gtk_text_iter_get_line_offset (&end));
	gtk_text_iter_forward_line (&end);
	gtk_text_buffer_apply_tag_by_name (buffer, "error", &start, &end);

	if (focus_on_error == TRUE) {
		gtk_text_buffer_get_iter_at_offset (	buffer, &start,
							error_idx_in_formula);
		gtk_text_buffer_place_cursor (buffer, &start);
		mark = gtk_text_buffer_create_mark (	buffer, NULL, &start,
							FALSE);
		gtk_text_view_scroll_mark_onscreen (	GTK_TEXT_VIEW (tview),
							mark);
		gtk_text_buffer_delete_mark (buffer, mark);
		gtk_widget_grab_focus (tview);
	}

	/* Popup an error window */
	if (popup_error == TRUE) {
		error_window (	_("Calendar syntax error"),
				cal_errcode_to_str (ret));
	}
	return -2;
}


/*
 * Refresh the canvas
 */
static void
refresh_calendar_canvas (GtkWidget *widget, gboolean change_cursor_pos)
{
	GtkWidget *text, *canvas;
	GtkTextBuffer *buffer;
	GtkTextIter start, end;
	calendar_canvas_t *cal_canvas;
	gchar *formula;
	GtkTextMark *mark;

	text = lookup_widget (widget, "textview_cal_formula");
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (text));
	gtk_text_buffer_get_start_iter (buffer, &start);
	gtk_text_buffer_get_end_iter (buffer, &end);
	formula = gtk_text_buffer_get_text (buffer, &start, &end, FALSE);

	canvas = lookup_widget (widget, "canvas_cal_preview");
	cal_canvas = (calendar_canvas_t *)g_object_get_data (G_OBJECT (canvas),
							"calendar");

	/*
	 * If the formula is correct the cursor is moved at the end (if
	 * it can be moved)
	 */
	if (	   set_formula_canvas (text, cal_canvas, formula,
				FALSE, change_cursor_pos) == 0
		&& change_cursor_pos == TRUE)
	{
		gtk_text_buffer_get_end_iter (buffer, &end);
		gtk_text_buffer_place_cursor (buffer, &end);
		mark = gtk_text_buffer_create_mark (buffer, NULL, &end, FALSE);
		gtk_text_view_scroll_mark_onscreen (GTK_TEXT_VIEW (text), mark);
		gtk_text_buffer_delete_mark (buffer, mark);
	}
	g_free (formula);
}


/*
 * Return the week day string corresponding to the provided week day
 */
static const gchar *
weekday_to_str (GDateWeekday wd)
{
	switch (wd) {
		case G_DATE_MONDAY:
			return "Monday";
		case G_DATE_TUESDAY:
			return "Tuesday";
		case G_DATE_WEDNESDAY:
			return "Wednesday";
		case G_DATE_THURSDAY:
			return "Thursday";
		case G_DATE_FRIDAY:
			return "Friday";
		case G_DATE_SATURDAY:
			return "Saturday";
		case G_DATE_SUNDAY:
			return "Sunday";
		default:
			return "*";
	}
}


/*
 * Callback functions for the calendar_canvas
 */
static gboolean
set_day (calendar_canvas_t *cal, GDateDay d, GDateMonth m, gpointer data)
{
	GtkWidget *text = data;
	GtkTextBuffer *buffer;
	GtkTextIter start, end;
	gchar *formula, *str, *tmp;

	/* Retrieve the current formula from the text view */
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (text));
	gtk_text_buffer_get_start_iter (buffer, &start);
	gtk_text_buffer_get_end_iter (buffer, &end);
	formula = gtk_text_buffer_get_text (buffer, &start, &end, FALSE);
	
	/* Remove the provided day from the formula */
	str = g_strdup_printf ("not %d/%d", m, d);
	tmp = remove_string (formula, str);
	g_free (str);
	g_free (formula);

	/* Add the day to the formula (if not already) */
	str = g_strdup_printf ("%d/%d", m, d);
	if (search_string (tmp, str) == FALSE) {
		formula = g_strconcat (tmp, "\n", str, NULL);
		g_free (tmp);
	}
	else {
		formula = tmp;
	}
	g_free (str);

	/* Update the text view and the calendar preview */
	gtk_text_buffer_set_text (buffer, formula, -1);
	g_free (formula);
	refresh_calendar_canvas (text, TRUE);
	return FALSE;
}

static gboolean
unset_day (calendar_canvas_t *cal, GDateDay d, GDateMonth m, gpointer data)
{
	GtkWidget *text = data;
	GtkTextBuffer *buffer;
	GtkTextIter start, end;
	gchar *formula, *str, *tmp;

	/* Retrieve the current formula from the text view */
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (text));
	gtk_text_buffer_get_start_iter (buffer, &start);
	gtk_text_buffer_get_end_iter (buffer, &end);
	formula = gtk_text_buffer_get_text (buffer, &start, &end, FALSE);
	
	/* Remove the provided day from the formula */
	str = g_strdup_printf ("%d/%d", m, d);
	tmp = remove_string (formula, str);
	g_free (str);
	g_free (formula);

	/* Add the day to the formula */
	str = g_strdup_printf ("not %d/%d", m, d);
	if (search_string (tmp, str) == FALSE) {
		formula = g_strconcat (tmp, "\n", str, NULL);
		g_free (tmp);
	}
	else {
		formula = tmp;
	}
	g_free (str);

	/* Update the text view and the calendar preview */
	gtk_text_buffer_set_text (buffer, formula, -1);
	g_free (formula);
	refresh_calendar_canvas (text, TRUE);
	return FALSE;
}

static gboolean
set_wday (calendar_canvas_t *cal, GDateWeekday wd, gpointer data)
{
	GtkWidget *text = data;
	GtkTextBuffer *buffer;
	GtkTextIter start, end;
	gchar *formula, *str, *tmp;
	const gchar *wd_str;

	wd_str = weekday_to_str (wd);

	/* Retrieve the current formula from the text view */
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (text));
	gtk_text_buffer_get_start_iter (buffer, &start);
	gtk_text_buffer_get_end_iter (buffer, &end);
	formula = gtk_text_buffer_get_text (buffer, &start, &end, FALSE);
	
	/* Remove the provided week day from the formula */
	str = g_strdup_printf ("not */%s", wd_str);
	tmp = remove_string (formula, str);
	g_free (str);
	g_free (formula);

	/* Add the week day to the formula */
	str = g_strdup_printf ("*/%s", wd_str);
	if (search_string (tmp, str) == FALSE) {
		formula = g_strconcat (tmp, "\n", str, NULL);
		g_free (tmp);
	}
	else {
		formula = tmp;
	}
	g_free (str);

	/* Update the text view and the calendar preview */
	gtk_text_buffer_set_text (buffer, formula, -1);
	g_free (formula);
	refresh_calendar_canvas (text, TRUE);
	return FALSE;
}

static gboolean
unset_wday (calendar_canvas_t *cal, GDateWeekday wd, gpointer data)
{
	GtkWidget *text = data;
	GtkTextBuffer *buffer;
	GtkTextIter start, end;
	gchar *formula, *str, *tmp;
	const gchar *wd_str;

	wd_str = weekday_to_str (wd);

	/* Retrieve the current formula from the text view */
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (text));
	gtk_text_buffer_get_start_iter (buffer, &start);
	gtk_text_buffer_get_end_iter (buffer, &end);
	formula = gtk_text_buffer_get_text (buffer, &start, &end, FALSE);
	
	/* Remove the provided week day from the formula */
	str = g_strdup_printf ("*/%s", wd_str);
	tmp = remove_string (formula, str);
	g_free (str);
	g_free (formula);

	/* Add the week day to the formula */
	str = g_strdup_printf ("not */%s", wd_str);
	if (search_string (tmp, str) == FALSE) {
		formula = g_strconcat (tmp, "\n", str, NULL);
		g_free (tmp);
	}
	else {
		formula = tmp;
	}
	g_free (str);

	/* Update the text view and the calendar preview */
	gtk_text_buffer_set_text (buffer, formula, -1);
	g_free (formula);
	refresh_calendar_canvas (text, TRUE);
	return FALSE;
}

static gboolean
set_month (calendar_canvas_t *cal, GDateMonth m, gpointer data)
{
	GtkWidget *text = data;
	GtkTextBuffer *buffer;
	GtkTextIter start, end;
	gchar *formula, *str, *tmp;

	/* Retrieve the current formula from the text view */
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (text));
	gtk_text_buffer_get_start_iter (buffer, &start);
	gtk_text_buffer_get_end_iter (buffer, &end);
	formula = gtk_text_buffer_get_text (buffer, &start, &end, FALSE);
	
	/* Remove the provided month from the formula */
	str = g_strdup_printf ("not %d/*", m);
	tmp = remove_string (formula, str);
	g_free (str);
	g_free (formula);

	/* Add the month to the formula */
	str = g_strdup_printf ("%d/*", m);
	if (search_string (tmp, str) == FALSE) {
		formula = g_strconcat (tmp, "\n", str, NULL);
		g_free (tmp);
	}
	else {
		formula = tmp;
	}
	g_free (str);

	/* Update the text view and the calendar preview */
	gtk_text_buffer_set_text (buffer, formula, -1);
	g_free (formula);
	refresh_calendar_canvas (text, TRUE);
	return FALSE;
}

static gboolean
unset_month (calendar_canvas_t *cal, GDateMonth m, gpointer data)
{
	GtkWidget *text = data;
	GtkTextBuffer *buffer;
	GtkTextIter start, end;
	gchar *formula, *str, *tmp;

	/* Retrieve the current formula from the text view */
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (text));
	gtk_text_buffer_get_start_iter (buffer, &start);
	gtk_text_buffer_get_end_iter (buffer, &end);
	formula = gtk_text_buffer_get_text (buffer, &start, &end, FALSE);
	
	/* Remove the provided month from the formula */
	str = g_strdup_printf ("%d/*", m);
	tmp = remove_string (formula, str);
	g_free (str);
	g_free (formula);

	/* Add the month to the formula */
	str = g_strdup_printf ("not %d/*", m);
	if (search_string (tmp, str) == FALSE) {
		formula = g_strconcat (tmp, "\n", str, NULL);
		g_free (tmp);
	}
	else {
		formula = tmp;
	}
	g_free (str);

	/* Update the text view and the calendar preview */
	gtk_text_buffer_set_text (buffer, formula, -1);
	g_free (formula);
	refresh_calendar_canvas (text, TRUE);
	return FALSE;
}


/*
 * Functions used to set values in the widgets
 */
static void
set_id (const char *value, void *data)
{
	/* Store the host id in the widget */
	g_object_set_data_full (G_OBJECT (data),
				"cal_id", g_strdup (value),
				g_free);
}

static void
set_name (const char *value, void *data)
{
	GtkWidget *entry;

	entry = lookup_widget (GTK_WIDGET (data), "entry_cal_name");
	gtk_entry_set_text (GTK_ENTRY (entry), value);
	gtk_window_set_title (GTK_WINDOW (data), value);
}

static void
set_description (const char *value, void *data)
{
	GtkWidget *text;
	GtkTextBuffer *buffer;

	text = lookup_widget (GTK_WIDGET (data), "textview_cal_description");
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (text));
	gtk_text_buffer_set_text (buffer, value, -1);
}

static void
set_formula (const char *value, void *data)
{
	GtkWidget *text, *canvas;
	GtkTextBuffer *buffer;
	calendar_canvas_t *cal_canvas;

	/* Copy the formula string in the Formula text view */ 
	text = lookup_widget (GTK_WIDGET (data), "textview_cal_formula");
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (text));
	gtk_text_buffer_set_text (buffer, value, -1);

	/* Compile the formula and mark the corresponding days in the canvas */
	canvas = lookup_widget (GTK_WIDGET (data), "canvas_cal_preview");
	cal_canvas = (calendar_canvas_t *)g_object_get_data (G_OBJECT (canvas),
							"calendar");
	set_formula_canvas (text, cal_canvas, value, TRUE, TRUE);
}


/*
 * Cell renderer function
 */
static void
cell_renderer_func (	GtkTreeViewColumn *tree_column,
			GtkCellRenderer *cell,
			GtkTreeModel *tree_model,
			GtkTreeIter *iter,
			gpointer data)
{
	gchar *name;
	gint type;

	gtk_tree_model_get (tree_model, iter, 0, &name, 2, &type, -1);
	if (type == 0 || type == 1) {
		/* Folder or calendar */
		g_object_set (cell, "text", name, NULL);
	}
	else {
		/* Empty folder */
		g_object_set (cell, "markup", _("    <i>(Empty)</i>"), NULL);
	}
	g_free (name);
}


/*
 * Add a column to the view
 */
static void
build_view (GtkTreeView *view)
{
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;
	GtkTreeSelection *select;

	/* Calendar name column */
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_title (column, _("Calendar"));
	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_set_cell_data_func (
				column, renderer,
				cell_renderer_func, NULL, NULL);
	gtk_tree_view_append_column (view, column);

	/* Selection setup */
	select = gtk_tree_view_get_selection (view);
	gtk_tree_selection_set_mode (select, GTK_SELECTION_MULTIPLE);
}


static void
text_buffer_changed_cb (GtkTextBuffer *textbuffer, gpointer user_data)
{
	refresh_calendar_canvas (GTK_WIDGET (user_data), FALSE);
}


/*
 * Initialize the window
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error (an error popup has been displayed for the user)
 */
static int
dialog_cal_init (	GtkTreeView *view, GtkWidget *dialog,
			const gchar *id, const gchar *parent_id)
{
	GtkWidget *cal_view, *label_year;
	GtkTreeModel *model;
	GnomeCanvas *canvas;
	calendar_canvas_t *cal_canvas;
	GDate *current_date;
	gchar *year_str;
	GtkTextView *text_view;
	GtkTextBuffer *buffer;

	gtk_window_set_type_hint (	GTK_WINDOW (dialog),
					GDK_WINDOW_TYPE_HINT_NORMAL);

	/*
	 * Store the tree view of the calendar list to be able to
	 * refresh it when a host is updated
	 */
	g_object_set_data (	G_OBJECT (dialog),
				"cal_view", (gpointer)view);

	/* Retrieve the model and associate it with the list */
	cal_view = lookup_widget (dialog, "treeview_cal_list");
	model = gtk_tree_view_get_model (view);
	gtk_tree_view_set_model (GTK_TREE_VIEW (cal_view), model);

	/* Add the column, cell and selection callbacks */
	build_view (GTK_TREE_VIEW (cal_view));


	/*
	 * Initialize the calendar canvas
	 */

	canvas = (GnomeCanvas *)lookup_widget (dialog, "canvas_cal_preview");
	cal_canvas = new_calendar_canvas (canvas, READWRITE);
	current_date = g_date_new ();
	g_date_set_time (current_date, time (NULL));
	draw_year_canvas (cal_canvas, g_date_get_year (current_date));

	/* Callbacks for the calendar canvas */
	text_view = (GtkTextView *) lookup_widget (dialog,
						"textview_cal_formula");
	set_callbacks_set_day (cal_canvas, set_day, text_view);
	set_callbacks_unset_day (cal_canvas, unset_day, text_view);
	set_callbacks_set_wday (cal_canvas, set_wday, text_view);
	set_callbacks_unset_wday (cal_canvas, unset_wday, text_view);
	set_callbacks_set_month (cal_canvas, set_month, text_view);
	set_callbacks_unset_month (cal_canvas, unset_month, text_view);

	/* Change the year label */
	label_year = lookup_widget (dialog, "label_cal_year_preview");
	year_str = g_strdup_printf (    "<big><b>%d</b></big>",
					g_date_get_year (current_date));
	gtk_label_set_markup (GTK_LABEL (label_year), year_str);
	g_free (year_str);
	g_date_free (current_date);

	/*
	 * The calendar_canvas_t object is associated with the canvas.
	 * It must be freed before the canvas is destroyed.  It's why the
	 * delete_event signal is catched.
	 * See calendar_cal_cancel_clicked() and calendar_cal_delete_event()
	 * and calendar_cal_ok_clicked()
	 */
	g_object_set_data (G_OBJECT (canvas), "calendar", cal_canvas);


	/*
	 * Initialize the Formula text view (add tags for marking errors
	 * and comments)
	 */
	buffer = gtk_text_view_get_buffer (text_view);
	gtk_text_buffer_create_tag (	buffer, "error",
					"background", ERROR_COLOR, NULL);
	gtk_text_buffer_create_tag (	buffer, "comment",
					"foreground", COMMENT_COLOR, NULL);
	g_signal_connect (	G_OBJECT (buffer), "changed",
				G_CALLBACK (text_buffer_changed_cb),
				text_view);


	/*
	 * New calendar
	 */
	if (id == NULL || id[0] == '\0') {
		g_object_set_data_full (G_OBJECT (dialog),
					"parent_id", g_strdup (parent_id),
					g_free);
		return 0;
	}

	/*
	 * Edit calendar
	 */

	/* Retrieve the calendar main parameters */
	if (sql_cal_get_details (
				id,
				set_id,
				NULL,
				set_name,
				NULL,
				set_description,
				set_formula,
				dialog,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error")) != 0)
	{
		return -1;
	}

	return 0;
}


/*
 * Create a new calendar dialog for the provided calendar ID
 *
 * Return:
 *   The GtkWidget of the new window (which has been `show'ed by this function)
 *   or NULL in case of error
 */
GtkWidget *
new_dialog_cal_calendar (	GtkTreeView *view,
				const gchar *id, const gchar *parent_id)
{
	GtkWidget *dialog;

	dialog = create_dialog_cal ();
	if (dialog_cal_init (view, dialog, id, parent_id) != 0) {
		gtk_widget_destroy (dialog);
		return NULL;
	}
	else {
		gtk_widget_show (dialog);
		return dialog;
	}
}


/*
 * Retrieve the calendar parameters from the widgets
 *
 * Return:
 *   0 --> No error (*name, *description and *formula are set - only
 *         *description and *formula must be freed by the caller)
 *  -1 --> Error (a required parameter is missing - an error popup has been
 *         displayed for the user)
 */
static gint
get_cal_values (	GtkWidget *dialog, const gchar **name,
			gchar **description, gchar **formula)
{
	GtkWidget *entry, *text, *canvas;
	GtkTextBuffer *buffer;
	GtkTextIter start, end;
	calendar_canvas_t *cal_canvas;


	/* Name */
	entry = lookup_widget (dialog, "entry_cal_name");
	*name = gtk_entry_get_text (GTK_ENTRY (entry));
	if (*name == NULL || (*name)[0] == '\0') {
		error_window (	_("Error in the Calendar definition"),
				_("The Calendar Name field is required."));
		return -1;
	}

	/* Formula */
	text = lookup_widget (dialog, "textview_cal_formula");
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (text));
	gtk_text_buffer_get_start_iter (buffer, &start);
	gtk_text_buffer_get_end_iter (buffer, &end);
	*formula = gtk_text_buffer_get_text (buffer, &start, &end, FALSE);

	canvas = lookup_widget (dialog, "canvas_cal_preview");
	cal_canvas = (calendar_canvas_t *)g_object_get_data (G_OBJECT (canvas),
							"calendar");
	if (set_formula_canvas (text, cal_canvas, *formula, TRUE, TRUE) != 0) {
		g_free (*formula);
		*formula = NULL;
		return -1;
	}

	/* Description */
	text = lookup_widget (dialog, "textview_cal_description");
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (text));
	gtk_text_buffer_get_start_iter (buffer, &start);
	gtk_text_buffer_get_end_iter (buffer, &end);
	*description = gtk_text_buffer_get_text (buffer, &start, &end, FALSE);

	return 0;
}


/*
 * Store a new calendar in the database
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error (an error popup has been displayed for the user)
 */
static gint
cal_create (GtkWidget *dialog, const gchar *parent_id)
{
	const gchar *name;
	gchar *description = NULL, *formula = NULL;

	/*
	 * Add the new calendar to the database
	 */
	if (	   get_cal_values (dialog, &name, &description, &formula) != 0
		|| sql_cal_new (name, parent_id, "0", description, formula,
				NULL, 
				(void (*)(void *, const char*, unsigned int))
						error_window_check_duplicate,
				_("Database error")) != 0)
	{
		g_free (formula);
		g_free (description);
		return -1;
	}
	g_free (formula);
	g_free (description);
	return 0;
}


/*
 * Update a calendar in the database
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error (an error popup has been displayed for the user)
 */
static gint
cal_update (GtkWidget *dialog, const gchar *id)
{
	const gchar *name;
	gchar *description = NULL, *formula = NULL;

	/*
	 * Update the calendar parameters
	 */
	if (	   get_cal_values (dialog, &name, &description, &formula) != 0
		|| sql_cal_update (id, name, description, formula,
				(void (*)(void *, const char*, unsigned int))
						error_window_check_duplicate,
				_("Database error")) != 0)
	{
		g_free (formula);
		g_free (description);
		return -1;
	}
	g_free (formula);
	g_free (description);
	return 0;
}


/*
 * Callback for the `OK' button
 */
void
calendar_cal_ok_clicked (GtkButton *button)
{
	GtkWidget *dialog, *canvas;
	GtkTreeView *view;
	gchar *id, *parent_id;
	calendar_canvas_t *cal_canvas;
	gint ret;

	cursor_busy (GTK_WIDGET (button));
	dialog = lookup_widget (GTK_WIDGET (button), "dialog_cal");
	id = g_object_get_data (G_OBJECT (dialog), "cal_id");
	parent_id = g_object_get_data (G_OBJECT (dialog), "parent_id");
	view = g_object_get_data (G_OBJECT (dialog), "cal_view");
	if (id == NULL) {
		ret = cal_create (dialog, parent_id);
	}
	else {
		ret = cal_update (dialog, id);
	}
	cursor_normal (GTK_WIDGET (button));
	if (ret == 0) {
		calendar_list_refresh (view);
 
		/* Retrieve and free the associated calendar_canvas_t object */
		canvas = lookup_widget (dialog, "canvas_cal_preview");
		cal_canvas = (calendar_canvas_t *)g_object_get_data (
							G_OBJECT (canvas),
							"calendar");
		destroy_calendar_canvas (cal_canvas);
		gtk_widget_destroy (dialog);
	}
}


/*
 * delete-event callback (when the user closes the window)
 */
void
calendar_cal_delete_event (GtkWidget *widget)
{
	GtkWidget *canvas;
	calendar_canvas_t *cal_canvas;

	/* Retrieve and free the calendar_canvas_t object */
	canvas = lookup_widget (widget, "canvas_cal_preview");
	cal_canvas = (calendar_canvas_t *)g_object_get_data (G_OBJECT (canvas),
								"calendar");
	destroy_calendar_canvas (cal_canvas);
}


/*
 * Cancel button callback
 */
void
calendar_cal_cancel_clicked (GtkButton *button)
{
	GtkWidget *canvas, *dialog;
	calendar_canvas_t *cal_canvas;

	/* Retrieve and free the calendar_canvas_t object */
	canvas = lookup_widget (GTK_WIDGET (button), "canvas_cal_preview");
	cal_canvas = (calendar_canvas_t *)g_object_get_data (G_OBJECT (canvas),
								"calendar");
	destroy_calendar_canvas (cal_canvas);

	/* Destroy (close) the Calendar window */
	dialog = lookup_widget (GTK_WIDGET (button), "dialog_cal");
	gtk_widget_destroy (dialog);
}


/*
 * Callback from the `View previous year' button
 */
void
calendar_cal_year_prev_clicked (GtkButton *button)
{
	GtkWidget *canvas, *label_year;
	gchar *year_str;
	calendar_canvas_t *cal_canvas;

	canvas = lookup_widget (GTK_WIDGET (button), "canvas_cal_preview");
	cal_canvas = (calendar_canvas_t *)g_object_get_data (G_OBJECT (canvas),
							"calendar");

	if (cal_canvas->year <= MIN_YEAR) {
		return;
	}

	/* Draw the days for the new year */
	draw_year_canvas (cal_canvas, cal_canvas->year - 1);

	/* Change the year label */
	label_year = lookup_widget (	GTK_WIDGET (button),
					"label_cal_year_preview");
	year_str = g_strdup_printf ("<big><b>%d</b></big>", cal_canvas->year);
	gtk_label_set_markup (GTK_LABEL (label_year), year_str);
	g_free (year_str);
}


/*
 * Callback from the `View next year' button
 */
void
calendar_cal_year_next_clicked (GtkButton *button)
{
	GtkWidget *canvas, *label_year;
	gchar *year_str;
	calendar_canvas_t *cal_canvas;

	canvas = lookup_widget (GTK_WIDGET (button), "canvas_cal_preview");
	cal_canvas = (calendar_canvas_t *)g_object_get_data (G_OBJECT (canvas),
							"calendar");

	/* Draw the days for the new year */
	draw_year_canvas (cal_canvas, cal_canvas->year + 1);

	/* Change the year label */
	label_year = lookup_widget (	GTK_WIDGET (button),
					"label_cal_year_preview");
	year_str = g_strdup_printf ("<big><b>%d</b></big>", cal_canvas->year);
	gtk_label_set_markup (GTK_LABEL (label_year), year_str);
	g_free (year_str);
}


/*
 * Append to the provided string a comment giving the name of the calendar
 * for which formula will be added
 */
static void
append_comment_to_str (GtkTreeModel *model, GtkTreeIter *iter, GString *str)
{
	gchar *name;
	GtkTreeIter it, parent;
	GString *tmp;

	tmp = g_string_new ("");
	parent = *iter;
	do {
		it = parent;
		gtk_tree_model_get (model, &it, 0, &name, -1);
		g_string_prepend (tmp, name);
		g_string_prepend (tmp, "/");
		g_free (name);
	} while (gtk_tree_model_iter_parent (model, &parent, &it) == TRUE);

	g_string_append (str, "\n\n# ");
	g_string_append (str, _("From "));
	g_string_append (str, tmp->str);
	g_string_append (str, "\n");
	g_string_free (tmp, TRUE);	
}


/*
 * Append a formula to the provided string
 */
static void
append_formula_to_str (const char *formula, void *data)
{
	g_string_append ((GString *)data, formula);
}


/*
 * Function called for each selected calendars in the list.  The formula
 * of each one is copied in the text view
 */
static void
foreach_calendar (	GtkTreeModel *model,
			GtkTreePath *path,
			GtkTreeIter *iter,
			gpointer data)
{
	GString *str = data;
	guint64 id;
	gint type;
	gchar *id_str;

	/* Retrieve the id and type of the selected row */
	gtk_tree_model_get (model, iter, 1, &id, 2, &type, -1);

	/* Only calendar are taken into account */
	if (type == 0) {
		/* Append a comment before the formula */ 
		append_comment_to_str (model, iter, str);

		/* Append the formula to the Formula text view */
		id_str = schedwi_ulltostr (id);
		sql_cal_get_details (	id_str, NULL, NULL, NULL, NULL, NULL,
					append_formula_to_str,
					str, NULL, NULL);
		g_free (id_str);
	}
}


/*
 * Callback of the `Copy Formula' button
 */
void
calendar_cal_copy_formula_clicked (GtkButton *button)
{
	GtkWidget *text, *view;
	GtkTreeSelection *selection;
	GtkTextBuffer *buffer;
	GtkTextIter text_iter;
	GString *str;

	cursor_busy (GTK_WIDGET (button));

	str = g_string_new ("");
	view = lookup_widget (GTK_WIDGET (button), "treeview_cal_list");
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));

	gtk_tree_selection_selected_foreach (	selection,
						foreach_calendar,
						str);

	text = lookup_widget (GTK_WIDGET (button), "textview_cal_formula");
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (text));
	gtk_text_buffer_get_end_iter (buffer, &text_iter);
	gtk_text_buffer_insert (buffer, &text_iter, str->str, str->len);
	g_string_free (str, TRUE);
	refresh_calendar_canvas (text, TRUE);
	cursor_normal (GTK_WIDGET (button));
}

/*------------------------======= End Of File =======------------------------*/
