/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* cache_pixbuf.c -- Functions to add an retrieve pixbufs form a cache */

#include <schedwi.h>

#include <crc32.h>
#include <cache_pixbuf.h>

#define PIXBUF_ENLIGTH_FACTOR	1.4
#define PIXBUF_DARKEN_FACTOR	0.5
#define PIXBUF_DARKEN_ENLIGTH_FACTOR	0.9

#define ENLIGTH_FACTOR		15.0
#define DARKEN_FACTOR		0.5	
#define DARKEN_ENLIGTH_FACTOR	0.7


/* Running states colors */
static GdkColor color[NUM_JOB_STATUS_STATE][4];
static gboolean color_set = FALSE;

/* List of cached pixbufs */
static GSList *pixbuf_list = NULL;

/* A cached pixbuf */
struct cached_pixbuf {
	GdkPixbuf *pixbuf;		/* Pixbuf */
	GdkPixbuf *pixbuf_high;		/* Same pixbuf but highlighted */
	GdkPixbuf *pixbuf_select;	/* Same pixbuf but darkened */
	GdkPixbuf *pixbuf_select_high;	/* Same pixbuf but darkened highligh */
	GdkPixbuf *pixbuf_cut;		/* Same pixbuf but with alpha layer */
	unsigned long int crc;		/* Key to find a pixbuf in the cache */
};
typedef struct cached_pixbuf cached_pixbuf_t;


/*
 * Destroy the provided cached_pixbuf_t object.  The associated pixbufs are
 * unreferenced (they will be destroyed if not used anymore)
 */
static void
destroy_cached_pixbuf_t (gpointer data, gpointer user_data)
{
	cached_pixbuf_t *ptr = data;

	if (ptr != NULL) {
		if (ptr->pixbuf != NULL) {
			g_object_unref (ptr->pixbuf);
		}
		if (ptr->pixbuf_high != NULL) {
			g_object_unref (ptr->pixbuf_high);
		}
		if (ptr->pixbuf_select != NULL) {
			g_object_unref (ptr->pixbuf_select);
		}
		if (ptr->pixbuf_select_high != NULL) {
			g_object_unref (ptr->pixbuf_select_high);
		}
		if (ptr->pixbuf_cut != NULL) {
			g_object_unref (ptr->pixbuf_cut);
		}
	}
}


/*
 * Empty the cache
 */
void
empty_cache_pixbuf ()
{
	g_slist_foreach (pixbuf_list, destroy_cached_pixbuf_t, NULL);
	g_slist_free (pixbuf_list);
	pixbuf_list = NULL;
}


/*
 * Find and return the cached_pixbuf_t object associated with the provided CRC
 *
 * Return:
 *   The object associated with CRC or
 *   NULL if not found in the cache
 */
static cached_pixbuf_t *
get_pixbuf_from_cache (unsigned long int crc)
{
	GSList *list;

	for (	list = pixbuf_list;
		list != NULL && ((cached_pixbuf_t *)list->data)->crc < crc;
		list = g_slist_next (list));
	if (list == NULL || ((cached_pixbuf_t *)list->data)->crc != crc) {
		return NULL;
	}
	else {
		return (cached_pixbuf_t *)list->data;
	}
}


/*
 * Compare to objects in the cache
 */
static gint
cached_pixbuf_t_cmp (gconstpointer a, gconstpointer b)
{
	const cached_pixbuf_t *ptr_a = a;
	const cached_pixbuf_t *ptr_b = b;

	return (gint)(ptr_a->crc - ptr_b->crc);
}
 

/*
 * Add a pixbuf to the cache (if it's not already in the cache).  The pixbuf to
 * add is provided by the serialized stream.
 *
 * Return:
 *   TRUE --> No error. pixbuf, pixbuf_high, pixbuf_select, pixbuf_select_high
 *            and pixbuf_cut are set (if not NULL)
 *  FALSE --> Error.  If err_msg is not NULL it contains an error message (it
 *            must be freed by the caller by g_free())
 */ 
gboolean
add_pixbuf_to_cache (	const char *stream, unsigned long int len,
			GdkPixbuf **pixbuf,
			GdkPixbuf **pixbuf_high,
			GdkPixbuf **pixbuf_select,
			GdkPixbuf **pixbuf_select_high,
			GdkPixbuf **pixbuf_cut,
			gchar **err_msg)
{
	unsigned long int crc;
	cached_pixbuf_t *ptr;
	GdkPixdata pixdata;
	GError *err = NULL;
	int width, height, x, y, rowstride;
	guchar *pixels, *p;

	/* Compute the CRC */
	crc = crc_string (stream, len, crc_init ());

	/* If the pixbuf is already in the cache */
	ptr = get_pixbuf_from_cache (crc);
	if (ptr != NULL) {
		if (pixbuf != NULL) {
			*pixbuf = ptr->pixbuf;
		}
		if (pixbuf_high != NULL) {
			*pixbuf_high = ptr->pixbuf_high;
		}
		if (pixbuf_select != NULL) {
			*pixbuf_select = ptr->pixbuf_select;
		}
		if (pixbuf_select_high != NULL) {
			*pixbuf_select_high = ptr->pixbuf_select_high;
		}
		if (pixbuf_cut != NULL) {
			*pixbuf_cut = ptr->pixbuf_cut;
		}
		return TRUE;
	}

	/* Build the new cached_pixbuf_t object */
	ptr = g_new0 (cached_pixbuf_t, 1);
	ptr->crc = crc;

	/* Build the pixbuf */
	if (gdk_pixdata_deserialize (	&pixdata, (guint)len, (guint8 *)stream,
					&err) == FALSE)
	{
		if (err_msg != NULL) {
			*err_msg = g_strdup (err->message);
		}
		g_error_free (err);
		g_free (ptr);
		return FALSE;
	}

	ptr->pixbuf = gdk_pixbuf_from_pixdata (&pixdata, TRUE, &err);
	if (ptr->pixbuf == NULL) {
		if (err_msg != NULL) {
			*err_msg = g_strdup (err->message);
		}
		g_error_free (err);
		g_free (ptr);
		return FALSE;
	}

	/* Compute the highlight and the selected pixbufs */
	ptr->pixbuf_high = gdk_pixbuf_new (
				gdk_pixbuf_get_colorspace (ptr->pixbuf),
				gdk_pixbuf_get_has_alpha (ptr->pixbuf),
				gdk_pixbuf_get_bits_per_sample (ptr->pixbuf),
				gdk_pixbuf_get_width (ptr->pixbuf),
				gdk_pixbuf_get_height (ptr->pixbuf)); 
	if (ptr->pixbuf_high == NULL) {
		/* Memory allocation error */
		g_object_unref (ptr->pixbuf);
		g_free (ptr);
		if (err_msg != NULL) {
			*err_msg = g_strdup (_("Memory allocation error"));
		}
		return FALSE;
	}
	gdk_pixbuf_saturate_and_pixelate (	ptr->pixbuf,
						ptr->pixbuf_high,
						PIXBUF_ENLIGTH_FACTOR,
						FALSE);

	ptr->pixbuf_select = gdk_pixbuf_new (
				gdk_pixbuf_get_colorspace (ptr->pixbuf),
				gdk_pixbuf_get_has_alpha (ptr->pixbuf),
				gdk_pixbuf_get_bits_per_sample (ptr->pixbuf),
				gdk_pixbuf_get_width (ptr->pixbuf),
				gdk_pixbuf_get_height (ptr->pixbuf)); 
	if (ptr->pixbuf_select == NULL) {
		/* Memory allocation error */
		g_object_unref (ptr->pixbuf_high);
		g_object_unref (ptr->pixbuf);
		g_free (ptr);
		if (err_msg != NULL) {
			*err_msg = g_strdup (_("Memory allocation error"));
		}
		return FALSE;
	}
	gdk_pixbuf_saturate_and_pixelate (	ptr->pixbuf,
						ptr->pixbuf_select,
						PIXBUF_DARKEN_FACTOR,
						FALSE);


	ptr->pixbuf_select_high = gdk_pixbuf_new (
				gdk_pixbuf_get_colorspace (ptr->pixbuf),
				gdk_pixbuf_get_has_alpha (ptr->pixbuf),
				gdk_pixbuf_get_bits_per_sample (ptr->pixbuf),
				gdk_pixbuf_get_width (ptr->pixbuf),
				gdk_pixbuf_get_height (ptr->pixbuf)); 
	if (ptr->pixbuf_select_high == NULL) {
		/* Memory allocation error */
		g_object_unref (ptr->pixbuf_select);
		g_object_unref (ptr->pixbuf_high);
		g_object_unref (ptr->pixbuf);
		g_free (ptr);
		if (err_msg != NULL) {
			*err_msg = g_strdup (_("Memory allocation error"));
		}
		return FALSE;
	}
	gdk_pixbuf_saturate_and_pixelate (	ptr->pixbuf,
						ptr->pixbuf_select_high,
						PIXBUF_DARKEN_ENLIGTH_FACTOR,
						FALSE);


	ptr->pixbuf_cut = gdk_pixbuf_copy (ptr->pixbuf);
	if (ptr->pixbuf_cut == NULL) {
		/* Memory allocation error */
		g_object_unref (ptr->pixbuf_select_high);
		g_object_unref (ptr->pixbuf_select);
		g_object_unref (ptr->pixbuf_high);
		g_object_unref (ptr->pixbuf);
		g_free (ptr);
		if (err_msg != NULL) {
			*err_msg = g_strdup (_("Memory allocation error"));
		}
		return FALSE;
	}

	if (  gdk_pixbuf_get_has_alpha (ptr->pixbuf_cut) == TRUE
	   && gdk_pixbuf_get_n_channels (ptr->pixbuf_cut) == 4
	   && gdk_pixbuf_get_colorspace (ptr->pixbuf_cut) == GDK_COLORSPACE_RGB)
	{
		width = gdk_pixbuf_get_width (ptr->pixbuf_cut);
		height = gdk_pixbuf_get_height (ptr->pixbuf_cut);
		rowstride = gdk_pixbuf_get_rowstride (ptr->pixbuf_cut);
		pixels = gdk_pixbuf_get_pixels (ptr->pixbuf_cut);
		for (y = 0; y < height; y++) {
			for (x = 0; x < width; x++) {
				p = pixels + y * rowstride + x * 4;
				p[3] /= 2;
			}
		}
	}

	/* Add the new object to the list */
	pixbuf_list = g_slist_insert_sorted (	pixbuf_list, ptr,
						cached_pixbuf_t_cmp);


	if (pixbuf != NULL) {
		*pixbuf = ptr->pixbuf;
	}
	if (pixbuf_high != NULL) {
		*pixbuf_high = ptr->pixbuf_high;
	}
	if (pixbuf_select != NULL) {
		*pixbuf_select = ptr->pixbuf_select;
	}
	if (pixbuf_select_high != NULL) {
		*pixbuf_select_high = ptr->pixbuf_select_high;
	}
	if (pixbuf_cut != NULL) {
		*pixbuf_cut = ptr->pixbuf_cut;
	}
	return TRUE;
}


/*
 * Saturate (highlight or darken) the provided color
 *
 * This function is inspired by the code of gdk_pixbuf_saturate_and_pixelate()
 * in gdk-pixbuf-util.c (www.gtk.org)
 */
static void
saturate_color (GdkColor *src, GdkColor *dst, gdouble factor)
{
	guint16 intensity;

	intensity = src->red * 0.30 + src->green * 0.59 + src->blue * 0.11;

	dst->red = CLAMP ((1.0 - factor) * intensity + factor * src->red,
				0, 65535);
	dst->green = CLAMP ((1.0 - factor) * intensity + factor * src->green,
				0, 65535);
	dst->blue = CLAMP ((1.0 - factor) * intensity + factor * src->blue,
				0, 65535);
	dst->pixel = src->pixel;
}


/*
 * Retrieve the colors associated with the provided running state
 */
void
get_color_from_cache (	job_status_state t,
			const GdkColor **normal,
			const GdkColor **high,
			const GdkColor **select,
			const GdkColor **select_high)
{
	job_status_state i;

	/* First call: load all the colors */
	if (color_set == FALSE) {
		for (	i = JOB_STATUS_STATE_UNDEFINED;
			i < NUM_JOB_STATUS_STATE;
			i++)
		{
			gdk_color_parse (	job_status_state_to_color (i),
						&(color[i][0]));
			saturate_color (&(color[i][0]), &(color[i][1]),
					ENLIGTH_FACTOR);
			saturate_color (&(color[i][0]), &(color[i][2]),
					DARKEN_FACTOR);
			saturate_color (&(color[i][0]), &(color[i][3]),
					DARKEN_ENLIGTH_FACTOR);
		}
		color_set = TRUE;
	}

	if (normal != NULL) {
		*normal = &(color[t][0]);
	}
	if (high != NULL) {
		*high = &(color[t][1]);
	}
	if (select != NULL) {
		*select = &(color[t][2]);
	}
	if (select_high != NULL) {
		*select_high = &(color[t][3]);
	}
}

/*-----------------============== End Of File ==============-----------------*/
