/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* sql_commands.c -- commands management functions */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <sql_common.h>
#include <sql_commands.h>


#define SQLCOMMAND_GET "SELECT DISTINCT workload_date,job_id,username,command,parameter FROM commands WHERE command_status=0 AND NOW() - INTERVAL 10 minute <= insert_time ORDER BY workload_date,job_id,insert_time"
#define SQLCOMMAND_ADD "INSERT INTO commands (workload_date,job_id,username,command,parameter) VALUES (%d,\"%s\",\"%s\",%d,%d)"
#define SQLCOMMAND_DONE "UPDATE commands SET command_status=1 WHERE workload_date=%d AND job_id=\"%s\" AND command=%d AND parameter=%d"
#define SQLCOMMAND_FAILED "UPDATE commands SET command_status=2 WHERE workload_date=%d AND job_id=\"%s\" AND command=%d AND parameter=%d"
#define SQLCOMMAND_PURGE "DELETE FROM commands WHERE NOW() - INTERVAL 24 hour > insert_time"


/*
 * Retrieve the commands to run.  For each command, the provided callback()
 * is called with the provided parameters:
 *   - the provided user_data parameter
 *   - the workload
 *   - the job ID
 *   - the command
 *   - the parameter
 * If the callback function return a value other than 0, the
 * sql_command_get_commands() function exits with a return code of 2.
 *
 * Return:
 *     0 --> No error
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *     2 --> The callback function returned a value other than 0
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_command_get_commands (
		int (*callback)(void *, const char *, const char *,
				const char *, int, int),
		void *user_data,
		void (*error_func)(void *, const char *, unsigned int),
		void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret;
	lwc_LL *rows;
	char **row;

#if HAVE_ASSERT_H
	assert (callback != NULL);
#endif

	ret = sql_select (	NULL, NULL, &err_msg, NULL, &rows, NULL,
				SQLCOMMAND_GET, SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	while ((row = (char **) lwc_delStartLL (rows)) != NULL) {
		/*
		 * row[0] --> Workload
		 * row[1] --> Job ID
		 * row[2] --> User name
		 * row[3] --> Command
		 * row[4] --> Parameter
		 */
		if (callback (	user_data, row[0], row[1], row[2],
				atoi (row[3]), atoi (row[4])) != 0)
		{
			sql_free_row (row);
			lwc_delLL (rows, (void (*)(const void *))sql_free_row);
			return 2;
		}
		sql_free_row (row);
	}
	lwc_delLL (rows, (void (*)(const void *))sql_free_row);

	return 0;
}


/*
 * Add the provided command for the Schedwi server to run
 *
 * Return:
 *     0 --> No error
 *     1 --> Memory allocation error.  If err_msg is not NULL it contains
 *           an error message to be freed by the caller by free()
 * other --> SQL error.  If err_msg is not NULL it contains
 *           an error message to be freed by the caller by free()
 */
unsigned int
sql_command_add (	int workload_date, const char *job_id,
			const char *user, int command, int parameter,
			char **err_msg)
{
#if HAVE_ASSERT_H
	assert (job_id != NULL);
#endif


	return sql_non_select (	NULL, NULL, err_msg, NULL, NULL,
				SQLCOMMAND_ADD,
				SQL_INT, (long int)workload_date,
				SQL_STRING, job_id,
				SQL_STRING, (user == NULL) ? "": user,
				SQL_INT, (long int)command,
				SQL_INT, (long int)parameter,
				SQL_END);
}


/*
 * Update the status of the command in the database
 *
 * Return:
 *     0 --> No error
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_command_done (	int workload_date, const char *job_id,
			int command, int parameter,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret;

#if HAVE_ASSERT_H
	assert (job_id != NULL);
#endif

	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, NULL,
				SQLCOMMAND_DONE,
				SQL_INT, (long int)workload_date,
				SQL_STRING, job_id,
				SQL_INT, (long int)command,
				SQL_INT, (long int)parameter,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Update the status of the command in the database
 *
 * Return:
 *     0 --> No error
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_command_failed (	int workload_date, const char *job_id,
			int command, int parameter,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret;

#if HAVE_ASSERT_H
	assert (job_id != NULL);
#endif

	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, NULL,
				SQLCOMMAND_FAILED,
				SQL_INT, (long int)workload_date,
				SQL_STRING, job_id,
				SQL_INT, (long int)command,
				SQL_INT, (long int)parameter,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Remove the old completed and failed commands
 *
 * Return:
 *     0 --> No error
 *     1 --> Memory allocation error.  If err_msg is not NULL it contains
 *           an error message to be freed by the caller by free()
 * other --> SQL error.  If err_msg is not NULL it contains
 *           an error message to be freed by the caller by free()
 */
unsigned int
sql_command_purge (char **err_msg)
{
	return sql_non_select (	NULL, NULL, err_msg, NULL, NULL,
				SQLCOMMAND_PURGE,
				SQL_END);
}

/*-----------------============== End Of File ==============-----------------*/
