/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301,
   USA.
*/

/* net_parse.c -- Parse a network request */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_STDIO_H
#include <stdio.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <sql_hierarchy.h>
#include <lib_functions.h>
#include <lwc_log.h>
#include <result_mgnt.h>
#include <schedwi_time.h>
#include <load_workload_class.h>
#include <job_status_state.h>
#include <workload_class.h>
#include <net_parse.h>

#define WORKLOAD_LEN 8


/*
 * Build a message and find out the new status of the job from the
 * provided result_t object.
 *
 * Return:
 *   0 --> No error.  new_state and job_err_msg are set.  job_err_msg must be
 *         freed by the caller (by free()).
 *  -1 --> Error.  An error message has been logged by lwc_writeLog()
 */
static int
result_to_status (	schedwi_date workload_date,
			result_t *ptr,
			job_status_state *new_state,
			char **job_err_msg)
{
	unsigned int len;
	char *status_msg, *msg, *err_msg, *success_ret;
	int ret, d;
	lwc_LL *hierarchy_list;

#if HAVE_ASSERT_H
	assert (ptr != NULL && new_state != NULL && job_err_msg != NULL);
#endif

	/* Build the status message */
	if (ptr->was_killed != 0) {
		status_msg = _("Killed by signal %d");
	}
	else {
		status_msg = _("Exit with return code %d");
	}

	len = schedwi_strlen (status_msg) + 11;
	msg = (char *) malloc (len);
	if (msg == NULL) {
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}
	ret = snprintf (msg, len, status_msg, ptr->exit_code);
	if (ret >= len || ret < 0) {
		msg[0] = '\0';
	}

	/* If the job was kill */
	if (ptr->was_killed != 0) {
		*job_err_msg = msg;
		*new_state = JOB_STATUS_STATE_FAILED;
		return 0;
	}

	/*
	 * Retrieve the error code from the database to find out if the
	 * job is successful or not
	 */
	d = schedwi_date_to_int (workload_date);
	err_msg = NULL;
	hierarchy_list = get_hierarchy_list (d, ptr->job_id + WORKLOAD_LEN + 1,
						&err_msg);
	if (hierarchy_list == NULL) {
		free (msg);
		if (err_msg != NULL) {
			lwc_writeLog (LOG_CRIT, err_msg);
			free (err_msg);
		}
		else {
			lwc_writeLog (	LOG_CRIT,
					_("Job %s: cannot retrieve details"),
					ptr->job_id + WORKLOAD_LEN + 1);
		}
		return -1;
	}

	err_msg = NULL;
	if (get_job_parameter (	d, hierarchy_list,
				"job_success_return_code_s", "success_ret",
				&success_ret, NULL, &err_msg) != 0)
	{
		free (msg);
		lwc_delLL (hierarchy_list, (void (*)(const void *))free);
		if (err_msg != NULL) {
			lwc_writeLog (LOG_CRIT, err_msg);
			free (err_msg);
		}
		else {
			lwc_writeLog (	LOG_CRIT,
	_("Job %s: error while retrieving the job success return code"),
					ptr->job_id + WORKLOAD_LEN + 1);
		}
		return -1;
	}
	lwc_delLL (hierarchy_list, (void (*)(const void *))free);

	if (success_ret == NULL || ptr->exit_code > atoi (success_ret)) {
		*new_state = JOB_STATUS_STATE_FAILED;
	}
	else {
		*new_state = JOB_STATUS_STATE_COMPLETED;
	}

	if (success_ret != NULL) {
		free (success_ret);
	}

	*job_err_msg = msg;
	return 0;
}


/*
 * Parse the client request
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error.  A message has been logged by lwc_writeLog()
 */
int
net_parse (schedwi_BIO *b, void *obj)
{
	load_workload_class_ptr workload_list = (load_workload_class_ptr) obj;
	char *buff, *err_msg;
	unsigned int len;
	int nb_read;
	char module[SCHEDWI_NET_FUNCTION_LENGHT + 1];
	lwc_LL *list;
	result_t *ptr;
	char workload[WORKLOAD_LEN + 1];
	schedwi_date workload_date;
	workload_class_ptr workload_obj;
	job_status_state new_state;

#if HAVE_ASSERT_H
	assert (workload_list != NULL);
#endif

	if (b == NULL) {
		return 0;
	}

	/* Read the request */
	buff = NULL;
	len = 0;
	nb_read = net_read (b, &buff, &len);
	if (nb_read < 0) {
		if (buff != NULL) {
			free (buff);
		}
		return -1;
	}

	/* Check the module name */
	if (nb_read < SCHEDWI_NET_FUNCTION_LENGHT) {
		if (buff != NULL) {
			free (buff);
		}
		lwc_writeLog (LOG_ERR,
			_("Wrong network request: not enough characters"));
		return -1;
	}

	strncpy (module, buff, SCHEDWI_NET_FUNCTION_LENGHT);
	module[SCHEDWI_NET_FUNCTION_LENGHT] = '\0';
	if (strcmp (module, "result") != 0) {
		if (buff != NULL) {
			free (buff);
		}
		lwc_writeLog (	LOG_ERR,
			_("Wrong network request: unknown module `%s'"),
				module);
		return -1;
	}

	/* Convert the string to a list of result_t structures */
	if (str_to_results (	buff + SCHEDWI_NET_FUNCTION_LENGHT,
				nb_read - SCHEDWI_NET_FUNCTION_LENGHT,
				&list) != 0)
	{
		if (buff != NULL) {
			free (buff);
		}
		err_msg = _("Memory allocation error");
		net_write (b, "1", 1);
		net_write (b, err_msg, schedwi_strlen (err_msg));
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}

	if (buff != NULL) {
		free (buff);
	}

	/* Update the status of the jobs */
	while ((ptr = (result_t *) lwc_delStartLL (list)) != NULL) {

		strncpy (workload, ptr->job_id, WORKLOAD_LEN);
		workload[WORKLOAD_LEN] = '\0';

		if (	   ptr->job_id_len < WORKLOAD_LEN + 2
			|| (ptr->job_id)[WORKLOAD_LEN] !=
						SCHEDWI_WORKLOAD_SEPARATOR
			|| schedwi_date_from_string (	workload,
							&workload_date) != 0)
		{
			lwc_delLL (list,
				(void (*)(const void *)) destroy_result);
			err_msg = _("Wrong job ID");
			net_write (b, "1", 1);
			net_write (b, err_msg, schedwi_strlen (err_msg));
			lwc_writeLog (	LOG_ERR,
					_("Wrong job ID `%s'"), ptr->job_id);
			destroy_result (ptr);
			return -1;
		}

		/* Convert the received result to a status */
		if (result_to_status (	workload_date, ptr,
					&new_state, &err_msg) != 0)
		{
			lwc_delLL (list,
				(void (*)(const void *)) destroy_result);
			destroy_result (ptr);
			err_msg = _("Server internal error");
			net_write (b, "1", 1);
			net_write (b, err_msg, schedwi_strlen (err_msg));
			return -1;
		}

		/* Retrieve the workload tree */
		workload_obj = load_workload_class_find (workload_list,
							workload_date);
		if (workload_obj == NULL) {
			/* Workload not found, ignore the status */
			free (err_msg);
			destroy_result (ptr);
			continue;
		}

		/* Update the status of the job */
		if (workload_class_job_finished (workload_obj,
				strtoull (	ptr->job_id + WORKLOAD_LEN + 1,
						NULL, 0),
				new_state,
				(long int)ptr->duration,
				err_msg) != 0)
		{
			free (err_msg);
			lwc_delLL (list,
				(void (*)(const void *)) destroy_result);
			destroy_result (ptr);
			err_msg = _("Server internal error");
			net_write (b, "1", 1);
			net_write (b, err_msg, schedwi_strlen (err_msg));
			return -1;
		}
		free (err_msg);
		destroy_result (ptr);
	}

	lwc_delLL (list, (void (*)(const void *)) destroy_result);
	net_write (b, "0", 1);
	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
