/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* parse_env_and_arg.c -- Parse an environement or an argument line */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lib_functions.h>
#include <lwc_linkedlist.h>
#include <lwc_log.h>
#include <parse_env_and_arg.h>


struct module_arg {
	const char *arg;
	int pos;
};
typedef struct module_arg module_arg_t;

struct module_env {
	const char *key;
	const char *value;
	unsigned long int pos;
};
typedef struct module_env module_env_t;


/*
 * Create a new module_arg_t object
 *
 * Return:
 *   The new object (to be freed by the caller using destroy_module_arg()) or
 *   NULL in case of memory allocation error
 */
static module_arg_t *
new_module_arg ()
{
	module_arg_t *ptr;

	ptr = (module_arg_t *) malloc (sizeof (module_arg_t));
	if (ptr != NULL) {
		ptr->arg = NULL;
		ptr->pos = 0;
	}
	return ptr;
}


/*
 * Free the provided module_arg_t object
 */
static void
destroy_module_arg (module_arg_t *ptr)
{
	if (ptr != NULL) {
		free (ptr);
	}
}


/*
 * Free a list of module_arg_t objects
 */
void
destroy_argument_list (lwc_LL *list)
{
	lwc_delLL (list, (void (*)(const void *))destroy_module_arg);
}


/*
 * Create a new module_env_t object
 *
 * Return:
 *   The new object (to be freed by the caller using destroy_module_env()) or
 *   NULL in case of memory allocation error
 */
static module_env_t *
new_module_env ()
{
	module_env_t *ptr;

	ptr = (module_env_t *) malloc (sizeof (module_env_t));
	if (ptr != NULL) {
		ptr->key = NULL;
		ptr->value = NULL;
		ptr->pos = 0;
	}
	return ptr;
}


/*
 * Free the provided module_env_t object
 */
static void
destroy_module_env (module_env_t *ptr)
{
	if (ptr != NULL) {
		free (ptr);
	}
}


/*
 * Free a list of module_env_t objects
 */
void
destroy_environment_list (lwc_LL *list)
{
	lwc_delLL (list, (void (*)(const void *))destroy_module_env);
}


/*
 * Comparison function used to sort the argument list
 */
static int
compar_argument (const module_arg_t *a, const module_arg_t *b)
{
	return a->pos - b->pos;
}


/*
 * Parse the argument line and fill the list with the parsed argument
 * The argument line looks like:
 *      ARG:2:-q
 * The part of the line provided to this function (in argument) starts after
 * the ARG: key word (it's `2:-q' then)
 *
 * Return:
 *   0 --> No error
 *  -1 --> Memory allocation error (lwc_writeLog() is used to display an error
 *         message)
 *  -2 --> Syntax error (duplicate argument position - lwc_writeLog() is used
 *         to display an error message) 
 */
int
parse_arguments (net_id *sock, char *argument, lwc_LL *list)
{
	int ret;
	module_arg_t *ptr;
	char *str, *err_msg;

#if HAVE_ASSERT_H
	assert (argument != NULL && list != NULL);
#endif

	ptr = new_module_arg ();
	if (ptr == NULL) {
		err_msg = _("Memory allocation error");
		net_write (sock, "1", 1);
		net_write (sock, err_msg, schedwi_strlen (err_msg));
		lwc_writeLog (LOG_CRIT, err_msg);
		return -1;
	}

	str = strchr (argument, ':');
	if (str == NULL) {
		destroy_module_arg (ptr);
		return 0;
	}

	*str = '\0';
	ptr->pos = atoi (argument);
	ptr->arg = str + 1;

	ret = lwc_addLL (list, ptr,
			(int (*)(const void *, const void *))compar_argument);
	if (ret != 0) {
		destroy_module_arg (ptr);
		if (ret == -1) {
			err_msg = _("Memory allocation error");
			net_write (sock, "1", 1);
			net_write (sock, err_msg, schedwi_strlen (err_msg));
			lwc_writeLog (LOG_CRIT, err_msg);
			return -1;
		}
		else {
			err_msg = _("Duplicate argument position");
			net_write (sock, "1", 1);
			net_write (sock, err_msg, schedwi_strlen (err_msg));
			lwc_writeLog (LOG_ERR, err_msg);
			return -2;
		}
	}
	return 0;
}


/*
 * Comparison function used to sort the environment list
 */
static int
compar_environment (const module_env_t *a, const module_env_t *b)
{
	return (a->pos != b->pos) ? a->pos - b->pos: 1;
}


/*
 * Parse the environment line and fill the list with the parsed environment
 * variable
 * An environment variable line looks like:
 *    ENV:7:PI:3.14
 * The part of this line provided in environment_var to this function starts
 * after ENV: (it's `7:PI:3.14' then)
 *
 * Return:
 *   0 --> No error
 *  -1 --> Memory allocation error (lwc_writeLog() is used to display an error
 *         message)
 */
int
parse_environment (net_id *sock, char *environment_var, lwc_LL *list)
{
	int ret;
	module_env_t *ptr;
	char *err_msg;
	char *pos, *key;

#if HAVE_ASSERT_H
	assert (environment_var != NULL && list != NULL);
#endif

	ptr = new_module_env ();
	if (ptr == NULL) {
		err_msg = _("Memory allocation error");
		net_write (sock, "1", 1);
		net_write (sock, err_msg, schedwi_strlen (err_msg));
		lwc_writeLog (LOG_CRIT, err_msg);
		return -1;
	}

	pos = strchr (environment_var, ':');
	if (pos == NULL) {
		destroy_module_env (ptr);
		return 0;
	}
	*pos = '\0';
	ptr->pos = atoi (environment_var);

	key = strchr (++pos, ':');
	if (key == NULL) {
		destroy_module_env (ptr);
		return 0;
	}
	*key = '\0';
	ptr->key = pos;
	ptr->value = key + 1;

	ret = lwc_addLL (list, ptr,
		(int (*)(const void *, const void *))compar_environment);
	if (ret != 0) {
		destroy_module_env (ptr);
		err_msg = _("Memory allocation error");
		net_write (sock, "1", 1);
		net_write (sock, err_msg, schedwi_strlen (err_msg));
		lwc_writeLog (LOG_CRIT, err_msg);
		return -1;
	}
	return 0;
}


/*
 * Convert (and destroy) an argument list to an argument_t object
 *
 * Return:
 *   0 --> No error (the list is destroyed and must not be used anymore)
 *  -1 --> Memory allocation error
 */
int
argument_list_to_job_parameter (lwc_LL *list, argument_t *arg)
{
	module_arg_t *arg_ptr;
	int ret;

#if HAVE_ASSERT_H
	assert (list != NULL);
#endif

	if (arg == NULL) {
		destroy_argument_list (list);
		return 0;
	}

	/* Add the argument list to the argument structure */
	while (	(arg_ptr = (module_arg_t *) lwc_delStartLL (list)) != NULL) {
		ret = add_arg (arg, arg_ptr->arg);
		destroy_module_arg (arg_ptr);
		if (ret != 0) {
			return -1;
		}
	}
	lwc_delLL (list, NULL);
	return 0;
}


/*
 * Convert (and destroy) an environment list to an environment_t object
 *
 * Return:
 *   0 --> No error (the list is destroyed and must not be used anymore)
 *  -1 --> Memory allocation error
 */
int
environment_list_to_job_env (lwc_LL *list, environment_t *env)
{
	module_env_t *env_ptr;
	int ret;

#if HAVE_ASSERT_H
	assert (list != NULL);
#endif

	if (env == NULL) {
		destroy_environment_list (list);
		return 0;
	}

	while (	(env_ptr = (module_env_t *) lwc_delStartLL (list)) != NULL) {
		ret = add_env (env, env_ptr->key, env_ptr->value);
		destroy_module_env (env_ptr);
		if (ret != 0) {
			return -1;
		}
	}
	lwc_delLL (list, NULL);
	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
