/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301,
   USA.
*/

/* net_utils_ssl.c -- Useful network functions (socket connection with SSL) */

#include <schedwi.h>

#if HAVE_LIBGNUTLS

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif

#if HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif

#if HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif

#if HAVE_NETDB_H
#include <netdb.h>
#endif

#if HAVE_TIME_H
#include <time.h>
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <gnutls/gnutls.h>
#include <gnutls/x509.h>

#include <lib_functions.h>
#include <lwc_log.h>
#include <net_utils_sock.h>
#include <net_utils_ssl.h>

#define BUFFER_LEN 1024
#define ENF_OF_REQUEST "\nbYe"


/* Credentials (client certificate and key and server certificate) */
gnutls_certificate_credentials_t x509_cred;
static char cerdentials_set = 0;

/* Diffie Hellman parameters */
static gnutls_dh_params_t dh_params;

/* RSA parameters */
static gnutls_rsa_params_t rsa_params;

static char params_set = 0;


/*
 * Initialize the Diffie Hellman and RSA algorithms (only required for server)
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error (a message has been logged by lwc_writeLog())
 */
int
net_init_server_ssl ()
{
	int ret;

	lwc_writeLog (LOG_INFO,
		_("Initializing the SSL layer (may take several minutes)..."));

	/* Build the Diffie Hellman parameters */
	ret = gnutls_dh_params_init (&dh_params);
	if  (ret != 0) {
		lwc_writeLog (	LOG_ERR,
		_("SSL: Failed to allocate the Diffie Hellman parameters: %s"),
				gnutls_strerror (ret));
		return -1;
	}

	ret = gnutls_dh_params_generate2 (dh_params, 1024);
	if  (ret != 0) {
		lwc_writeLog (	LOG_ERR,
_("SSL: Failed to generate the parameters for the Diffie Hellman algorithms: %s"),
				gnutls_strerror (ret));
		gnutls_dh_params_deinit (dh_params);
		return -1;
	}

	/* Build the RSA parameters */
	ret = gnutls_rsa_params_init (&rsa_params);
	if  (ret != 0) {
		lwc_writeLog (	LOG_ERR,
		_("SSL: Failed to allocate the RSA parameters: %s"),
				gnutls_strerror (ret));
		gnutls_dh_params_deinit (dh_params);
		return -1;
	}

	ret = gnutls_rsa_params_generate2 (rsa_params, 512);
	if  (ret != 0) {
		lwc_writeLog (	LOG_ERR,
_("SSL: Failed to generate the parameters for the RSA algorithms: %s"),
				gnutls_strerror (ret));
		gnutls_rsa_params_deinit (rsa_params);
		gnutls_dh_params_deinit (dh_params);
		return -1;
	}

	if (cerdentials_set != 0) {
		gnutls_certificate_set_dh_params (x509_cred, dh_params);
		gnutls_certificate_set_rsa_export_params (	x509_cred,
								rsa_params);
	}

	params_set = 1;
	lwc_writeLog (LOG_INFO, _("Done"));
	return 0;
}


/*
 * Initialize the SSL layer
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error (a message has been logged by lwc_writeLog())
 */
int
net_init_ssl (	const char *clnt_crt_file, const char *clnt_key_file,
		const char *server_crt)
{
	int ret;

#if HAVE_ASSERT_H
	assert (   clnt_crt_file != NULL && clnt_key_file != NULL
		&& server_crt != NULL);
#endif

	if (cerdentials_set != 0) {
		gnutls_certificate_free_credentials (x509_cred);
		cerdentials_set = 0;
	}

	if (params_set != 0) {
		gnutls_rsa_params_deinit (rsa_params);
		gnutls_dh_params_deinit (dh_params);
		params_set = 0;
	}

	/* GNUTLS init */
	ret = gnutls_global_init ();
	if (ret != 0) {
		lwc_writeLog (LOG_CRIT,
			_("SSL: Cannot initialize the SSL library: %s"),
			gnutls_strerror (ret));
		return -1;
	}

	/* Allocate the credentials for the provided certificate */
	ret = gnutls_certificate_allocate_credentials (&x509_cred);
	if (ret != 0) {
		lwc_writeLog (LOG_ERR,
	_("SSL: Cannot initialize the credentials for the current host: %s"),
			gnutls_strerror (ret));
		gnutls_global_deinit ();
		return -1;
	}

	/*
	 * Try to load the certificate in PEM format.  If it does not work,
	 * try to load it in DER format
	 */
	ret = gnutls_certificate_set_x509_key_file (	x509_cred,
							clnt_crt_file,
							clnt_key_file,
							GNUTLS_X509_FMT_PEM);
	if (	   ret != 0
		&& gnutls_certificate_set_x509_key_file (x509_cred,
						clnt_crt_file,
						clnt_key_file,
						GNUTLS_X509_FMT_DER) != 0)
	{
		lwc_writeLog (LOG_ERR,
_("SSL: Cannot load the client certificat `%s' with the key `%s': %s"),
				clnt_crt_file, clnt_key_file,
				gnutls_strerror (ret));
		gnutls_certificate_free_credentials (x509_cred);
		gnutls_global_deinit ();
		return -1;
	}

	/* Load the server (trusted) certificate */
	ret = gnutls_certificate_set_x509_trust_file (	x509_cred, server_crt,
							GNUTLS_X509_FMT_PEM);
	if (	   ret < 0
		&& gnutls_certificate_set_x509_trust_file (x509_cred,
					server_crt, GNUTLS_X509_FMT_DER) < 0)
	{
		lwc_writeLog (LOG_ERR,
			_("SSL: Cannot load the server certificate `%s': %s"),
			server_crt, gnutls_strerror (ret));
		gnutls_certificate_free_credentials (x509_cred);
		gnutls_global_deinit ();
		return -1;
	}

	if (params_set != 0) {
		gnutls_certificate_set_dh_params (x509_cred, dh_params);
		gnutls_certificate_set_rsa_export_params (	x509_cred,
								rsa_params);
	}

	cerdentials_set = 1;
	return 0;
}


/*
 * Free the memory used by SSL
 */
void
net_destroy_ssl ()
{
	if (cerdentials_set != 0) {
		gnutls_certificate_free_credentials (x509_cred);
		cerdentials_set = 0;
	}

	if (params_set != 0) {
		gnutls_rsa_params_deinit (rsa_params);
		gnutls_dh_params_deinit (dh_params);
		params_set = 0;
	}
	gnutls_global_deinit ();
}


/*
 * Compare two hostnames
 *
 * Return:
 *   0 --> The two hosts are the same
 *   1 --> The two hosts are not the same
 *  -1 --> Error (a message is displayed using lwc_writeLog())
 */
static
int hostname_cmp (const char *h1, const char *h2)
{
	int in_num_dot, i, try_no;
	struct hostent *he;
	struct in_addr in;
#if HAVE_NANOSLEEP
	struct timespec req;

	req.tv_sec = SCHEDWI_HOSTLOOKUP_SLEEP;
	req.tv_nsec = 0;
#endif

	if (h1 == NULL || h2 == NULL) {
		return 1;
	}

	if (schedwi_strcasecmp (h1, h2) == 0) {
		return 0;
	}

	/* If the hostname is in number-dot notation, convert it */
	in_num_dot = inet_aton (h1, &in);

	try_no = 0;
	do {
		if (in_num_dot == 0) {
			he = gethostbyname (h1);
		}
		else {
			he = gethostbyaddr (&in, sizeof (in), AF_INET);
		}
		if (he == NULL) {
			if (h_errno == TRY_AGAIN) {
				try_no++;
				if (try_no < SCHEDWI_HOSTLOOKUP_RETRIES) {
#if HAVE_NANOSLEEP
					nanosleep (&req, NULL);
#elif HAVE_SLEEP
					sleep (SCHEDWI_HOSTLOOKUP_SLEEP);
#elif HAVE_USLEEP
					usleep (  SCHEDWI_HOSTLOOKUP_SLEEP
						* 1000000);
#endif
				}
			}
			else {
			if (h_errno == NO_RECOVERY) {
				lwc_writeLog (	LOG_ERR,
						_("Name server error"));
				return -1;
			}
			else {
				return 1;
			}}
		}
	} while (he == NULL && try_no < SCHEDWI_HOSTLOOKUP_RETRIES);
	if (he == NULL) {
		lwc_writeLog (LOG_WARNING, _("Name server unavailable"));
		return -1;
	}

	if (schedwi_strcasecmp (he->h_name, h2) == 0) {
		return 0;
	}

	for (i = 0; (he->h_aliases)[i] != NULL; i++) {
		if (schedwi_strcasecmp ((he->h_aliases)[i], h2) == 0) {
			return 0;
		}
	}
	return 1;
}


/*
 * Verify the remote certificate
 *
 * Return:
 *   0 --> No error, certificate verified
 *  -1 --> Error (an error message is displayed using lwc_writeLog())
 */
static int
net_utils_ssl_check_certificate (gnutls_session_t *session, const char *hostname)
{
	int ret, i;
	unsigned int status, cert_list_size;
	gnutls_x509_crt_t cert;
	const gnutls_datum_t *cert_list;
	time_t now;
	char dnsname[512];
	size_t dnsnamesize;
	char found_dnsname;

#if HAVE_ASSERT_H
	assert (hostname != NULL && session != NULL);
#endif

	/* Verifiy the certificate */
	ret = gnutls_certificate_verify_peers2 (*session, &status);
	if (ret < 0) {
		lwc_writeLog (LOG_ERR,
			_("SSL: Failed to verify the certificate for %s: %s"),
			hostname, gnutls_strerror (ret));
		return -1;
	}

	if (status & GNUTLS_CERT_INVALID) {
		lwc_writeLog (	LOG_ERR,
				_("SSL: The certificate for %s is invalid."),
				hostname);
		return -1;
	}

	if (status & GNUTLS_CERT_SIGNER_NOT_FOUND) {
		lwc_writeLog (	LOG_ERR,
		_("SSL: The certificate for %s does not have a known issuer."),
				hostname);
		return -1;
	}

	if (status & GNUTLS_CERT_REVOKED) {
		lwc_writeLog (LOG_ERR,
			_("SSL: The certificate for %s has been revoked."),
			hostname);
		return -1;
	}

	/* Retrieve the x509 certificate */
	ret = gnutls_x509_crt_init (&cert);
	if (ret != 0) {
		lwc_writeLog (LOG_ERR,
			_("SSL: Cannot initialize the certificate for %s: %s"),
			hostname, gnutls_strerror (ret));
		return -1;
	}

	cert_list = gnutls_certificate_get_peers (*session, &cert_list_size);
	if (cert_list == NULL) {
		lwc_writeLog (	LOG_ERR,
				_("SSL: No certificate found for %s"),
				hostname);
		gnutls_x509_crt_deinit (cert);
		return -1;
	}

	/* Only check the first certificate */
	ret = gnutls_x509_crt_import (	cert, &cert_list[0],
					GNUTLS_X509_FMT_DER);
	if (ret != 0) {
		lwc_writeLog (	LOG_ERR,
			_("SSL: Cannot retrieve the certificate for %s: %s"),
				hostname, gnutls_strerror (ret));
		gnutls_x509_crt_deinit (cert);
		return -1;
	}

	/* Check the dates */
	now = time (NULL);
	if (gnutls_x509_crt_get_expiration_time (cert) < now) {
		lwc_writeLog (	LOG_ERR,
				_("SSL: The certificate for %s has expired."),
				hostname);
		gnutls_x509_crt_deinit (cert);
		return -1;
	}

	if (gnutls_x509_crt_get_activation_time (cert) > now) {
		lwc_writeLog (LOG_ERR,
			_("SSL: The certificate for %s is not yet activated."),
			hostname);
		gnutls_x509_crt_deinit (cert);
		return -1;
	}

	/* Log a message if the certificate expires in less than a month */
	if (gnutls_x509_crt_get_expiration_time (cert)
			< now + 30 * 24 * 60 * 60)
	{
		lwc_writeLog (	LOG_NOTICE,
_("SSL: The certificate for %s is going to expire in less than a month."),
				hostname);
	}

	/*
	 * Check the hostname.  One could use the provided
	 * gnutls_x509_crt_check_hostname() function.  However, we will check
	 * all the known names of the host (from the DNS, /etc/hosts, ...)
	 * The following lines are then inspired from this function
	 * in GNUTLS (see rfc2818_hostname.c)
	 */
	found_dnsname = 0;
	for (i = ret = 0; ret >= 0; i++) {
		dnsnamesize = sizeof (dnsname);
		ret = gnutls_x509_crt_get_subject_alt_name (cert, i,
						dnsname, &dnsnamesize, NULL);
		if (ret == GNUTLS_SAN_DNSNAME) {
			found_dnsname = 1;
			if (hostname_cmp (hostname, dnsname) == 0) {
				gnutls_x509_crt_deinit (cert);
				return 0;
			}
		}
	}

	if (found_dnsname == 0) {
		dnsnamesize = sizeof (dnsname);
		ret = gnutls_x509_crt_get_dn_by_oid (cert,
						GNUTLS_OID_X520_COMMON_NAME,
						0, 0,
						dnsname, &dnsnamesize);
		if (ret < 0) {
			lwc_writeLog (LOG_ERR,
		_("SSL: Cannot retrieve the certificate hostname for %s: %s"),
				hostname, gnutls_strerror (ret));
			gnutls_x509_crt_deinit (cert);
			return -1;
		}
		if (hostname_cmp (hostname, dnsname) == 0) {
			gnutls_x509_crt_deinit (cert);
			return 0;
		}
	}

	gnutls_x509_crt_deinit (cert);
	lwc_writeLog (	LOG_ERR,
			_("SSL: The certificate hostname does not match %s."),
			hostname);
	return -1;
}


/*
 * Create a SSL session connected to the Schedwi server (which is the network
 * client here)
 *
 * Return:
 *    0 on success
 *    1 on an authentication error (an error message is displayed
 *      using lwc_writeLog())
 *   -1 in case of error (an error message is displayed using lwc_writeLog())
 */
int
net_accept_ssl (int fd, gnutls_session_t *session, const char *server_hostname)
{
	int ret;
	const int cert_type_priority[2] = { GNUTLS_CRT_X509, 0 };

#if HAVE_ASSERT_H
	assert (session != NULL && server_hostname != NULL);
#endif

	/* Initialize the session */
	ret = gnutls_init (session, GNUTLS_SERVER);
	if (ret != 0) {
		lwc_writeLog (LOG_ERR,
		_("SSL: Cannot initialize the session with the server: %s"),
			gnutls_strerror (ret));
		return -1;
	}

	/* Use default priority */
	gnutls_set_default_priority (*session);
	gnutls_certificate_type_set_priority (*session, cert_type_priority);

	/* Put the x509 credentials to the current session */
	ret = gnutls_credentials_set (	*session, GNUTLS_CRD_CERTIFICATE,
					x509_cred);
	if (ret != 0) {
		lwc_writeLog (LOG_ERR,
		_("SSL: Cannot associate certificate with the session: %s"),
			gnutls_strerror (ret));
		gnutls_deinit (*session);
		return -1;
	}

	/* The remote client certificate is required */
	gnutls_certificate_server_set_request (*session, GNUTLS_CERT_REQUIRE);
	gnutls_dh_set_prime_bits (*session, 1024);

	/* Associate the socket to the session */
	gnutls_transport_set_ptr (*session, (gnutls_transport_ptr_t) fd);

	/* Handshake */
	do {
		ret = gnutls_handshake (*session);
	} while (ret == GNUTLS_E_AGAIN || ret == GNUTLS_E_INTERRUPTED);
	if (ret < 0) {
		lwc_writeLog (	LOG_ERR,
		_("SSL: Connection handshake failed with the server: %s"),
				gnutls_strerror (ret));
		gnutls_bye (*session, GNUTLS_SHUT_RDWR);
		gnutls_deinit (*session);
		return 1;
	}

	/* Check the client name against the received certificate */
	if (net_utils_ssl_check_certificate (session, server_hostname) != 0) {
		gnutls_bye (*session, GNUTLS_SHUT_RDWR);
		gnutls_deinit (*session);
		return 1;
	}
	return 0;
}


/*
 * Create a SSL session connected to a server
 * port_number is the port number to connect to (a number or a service name
 *             in /etc/services)
 *
 * Return:
 *   0 on success
 *   -1 in case of error (an error message is displayed using lwc_writeLog())
 */
int
net_client_ssl (int fd, gnutls_session_t *session, const char *server_hostname)
{
	int ret;
	const int cert_type_priority[2] = { GNUTLS_CRT_X509, 0 };

#if HAVE_ASSERT_H
	assert (server_hostname != NULL && session != NULL);
#endif

	/* Initialize the session */
	ret = gnutls_init (session, GNUTLS_CLIENT);
	if (ret != 0) {
		lwc_writeLog (LOG_ERR,
			_("SSL: Cannot initialize the session with %s: %s"),
			server_hostname, gnutls_strerror (ret));
		return -1;
	}

	/* Use default priority */
	gnutls_set_default_priority (*session);
	gnutls_certificate_type_set_priority (*session, cert_type_priority);

	/* Put the x509 credentials to the current session */
	ret = gnutls_credentials_set (	*session, GNUTLS_CRD_CERTIFICATE,
					x509_cred);
	if (ret != 0) {
		lwc_writeLog (LOG_ERR,
	_("SSL: Cannot associate certificate with the session for %s: %s"),
			server_hostname, gnutls_strerror (ret));
		gnutls_deinit (*session);
		return -1;
	}

	/* Associate the socket to the session */
	gnutls_transport_set_ptr (*session, (gnutls_transport_ptr_t) fd);

	/* Handshake */
	do {
		ret = gnutls_handshake (*session);
	} while (ret == GNUTLS_E_AGAIN || ret == GNUTLS_E_INTERRUPTED);
	if (ret < 0) {
		lwc_writeLog (LOG_ERR,
			_("SSL: Connection handshake failed with %s: %s"),
			server_hostname, gnutls_strerror (ret));
		gnutls_bye (*session, GNUTLS_SHUT_RDWR);
		gnutls_deinit (*session);
		return -1;
	}

	/* Check the certificate */
	if (net_utils_ssl_check_certificate (session, server_hostname) != 0) {
		gnutls_bye (*session, GNUTLS_SHUT_RDWR);
		gnutls_deinit (*session);
		return -1;
	}
	return 0;
}


/*
 * Read data from the provided SSL session handler.
 * The data are store in *buff (of size *len). If buffer is NULL or not big
 * enough, it is reallocated to a bigger size. It must be freed by the caller
 *
 * Return:
 *    The total number of characters read and stored in *buff or
 *    -1 on error (a message is displayed using lwc_writeLog())
 *    -2 if too many data are received (a message is displayed using
 *       lwc_writeLog())
 */
ssize_t
net_read_ssl (gnutls_session_t session, char **buff, size_t *len)
{
	ssize_t nb_read, total;
	size_t to_read;
	char *b, *tmp;

#if HAVE_ASSERT_H
	assert (buff != NULL && len != NULL);
#endif

	if (*buff == NULL || *len == 0) {
		*buff = (char *) malloc (BUFFER_LEN);
		if (*buff == NULL) {
			*len = 0;
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			return -1;
		}
		*len = BUFFER_LEN;
	}

	total = 0;
	to_read = *len;
	b = *buff;
	do {
		nb_read = gnutls_record_recv (session, b, to_read);
		if (nb_read == 0) {
			return total;
		}

		if (nb_read < 0) {
			if (	   nb_read != GNUTLS_E_AGAIN
				&& nb_read != GNUTLS_E_INTERRUPTED)
			{
				lwc_writeLog (LOG_ERR,
					_("Error while reading data: %s"),
					gnutls_strerror (nb_read));
				return nb_read;
			}
			continue;
		}

		total += nb_read;

		/* Find the EOF tag */
		tmp = schedwi_strnstr (*buff, ENF_OF_REQUEST, total);
		if (tmp != NULL) {
			return (ssize_t)(tmp - *buff);
		}

		if (nb_read < to_read) {
			to_read -= nb_read;
			b += nb_read;
		}
		else {
			if (*len + BUFFER_LEN > SCHEDWI_MAX_BUFFER_LEN) {
				/* Too many data */
				lwc_writeLog (LOG_NOTICE,
		_("Failed to read the network request: buffer overflow"));
				return -2;
			}

			tmp = (char *) realloc (*buff, *len + BUFFER_LEN);
			if (tmp == NULL) {
				lwc_writeLog (	LOG_CRIT,
						_("Memory allocation error"));
				return -1;
			}
			*buff = tmp;
			*len += BUFFER_LEN;
			b = *buff + total;
			to_read = BUFFER_LEN;
		}
	} while (1);

	return total;
}


/*
 * Read the reply
 *
 * The format of the reply must be as follow:
 *    The first character: '0' means success, an other character means failure
 *    The next characters: An optional explanation message (which is returned
 *                         in result_msg)
 * No reply (nothing to read) means failure.
 *
 * Return:
 *   0 --> The reply means the remote action was successfull. If result_msg is
 *         not NULL, it is set with the message sent by the remote socket. It
 *         must be freed by the caller.
 *   1 --> The reply means the remote action failed. If result_msg is not NULL,
 *         it is set with the message sent by the remote socket. It may be NULL
 *         if no message was sent. It must be freed by the caller.
 *  -1 --> Error (a message is diplayed using lwc_writeLog())
 */
int
net_read_result_ssl (gnutls_session_t session, char **result_msg)
{
	char *buff, ret_code;
	size_t len;
	ssize_t nb_read;
	int i;

	if (result_msg != NULL) {
		*result_msg = NULL;
	}

	/* Read the reply */
	buff = NULL;
	len = 0;
	nb_read = net_read_ssl (session, &buff, &len);
	if (nb_read < 0 ) {
		if (buff != NULL) {
			free (buff);
		}
		return -1;
	}

	if (nb_read > 0) {
		/* Return code sent by the remote process */
		ret_code = buff[0];

		if (result_msg != NULL) {
			/* Shift the message 1 character to the left */
			for (i = 1; i < nb_read; i++) {
				buff[i - 1] = buff[i];
			}
			buff [nb_read - 1] = '\0';
			*result_msg = buff;
		}
		else {
			free (buff);
		}

		/* Return code '0' means success */
		if (ret_code == '0') {
			return 0;
		}
	}
	else {
		if (buff != NULL) {
			free (buff);
		}
	}
	return 1;
}


/*
 * Write data to the provided SSL session handler.
 *
 * Return:
 *   0 --> No error
 *  -1 --> Write error (a message is displayed using lwc_writeLog())
 */
int
net_write_ssl (gnutls_session_t session, const char *buff, size_t len)
{
	ssize_t nb_write;

	if (buff == NULL || len == 0) {
		return 0;
	}

	do {
		nb_write = gnutls_record_send (session, buff, len);
		if (nb_write == len) {
			return 0;
		}

		if (nb_write < 0) {
			if (	   nb_write != GNUTLS_E_AGAIN
				&& nb_write != GNUTLS_E_INTERRUPTED)
			{
				lwc_writeLog (LOG_ERR,
					_("Error while sending data: %s"),
					gnutls_strerror (nb_write));
				return -1;
			}
		}
		else {
			buff += nb_write;
			len -= nb_write;
		}

	} while (1);

	return 0;
}


/*
 * Write the End of Request tag
 *
 * Return:
 *   0 --> No error
 *  -1 --> Write error (a message is displayed using lwc_writeLog())
 */
int
net_write_eof_ssl (gnutls_session_t session)
{
	return net_write_ssl (session, ENF_OF_REQUEST,
				(size_t)schedwi_strlen (ENF_OF_REQUEST));  
}


/*
 * Close the SSL handler 
 */
int
net_close_ssl (gnutls_session_t session, int fd)
{
	int ret;

	do {
		ret = gnutls_bye (session, GNUTLS_SHUT_RDWR);
	} while (ret == GNUTLS_E_AGAIN || ret == GNUTLS_E_INTERRUPTED);
	net_close_sock (fd);
	gnutls_deinit (session);
	return 0;
}

#endif /* HAVE_LIBGNUTLS */

/*------------------------======= End Of File =======------------------------*/
