/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* net_module.c -- Manages module functions */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lib_functions.h>
#include <net_module.h>

struct modules {
	char name[SCHEDWI_NET_FUNCTION_LENGHT];
	int (*function)(net_id *, char *, size_t);
};

static struct modules *module_array = NULL;
static size_t module_array_size = 0;
static int module_array_idx = 0;


/*
 * Construct the buffer containing the module name
 */
static void
build_module_name_static (const char *module_name, char buffer[])
{
	schedwi_memset (buffer, 0, SCHEDWI_NET_FUNCTION_LENGHT);
	if (module_name != NULL) {
		strncpy (buffer, module_name, SCHEDWI_NET_FUNCTION_LENGHT);
	}
}


/*
 * Construct the buffer containing the module name.  This buffer must then
 * be sent to the remote socket
 *
 * Return:
 *   The size of the buffer (which must be freed by the caller) or
 *   -1 in case of memory allocation error
 */
ssize_t
build_module_name (const char *module_name, char **buffer)
{
	char *ret;

#if HAVE_ASSERT_H
	assert (buffer != NULL);
#endif

	ret = (char *) malloc (SCHEDWI_NET_FUNCTION_LENGHT);
	if (ret == NULL) {
		return -1;
	}
	build_module_name_static (module_name, ret);
	*buffer = ret;
	return SCHEDWI_NET_FUNCTION_LENGHT;
}


/*
 * Return the function associated this the provided module name
 *
 * Return:
 *   A pointer to the function or
 *   NULL if the module name is not found
 */
int
(*net_module (const char *module_name))(net_id *, char *, size_t)
{
	int i;

	if (module_name == NULL || module_array == NULL) {
		return NULL;
	}

	for (i = 0; i < module_array_idx; i++) {
		if (schedwi_strncasecmp (module_array[i].name,
					module_name,
					SCHEDWI_NET_FUNCTION_LENGHT) == 0)
		{
			return module_array[i].function;
		}
	}
	return NULL;
}


/*
 * Add a module to the module list
 *
 * Return:
 *   0 --> No error
 *  -1 --> Memory allocation error
 */
int
add_module (const char *module_name, int (*module)(net_id *, char *, size_t))
{
	struct modules *tmp;

#if HAVE_ASSERT_H
	assert (module_name != NULL && module != NULL);
#endif

	if (module_array_idx >= module_array_size) {
		tmp = (struct modules *) realloc (module_array,
			(module_array_size + 5) * sizeof (struct modules));
		if (tmp == NULL) {
			return -1;
		}
		module_array = tmp;
		module_array_size += 5;
	}
	module_array[module_array_idx].function = module;
	build_module_name_static (	module_name,
					module_array[module_array_idx].name); 
	module_array_idx++;
	return 0;
}


/*
 * Destroy/free the module list
 */
void
destroy_modules ()
{
	if (module_array != NULL) {
		free (module_array);
	}
	module_array = NULL;
	module_array_idx = 0;
	module_array_size = 0;
}

/*-----------------============== End Of File ==============-----------------*/
