/* Logwi
   Copyright (C) 2007 Herve Quatremain
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301,
   USA.
*/

/*
 * lwc_log.c -- Function to log messages
 *
 * Example:
 *    // Using the syslog mechanism
 *    int n = 0;
 *    lwc_newLog ("my_program", 1, LOG_LOCAL0, NULL, NULL);
 *    lwc_writeLog (LOG_INFO, "sample information message number %d", n++); 
 *    lwc_writeLog (LOG_INFO, "sample information message number %d", n++); 
 *    lwc_closeLog ();
 *
 *    // Using a log file 
 *    lwc_newLog (NULL, 0, 0, "/tmp/my_program.log", "%m/%d/%Y");
 *    lwc_writeLog (LOG_INFO, "sample information message number %d", n++); 
 *    lwc_writeLog (LOG_INFO, "sample information message number %d", n++); 
 *    lwc_closeLog ();
 */

#include <schedwi.h>

#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#if STDC_HEADERS
#include <string.h>
#endif

#if HAVE_STDIO_H
#include <stdio.h>
#endif

#if HAVE_TIME_H
#include <time.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

/*PTHREAD #include <pthread.h> */

#include <lwc_log.h>


/* Static variables (ie. One log by prog !) */
static FILE  *logFile    = NULL; /* Output stream of the log file   */
static char   useSyslog  = 0;    /* Do we use syslog (or a file ) ? */

#if !HAVE_LOCALTIME || !HAVE_TIME
#undef SCHEDWI_DEFAULT_DATEFORMAT
#define SCHEDWI_DEFAULT_DATEFORMAT ""
#endif
/* Date format (see strftime (3))  */
static char dateFormat[MAX_LINE_LENGTH] = SCHEDWI_DEFAULT_DATEFORMAT;


/*PTHREAD static pthread_mutex_t lock    = PTHREAD_MUTEX_INITIALIZER; */


/*
 * Reinitialize the log mechanism (close the current open log and reset the
 * default date/time format
 *
 * Return:
 *      0 --> OK
 *  other --> fclose error (errno is set)
 */
static int
lwc_initLog ()
{
	int ret = 0;

/*PTHREAD	pthread_mutex_lock(&lock); */
	if (logFile != NULL) {
		ret = fclose (logFile);
	}
	if (useSyslog == 1) {
#if HAVE_CLOSELOG
		closelog ();
#endif
	}

	logFile = NULL;
	useSyslog = 0;
	strncpy (dateFormat, SCHEDWI_DEFAULT_DATEFORMAT, MAX_LINE_LENGTH);
	dateFormat[MAX_LINE_LENGTH - 1] = '\0';
/*PTHREAD	pthread_mutex_unlock(&lock); */
	return ret;
}


/*
 * Open the log file (or the syslog)
 *
 *       ident: The string pointed to by ident is added to each message, and is
 *              typically set to the program name (see openlog(3)).  ident is
 *              only used if the log mechanism is syslog (inSyslog is 1)
 *    inSyslog: If 1, use the syslog mechanism. If 0 use a log file (its name
 *              is provided by the logFileName argument)
 *    facility: The facility argument is used to specify what type of program
 *              is logging the message (see openlog(3)).  facility is only
 *              used if the log mechanism is syslog (inSyslog is 1)
 * logFileName: File name used to log message. Only used if inSyslog is 0. If
 *              NULL, stderr is used instead
 *     dateFmt: Date/time format (see strftime(3)) to use when logging
 *              messages. If NULL or empty the default format is "%c".
 *              dateFmt is only used if inSyslog is 0
 *
 * Return:
 *   0 --> OK
 *  -1 --> fopen error (errno is set)
 */
int
lwc_newLog (	const char * ident,
		char inSyslog,
		int facility,
		const char * logFileName,
		const char * dateFmt)
{


	(void)lwc_initLog ();

/*PTHREAD	pthread_mutex_lock(&lock); */

	/*
	 * Use syslog or a file?
	 */
	if (inSyslog != 0) {
#if HAVE_OPENLOG
		if (ident != NULL) {
			openlog (ident, LOG_PID, facility);
		}
		else {
			openlog ("", LOG_PID, facility);
		}
#endif
		useSyslog = 1;
	}
	else {
		if (logFileName != NULL) {
			logFile = fopen (logFileName, "a");
			if (logFile == NULL) {
/*PTHREAD				pthread_mutex_unlock(&lock); */
				return -1;
			}
			/* Line buffer */
#if HAVE_SETVBUF
#if SETVBUF_REVERSED
			(void)setvbuf (logFile, _IOLBF, NULL, 0);
#else
			(void)setvbuf (logFile, NULL, _IOLBF, 0);
#endif
#endif
		}
		if (dateFmt != NULL) {
			strncpy (dateFormat, dateFmt, MAX_LINE_LENGTH);
			dateFormat[MAX_LINE_LENGTH - 1] = '\0';
		}
	}
/*PTHREAD	pthread_mutex_unlock(&lock); */
	return 0;
}


/*
 * Close the log file
 *
 * Return:
 *      0 --> OK
 *  other --> fclose error (errno is set)
 */
int
lwc_closeLog ()
{
	lwc_writeLog (LOG_INFO, _("Closing log"));
	return lwc_initLog ();
}


/*
 * Write a message to the log
 * If a log was not previously open by lwc_newLog, the message is simply send
 * to stderr (without date, time, level, ...)
 *
 *        level: see syslog(3) (LOG_EMERG, LOG_ALERT, LOG_CRIT, LOG_ERR, ...)
 *       format: see printf(3)
 */
void
lwc_vwriteLog (int level, const char *format, va_list ap)
{
	char buffer[MAX_LINE_LENGTH];
	char dateStr[MAX_LINE_LENGTH];
	int num, i;

#if HAVE_STRFTIME && HAVE_LOCALTIME && HAVE_TIME
	time_t now;
#endif

#if HAVE_ASSERT_H
	assert (format != NULL);
#endif

	if (useSyslog == 0 && logFile == NULL) {
		vfprintf (stderr, format, ap);
		fputc ('\n', stderr);
		return;
	}

	num = vsnprintf (buffer, MAX_LINE_LENGTH, format, ap);
	if (num >= MAX_LINE_LENGTH || num < 0) { /* buffer is truncated */
		buffer[MAX_LINE_LENGTH - 1] = '\0';
	}
	else {
		buffer[num] = '\0';
	}

	/* Replace all the \n, \r, \a, \b, \v, \f by spaces */
	for (i = 0; buffer[i] != '\0'; i++) {
		if (	   buffer[i] == '\n'
			|| buffer[i] == '\r'
			|| buffer[i] == '\a'
			|| buffer[i] == '\b'
			|| buffer[i] == '\v'
			|| buffer[i] == '\f')
		{
			 buffer[i] = ' ';
		}
	}

/*PTHREAD	pthread_mutex_lock(&lock); */
	if (useSyslog == 0) {

		/* Lock the file */
#if HAVE_LOCKF && F_LOCK && F_ULOCK
		fseek (logFile, 0L, SEEK_SET);
		lockf (fileno (logFile), F_LOCK, 0);
		fseek (logFile, 0L, SEEK_END);
#elif HAVE_FLOCK && LOCK_EX && LOCK_UN
		flock (fileno (logFile), LOCK_EX);
#endif

#if HAVE_STRFTIME && HAVE_LOCALTIME && HAVE_TIME
		now = time (NULL);
		if (strftime (	dateStr,
				MAX_LINE_LENGTH,
				dateFormat,
				localtime (&now)) != 0)
		{
			fputs (dateStr, logFile);
			fputs (" - ",   logFile);
		}
#endif
		switch (level) {
			case LOG_EMERG:
				fputs (_("Emergency"), logFile);
				fputs (" - ", logFile);
				break;
			case LOG_ALERT:
				fputs (_("Alert"), logFile);
				fputs (" - ", logFile);
				break;
			case LOG_CRIT:
				fputs (_("Critical"), logFile);
				fputs (" - ", logFile);
				break;
			case LOG_ERR:
				fputs (_("Error"), logFile);
				fputs (" - ", logFile);
				break;
			case LOG_WARNING:
				fputs (_("Warning"), logFile);
				fputs (" - ", logFile);
				break;
			case LOG_NOTICE:
				fputs (_("Notice"), logFile);
				fputs (" - ", logFile);
				break;
			case LOG_INFO:
				fputs (_("Information"), logFile);
				fputs (" - ", logFile);
				break;
			case LOG_DEBUG:
				fputs (_("Debug"), logFile);
				fputs (" - ", logFile);
				break;
		}
		fprintf (logFile, "%ld - %s\n", (long int)getpid (), buffer);

		/* Unlock the file */
#if HAVE_LOCKF && F_LOCK && F_ULOCK
		fseek (logFile, 0L, SEEK_SET);
		lockf (fileno (logFile), F_ULOCK, 0);
		fseek (logFile, 0, SEEK_END);
#elif HAVE_FLOCK && LOCK_EX && LOCK_UN
		flock (fileno (logFile), LOCK_UN);
#endif
	}
	else {
#if HAVE_SYSLOG
		syslog (level, buffer);
#endif
	}
/*PTHREAD	pthread_mutex_unlock(&lock); */
}


/*
 * Write a message to the log
 * If a log was not previously open by lwc_newLog, the message is send to
 * stderr
 */
void
lwc_writeLog (int level, const char * format, ...)
{
	va_list ap;

#if HAVE_ASSERT_H
	assert (format != NULL);
#endif

	va_start (ap, format);
	lwc_vwriteLog (level, format, ap);
	va_end (ap);
}

/******************************** End Of File ********************************/
