/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* references_cb.c -- GUI functions for the references window */

#include <schedwi.h>

#include "support.h"
#include "interface.h"

#include <sql_where_used.h>
#include <message_windows.h>
#include <references_cb.h>


/*
 * Initialize the tree view
 */
static void
dialog_host_references_build_view (	GtkWidget *dialog_ref,
					lwc_LL *list_job, lwc_LL *list_file)
{
	GtkWidget *view;
	GtkListStore *store;
	referenced_object_t *obj;
	GtkTreeIter iter;
	GtkCellRenderer *renderer;
	GtkTreeSelection *select;

	view = lookup_widget (	GTK_WIDGET (dialog_ref),
				"treeview_host_references");

	/* Create and fill the GtkListStore object */
	store = gtk_list_store_new (	2,
					G_TYPE_STRING,	/* Object type */
					G_TYPE_STRING); /* Object */

	lwc_rewindLL (list_job);
	while ((obj =(referenced_object_t *)lwc_nextLL (list_job)) != NULL)
	{
		gtk_list_store_append (store, &iter);
		if (obj->obj_type == JOBSET) {
			gtk_list_store_set (	store, &iter,
						0, _("Jobset"),
						1, obj->name,
						-1);
		}
		else {
			gtk_list_store_set (	store, &iter,
						0, _("Job"),
						1, obj->name,
						-1);
		}
	}

	lwc_rewindLL (list_file);
	while ((obj=(referenced_object_t *)lwc_nextLL (list_file)) != NULL)
	{
		gtk_list_store_append (store, &iter);
		gtk_list_store_set (	store, &iter,
					0, _("File"),
					1, obj->name,
					-1);
	}

	/* Associate the model with the view */
	gtk_tree_view_set_model (GTK_TREE_VIEW (view), GTK_TREE_MODEL (store));
	g_object_unref (store);

	/* Add the columns and cells */
	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_insert_column_with_attributes (	GTK_TREE_VIEW (view),
							-1,
							_("Type"),
							renderer,
							"text", 0,
							NULL);
	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_insert_column_with_attributes (	GTK_TREE_VIEW (view),
							-1,
							_("Object"),
							renderer,
							"text", 1,
							NULL);

	/* Selection */
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
	gtk_tree_selection_set_mode (select, GTK_SELECTION_NONE);
}


/*
 * Initialize a reference window (for a host)
 *
 * Return:
 *       0 --> No error
 *       1 --> Host not used (an information popup has been displayed)
 *   other --> Error (an error popup has been displayed)
 */
static int
dialog_host_references_init (	GtkWidget *dialog_ref,
				const gchar *host_id,
				const gchar *host_name,
				const gchar *portnum,
				int workload_date)
{
	GString *message;
	GtkWidget *label;
	lwc_LL *list_job, *list_file;
	int ret;

	/* Retrieve the objects used by the provided host id */
	ret = sql_host_job_list (workload_date, host_id, &list_job,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error"));
	if (ret != 0) {
		return ret;
	}

	ret = sql_host_file_list (workload_date, host_id, &list_file,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error"));
	if (ret != 0) {
		lwc_delLL (list_job, (void (*)(const void *))
						destroy_referenced_object);
		return ret;
	}

	/* Host not used */
	if (lwc_getNumNode (list_job) + lwc_getNumNode (list_file) <= 0) {
		info_window (NULL,
			_("This host is not used by any other object"));
		lwc_delLL (list_job, (void (*)(const void *))
						destroy_referenced_object);
		lwc_delLL (list_file, (void (*)(const void *))
						destroy_referenced_object);
		return 1;
	}

	/* Label */
	message = g_string_new ("");
	g_string_printf (message,
	_("<b>%s (port %s) is used in the following objects:</b>"),
			host_name, portnum);
	label = lookup_widget (GTK_WIDGET (dialog_ref),
				"label_host_references");
	gtk_label_set_markup (GTK_LABEL (label), message->str);
	g_string_free (message, TRUE);

	/* Model and view */
	dialog_host_references_build_view (dialog_ref, list_job, list_file);
	lwc_delLL (list_job, (void (*)(const void *))
						destroy_referenced_object);
	lwc_delLL (list_file, (void (*)(const void *))
						destroy_referenced_object);
	return 0;
}


/*
 * Create a new host reference window
 *
 * Return:
 *   The new GtkWidget object or
 *   NULL in case of error (an error popup has been displayed)
 */
GtkWidget *
new_dialog_host_references (	const gchar *host_id,
				const gchar *host_name,
				const gchar *portnum,
				int workload_date)
{
	GtkWidget *dialog_ref;

	dialog_ref = create_dialog_host_references ();
	if (dialog_host_references_init (dialog_ref,
					host_id, host_name, portnum,
					workload_date) != 0)
	{
		gtk_widget_destroy (dialog_ref);
		return NULL;
	}
	else {
		gtk_widget_show (dialog_ref);
		return dialog_ref;
	}
}


/*
 * Create a new error host reference window
 *
 * Return:
 *   The new GtkWidget object or
 *   NULL in case of error (an error popup has been displayed)
 */
GtkWidget *
new_error_dialog_host_references (	const gchar *host_name,
					const gchar *portnum,
					lwc_LL *list_job, lwc_LL *list_file)
{
	GtkWidget *dialog_ref;
	GString *message;
	GtkWidget *label, *image;

	dialog_ref = create_dialog_host_references ();

	/* Label */
	message = g_string_new ("");
	g_string_printf (message,
_("<b>Cannot delete %s (port %s) as it is used in the following objects:</b>"),
			host_name, portnum);
	label = lookup_widget (	GTK_WIDGET (dialog_ref),
				"label_host_references");
	gtk_label_set_markup (GTK_LABEL (label), message->str);
	g_string_free (message, TRUE);

	/* Image */
	image = lookup_widget (	GTK_WIDGET (dialog_ref),
				"image_host_references");
	gtk_image_set_from_stock (	GTK_IMAGE (image),
					"gtk-dialog-error",
					GTK_ICON_SIZE_DIALOG);

	/* Model and view */
	dialog_host_references_build_view (dialog_ref, list_job, list_file);
	gtk_widget_show (dialog_ref);
	return dialog_ref;
}


/*
 * Initialize the tree view
 */
static void
dialog_env_references_build_view (	GtkWidget *dialog_ref,
					const gchar *title,
					lwc_LL *list_job, lwc_LL *list_host)
{
	GtkWidget *view;
	GtkListStore *store;
	referenced_object_t *obj;
	GtkTreeIter iter;
	GtkCellRenderer *renderer;
	GtkTreeSelection *select;

	/* Title */
	if (title != NULL) {
		gtk_window_set_title (GTK_WINDOW (dialog_ref), title);
	}

	/* Create and fill the GtkListStore object */
	store = gtk_list_store_new (	2,
					G_TYPE_STRING,	/* Object type */
					G_TYPE_STRING);	/* Object */

	lwc_rewindLL (list_job);
	while ((obj =(referenced_object_t *)lwc_nextLL (list_job)) != NULL)
	{
		gtk_list_store_append (store, &iter);
		if (obj->obj_type == JOBSET) {
			gtk_list_store_set (	store, &iter,
						0, _("Jobset"),
						1, obj->name,
						-1);
		}
		else {
			gtk_list_store_set (	store, &iter,
						0, _("Job"),
						1, obj->name,
						-1);
		}
	}

	lwc_rewindLL (list_host);
	while ((obj=(referenced_object_t *)lwc_nextLL (list_host)) != NULL)
	{
		gtk_list_store_append (store, &iter);
		gtk_list_store_set (	store, &iter,
					0, _("Host"),
					1, obj->name,
					-1);
	}

	/* Associate the model with the view */
	view = lookup_widget (	GTK_WIDGET (dialog_ref),
				"treeview_host_references");
	gtk_tree_view_set_model (GTK_TREE_VIEW (view), GTK_TREE_MODEL (store));
	g_object_unref (store);

	/* Add the columns and cells */
	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_insert_column_with_attributes (	GTK_TREE_VIEW (view),
							-1,
							_("Type"),
							renderer,
							"text", 0,
							NULL);
	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_insert_column_with_attributes (	GTK_TREE_VIEW (view),
							-1,
							_("Object"),
							renderer,
							"text", 1,
							NULL);

	/* Selection */
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
	gtk_tree_selection_set_mode (select, GTK_SELECTION_NONE);
}


/*
 * Initialize a reference window (for an environment)
 *
 * Return:
 *       0 --> No error
 *       1 --> Environment not used (an information popup has been displayed)
 *   other --> Error (an error popup has been displayed)
 */
static int
dialog_env_references_init (	GtkWidget *dialog_ref,
				const gchar *env_id,
				const gchar *env_name,
				int workload_date)
{
	GString *message;
	GtkWidget *label;
	lwc_LL *list_job, *list_host;
	int ret;

	/* Retrieve the objects used by the provided environment */
	ret = sql_env_job_list (workload_date, env_id, &list_job,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error"));
	if (ret != 0) {
		return ret;
	}

	ret = sql_env_host_list (workload_date, env_id, &list_host,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error"));
	if (ret != 0) {
		lwc_delLL (list_job, (void (*)(const void *))
						destroy_referenced_object);
		return ret;
	}

	/* Environment not used */
	if (lwc_getNumNode (list_job) + lwc_getNumNode (list_host) <= 0) {
		info_window (NULL,
			_("This environment is not used by any other object"));
		lwc_delLL (list_job, (void (*)(const void *))
						destroy_referenced_object);
		lwc_delLL (list_host, (void (*)(const void *))
						destroy_referenced_object);
		return 1;
	}

	/* Label */
	message = g_string_new ("");
	g_string_printf (message,
			_("<b>%s is used in the following objects:</b>"),
			env_name);
	label = lookup_widget (GTK_WIDGET (dialog_ref),
				"label_host_references");
	gtk_label_set_markup (GTK_LABEL (label), message->str);
	g_string_free (message, TRUE);

	dialog_env_references_build_view (	dialog_ref,
						_("Environment references"),
						list_job, list_host);
	lwc_delLL (list_job, (void (*)(const void *))
						destroy_referenced_object);
	lwc_delLL (list_host, (void (*)(const void *))
						destroy_referenced_object);
	return 0;
}


/*
 * Create a new environment reference window
 *
 * Return:
 *   The new GtkWidget object or
 *   NULL in case of error (an error popup has been displayed)
 */
GtkWidget *
new_dialog_env_references (	const gchar *env_id, const gchar *env_name,
				int workload_date)
{
	GtkWidget *dialog_ref;

	dialog_ref = create_dialog_host_references ();
	if (dialog_env_references_init (dialog_ref, env_id, env_name,
					workload_date) != 0)
	{
		gtk_widget_destroy (dialog_ref);
		return NULL;
	}
	else {
		gtk_widget_show (dialog_ref);
		return dialog_ref;
	}
}


/*
 * Create a new error environment reference window
 *
 * Return:
 *   The new GtkWidget object or
 *   NULL in case of error (an error popup has been displayed)
 */
GtkWidget *
new_error_dialog_env_references (	const gchar *name,
					lwc_LL *list_job, lwc_LL *list_host)
{
	GtkWidget *dialog_ref;
	GString *message;
	GtkWidget *label, *image;

	dialog_ref = create_dialog_host_references ();

	/* Label */
	message = g_string_new ("");
	g_string_printf (message,
_("<b>Cannot delete %s as it is used in the following objects:</b>"),
			name);
	label = lookup_widget (	GTK_WIDGET (dialog_ref),
				"label_host_references");
	gtk_label_set_markup (GTK_LABEL (label), message->str);
	g_string_free (message, TRUE);

	/* Image */
	image = lookup_widget (	GTK_WIDGET (dialog_ref),
				"image_host_references");
	gtk_image_set_from_stock (	GTK_IMAGE (image),
					"gtk-dialog-error",
					GTK_ICON_SIZE_DIALOG);

	/* Model and view */
	dialog_env_references_build_view (	dialog_ref,
						_("Environment references"),
						list_job, list_host);
	gtk_widget_show (dialog_ref);
	return dialog_ref;
}


/*
 * Create a new calendar reference window
 *
 * Return:
 *   The new GtkWidget object or
 *   NULL in case of error (an error popup has been displayed)
 */
GtkWidget *
new_dialog_cal_references (const gchar *cal_name, lwc_LL *list_job)
{
	GtkWidget *dialog_ref;
	GtkWidget *label;
	GString *message;

	dialog_ref = create_dialog_host_references ();

	/* Label */
	message = g_string_new ("");
	g_string_printf (message,
			_("<b>%s is used in the following objects:</b>"),
			cal_name);
	label = lookup_widget (GTK_WIDGET (dialog_ref),
				"label_host_references");
	gtk_label_set_markup (GTK_LABEL (label), message->str);
	g_string_free (message, TRUE);

	/* Model and view */
	dialog_env_references_build_view (	dialog_ref,
						_("Calendar references"),
						list_job, NULL);
	gtk_widget_show (dialog_ref);
	return dialog_ref;
}


/*
 * Create a new error calendar reference window
 *
 * Return:
 *   The new GtkWidget object or
 *   NULL in case of error (an error popup has been displayed)
 */
GtkWidget *
new_error_dialog_cal_references (	const gchar *error_message,
					lwc_LL *list_job)
{
	GtkWidget *dialog_ref;
	GtkWidget *label, *image;

	dialog_ref = create_dialog_host_references ();

	/* Label */
	label = lookup_widget (	GTK_WIDGET (dialog_ref),
				"label_host_references");
	gtk_label_set_markup (GTK_LABEL (label), error_message);

	/* Image */
	image = lookup_widget (	GTK_WIDGET (dialog_ref),
				"image_host_references");
	gtk_image_set_from_stock (	GTK_IMAGE (image),
					"gtk-dialog-error",
					GTK_ICON_SIZE_DIALOG);

	/* Model and view */
	dialog_env_references_build_view (	dialog_ref,
						_("Calendar references"),
						list_job, NULL);
	gtk_widget_show (dialog_ref);
	return dialog_ref;
}

/*------------------------======= End Of File =======------------------------*/
