/* Schedwi
   Copyright (C) 2007 Herve Quatremain
     
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
     
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.
     
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/*
 * parsevar.c -- Parse an environment array
 *
 * The `parse_param' function is used to parse an environment array, doing
 * all the variable substitutions.  A new environment array is built and
 * returned.
 * The input environment array is an array of strings.  These strings have
 * the form `name=value'.  The array is terminated by a NULL pointer.
 * The output environment array has the same format but is embedded in a
 * environment_t structure (it's the env member of the structure).
 *
 * In the value part of each environment variable, parameter substitutions
 * are done using a similar syntax of the bash shell parameter expansion
 * (with limitations):
 *     $parameter or ${parameter}
 *           The  value of `parameter' is substituted.  The braces are required
 *           when `parameter' is followed by a character which is not to be
 *           interpreted as part of its name.
 *     ${parameter:-word}
 *           If `parameter' is unset or null, the expansion of `word' is
 *           substituted.  Otherwise, the value of `parameter' is
 *           substituted.
 *     ${parameter:=word}
 *           If `parameter' is unset or null, the expansion of `word' is
 *           assigned to `parameter'.  The value of `parameter' is then
 *           substituted.
 *     ${parameter:?word}
 *           If `parameter' is unset or null, the expansion of `word' is
 *           substituted if not empty.  Otherwise the string `parameter
 *           null or n ot set' is substituted.  THIS IS A DIFFERENT BEHAVIOR
 *           than bash.
 *     ${parameter:+word}
 *           If `parameter' is unset or null, nothing is substituted,
 *           otherwise the expansion of `word' is substituted.
 */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_CTYPE_H
#include <ctype.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lib_functions.h>
#include <parsevar.h>

/*
 * Copy a string
 * If *dst is NULL, copy_str will allocate a buffer for containing the
 * copy, which must be freed by the user program.  Alternatively, before
 * calling copy_str, *dst can contain a pointer to a malloc()-allocated
 * buffer *dst_len in size.  If the buffer is not large enough, copy_str
 * resizes the buffer, updating *dst and *dst_len.
 *
 * Return:
 *    0 --> No error
 *   -1 --> Memory allocation error (dst is not modified)
 */
static int
copy_str (	char **dst, size_t *dst_len, int dst_idx,
		const char *src, size_t src_len) 
{
	char *tmp;
	size_t l;

#if HAVE_ASSERT_H
	assert (dst != NULL && dst_len != NULL && src != NULL);
#endif

	if (*dst == NULL || *dst_len == 0) {
		*dst = NULL;
		*dst_len = 0;
	}

	if (*dst_len - dst_idx <= src_len) {
		l = src_len + dst_idx + 128;
		tmp = (char *) malloc (l);
		if (tmp == NULL) {
			return -1;
		}
		if (*dst != NULL && *dst_len > 0) {
			schedwi_memcpy (tmp, *dst, *dst_len);
			free (*dst);
		}
		*dst = tmp;
		*dst_len = l;
	}

	schedwi_memcpy (*dst + dst_idx, src, src_len);
	(*dst)[dst_idx + src_len] = '\0';
	return 0;
}


/*
 * Parse a name ([_A-Za-z][_A-Za-z0-9]*) and return the index of the first
 * character not belonging to the name
 *
 * Return:
 *    The index of the character after the name or
 *    -1 in syntax error
 */
static ssize_t
parse_name (const char *param, size_t param_len)
{
	ssize_t i;

#if HAVE_ASSERT_H
	assert (param != NULL);
#endif

	if (param_len == 0) {
		return -1;
	}

	if (param_len == 0 || (param[0] != '_' && !isalpha (param[0]))) {
		return -1;
	}

	for (	i = 1;
		i < param_len && (param[i] == '_' || isalnum (param[i]));
		i++);
	return i;
}


/*
 * Parse a variable name and return in *out and *out_len its value (from
 * the env array).
 * If *out is NULL, parse_variable will allocate a buffer for containing the
 * value of the variable, which must be freed by the user program.
 * Alternatively, before calling parse_variable, *out can contain a pointer to
 * a malloc()-allocated buffer *out_len in size.  If the buffer is not large
 * enough, parse_variable resizes the buffer, updating *out and *out_len.
 *
 * Return:
 *    The number of characters parsed in the param string or
 *    -1 in case of memory allocation error (the previous value of *out and
 *    *out_len are not modified)
 *    -2 in case of syntax error
 */
static ssize_t
parse_variable (const char *param, size_t param_len,
		environment_t *env, char **out, size_t *out_len)
{
	ssize_t i;
	int idx_env;
	char *equal, *tmp;
	const char *env_val;
	size_t val_len;

#if HAVE_ASSERT_H
	assert (param != NULL && out != NULL && out_len != NULL);
#endif

	/* Parse the variable name */
	i = parse_name (param, param_len);
	if (i < 0) {
		return -2;
	}

	/* Search the variable name in the env array */
	idx_env = search_environment (env, param, i);
	/* Not found, the associated value will be the empty string */
	if (idx_env == -1) {
		if (*out_len < 1) {
			*out = (char *) malloc (128);
			if (*out == NULL) {
				return -1;
			}
			*out_len = 128;
		}
		(*out)[0] = '\0';
		return i;
	}

	/* Parse the associated value */
	env_val = getvar_environment (env, idx_env);
	equal = strchr (env_val, '=');
	equal++;
	val_len = schedwi_strlen (equal);
	if (*out_len < val_len + 1) {
		tmp = (char *) malloc (val_len + 128);
		if (tmp == NULL) {
			return -1;
		}
		if (*out_len > 0 && *out != NULL) {
			free (*out);
		}
		*out = tmp;
		*out_len = val_len + 128;
	}
	strcpy (*out, equal);
	return i;
}


/*
 * Find the closing '}' in the provided string
 *
 * Return:
 *    The index in s of the closing bracket or
 *    -1 if not found
 */
static ssize_t
find_matching_bracket (const char *s, size_t len)
{
	int nb;
	ssize_t i;

#if HAVE_ASSERT_H
	assert (s != NULL);
#endif

	nb = 1;
	for (i = 0; i< len && s[i] != '\0'; i++) {
		if (s[i] == '{') {
			nb++;
		}
		if (s[i] == '}') {
			nb--;
			if (nb == 0) {
				return i;
			}
		}
	}
	return -1;
}


/*
 * Parse and expand the parameter.  Set its value int *out and return the
 * index in param of the character following the parsed parameter
 * If *out is NULL, expand_param will allocate a buffer for containing the
 * value of the parameter, which must be freed by the user program.
 * Alternatively, before calling expand_param, *out can contain a pointer to
 * a malloc()-allocated buffer *out_len in size.  If the buffer is not large
 * enough, expand_param resizes the buffer, updating *out and *out_len.
 *
 * Return:
 *    The index in param of the caracter following the parsed parameter OR
 *    -1 in case of memory allocation error (the previous value of *out and
 *       *out_len are not modified) OR
 *    -2 in case of syntax error (The content of *out and *out_len is
 *       unspecified)
 */
static ssize_t
expand_param (	const char *param, size_t param_len,
		environment_t *env, char **out, size_t *out_len)
{
	ssize_t i, end_key_idx;
	char *val, *msg;
	ssize_t end;
	int ret;
	size_t val_len;

#if HAVE_ASSERT_H
	assert (param != NULL && out != NULL && out_len != NULL);
#endif

	/* $VAR */
	if (param[0] != '{') {
		return parse_variable (param, param_len, env, out, out_len);
	}

	i = parse_variable (param + 1, param_len - 1, env, out, out_len);
	if (i < 0) {
		return i;
	}

	/* ${VAR} */
	if (param[i + 1] == '}') {
		return i + 2;
	}

	if (param[i + 1] != ':' || (	   param[i + 2] != '-'
					&& param[i + 2] != '='
					&& param[i + 2] != '?'
					&& param[i + 2] != '+'))
	{
		return -2;
	}

	/* ${VAR:-word}, ${VAR:?word}, ... */
	end = find_matching_bracket (param + i + 2, param_len - (i + 2));
	if (end < 0) {
		return -2;
	}

	val = NULL;
	val_len = 0;

	ret = parse_string (param + i + 3, end - 1, env, &val, &val_len);
	if (ret < 0) {
		if (val != NULL && val_len > 0) {
			free (val);
		}
		return ret;
	}

	switch (param[i + 2]) {
		case '-':
			if ((*out)[0] != '\0') {
				if (val != NULL && val_len > 0) {
					free (val);
				}
				return end + i + 3;
			}
			if (*out != NULL && *out_len > 0) {
				free (*out);
			}
			*out = val;
			*out_len = val_len;
			return end + i + 3;

		case '=':
			if ((*out)[0] != '\0') {
				if (val != NULL && val_len > 0) {
					free (val);
				}
				return end + i + 3;
			}
			if (*out != NULL) {
				if (*out_len > 0) {
					free (*out);
				}
				*out = NULL;
				*out_len = 0;
			}

			end_key_idx = parse_name (param + 1, param_len - 1);
			ret = add_env_s (	env,
						param + 1, end_key_idx,
						val, schedwi_strlen (val));
			if (ret < 0) {
				free (val);
				return ret;
			}

			*out = val;
			*out_len = val_len;
			return end + i + 3;

		case '?':
			if ((*out)[0] != '\0') {
				if (val != NULL && val_len > 0) {
					free (val);
				}
				return end + i + 3;
			}
			if (*out != NULL && *out_len > 0) {
				free (*out);
			}
			if (	   val_len == 0
				|| val == NULL
				|| *val == '\0')
			{
				if (val != NULL) {
					free (val);
				}
				msg = _("parameter null or not set");
				*out = (char *)malloc (	  schedwi_strlen (msg)
							+ 1);
				if (*out == NULL) {
					*out_len = 0;
					return -1;
				}
				strcpy (*out, msg);
				*out_len = schedwi_strlen (msg) + 1;
			}
			else {
				*out = val;
				*out_len = val_len;
			}
			return end + i + 3;

		case '+':
			if ((*out)[0] == '\0') {
				if (val != NULL && val_len > 0) {
					free (val);
				}
				return end + i + 3;
			}
			if (*out != NULL && *out_len > 0) {
				free (*out);
			}
			*out = val;
			*out_len = val_len;
			return end + i + 3;
	}

	return -2;
}


/*
 * Parse the variable value.  Set *out with it.
 * If *out is NULL, parse_string will allocate a buffer for containing the
 * value of the parameter, which must be freed by the user program.
 * Alternatively, before calling parse_string, *out can contain a pointer to
 * a malloc()-allocated buffer *out_len in size.  If the buffer is not large
 * enough, parse_string resizes the buffer, updating *out and *out_len.
 *
 * Return:
 *    0 --> No error
 *   -1 --> Memory allocation error (the previous value of *out and
 *          *out_len are not modified)
 *   -2 --> Syntax error (the previous value of *out and *out_len are not
 *          modified)
 */
int
parse_string (	const char *string, size_t string_len,
		environment_t *env, char **out, size_t *out_len)
{
	const char *val;
	char *dollar, *expand;
	size_t expand_len, l;
	ssize_t substr_len;
	int idx_out;

	if (out == NULL || out_len == NULL) {
		return 0;
	}

	if (string == NULL) {
		if (copy_str (out, out_len, 0, "", 0) < 0) {
			return -1;
		}
		return 0;
	}

	expand = NULL;
	expand_len = 0;
	idx_out = 0;
	val = string;

	/* Look for parameters to expand ($VAR) */
	dollar = strchr (val, '$');
	while (dollar != NULL && dollar < string + string_len) {

		/* \$ */
		if (dollar != val && *(dollar-1) == '\\') {

			/*
			 * Copy until the dollar (included) but
			 * without the \ character
			 */
			if (	   copy_str (	out, out_len, idx_out,
						val, dollar - 1 - val) < 0
				|| copy_str (	out, out_len,
						idx_out + dollar - 1 - val,
						"$", 1) < 0)
			{
				if (expand != NULL && expand_len > 0) {
					free (expand);
				}
				return -1;
			}
			idx_out += dollar - val;
			val = dollar + 1;
		}

		/* $VAR */
		else {
			/* Copy up to the dollar (not included) */
			if (copy_str (	out, out_len, idx_out,
					val, dollar - val) < 0)
			{
				if (expand != NULL && expand_len > 0) {
					free (expand);
				}
				return -1;
			}

			idx_out += dollar - val;
			val = dollar + 1;

			/* Expand the parameter */
			substr_len = expand_param (val,
						string_len - (val - string),
						env, &expand, &expand_len);
			if (substr_len < 0) {
				if (expand != NULL && expand_len > 0) {
					free (expand);
				}
				return substr_len;
			}

			/* Copy the expanded value */
			l = schedwi_strlen (expand);
			if (copy_str (out, out_len, idx_out, expand, l) < 0) {
				free (expand);
				return -1;
			}

			idx_out += l;
			val += substr_len;
		}

		dollar = strchr (val, '$');
	}

	if (expand != NULL && expand_len > 0) {
		free (expand);
	}

	/* Copy the end (no more expansion to do) */
	if (copy_str (	out, out_len, idx_out,
			val, string_len - (val - string)) < 0)
	{
		return -1;
	}
	return 0;
}


/*
 * Parse the variable name.  Set *out with it.
 * If *out is NULL, parse_key will allocate a buffer for containing the
 * variable name, which must be freed by the user program.
 * Alternatively, before calling parse_key, *out can contain a pointer to
 * a malloc()-allocated buffer *out_len in size.  If the buffer is not large
 * enough, parse_key resizes the buffer, updating *out and *out_len.
 *
 * Return:
 *    0 --> No error
 *   -1 --> Memory allocation error (the previous value of *out and
 *          *out_len are not modified)
 *   -2 --> Syntax error (the previous value of *out and *out_len are not
 *          modified)
 */
static int
parse_key (const char *key, char **out, size_t *out_len)
{
	int i, j;
	size_t l;
	char *tmp;

#if HAVE_ASSERT_H
	assert (key != NULL && out != NULL && out_len != NULL);
#endif

	/* Skip the spaces */
	for (i = 0; isspace (key[i]); i++);

	/* The variable must start by _ or a letter (A-Za-z) */
	if (key[i] != '_' && !isalpha (key[i])) {
		return -2;
	}

	/* Allocate a memory buffer for copying the variable name */
	l = schedwi_strlen (key + i) + 1;
	if (*out_len < l) {
		tmp = (char *) malloc (l + 128);
		if (tmp == NULL) {
			return -1;
		}
		if (*out != NULL && *out_len > 0) {
			free (*out);
		}
		*out = tmp;
		*out_len = l + 128;
	}

	/* Copy the key (remove all the invalid characters) */
	for (j = 0; key[i] != '\0' && key[i] != '='; i++) {
		if (isalnum (key[i]) || key[i] == '_') {
			(*out)[j++] = key[i];
		}
	}
	(*out)[j] = '\0';
	return 0;
}


/*
 * Parse each variable of the provided environment array.  The parameters
 * in the variables are expanded using the previously parsed variable
 * (ie. $foo is expanded if it was defined in a previous variable in the
 * environment array)
 * In case of error, *error_idx contains the index in the env array of the
 * invalid environment parameter.
 *
 * Example:
 *      env -----------------------------> new_env
 *      A=ABC                     -->      A=ABC
 *      B=A is $A                 -->      B=A is ABC
 *      C=C is ${D:-not set}      -->      C=C is not set
 *      D=\$D is $E               -->      D=$D is 
 *      E=foo                     -->      E=foo
 *
 * Return:
 *    0 --> No error
 *   -1 --> Memory allocation error
 *   -2 --> Syntax error in the key part (variable name)
 *   -3 --> Syntax error in the value part
 */
int
parse_param (const environment_t *env, environment_t *new_env, int *error_idx)
{
	int i, ret;
	char *val, *key, *equal;
	size_t key_len, val_len;

	if (env == NULL || new_env == NULL) {
		return 0;
	}

	key = NULL;
	key_len = 0;
	val = NULL;
	val_len = 0;

	for (i = 0; i < env->idx; i++) {

		equal = strchr ((env->env)[i], '=');
		if (equal == NULL) {
			if (key_len > 0 && key != NULL) {
				free (key);
			}
			if (val_len > 0 && val != NULL) {
				free (val);
			}
			if (error_idx != NULL) {
				*error_idx = i;
			}
			return -2;
		}

		/* Get the variable name */
		ret = parse_key ((env->env)[i], &key, &key_len);
		if (ret != 0) {
			if (key_len > 0 && key != NULL) {
				free (key);
			}
			if (val_len > 0 && val != NULL) {
				free (val);
			}
			if (error_idx != NULL) {
				*error_idx = i;
			}
			return ret;
		}

		/* Get the variable value */
		ret = parse_string (	equal + 1, schedwi_strlen (equal + 1),
					new_env, &val, &val_len);
		if (ret != 0 ) {
			free (key);
			if (val_len > 0 && val != NULL) {
				free (val);
			}
			if (error_idx != NULL) {
				*error_idx = i;
			}
			return (ret == -2) ? -3 : ret;
		}

		/* Add the variable to the environment array */
		ret = add_env (new_env, key, val);
		if (ret != 0 ) {
			free (key);
			free (val);
			if (error_idx != NULL) {
				*error_idx = i;
			}
			return ret;
		}
	}
	if (key_len > 0 && key != NULL) {
		free (key);
	}
	if (val_len > 0 && val != NULL) {
		free (val);
	}
	return 0;
}


/*
 * Parse each variable of the provided argument array.  The parameters
 * in the variables are expanded using the provided environment object.
 * A new argument array is returned int arg.
 *
 * Return:
 *    0 --> No error
 *   -1 --> Memory allocation error
 *   -2 --> Syntax error in the value part
 */
int
parse_args (const argument_t *arg, argument_t *new_arg, environment_t *env,
		int *error_idx)
{
	int i, ret;
	char *val;
	size_t val_len;

	if (arg == NULL || new_arg == NULL) {
		return 0;
	}

	val = NULL;
	val_len = 0;

	for (i = 0; i < arg->idx; i++) {

		/* Get the variable value */
		ret = parse_string (	(arg->arg)[i],
					schedwi_strlen ((arg->arg)[i]),
					env, &val, &val_len);
		if (ret != 0 ) {
			if (val_len > 0 && val != NULL) {
				free (val);
			}
			if (error_idx != NULL) {
				*error_idx = i;
			}
			return ret;
		}

		/* Add the variable to the environment array */
		ret = add_arg (new_arg, val);
		if (ret != 0 ) {
			free (val);
			if (error_idx != NULL) {
				*error_idx = i;
			}
			return ret;
		}
	}
	if (val_len > 0 && val != NULL) {
		free (val);
	}
	return 0;
}


/*
 * Check the syntax of a key
 *
 * Return:
 *   0 --> Syntax OK
 *  -1 --> Syntax error
 */
int
check_key_syntax (const char *key)
{
	int i;

	if (	   key == NULL
		|| key[0] == '0'
		|| (key[0] != '_' && isalpha (key[0]) == 0))
	{
		return -1;
	}
	for (i = 1; key[i] != 0; i++) {
		if (key[i] != '_' && isalnum (key[i]) == 0) {
			return -1;
		}
	}
	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
