/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/*
 * lib_functions.c -- Library functions that may be not available on every
 *                    operating system. Their behavior may be slightly
 *                    different than the library one but their are alright
 *                    for schedwi
 */

#include <lib_functions.h>

/*
 * memset - fill memory with a constant byte
 */
void *
schedwi_memset (void *s, int c, unsigned int n)
{
	char *tmp = s;

	if (s != 0) {
		while (n/10 != 0) {
			*(tmp++) = c;
			*(tmp++) = c;
			*(tmp++) = c;
			*(tmp++) = c;
			*(tmp++) = c;
			*(tmp++) = c;
			*(tmp++) = c;
			*(tmp++) = c;
			*(tmp++) = c;
			*(tmp++) = c;
			n -= 10;
		}
		while (n > 0) {
			*(tmp++) = c;
			n--;
		}
	}
	return s;
}


/*
 * memcpy - copy non overlaping memory area
 */
void *
schedwi_memcpy (void *dest, const void *src, unsigned int n)
{
	char *d = dest;
	const char *s = src;

	if (dest != 0 && src != 0) {
		while (n/10 != 0) {
			*(d++) = *(s++); 
			*(d++) = *(s++); 
			*(d++) = *(s++); 
			*(d++) = *(s++); 
			*(d++) = *(s++); 
			*(d++) = *(s++); 
			*(d++) = *(s++); 
			*(d++) = *(s++); 
			*(d++) = *(s++); 
			*(d++) = *(s++); 
			n -= 10;
		}
		while (n > 0) {
			*(d++) = *(s++);
			n--;
		}
	}
	return dest;
}


/*
 * toupper - convert letter to upper case
 */
int
schedwi_toupper (int c)
{
	if (c >= 'a' && c <= 'z') {
		if ('a'  > 'A') {
			return c  - ('a' - 'A');
		}
		else {
			return c  + ('A' - 'a');
		}
	}
	return c;
}


/*
 * strcasecmp - compare two strings ignoring case
 */
int
schedwi_strcasecmp (const char *s1, const char *s2)
{
	unsigned int i;

	if (s1 == 0) {
		if (s2 == 0) {
			return 0;
		}
		return -1;
	}
	if (s2 == 0) {
		return 1;
	}

	for (	i = 0;
		   s1[i] != '\0' && s2[i] != '\0'
		&& schedwi_toupper (s1[i]) ==  schedwi_toupper (s2[i]);
		i++);
	return schedwi_toupper (s1[i]) - schedwi_toupper (s2[i]);
}


/*
 * strncasecmp - compare two strings ignoring case
 */
int
schedwi_strncasecmp (const char *s1, const char *s2, unsigned int n)
{
	unsigned int i;

	if (s1 == 0) {
		if (s2 == 0) {
			return 0;
		}
		return -1;
	}
	if (s2 == 0) {
		return 1;
	}

	for (	i = 0;
		   i < n
		&& s1[i] != '\0' && s2[i] != '\0'
		&& schedwi_toupper (s1[i]) ==  schedwi_toupper (s2[i]);
		i++);
	return (i == n) ? 0: schedwi_toupper (s1[i]) - schedwi_toupper (s2[i]);
}


/*
 * strlen - calculate the length of a string
 */
unsigned int
schedwi_strlen (const char *s)
{
	unsigned int i;

	if (s == 0) {
		return 0;
	}

	for (i = 0; s[i] != '\0'; i++);
	return i;
}


/*
 * strncmp -- compare two strings
 */
int
schedwi_strncmp (const char *s1, const char *s2, unsigned int n)
{
	unsigned int i;

	if (s1 == 0) {
		if (s2 == 0) {
			return 0;
		}
		return -1;
	}
	if (s2 == 0) {
		return 1;
	}

	for (	i = 0;
		   i < n
		&& s1[i] != '\0' && s2[i] != '\0'
		&& s1[i] == s2[i];
		i++);
	return (i == n) ? 0 : s1[i] - s2[i];
}


/*
 * strnstr - locate a substring in the first n characters
 */
char *
schedwi_strnstr (char *haystack, const char *needle, unsigned int n)
{
	unsigned int len;

	if (needle == 0 || *needle == '\0' || haystack == 0) {
		return 0;
	}

	len = schedwi_strlen (needle);

	while (len <= n) {
		if (schedwi_strncmp (needle, haystack, len) == 0) {
			return haystack;
		}
		n--;
		haystack++;
	}

	return 0;
}

/*------------------------======= End Of File =======------------------------*/
