/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* job_select.c -- Job and link selection management */

#include <schedwi.h>

#include <schedwi_main.h>
#include <job_select.h>


/*
 * Build and return a new job_select_t object.  This object must be freed by
 * the caller by destroy_job_select()
 */
job_select_t *
new_job_select ()
{
	job_select_t *ptr;

	ptr = g_new0 (job_select_t, 1);
	ptr->set = FALSE;
	return ptr;
}


/*
 * Free the provided job_select_t object
 */
void
destroy_job_select (job_select_t *ptr)
{
	if (ptr != NULL) {
		job_select_unselect_all (ptr, FALSE);
		g_free (ptr);
	}
}


/*
 * Select the provided job.  If select_item is TRUE, a select request is sent
 * to the job
 */
void
job_select_job (job_select_t *ptr, child_job_t *job, gboolean select_item)
{
	if (ptr->set == TRUE) {
		if (ptr->item_type == JOB_SELECT_TYPE_JOB) {
			if (ptr->item.job != job) {
				child_job_unselect (ptr->item.job);
			}
		}
		else {
			link_unselect (ptr->item.link);
		}
	}
	else {
		ptr->set = TRUE;
	}
	ptr->item_type = JOB_SELECT_TYPE_JOB;
	ptr->item.job = job;
	if (select_item == TRUE) {
		child_job_select (job);
	}
	main_copy_cut_set_sensitive (TRUE);
	main_delete_set_sensitive (TRUE);
}


/*
 * Select the provided link.  If select_item is TRUE, a select request is sent
 * to the link
 */
void
job_select_link (job_select_t *ptr, link_t *link, gboolean select_item)
{
	if (ptr->set == TRUE) {
		if (ptr->item_type == JOB_SELECT_TYPE_JOB) {
			child_job_unselect (ptr->item.job);
		}
		else {
			if (ptr->item.link != link) {
				link_unselect (ptr->item.link);
			}
		}
	}
	else {
		ptr->set = TRUE;
	}
	ptr->item_type = JOB_SELECT_TYPE_LINK;
	ptr->item.link = link;
	if (select_item == TRUE) {
		link_select (link);
	}
	main_copy_cut_set_sensitive (FALSE);
	main_delete_set_sensitive (TRUE);
}


/*
 * Unselect the currently selected item
 */
void
job_select_unselect (job_select_t *ptr)
{
	if (ptr->set == TRUE) {
		if (ptr->item_type == JOB_SELECT_TYPE_JOB) {
			child_job_unselect (ptr->item.job);
		}
		else {
			link_unselect (ptr->item.link);
		}
		ptr->set = FALSE;
	}
	main_copy_cut_set_sensitive (FALSE);
	main_delete_set_sensitive (FALSE);
}


/*
 * Unselect the selected item.  If redraw is TRUE, the job or link is
 * redrawn
 */
void
job_select_unselect_all (job_select_t *ptr, gboolean redraw)
{
	if (ptr != NULL && ptr->set == TRUE && redraw == TRUE) {
		if (ptr->item_type == JOB_SELECT_TYPE_JOB) {
			child_job_unselect (ptr->item.job);
		}
		else {
			link_unselect (ptr->item.link);
		}
		ptr->set = FALSE;
	}
	main_copy_cut_set_sensitive (FALSE);
	main_delete_set_sensitive (FALSE);
}


/*
 * Is the provided job/jobset selected?
 */
gboolean
job_select_is_job_selected (job_select_t *ptr, child_job_t *job)
{
	if (	   ptr->set == TRUE
		&& ptr->item_type == JOB_SELECT_TYPE_JOB
		&& ptr->item.job == job)
	{
		return TRUE;
	}
	return FALSE;
}

/*-----------------============== End Of File ==============-----------------*/
