/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* env_window_cb.c - GUI functions for the environment window */

#include <schedwi.h>

#include "support.h"
#include "interface.h"

#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#include <lib_functions.h>
#include <parsevar.h>
#include <message_windows.h>
#include <cursor.h>
#include <sql_env.h>
#include <env_list_cb.h>
#include <env_window_cb.h>


/*
 * Free a GString
 */
static void
string_free (gpointer string)
{
	g_string_free ((GString *)string, TRUE);
}


/*
 * Functions used to set values in the `Main' tab
 */
static void
set_id (const char *value, void *data)
{
	GString *value_str;

	/* Store the environment id in the widget */
	value_str = g_string_new (value);
	g_object_set_data_full (G_OBJECT (data), "env_id",
				(gpointer)value_str,
				string_free);
}

static void
set_name (const char *value, void *data)
{
	GtkWidget *label;
	gchar *s;

	label = lookup_widget (GTK_WIDGET (data), "label_env_name");
	s = g_strdup_printf ("<b>%s</b>", value);
	gtk_label_set_markup (GTK_LABEL (label), s);
	g_free (s);
	gtk_window_set_title (GTK_WINDOW (data), value);
}

static void
set_description (const char *value, void *data)
{
	GtkWidget *text;
	GtkTextBuffer *buffer;

	text = lookup_widget (GTK_WIDGET (data), "textview_env_description");
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (text));
	gtk_text_buffer_set_text (buffer, value, -1);
}

/*
 * Callback when the selection changes
 * The buttons are made sensitive depending of the item selected
 */
static void
tree_selection_changed_cb (GtkTreeSelection *selection, gpointer data)
{
	GtkWidget *widget = data;
	GtkWidget *remove, *up, *down, *top, *bottom;
	GtkTreeModel *model;
	GtkTreeIter iter;
	GtkTreePath *path;

	remove = lookup_widget (widget, "button_env_del_var");
	up = lookup_widget (widget, "button_env_up_var");
	down = lookup_widget (widget, "button_env_down_var");
	top = lookup_widget (widget, "button_env_var_top");
	bottom = lookup_widget (widget, "button_env_var_bottom");

	/* Nothing is selected */
	if (gtk_tree_selection_get_selected (selection, &model, &iter) == FALSE)
	{
		gtk_widget_set_sensitive (remove, FALSE);
		gtk_widget_set_sensitive (up, FALSE);
		gtk_widget_set_sensitive (down, FALSE);
		gtk_widget_set_sensitive (top, FALSE);
		gtk_widget_set_sensitive (bottom, FALSE);
		return;
	}

	/* Is there an item before the selected one? */
	gtk_widget_set_sensitive (remove, TRUE);
	path =  gtk_tree_model_get_path (model, &iter);
	if (gtk_tree_path_prev (path) == TRUE) {
		gtk_widget_set_sensitive (up, TRUE);
		gtk_widget_set_sensitive (top, TRUE);
	}
	else {
		gtk_widget_set_sensitive (up, FALSE);
		gtk_widget_set_sensitive (top, FALSE);
	}
	gtk_tree_path_free (path);

	/* Is there an item bellow the selected one? */
	if (gtk_tree_model_iter_next (model, &iter) == TRUE) {
		gtk_widget_set_sensitive (down, TRUE);
		gtk_widget_set_sensitive (bottom, TRUE);
	}
	else {
		gtk_widget_set_sensitive (down, FALSE);
		gtk_widget_set_sensitive (bottom, FALSE);
	}
}


/*
 * Append an environment parameter (key=value) to the GtkListStore model
 *
 * Return:
 *   Always 0
 */
static int
env_add_variable_row (void *ptr, int pos, const char *key, const char *value)
{
	GtkListStore *store = ptr;
	GtkTreeIter iter;

	gtk_list_store_append (store, &iter);
	gtk_list_store_set (store, &iter, 0, key, 1, value, -1);
	return 0;
}


/*
 * Check the syntax of the parameter value
 *
 * Return:
 *      0 --> Syntax OK
 *  other --> Syntax error
 */
static int
check_value_syntax (const gchar *value)
{
	char *out;
	size_t out_len;
	int ret;

	out = NULL;
	out_len = 0;
	ret = parse_string (	value, (size_t)schedwi_strlen (value), NULL,
				&out, &out_len);
	if (out != NULL) {
		free (out);
	}
	return ret;
}


/*
 * Function called when a key cell is rendered
 * A syntax check is done.  If it fails (syntax error), the cell is rendered
 * in red
 */
static void
column_key_func (	GtkTreeViewColumn *tree_column,
			GtkCellRenderer *cell,
			GtkTreeModel *tree_model,
			GtkTreeIter *iter,
			gpointer data)
{
	gchar *key;

	/* Retrieve the key from the model */
	gtk_tree_model_get (tree_model, iter, 0, &key, -1);
	if (key != NULL) {
		g_object_set (cell, "text", key, NULL);

		/* Check the syntax */
		if (check_key_syntax (key) != 0) {
			g_object_set (cell, "background-set", TRUE, NULL);
		}
		else {
			g_object_set (cell, "background-set", FALSE, NULL);
		}
		g_free (key);
	}
	else {
		g_object_set (cell, "text", "",  NULL);
	}
}


/*
 * Function called when a value cell is rendered
 * A syntax check is done.  If it fails (syntax error), the cell is rendered
 * in red
 */
static void
column_value_func (	GtkTreeViewColumn *tree_column,
			GtkCellRenderer *cell,
			GtkTreeModel *tree_model,
			GtkTreeIter *iter,
			gpointer data)
{
	gchar *value;

	/* Retrieve the value from the model */
	gtk_tree_model_get (tree_model, iter, 1, &value, -1);
	if (value != NULL) {
		g_object_set (cell, "text", value, NULL);

		/* Check the syntax */
		if (check_value_syntax (value) != 0) {
			g_object_set (cell, "background-set", TRUE, NULL);
		}
		else {
			g_object_set (cell, "background-set", FALSE, NULL);
		}
		g_free (value);
	}
	else {
		g_object_set (cell, "text", "",  NULL);
	}
}


/*
 * Initialize the environment parameter list
 *
 * Return:
 *       0 --> No error
 *   other --> SQL error (an error popup has been displayed for the user)
 */
static int
env_variables_init (GtkWidget *view, const gchar *id, int workload_date)
{
	GtkListStore *store;
	GtkTreeViewColumn *col;
	GtkCellRenderer *renderer;
	GtkTreeSelection *select;
	int ret;

	/* Create and fill the GtkListStore object */
	store = gtk_list_store_new (	2,
					G_TYPE_STRING,	/* Key */	
					G_TYPE_STRING);	/* Value */

	if (id != NULL && id[0] != '\0') {
		ret = sql_env_get_variables (workload_date, id,
				env_add_variable_row, store,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error"));
		if (ret != 0) {
			g_object_unref (store);
			return ret;
		}
	}

	/* Associate the model with the view */
	gtk_tree_view_set_model (GTK_TREE_VIEW (view), GTK_TREE_MODEL (store));

	/* Add the column and cell for the key */
	col = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_title (col, _("Name"));
	gtk_tree_view_append_column (GTK_TREE_VIEW (view), col);

	renderer = gtk_cell_renderer_text_new ();
	g_object_set (renderer, "editable", TRUE,
				"background", "red",
				"background-set", FALSE,
				NULL);

	gtk_tree_view_column_pack_start (col, renderer, FALSE);
	gtk_tree_view_column_set_cell_data_func (col, renderer,
				column_key_func, NULL, NULL);

	
	/* Add the column and cell for the value */
	col = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_title (col, _("Value"));
	gtk_tree_view_append_column (GTK_TREE_VIEW (view), col);

	renderer = gtk_cell_renderer_text_new ();
	g_object_set (renderer, "editable", TRUE,
				"background", "red",
				"background-set", FALSE,
				NULL);

	gtk_tree_view_column_pack_start (col, renderer, FALSE);
	gtk_tree_view_column_set_cell_data_func (col, renderer,
				column_value_func, NULL, NULL);

	g_object_unref (store);

	/* Selection */
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
	gtk_tree_selection_set_mode (select, GTK_SELECTION_NONE);
	return 0;
}


/*
 * Initialize the window
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error (an error popup has been displayed for the user)
 */
static int
dialog_env_init (	GtkTreeView *view, GtkWidget *dialog_edit,
			const gchar *env_id, int workload_date)
{
	GtkWidget *env_view;

	gtk_window_set_type_hint (	GTK_WINDOW (dialog_edit),
					GDK_WINDOW_TYPE_HINT_NORMAL);

	/*
	 * Store the tree view of the environment list to be able to
	 * refresh it when an environment is updated
	 */
	g_object_set_data (	G_OBJECT (dialog_edit),
				"env_view", (gpointer)view);

	/* Initialize the variable tree view */
	env_view = lookup_widget (dialog_edit, "treeview_env_variables");
	if (env_variables_init (env_view, env_id, workload_date) != 0) {
		return -1;
	}

	if (env_id == NULL || env_id[0] == '\0') {
		return 0;
	}

	/* Retrieve the main parameters of the environment */
	if (sql_env_get_main (	workload_date,
				env_id,
				set_id,
				set_name,
				set_description,
				dialog_edit,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error")) != 0)
	{
		return -1;
	}

	return 0;
}


/*
 * Create a new environment dialog for the provided environment id
 *
 * Return:
 *   The GtkWidget of the new window (which has been `show'ed by this function)
 *   or NULL in case of error
 */
GtkWidget *
new_dialog_env (GtkTreeView *view, const gchar *env_id, int workload_date)
{
	GtkWidget *dialog_edit;

	dialog_edit = create_dialog_env ();
	if (dialog_env_init (view, dialog_edit, env_id, workload_date) != 0) {
		gtk_widget_destroy (dialog_edit);
		return NULL;
	}
	else {
		gtk_widget_show (dialog_edit);
		return dialog_edit;
	}
}


/*
 * Check that the provided values for the main parameters are alright and
 * can be stored in the database
 *
 * Return:
 *   0 --> Parameters OK
 *  -1 --> Error (an error popup has been displayed)
 */
static int
check_env_values (const gchar *name)
{
	if (name == NULL || name[0] == '\0') {
		error_window (	_("A required information is missing"),
				_("The Environment Name field is required."));
		return -1;
	}
	return 0;
}


/*
 * Retrieve the main environment parameters from the widgets
 *
 * Return:
 *   0 --> No error (*name and *description are set - only *description must
 *         be freed by the caller)
 *  -1 --> Error (a required parameter is missing - an error popup has been
 *         displayed for the user)
 */
static int
get_env_values (	GtkWidget *dialog_edit,
			const gchar **name,
			gchar **description)
{
	GtkWidget *entry, *text;
	GtkTextBuffer *buffer;
	GtkTextIter start, end;

	entry = lookup_widget (dialog_edit, "entry_env_name");
	*name = gtk_entry_get_text (GTK_ENTRY (entry));

	text = lookup_widget (dialog_edit, "textview_env_description");
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (text));
	gtk_text_buffer_get_start_iter (buffer, &start);
	gtk_text_buffer_get_end_iter (buffer, &end);
	*description = gtk_text_buffer_get_text (buffer, &start, &end, FALSE);

	return check_env_values (*name);
}


/*
 * Free an environment variable (an array of two items - the key and the value)
 */
void
free_variable_array (gchar **var)
{
	if (var != NULL) {
		g_free (var[0]);
		g_free (var[1]);
		g_free (var);
	}
}


/*
 * Append the provided environment variable to the end of the list of variables
 * (arrays of two items - the key and the value)
 *
 * Return:
 *   FALSE --> No error
 *    TRUE --> Memory allocation error (this return value stops the
 *             gtk_tree_model_foreach() calling function)
 */
static gboolean
add_variable_to_list (	GtkTreeModel *model,
			GtkTreePath *path,
			GtkTreeIter *iter,
			gpointer data)
{
	lwc_LL *current_lst = data;
	gchar **var;

	var = (gchar **) g_malloc (sizeof (gchar *) * 2);
	gtk_tree_model_get (model, iter, 0, &(var[0]), 1, &(var[1]), -1);
	if (lwc_addEndLL (current_lst, var) != 0) {
		free_variable_array (var);
		lwc_emptyLL (	current_lst,
				(void (*)(const void *))free_variable_array);
		error_window (_("Memory allocation error"), NULL);
		return TRUE;
	}
	return FALSE;
}


/*
 * Store the environment variables in the database
 *
 * Return:
 *       0 --> No error
 *      -1 --> Memory allocation error (an error popup has been displayed
 *             for the user)
 *   other --> SQL error (an error popup has been displayed for the user)
 */
static int
env_variables_save (GtkWidget *view, const gchar *id)
{
	lwc_LL *list;
	GtkTreeModel *model;
	int ret;

	list = lwc_newLL ();
	if (list == NULL) {
		error_window (_("Memory allocation error"), NULL);
		return -1;
	}

	model = gtk_tree_view_get_model (GTK_TREE_VIEW (view));

	/* Built the list of the environment variables */
	gtk_tree_model_foreach (model, add_variable_to_list, list);

	/* Update the database */
	ret = sql_env_set_variables (id, list,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error"));
	lwc_delLL (list, (void (*)(const void *)) free_variable_array);
	return ret;
}


/*
 * Store a new environment in the database
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error (an error popup has been displayed for the user)
 */
static int
env_create (GtkWidget *dialog_edit)
{
	GtkWidget *env_view;
	const gchar *name;
	gchar *description = NULL;
	unsigned long int env_id;
	GString *env_id_str;

	/*
	 * Add the new environment main parameters to the database
	 */
	if (	   get_env_values (dialog_edit, &name, &description) != 0
		|| sql_env_new (name, description, &env_id,
				(void (*)(void *, const char*, unsigned int))
						error_window_check_duplicate,
				_("Database error")) != 0)
	{
		g_free (description);
		return -1;
	}
	g_free (description);


	/*
	 * Add the environment variables to the database
	 */
	env_id_str = g_string_new ("");
	g_string_printf (env_id_str, "%lu", env_id);
	env_view = lookup_widget (dialog_edit, "treeview_env_variables");
	if (env_variables_save (env_view, env_id_str->str) != 0) {
		g_string_free (env_id_str, TRUE);
		return -1;
	}
	g_string_free (env_id_str, TRUE);
	return 0;
}


/*
 * Update an environment in the database
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error (an error popup has been displayed for the user)
 */
static int
env_update (GtkWidget *dialog_edit, GString *env_id)
{
	GtkWidget *env_view;
	const gchar *name;
	gchar *description = NULL;

	/*
	 * Update the enviroment main parameters
	 */
	if (	   get_env_values (dialog_edit, &name, &description) != 0
		|| sql_env_update (env_id->str, name, description,
				(void (*)(void *, const char*, unsigned int))
						error_window_check_duplicate,
					_("Database error")) != 0)
	{
		g_free (description);
		return -1;
	}
	g_free (description);


	/*
	 * Update the environment variables
	 */

	/* First, delete the old environment list from the database */
	if (sql_env_del_variables (env_id->str,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error")) != 0)
	{
		return -1;
	}

	/* Now, add the environment variables list */
	env_view = lookup_widget (dialog_edit, "treeview_env_variables");
	if (env_variables_save (env_view, env_id->str) != 0) {
		return -1;
	}
	return 0;
}


/*
 * Callback for the `OK' button
 */
void
env_ok_clicked (GtkButton *button)
{
	GtkWidget *dialog_edit;
	GtkTreeView *view;
	GString *env_id;
	int ret;

	cursor_busy (GTK_WIDGET (button));
	dialog_edit = lookup_widget (GTK_WIDGET (button), "dialog_env");
	env_id = g_object_get_data (G_OBJECT (dialog_edit), "env_id");
	view = g_object_get_data (G_OBJECT (dialog_edit), "env_view");
	if (env_id == NULL) {
		ret = env_create (dialog_edit);
	}
	else {
		ret = env_update (dialog_edit, env_id);
	}
	cursor_normal (GTK_WIDGET (button));
	if (ret == 0) {
		env_list_refresh (view);
		gtk_widget_destroy (dialog_edit);
	}
}


/*
 * Callback for the `Move to top' button
 */
void
env_var_top_clicked (GtkButton *button)
{
	GtkWidget *view;
	GtkTreeSelection *select;
	GtkTreeModel *model;
	GtkTreeIter iter;

	view = lookup_widget (GTK_WIDGET (button), "treeview_env_variables");
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
	if (gtk_tree_selection_get_selected (select, &model, &iter) == TRUE)
	{
		gtk_list_store_move_after (	GTK_LIST_STORE (model),
						&iter, NULL);
		tree_selection_changed_cb (select, (gpointer)button);
	}

}


/*
 * Callback for the `Move to bottom' button
 */
void
env_var_bottom_clicked (GtkButton *button)
{
	GtkWidget *view;
	GtkTreeSelection *select;
	GtkTreeModel *model;
	GtkTreeIter iter;

	view = lookup_widget (GTK_WIDGET (button), "treeview_env_variables");
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
	if (gtk_tree_selection_get_selected (select, &model, &iter) == TRUE)
	{
		gtk_list_store_move_before (	GTK_LIST_STORE (model),
						&iter, NULL);
		tree_selection_changed_cb (select, (gpointer)button);
	}
}


/*
 * Callback for the `Move up' button
 */
void
env_var_up_clicked (GtkButton *button)
{
	GtkWidget *view;
	GtkTreeSelection *select;
	GtkTreeModel *model;
	GtkTreeIter iter, prev;
	GtkTreePath *path;

	view = lookup_widget (GTK_WIDGET (button), "treeview_env_variables");
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
	if (gtk_tree_selection_get_selected (select, &model, &iter) == TRUE)
	{
		path = gtk_tree_model_get_path (model, &iter);
		gtk_tree_path_prev (path);
		gtk_tree_model_get_iter (model, &prev, path);
		gtk_list_store_swap (GTK_LIST_STORE (model), &iter, &prev);
		gtk_tree_path_free (path);
		tree_selection_changed_cb (select, (gpointer)button);
	}
}


/*
 * Callback for the `Move down' button
 */
void
env_var_down_clicked (GtkButton *button)
{
	GtkWidget *view;
	GtkTreeSelection *select;
	GtkTreeModel *model;
	GtkTreeIter iter, next;

	view = lookup_widget (GTK_WIDGET (button), "treeview_env_variables");
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
	if (gtk_tree_selection_get_selected (select, &model, &iter) == TRUE)
	{
		next = iter;
		gtk_tree_model_iter_next (model, &next);
		gtk_list_store_swap (GTK_LIST_STORE (model), &iter, &next);
		tree_selection_changed_cb (select, (gpointer)button);
	}
}


/*
 * Callback for the `Remove' button
 */
void
env_var_remove_clicked (GtkButton *button)
{
	GtkWidget *view;
	GtkTreeSelection *select;
	GtkTreeModel *model;
	GtkTreeIter iter, next;

	view = lookup_widget (GTK_WIDGET (button), "treeview_env_variables");
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
	if (gtk_tree_selection_get_selected (select, &model, &iter) == TRUE)
	{
		/* Move the selection to the next environment variable */
		next = iter;
		if (gtk_tree_model_iter_next (model, &next) == TRUE) {
			gtk_tree_selection_select_iter (select, &next);
		}
		/* Remove the environment variable */
		gtk_list_store_remove (GTK_LIST_STORE (model), &iter);
		tree_selection_changed_cb (select, (gpointer)button);
	}
}


/*
 * Callback for the `Add' button
 */
void
env_var_add_clicked (GtkButton *button)
{
	GtkWidget *view;
	GtkTreeSelection *select;
	GtkTreeModel *model;
	GtkTreeIter iter, new_iter;
	GtkTreePath *path;
	GtkTreeViewColumn *column;

	view = lookup_widget (GTK_WIDGET (button), "treeview_env_variables");
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));

	/*
	 * If an environment variable is selected, add the new one just
	 * after it
	 */
	if (gtk_tree_selection_get_selected (select, &model, &iter) == TRUE) {
		gtk_list_store_insert_after (	GTK_LIST_STORE (model),
						&new_iter,
						&iter);
	}
	else {
		/* Otherwise, add it to the end */
		model = gtk_tree_view_get_model (GTK_TREE_VIEW (view));
		gtk_list_store_append (GTK_LIST_STORE (model), &new_iter);
	}

	/* Edit the key parameter of the new environment variable */
	path = gtk_tree_model_get_path (model, &new_iter);
	column = gtk_tree_view_get_column (GTK_TREE_VIEW (view), 0);
	gtk_tree_view_set_cursor (	GTK_TREE_VIEW (view),
					path,
					column,
					TRUE);
	gtk_tree_path_free (path);
}

/*------------------------======= End Of File =======------------------------*/
