/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* calendar_folder_cb.c -- GUI functions for the folder window */

#include <schedwi.h>

#include "support.h"
#include "interface.h"

#include <message_windows.h>
#include <cursor.h>
#include <sql_calendar.h>
#include <calendar_list_cb.h>
#include <calendar_folder_cb.h>


/*
 * Functions used to set values in the dialog widgets
 */
static void
set_id (const char *value, void *data)
{
	/* Store the calendar folder id in the widget */
	g_object_set_data_full (G_OBJECT (data), "folder_id",
				g_strdup (value),
				g_free);
}

static void
set_name (const char *value, void *data)
{
	GtkWidget *label;
	gchar *s;

	label = lookup_widget (GTK_WIDGET (data), "label_cal_folder_name");
	s = g_strdup_printf ("<b>%s</b>", value);
	gtk_label_set_markup (GTK_LABEL (label), s);
	g_free (s);
	gtk_window_set_title (GTK_WINDOW (data), value);
}

static void
set_description (const char *value, void *data)
{
	GtkWidget *text;
	GtkTextBuffer *buffer;

	text = lookup_widget (	GTK_WIDGET (data),
				"textview_cal_folder_description");
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (text));
	gtk_text_buffer_set_text (buffer, value, -1);
}


/*
 * Initialize the window
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error (an error popup has been displayed for the user)
 */
static int
dialog_cal_folder_init (	GtkTreeView *view, GtkWidget *dialog,
				const gchar *id, const gchar *parent_id,
				int workload_date)
{
	gtk_window_set_type_hint (	GTK_WINDOW (dialog),
					GDK_WINDOW_TYPE_HINT_NORMAL);

	/*
	 * Store the tree view of the calendar list to be able to
	 * refresh it when the folder is updated
	 */
	g_object_set_data (	G_OBJECT (dialog),
				"cal_view", (gpointer)view);

	/*
	 * New folder
	 */
	if (id == NULL || id[0] == '\0') {
		g_object_set_data_full (G_OBJECT (dialog), "parent_id",
					g_strdup (parent_id),
					g_free);
		return 0;
	}

	/*
	 * Edit folder
	 */

	/* Retrieve the folder main parameters */
	if (sql_cal_get_details (workload_date,
				id,
				set_id,
				NULL,	/* Parent ID */
				set_name,
				NULL,	/* Type */
				set_description,
				NULL,
				dialog,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error")) != 0)
	{
		return -1;
	}

	return 0;
}


/*
 * Create a new folder dialog for the provided folder (if not NULL)
 *
 * Return:
 *   The GtkWidget of the new window (which has been `show'ed by this function)
 *   or NULL in case of error
 */
GtkWidget *
new_dialog_cal_folder (	GtkTreeView *view,
			const gchar *id, const gchar *parent_id,
			int workload_date)
{
	GtkWidget *widget;

	widget = create_dialog_cal_folder ();
	if (dialog_cal_folder_init (	view, widget, id, parent_id,
					workload_date) != 0) {
		gtk_widget_destroy (widget);
		return NULL;
	}
	else {
		gtk_widget_show (widget);
		return widget;
	}
}


/*
 * Retrieve the folder parameters from the widgets
 *
 * Return:
 *   0 --> No error (*name and *description are set - only *description must
 *         be freed by the caller)
 *  -1 --> Error (the folder name is missing - an error popup has been
 *         displayed for the user)
 */
static int
get_folder_values (	GtkWidget *dialog,
			const gchar **name,
			gchar **description)
{
	GtkWidget *entry, *text;
	GtkTextBuffer *buffer;
	GtkTextIter start, end;

	/* Retrieve the folder name */
	entry = lookup_widget (dialog, "entry_cal_folder_name");
	*name = gtk_entry_get_text (GTK_ENTRY (entry));
	if (*name == NULL || (*name)[0] == '\0') {
		error_window (_("A folder name is required"), NULL);
		gtk_widget_grab_focus (entry);
		return -1;
	}

	/* Retrieve the description */
	text = lookup_widget (dialog, "textview_cal_folder_description");
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (text));
	gtk_text_buffer_get_start_iter (buffer, &start);
	gtk_text_buffer_get_end_iter (buffer, &end);
	*description = gtk_text_buffer_get_text (buffer, &start, &end, FALSE);

	return 0;
}


/*
 * Store a new folder in the database
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error (an error popup has been displayed for the user)
 */
static int
folder_create (GtkWidget *dialog, const gchar *parent_id)
{
	const gchar *name;
	gchar *description = NULL;

	/*
	 * Add the new folder parameters to the database
	 */
	if (	   get_folder_values (dialog, &name, &description) != 0
		|| sql_cal_new (name, parent_id, "1", description, NULL,
				NULL,
				(void (*)(void *, const char*, unsigned int))
						error_window_check_duplicate,
				_("Database error")) != 0)
	{
		g_free (description);
		return -1;
	}
	g_free (description);
	return 0;
}


/*
 * Update a folder in the database
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error (an error popup has been displayed for the user)
 */
static int
folder_update (GtkWidget *dialog, const gchar *folder_id)
{
	const gchar *name;
	gchar *description = NULL;

	/*
	 * Update the folder main parameters
	 */
	if (	   get_folder_values (dialog, &name, &description) != 0
		|| sql_cal_update (folder_id, name, description, NULL,
				(void (*)(void *, const char*, unsigned int))
						error_window_check_duplicate,
				_("Database error")) != 0)
	{
		g_free (description);
		return -1;
	}
	g_free (description);
	return 0;
}


/*
 * Callback for the `OK' button
 */
void
cal_folder_ok_clicked (GtkButton *button)
{
	GtkWidget *dialog;
	GtkTreeView *view;
	gchar *folder_id, *parent_id;
	int ret;

	cursor_busy (GTK_WIDGET (button));
	dialog = lookup_widget (GTK_WIDGET (button), "dialog_cal_folder");
	folder_id = (gchar *) g_object_get_data (	G_OBJECT (dialog),
							"folder_id");
	parent_id = (gchar *) g_object_get_data (	G_OBJECT (dialog),
							"parent_id");
	view = (GtkTreeView *) g_object_get_data (	G_OBJECT (dialog),
							"cal_view");
	if (folder_id == NULL) {
		ret = folder_create (dialog, parent_id);
	}
	else {
		ret = folder_update (dialog, folder_id);
	}
	cursor_normal (GTK_WIDGET (button));
	if (ret == 0) {
		calendar_list_refresh (view);
		gtk_widget_destroy (dialog);
	}
}

/*------------------------======= End Of File =======------------------------*/
