/* Schedwi
   Copyright (C) 2007 Herve Quatremain
     
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
     
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.
     
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* background.c -- Functions to draw jobset backgrounds */

#include <schedwi.h>

#include "support.h"
#include "interface.h"

#include <sql_common.h>
#include <sql_hierarchy.h>
#include <message_windows.h>
#include <cache_pixbuf.h>
#include <job_cb.h>
#include <main_cb.h>
#include <copy_paste.h>
#include <background.h>


/*
 * Convert a string to the corresponding background_t value
 */
static background_t
str_to_background_t (const gchar *str)
{
	if (str == NULL) {
		return SOLID_COLOR;
	}
	switch (str[0]) {
		case '1':	return VERTICAL_GRADIENT;
		case '2':	return HORIZONTAL_GRADIENT;
		case '3':	return WALLPAPER_UPPERLEFT;
		case '4':	return WALLPAPER_CENTERED;
		case '5':	return WALLPAPER_TILED;
		case '6':	return WALLPAPER_SCALED;
		case '7':	return WALLPAPER_STRETCHED;
		default:	return SOLID_COLOR;
	}
}


/*
 * Destroy the provided jobset_background_t object
 */
void
destroy_jobset_background (jobset_background_t *ptr)
{
	if (ptr != NULL) {
		if (ptr->canvas_bg != NULL) {
			gtk_object_destroy (GTK_OBJECT (ptr->canvas_bg));
		}
		g_free (ptr->jobset_id);
		g_free (ptr);
	}
}


/*
 * Fill the provided jobset_background_t structure with the background
 * details retrieved from the database
 *
 * Return:
 *   TRUE --> No error
 *  FALSE --> Error.  An error message has been displayed
 */
jobset_background_t *
new_jobset_background (	int workload_date,
			lwc_LL *hierarchy_list, const gchar *jobset_id,
			void (*refresh_func)(void *),
			void *refresh_func_user_data)
{
	jobset_background_t *ptr;
	char *err_msg = NULL;
	char **columns;
	unsigned long int *columns_len;
	gchar *s;

	/* Retrieve the background/wallpaper parameters */
	if (get_job_parameters (workload_date, hierarchy_list,
				"jobset_background_s",	/* Database table */
	"background_mode,image,gradient_color_primary,gradient_color_secondary",
				&columns, &columns_len, &err_msg) != 0)
	{
		error_window (_("Database error"), err_msg);
		if (err_msg != NULL) {
			free (err_msg);
		}
		return NULL;
	}

	ptr = g_new0 (jobset_background_t, 1);

	/*
	 * The background parameters are not found in the database.  This
	 * is an error as the database is loaded during the installation with
	 * default parameters for the "Root" jobset.  A warning message is
	 * then displayed and sensible default values are used
	 */
	if (columns == NULL || columns_len == NULL) {
		warning_window (
		_("Could not load the parameters of the jobset background"),
		_("The database may be corrupted.\nReplacement parameters will be used instead."));

		/* Solid white background */
		ptr->background_mode = SOLID_COLOR;
		gdk_color_parse ("#FFFFFF",
				&(ptr->background_gradient_color_primary));
		ptr->background_gradient_color_secondary =
					ptr->background_gradient_color_primary;
		ptr->background_image = NULL;
	}
	else {
	
		ptr->background_mode = str_to_background_t (columns[0]);
	
		/* Background colors */
		if (gdk_color_parse (columns[2],
				&(ptr->background_gradient_color_primary))
			!= TRUE)
		{
			s = g_strdup_printf (
		_("Invalid color \"%s\" for the jobset background"),
				columns[2]);
			warning_window (s,
			_("The color stored in the database may be corrupted.\nA replacement color will be used instead."));
			g_free (s);
			gdk_color_parse ("#FFFFFF",
				&(ptr->background_gradient_color_primary));
		}
		if (gdk_color_parse (columns[3],
				&(ptr->background_gradient_color_secondary))
			 != TRUE)
		{
			s = g_strdup_printf (
		_("Invalid color \"%s\" for the jobset gradient background"),
				columns[3]);
			warning_window (s,
			_("The color stored in the database may be corrupted.\nA replacement color will be used instead."));
			g_free (s);
			gdk_color_parse ("#808080",
				&(ptr->background_gradient_color_secondary));
		}

		/* Background image */
		if (columns[1] != NULL && columns_len[1] != 0) {
			if (add_pixbuf_to_cache (columns[1], columns_len[1],
						&(ptr->background_image),
						NULL, NULL, NULL, NULL,
						&err_msg) != TRUE)
			{
				s = g_strdup_printf (
				_("%s.\nNo image will be displayed."),
						err_msg);
				g_free (err_msg);
				warning_window (
				_("Could not load the background image"), s);
				g_free (s);
				ptr->background_image = NULL;
				if (	   ptr->background_mode ==
							WALLPAPER_UPPERLEFT
					|| ptr->background_mode ==
							WALLPAPER_CENTERED
					|| ptr->background_mode ==
							WALLPAPER_TILED
					|| ptr->background_mode ==
							WALLPAPER_SCALED 
					|| ptr->background_mode ==
							WALLPAPER_STRETCHED)
				{
					 ptr->background_mode = SOLID_COLOR;
				}
			}
		}
		else {
			ptr->background_image = NULL;
			if (	   ptr->background_mode == WALLPAPER_UPPERLEFT
				|| ptr->background_mode == WALLPAPER_CENTERED
				|| ptr->background_mode == WALLPAPER_TILED
				|| ptr->background_mode == WALLPAPER_SCALED 
				|| ptr->background_mode == WALLPAPER_STRETCHED)
			{
				 ptr->background_mode = SOLID_COLOR;
			}
		}

		sql_free_row (columns);
		free (columns_len);
	}

	ptr->jobset_id = g_strdup (jobset_id);
	ptr->refresh_func = refresh_func;
	ptr->refresh_func_user_data = refresh_func_user_data;
	return ptr;
}


/*
 * Draw the background on the provided canvas
 */
void
jobset_background_draw (	jobset_background_t *ptr,
				GnomeCanvas *canvas,
				gint width, gint height)
{
	GdkPixbuf *scaled;
	int pixbuf_w, pixbuf_h, x, y;
	GnomeCanvasPoints *points;
	gdouble ratio_r, ratio_g, ratio_b, ratio;
	GdkColor c, primary, secondary;

	if (ptr->canvas_bg != NULL) {
		gtk_object_destroy (GTK_OBJECT (ptr->canvas_bg));
	}

	switch (ptr->background_mode) {
		case SOLID_COLOR:
			ptr->canvas_bg = gnome_canvas_item_new (
					gnome_canvas_root (canvas),
					gnome_canvas_rect_get_type (),
					"x1", (gdouble)0,
					"y1", (gdouble)0,
					"x2", (gdouble)width,
					"y2", (gdouble)height,
					"fill-color-gdk",
				&(ptr->background_gradient_color_primary), 
					NULL);
			gnome_canvas_item_lower_to_bottom (ptr->canvas_bg);
			break;

		case WALLPAPER_UPPERLEFT:
			pixbuf_w = gdk_pixbuf_get_width (
							ptr->background_image);
			pixbuf_h = gdk_pixbuf_get_height (
							ptr->background_image);
			scaled = NULL;
			if (pixbuf_w > width || pixbuf_h > height) {
				scaled = gdk_pixbuf_new (
						gdk_pixbuf_get_colorspace (
							ptr->background_image),
						gdk_pixbuf_get_has_alpha (
							ptr->background_image),
						gdk_pixbuf_get_bits_per_sample (
							ptr->background_image),
						MIN (width, pixbuf_w),
						MIN (height, pixbuf_h));
				if (scaled != NULL) {
					gdk_pixbuf_copy_area (
						ptr->background_image,
						0, 0,
						MIN (width, pixbuf_w),
						MIN (height, pixbuf_h),
						scaled,
						0, 0);
				}
			}
			ptr->canvas_bg = gnome_canvas_item_new (
					gnome_canvas_root (canvas),
					gnome_canvas_group_get_type (),
					"x", (gdouble)0,
					"y", (gdouble)0,
					NULL);
			gnome_canvas_item_new (
					GNOME_CANVAS_GROUP (ptr->canvas_bg),
					gnome_canvas_rect_get_type (),
					"x1", (gdouble)0,
					"y1", (gdouble)0,
					"x2", (gdouble)width,
					"y2", (gdouble)height,
					"fill-color-gdk",
				&(ptr->background_gradient_color_primary), 
					NULL);
			gnome_canvas_item_new (
					GNOME_CANVAS_GROUP (ptr->canvas_bg),
					gnome_canvas_pixbuf_get_type (),
					"x", (gdouble)0,
					"y", (gdouble)0,
					"pixbuf", (scaled != NULL)
							? scaled
							: ptr->background_image,
					NULL);
			gnome_canvas_item_lower_to_bottom (ptr->canvas_bg);
			if (scaled != NULL) {
				g_object_unref (scaled);
			}
			break;

		case WALLPAPER_CENTERED:
			pixbuf_w = gdk_pixbuf_get_width (
							ptr->background_image);
			pixbuf_h = gdk_pixbuf_get_height (
							ptr->background_image);
			scaled = NULL;
			if (pixbuf_w > width || pixbuf_h > height) {
				scaled = gdk_pixbuf_new (
						gdk_pixbuf_get_colorspace (
							ptr->background_image),
						gdk_pixbuf_get_has_alpha (
							ptr->background_image),
						gdk_pixbuf_get_bits_per_sample (
							ptr->background_image),
						MIN (width, pixbuf_w),
						MIN (height, pixbuf_h));
				if (scaled != NULL) {
					gdk_pixbuf_copy_area (
						ptr->background_image,
						0, 0,
						MIN (width, pixbuf_w),
						MIN (height, pixbuf_h),
						scaled,
						0, 0);
				}
			}
			ptr->canvas_bg = gnome_canvas_item_new (
					gnome_canvas_root (canvas),
					gnome_canvas_group_get_type (),
					"x", (gdouble)0,
					"y", (gdouble)0,
					NULL);
			gnome_canvas_item_new (
					GNOME_CANVAS_GROUP (ptr->canvas_bg),
					gnome_canvas_rect_get_type (),
					"x1", (gdouble)0,
					"y1", (gdouble)0,
					"x2", (gdouble)width,
					"y2", (gdouble)height,
					"fill-color-gdk",
				&(ptr->background_gradient_color_primary), 
					NULL);
			gnome_canvas_item_new (
					GNOME_CANVAS_GROUP (ptr->canvas_bg),
					gnome_canvas_pixbuf_get_type (),
					"anchor", GTK_ANCHOR_CENTER,
					"x", (gdouble)width/2,
					"y", (gdouble)height/2,
					"pixbuf", (scaled != NULL)
							? scaled
							: ptr->background_image,
					NULL);
			gnome_canvas_item_lower_to_bottom (ptr->canvas_bg);
			if (scaled != NULL) {
				g_object_unref (scaled);
			}
			break;

		case WALLPAPER_STRETCHED:
			scaled =  gdk_pixbuf_scale_simple (
						ptr->background_image,
						width,
						height,
						GDK_INTERP_HYPER);

			ptr->canvas_bg = gnome_canvas_item_new (
					gnome_canvas_root (canvas),
					gnome_canvas_group_get_type (),
					"x", (gdouble)0,
					"y", (gdouble)0,
					NULL);
			gnome_canvas_item_new (
					GNOME_CANVAS_GROUP (ptr->canvas_bg),
					gnome_canvas_pixbuf_get_type (),
					"x", (gdouble)0,
					"y", (gdouble)0,
					"pixbuf", 	(scaled != NULL)?
							scaled:
							ptr->background_image,
					NULL);
			if (scaled != NULL) {
				g_object_unref (scaled);
			}
			gnome_canvas_item_lower_to_bottom (ptr->canvas_bg);
			break;

		case WALLPAPER_SCALED:
			pixbuf_w = gdk_pixbuf_get_width (
							ptr->background_image);
			pixbuf_h = gdk_pixbuf_get_height (
							ptr->background_image);
			ratio = (gdouble)pixbuf_w / (gdouble)pixbuf_h;
			if (width / ratio > height) {
				pixbuf_h = height;
				pixbuf_w = ratio * height;
			}
			else {
				pixbuf_h = width / ratio;
				pixbuf_w = width;
			}
			scaled =  gdk_pixbuf_scale_simple (
						ptr->background_image,
						pixbuf_w,
						pixbuf_h,
						GDK_INTERP_HYPER);

			ptr->canvas_bg = gnome_canvas_item_new (
					gnome_canvas_root (canvas),
					gnome_canvas_group_get_type (),
					"x", (gdouble)0,
					"y", (gdouble)0,
					NULL);
			gnome_canvas_item_new (
					GNOME_CANVAS_GROUP (ptr->canvas_bg),
					gnome_canvas_rect_get_type (),
					"x1", (gdouble)0,
					"y1", (gdouble)0,
					"x2", (gdouble)width,
					"y2", (gdouble)height,
					"fill-color-gdk",
				&(ptr->background_gradient_color_primary), 
					NULL);
			gnome_canvas_item_new (
					GNOME_CANVAS_GROUP (ptr->canvas_bg),
					gnome_canvas_pixbuf_get_type (),
					"x", (gdouble)0,
					"y", (gdouble)0,
					"pixbuf", 	(scaled != NULL)?
							scaled:
							ptr->background_image,
					NULL);
			if (scaled != NULL) {
				g_object_unref (scaled);
			}
			gnome_canvas_item_lower_to_bottom (ptr->canvas_bg);
			break;

		case WALLPAPER_TILED:
			pixbuf_w = gdk_pixbuf_get_width (
							ptr->background_image);
			pixbuf_h = gdk_pixbuf_get_height (
							ptr->background_image);
			scaled = NULL;
			if (pixbuf_w > width || pixbuf_h > height) {
				scaled = gdk_pixbuf_new (
						gdk_pixbuf_get_colorspace (
							ptr->background_image),
						gdk_pixbuf_get_has_alpha (
							ptr->background_image),
						gdk_pixbuf_get_bits_per_sample (
							ptr->background_image),
						MIN (width, pixbuf_w),
						MIN (height, pixbuf_h));
				if (scaled != NULL) {
					gdk_pixbuf_copy_area (
						ptr->background_image,
						0, 0,
						MIN (width, pixbuf_w),
						MIN (height, pixbuf_h),
						scaled,
						0, 0);
				}
			}
			pixbuf_w = MIN (width, pixbuf_w);
			pixbuf_h = MIN (height, pixbuf_h);
			ptr->canvas_bg = gnome_canvas_item_new (
					gnome_canvas_root (canvas),
					gnome_canvas_group_get_type (),
					"x", (gdouble)0,
					"y", (gdouble)0,
					NULL);

			for (y = 0; y < height; y += pixbuf_h) {
				for (x = 0; x < width; x += pixbuf_w) {
					gnome_canvas_item_new (
					GNOME_CANVAS_GROUP (ptr->canvas_bg),
					gnome_canvas_pixbuf_get_type (),
					"x", (gdouble)x,
					"y", (gdouble)y,
					"pixbuf", (scaled != NULL)
							? scaled
							: ptr->background_image,
					NULL);
				}
			}
			gnome_canvas_item_lower_to_bottom (ptr->canvas_bg);
			if (scaled != NULL) {
				g_object_unref (scaled);
			}
			break;

		case HORIZONTAL_GRADIENT:

			c = primary = ptr->background_gradient_color_primary; 
			secondary = ptr->background_gradient_color_secondary;

			ratio_r = (gdouble)width /
					(secondary.red - primary.red);
			ratio_g = (gdouble)width /
					(secondary.green - primary.green);
			ratio_b = (gdouble)width /
					(secondary.blue - primary.blue);

			points = gnome_canvas_points_new (2);
			points->coords[1] = (gdouble)0;
			points->coords[3] = (gdouble)height;

			ptr->canvas_bg = gnome_canvas_item_new (
					gnome_canvas_root (canvas),
					gnome_canvas_group_get_type (),
					"x", (gdouble)0,
					"y", (gdouble)0,
					NULL);

			for (x = 0; x < width; x++) {
				c.red = primary.red + x / ratio_r;
				c.green = primary.green + x / ratio_g;
				c.blue = primary.blue + x / ratio_b;
				points->coords[0] =
					points->coords[2] = (gdouble)x;
				gnome_canvas_item_new (
					GNOME_CANVAS_GROUP (ptr->canvas_bg),
					gnome_canvas_line_get_type (),
					"points", points,
					"width_units", (double)1,
					"fill-color-gdk", &c,
					NULL);
			}
			gnome_canvas_points_free (points);
			gnome_canvas_item_lower_to_bottom (ptr->canvas_bg);
			break;

		case VERTICAL_GRADIENT:

			c = primary = ptr->background_gradient_color_primary; 
			secondary = ptr->background_gradient_color_secondary;

			ratio_r = (gdouble)height /
					(secondary.red - primary.red);
			ratio_g = (gdouble)height /
					(secondary.green - primary.green);
			ratio_b = (gdouble)height /
					(secondary.blue - primary.blue);

			points = gnome_canvas_points_new (2);
			points->coords[0] = (gdouble)0;
			points->coords[2] = (gdouble)width;

			ptr->canvas_bg = gnome_canvas_item_new (
					gnome_canvas_root (canvas),
					gnome_canvas_group_get_type (),
					"x", (gdouble)0,
					"y", (gdouble)0,
					NULL);

			for (y = 0; y < height; y++) {
				c.red = primary.red + y / ratio_r;
				c.green = primary.green + y / ratio_g;
				c.blue = primary.blue + y / ratio_b;
				points->coords[1] =
					points->coords[3] = (gdouble)y;
				gnome_canvas_item_new (
					GNOME_CANVAS_GROUP (ptr->canvas_bg),
					gnome_canvas_line_get_type (),
					"points", points,
					"width_units", (double)1,
					"fill-color-gdk", &c,
					NULL);
			}
			gnome_canvas_points_free (points);
			gnome_canvas_item_lower_to_bottom (ptr->canvas_bg);
			break;

		default:
			break;
	}
}

/*------------------------======= End Of File =======------------------------*/
