#!/usr/bin/perl -w -T
use strict ;

=head1 CGI sched_get_gantt.cgi

    Ce programme permet visualiser l'hitorique via un diag de Gantt

=cut

$ENV{PATH}='/bin:/usr/bin' ;

use CGI ;
use DBI ;

use Time::ParseDate qw/parsedate/;
use POSIX qw/strftime/ ;

use Project::Gantt ;
use Project::Gantt::Skin ;

#die "E : initialisation impossible" if (!CGI::param()) ;

##########################################################

use Sched ;
Sched::init('cgi', "$Sched::prefix_etc/cgi.cfg") ;

##########################################################
# traitement des param
my @sql_arg = () ;
my @where_clause = () ;
my $title = "Job reporting" ;

################################################################

my $id = CGI::param('id') || '' ;

if ($id) {
    if ($id !~ /^$Sched::re_jobid$/) { 
	$Sched::log->write("E : error on jobid") ;
	print "E : error on jobid" ;
	exit 0 ;
    }
    push @sql_arg, $1 ;
    push @where_clause, 'job.id = ?' ;
}

################################################################

my $host = CGI::param('host') || '' ;

if ($host) {
    if ($host !~ /^$Sched::re_host$/) { 
	$Sched::log->write("E : error on host") ;
	print "E : error on host" ;
	exit 0 ;
    }
    $title .= " on $host" ;
    push @sql_arg, $1 ;
    push @where_clause, 'job_hist.host = ?' ;
}

################################################################

my $sid = CGI::param('sid') || '' ;

if ($sid) {
    if ($sid !~ /^$Sched::re_jobsid$/) { 
	$Sched::log->write("E : error on sid") ;
	print "E : error on sid" ;
	exit 0 ;
    }
    $title .= " ($sid)" ;
    push @sql_arg, $1 ;
    push @where_clause, 'job_hist.job_sid = ?' ;
}

################################################################

my $start_date = CGI::param('start_date') 
    || strftime('%F 00:00:00', localtime(parsedate("-3 months"))) ;

if ($start_date !~ /^(\d{4}-\d{2}-\d{2} (\d{1,2}:\d{1,2}:\d{1,2}))$/) { 
    $Sched::log->write("E : error on start_date") ;
    print "E : error on start_date" ;
    exit 0 ;
}
$start_date = $1 ;
push @sql_arg, $1 ;
push @where_clause, 'job_hist.start_date >= ?' ;

################################################################

my $end_date = CGI::param('end_date') || '' ;

if ($end_date) {
    if ($end_date !~ /^(\d{4}-\d{2}-\d{2} (\d{1,2}:\d{1,2}:\d{1,2}))$/) { 
	$Sched::log->write("E : error on end_date") ;
	print "E : error on end_date" ;
	exit 0 ;
    }
    push @sql_arg, $1 ;
    push @where_clause, 'job_hist.end_date <= ?' ;
}

################################################################

my $limit = CGI::param('limit') || '' ;

if ($limit =~ /^(\d+)$/) {
    $limit = $1 ;
} else {
    $limit = 35 ;
}

################################################################

my $offset = CGI::param('offset') || '' ;

if ($offset =~ /^(\d+)$/) {
    $offset = $1 ;
} else {
    $offset = 0 ;
}

################################################################

my %can_sort = ('host' => 'job_hist.host',
		'start_date' => 'job_hist.start_date',
		'end_date' => 'job_hist.end_date',
		'id' => 'job.id',
		) ;

my $order = CGI::param('order') || 'host' ;

if ($order =~ /^(\w+)$/) {
    $order = $can_sort{$1} || $can_sort{'host'} ;
} else {
    $order =  $can_sort{'host'} ;
}

################################################################

my $where_clause = join(' AND ', @where_clause) ;
$where_clause = (($where_clause)?'WHERE ':'') . $where_clause ;

################################################################

my $dsn = Sched::cfg('dsn') ;

# postgresql
my $dbh = DBI->connect("DBI:Pg:$dsn",'', '');

if (!$dbh)
{
    $Sched::log->write("E : error of connection " . $dbh->errstr) ;
    print "E : can't connect to database" ;
    return 0 ;
}

print CGI::header('image/png') ;

my $mode = 'months' ;
my $interval ;

if ($start_date and $end_date)
{
    $interval = parsedate($end_date) - parsedate($start_date) ;

} else {

    # autodetection de l'echelle

    my $req = "
SELECT  
  date_trunc('second', max(job_hist.end_date) - min(job_hist.start_date))
FROM job_hist
$where_clause
" ;
    my $sth = $dbh->prepare($req) ;
    if ($sth->execute(@sql_arg)) {
	($interval) = $sth->fetchrow_array() ;
    }

    if ($interval) {
	$interval = parsedate($interval, NOW => 0) ;
    } else {
	$interval = 0 ;
    }
}

if ($interval <= (60*60*24*2)) {
    $mode = 'hours' ;
} elsif ($interval <= (60*60*24*7)) {
    $mode = 'days' ;
} else {
    $mode = 'months' ;
}

my $chart = new Project::Gantt(
        description     =>      $title,
        mode            =>      $mode,
        file            =>      "png:/tmp/toto$$.png");

my %ress ;

################################################################

my $req = "
SELECT job.id, job_hist.host, job_hist.status,
       date_trunc('second', job_hist.start_date),
       date_trunc('second', job_hist.end_date + '00:00:01')
FROM job_hist INNER JOIN job on job.job_no = job_hist.job_no
$where_clause
ORDER BY $order
LIMIT ? OFFSET ?
" ;

my $sth = $dbh->prepare($req) ;
if (! $sth->execute(@sql_arg, $limit, $offset)) {
    $Sched::log->write("E : error on execute " . $sth->errstr) ;
    print "E : error on execute" ;
    exit 0 ;
}

while (my ($job_id, $target, $status, $sd, $ed) = $sth->fetchrow_array())
{
    # on n'affiche pas si le job n'est pas fini
    next unless($ed) ;

    unless (defined $ress{$target}) {
	$ress{$target} = $chart->addResource( name => $target) ;
    }

    $status = ($status == 0)?' OK':' ERR' ;

    $chart->addTask(
		    description => $job_id . $status,
		    resource =>      $ress{$target},
		    start =>      $sd,
		    end =>      $ed);
}

$chart->display();

# bug
system("cat /tmp/toto$$.png") ;
unlink("/tmp/toto$$.png") ;

exit 0 ;

__END__

=head1 AUTHOR

(C) 2004-2005 Eric Bollengier

You may reach me through the contact info at eric@eb.homelinux.org

=head1 LICENSE

    Part of the network scheduling system (Sched)
    Copyright (C) 2004-2005 Eric Bollengier
        All rights reserved.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

=cut

