package Sched::Job ;

=head1 NAME Job

Ce module est utilise pour gerer des job et recuperer leur
sortie.

Version $Id: Job.pm,v 1.2 2005/04/26 19:41:28 mcgregor Exp $

=cut


use strict ;
use Sched ;
use Sched::Cond ;
use Sched::Var ;
use XML::Mini::Document ;
use Digest::MD5 ;

=head1 FUNCTION Sched::Job->new

Cette methode cree un nouveau Job.

=head2 USAGE

    * self     => 

      file     => fichier de description
      xml      => XML::Mini::Element
      serial   => identifiant

=head2 RETURN

    $obj : OK

=cut

sub new
{
    my $class = shift ;
    my (%arg) = @_ ;

    return undef if (!defined $arg{serial}) ;
    
    if (defined $arg{file} and -r $arg{file}) {
	my $sav = $XML::Mini::AutoSetParent ;
	$XML::Mini::AutoSetParent = 1 ;
	{
	    my $doc = XML::Mini::Document->new() ;
	    $doc->parse($arg{file}) ;

	    $arg{xml} = $doc->getRoot()->getElement('job') if ($doc) ;
	}
	$XML::Mini::AutoSetParent = $sav ;
	
	if (not defined $arg{xml}) {
	    print "E : Impossible de charger $arg{file} $! $@\n" ;
	    return undef ;
	}
    }

    my $self = bless {}, $class ;

    $self->{xml} = $arg{xml} ;

    if (not defined $self->{xml} or not defined $self->attribute('id')) {
	print "E : erreur d'initialisation Sched::Job::new\n" ;
	return undef ;
    }

    Sched::NS::register_id("=xml-root=", $self->{xml}) ;

    $self->attribute('serial', $arg{serial}) ;

    #if (!$self->apply_dtd($Sched::cfg->val('main', 'dtd_job'))) {
    #print "E : Erreur d'application de la DTD\n" ;
    #return undef ;
    #}

    Sched::Var::init($self->{xml}) ;
    Sched::Var::add_var('JOB', $self->attribute('id')) ;

    $self->attribute('start_date', time()) ;
    $self->attribute('pid', $$) ;

    return bless ($self, $class) ;
}

=head1 FUNCTION Sched::Job::is_running

Verifie que le job est en cours

=head2 USAGE

    $obj->is_running() ;

=head2 RETURN

    0 : NON
    1 : OUI

=cut

sub is_running($)
{
    my $self = shift ;
    
    my $state = $self->{xml}->attribute('state') || '' ;

    if ($state eq 'failed') {
	return 0 ;
    } else {
	return 1 ;
    }
}

=head1 FUNCTION Sched::Job::attribute

Accede au attribut XML

=head2 USAGE

    $obj->is_running() ;

=head2 RETURN

    0 : NON
    1 : OUI

=cut

my %attr_base64 = map { $_ => 1 } qw/comment cond env/ ;

sub attribute
{
    my $self = shift ,
    my ($attr, $val) = @_ ;

    if ($attr_base64{$attr}) {
	if (defined $val) {
	  return $self->{xml}->attribute($attr, 
					 MIME::Base64::encode_base64($val,''));
	} else {
	    my $v = $self->{xml}->attribute($attr) ;

	    if ($v) {
		$v = MIME::Base64::decode_base64($v) ;
	    } 
	    return $v ;
	}
    } else {
	return $self->{xml}->attribute(@_) ;
    }
}

=head1 FUNCTION Sched::Job::run

Lancement de la commande

=head2 USAGE

    $obj->run() ;

=head2 RETURN

    0 : ERR
    1 : OK

=cut

sub run ( $ )
{
    my ($self) = @_ ;

    return 1 ;
}

=head1 FUNCTION Sched::Job::sync

Sauvegarde le XML dans le fichier status:serial:id

=head2 USAGE

    $obj->sync() ;

=head2 RETURN

    1 : OK
=cut

sub sync
{
    my ($self) = @_ ;
    $self->attribute('sync_date', time()) ;
    my $file = "status_" 
	       . $self->attribute('serial') 
	       . "_" 
	       . $self->attribute('id') ;
    open(SAV, ">$file") ;
    print SAV $self->{xml}->toString() ;
    close(SAV) ;
}

=head1 FUNCTION Sched::Job::can_run

Verifie si un host peut utiliser ce job

=head2 USAGE

<job id='test' host='betty,boop' />

$job->can_run('betty') ;
or
can_run($xml->{job}, 'betty') ;

=head2 RETURN

    O => non
    1 => ok

=cut

sub can_run($$)
{
    my ($self, $host) = @_ ;

    return 1 if (not defined $self->attribute('host')) ;
    
    if ($self->attribute('host') =~ /(^|,)$host(,|$)/i) {
	return 1 ;
    } else {
	return 0 ;
    }
}


=head1 FUNCTION Sched::Job::cancel

Termine toutes les taches en cours.

=head2 USAGE

    $obj->cancel()

=cut


sub cancel($)
{
    my $self = shift ;
    $self->attribute('state', 'failed') ;
    $self->attribute('status', 255) ;
    $self->sync() ;
}

=head1 FUNCTION Sched::Job::validate

Test si le job semble ok

=head2 USAGE

    $obj->validate() ;

    remember_template_id($obj) ;
    remember_var_name($obj) ;
    remember_task_id($obj) ;

    %Sched::Cond::result = %list_id ;

=head2 DESCRIPTION

    %list_id : liste des identifiants Task XML
    id => 1

    %list_var : liste des variables
    name => 1

    %list_template : liste des id template
    id => 1

=cut

my %list_id ;
my %list_var ;
my %list_template ;
my $max_time = 0 ;

=head1 FUNCTION Sched::Job::get_all_id

    Recupere tous les task id d'un job

=head2 USAGE

    @ids = Sched::Job::get_all_id(xmlroot) ;

=cut

sub get_all_id
{
    my $root = shift ;
    
    %list_id = () ;
    remember_task_id($root) ;
    return keys %list_id ;
}

=head1 FUNCTION Sched::Job::remember_template_id

    Recupere tous les identifiants de template et verifie
    qu'il n'y a pas de doublon.

    met  jour %list_template

=head2 USAGE

    $job->remember_template_id() ;

=cut

sub remember_template_id
{
    my $root = shift ;
    my $ret ;

    for my $t (@{ $root->getAllChildren('template') }) {
	my $id = $t->attribute('id') ;

	if (exists $list_template{$id}) {
	    print "E : template $id is already defined elsewhere\n" ;
	    return 0 ;
	}
	
	$list_template{$id} = 1 ;
    }

    return 1 ;
}

=head1 FUNCTION Sched::Job::remember_task_id

    Recupere tous les identifiants de task et verifie
    qu'il n'y a pas de doublon.

    met  jour %list_id

=head2 USAGE

    $job->remember_task_id() ;

=cut

sub remember_task_id
{
    my $root = shift ;
    my $ret ;

    for my $t (@{ $root->getAllChildren('task') }) {
	my $id = $t->attribute('id') ;

	if (exists $list_id{$id}) {
	    print "E : task $id is already defined elsewhere\n" ;
	    return 0 ;
	}
	
	if ($id !~ /^[\w\d\.-]+$/) {
	    print "E : task ($id) is not [\\w\\d\\.-]+ compliant\n" ;
	    return 0 ;
	}

	$list_id{$id} = 1 ;

	if (!remember_task_id($t)) {
	    return 0 ;
	}

    }

    return 1 ;
}

=head1 FUNCTION Sched::Job::remember_var_name

    Recupere tous les identifiants de var, verifie
    qu'il n'y a pas de doublon et que les identifiants sont
    syntaxiquement correctes.

    met  jour %list_id

=head2 USAGE

    $job->remember_var_name() ;

=cut 

sub remember_var_name
{
    my $root = shift ;

     for my $t (@{ $root->getAllChildren('var') }) {
	my $n = $t->attribute('name') ;
	if ($n !~ /^[\w\d\.-]+$/) {
	    print "E : warning [$n] is not a valid variable name\n" ;
	    return 0 ;
	}
	if (exists $list_var{$n}) {
	    print "E : warning [$n] is already defined elsewhere\n" ;
	    return 0 ;
	}
	$list_var{$n} = 1 ;
    }

    return 1 ;
}

=head1 FUNCTION Sched::Job::check_task_id

    Verifie que on peut creer un Sched::Cmd avec
    la description xml et que la condition est valide.
    

=head2 USAGE

    $job->remember_task_id() ;
    $job->check_task_id() ;

=cut 

sub check_task_id
{
    my $root = shift ;


    for my $t (@{ $root->getAllChildren('task') }) {
	my $id = $t->attribute('id') ;

	if (defined $t->attribute('use')) {
	    my $u = $t->attribute('use') ;
	    if (not exists $list_template{$u}) {
		print "E : task $id use $u template, but it isn't defined\n" ;
		return 0 ;
	    }
	}
	
	if ($id !~ /^[\w\d\.-]+$/) {
	    print "E : task ($id) is not [\\w\\d\\.-]+ compliant\n" ;
	    return 0 ;
	}

	my $cmd = new Sched::Cmd(serial => 'test', xml => $t) ;
	
	if (!$cmd) {
	    print "E : $id is not a valide task description\n" ;
	    return 0 ;
	}

	my $ret = Sched::Cond::have_all_result($t) ;

	if (not $ret) {
	    print "W : $id condition seems to be strange...\n" ;
	}

	if ($t->attribute('maxtime') > $max_time) {
	    print "W : $id max time is bigger than job max time\n" ;
	}

	if (! check_task_id($t)) {
	    return 0 ;
	}

    }

    return 1 ;
}

=head1 FUNCTION Sched::Job::validate

    Verifie que le job est ok.

=head2 USAGE

    $job->validate() ;

=cut

sub validate
{
    my $self = shift ;
    my $root = $self->{xml} ;

    # init
    %list_template = %list_id = %list_var = () ;

    # on enregistre l'id du job
    $list_id{$root->attribute('id')} = 1 ;

    $max_time = $root->attribute('maxtime') ;

    if (! remember_var_name($root)) {
	print "E : verification failed\n" ;
	return 0 ;
    }

    if (! remember_template_id($root)) {
	print "E : verification failed\n" ;
	return 0 ;
    }

    if (! remember_task_id($root)) {
	print "E : verification failed\n" ;
	return 0 ;
    }

    %Sched::Cond::result = %list_id ;

    if (! check_task_id($root)) {
	print "E : verification failed\n" ;
	return 0 ;
    }

    print "I : verification ok\n" ;
    return 1 ;
}

=head1 FUNCTION Sched::Job::get_md5

    Renvoi le MD5 d'un fichier Job

=head2 USAGE

    $md5 = Sched::Job::get_md5($filename) ;

=cut


sub get_md5($)
{
    my $file = shift ;
    open(FILE, $file) or return undef ;
    binmode(FILE);
    
    my $ret = Digest::MD5->new->addfile(*FILE)->hexdigest ;
    close(FILE) ;
    return $ret ;
}


1;

__END__

=head1 AUTHOR

(C) 2004-2005 Eric Bollengier

You may reach me through the contact info at eric@eb.homelinux.org

=head1 LICENSE

    Part of the network scheduling system (Sched)
    Copyright (C) 2004-2005 Eric Bollengier
        All rights reserved.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

=cut
