package Sched::Cond ;

use strict ;

=head1 NAME Cond

Pour executer un script (condition par ex), on fait un reval avec verification
des dependances

=cut

use Sched::NS ;
use Sched::Callback ;
use Sched::Cmd ;
use Sched::Var ;

use MIME::Base64 ;

our %result ;
our $version = '$Id: Cond.pm,v 1.2 2005/04/09 09:39:32 mcgregor Exp $' ;


=head1 FUNCTION Sched::Cond::depend_on_ok|err|finish

Cette methode renvoie toutes les dependances d'un script err ou ok

=head2 USAGE

      cond : string contenant du code perl

=head2 RETURN

    @list task_id

=cut


sub depend_on_ok
{
    my ($cond) = shift ;
    return ($cond =~ /ok\s*\(\s*['"]([\w\d\.-]+)["']\s*\)/mg) ;
}

sub depend_on_err
{
    my ($cond) = shift ;
    return ($cond =~ /err\s*\(\s*['"]([\w\d\.-]+)["']\s*\)/mg) ;
}

sub depend_on_finish
{
    my ($cond) = shift ;
    return ($cond =~ /finish\s*\(\s*['"]([\w\d\.-]+)["']\s*\)/mg) ;
}

=head1 FUNCTION Sched::Cond::depend_of

Cette methode renvoie toutes les dependances d'un script

=head2 USAGE

      node : noeud XML::Mini::Element ayant un attribut attr=''
      [attr : attribut du noeud] ('cond' par defaut)

=head2 RETURN

    @list task_id

=cut

sub depend_of
{
    my $node = shift ;
    my $attr = shift || 'cond' ;

    my @res ;
    
    if (defined $node->attribute($attr)) {
	my $expr = MIME::Base64::decode_base64($node->attribute($attr)) ;
	push @res, ($expr =~ /result{["']?([\w\d_,\.-]+?)['"]?}/mg) ;
	push @res, depend_on_finish($expr) ;
	push @res, depend_on_ok($expr) ;
	push @res, depend_on_err($expr) ;
    }

    return @res ;
}

=head1 FUNCTION Sched::Cond::how_depend_of

Cette methode renvoie toutes les dependances d'un script dans un
hash qui precise le type de dpendance (ok, err, result...)

=head2 USAGE

      node : noeud XML::Mini::Element ayant un attribut attr=''
      [attr : attribut du noeud] ('cond' par defaut)

=head2 RETURN

    %list : task_id => type

=cut

sub how_depend_of
{
    my $node = shift ;
    my $attr = shift || 'cond' ;

    my @res ;
    
    if (defined $node->attribute($attr)) {
	my $expr = MIME::Base64::decode_base64($node->attribute($attr)) ;
 push @res, 
   map { $_ => 'result' } ($expr =~ /result{["']?([\w\d_,\.-]+?)['"]?}/mg) ;

	push @res, map { $_ => 'finish' } depend_on_finish($expr) ;
	push @res, map { $_ => 'ok'  } depend_on_ok($expr) ;
	push @res, map { $_ => 'err' } depend_on_err($expr) ;
    }

    return @res ;
}


=head1 FUNCTION Sched::Cond::have_all_result

Cette methode test si toutes les resultats demandes sont
disponibles.

=head2 USAGE

      node : noeud XML::Mini::Element ayant un attribut attr=''
      [attr : attribut du noeud] ('cond' par defaut)


=head2 RETURN

    1 : OK
    0 : ERR

=cut

sub have_all_result
{
    my $node = shift ;
    my $attr = shift || 'cond' ;

    my @res = depend_of($node, $attr) ;
    my $ret = 1 ;

    # validation de tous les resultats
    for my $r (@res) {
	next if (exists $result{$r}) ;

	my $val = get_result($node, $r) ;
	if (not defined $val) {
	    $ret = 0 ;
	} else {
	    $result{$r} = $val ;
	}
    }

    return $ret ;
}


=head1 FUNCTION Sched::Cond::get_result

Cette methode essaye de recuperer le status d'une commande
disponibles. 

=head2 USAGE

      node : noeud XML::Mini
      "(job_id:)?task_id" : id demande

=head2 RETURN

    status : OK
    undef  : ERR

=cut

sub get_result($$)
{
    my ($node, $r) = @_ ;
    my ($id_task, $id_job) = reverse split(/:/, $r) ;
    
    my $root = Sched::NS::aa('=xml-root=') ;

    if ($id_job) {
	if ($root->attribute('id') ne $id_job) {
	    die "E : demande d'un resulat sur un autre graph non implment"; 
	}
    }
    
    my $cmd = Sched::Cmd::find_task_by_id($root, $id_task) ;

    if (!$cmd) {
	$Sched::log->write("E : impossible de trouver $id_task") ;
	return undef ;
    }

    my $state = $cmd->attribute('state') ;

    if (not defined $state) {
	return undef ;
    }

    if ($state eq 'finish' || $state eq 'failed') {
	return $cmd->attribute('status') ;
    }

    return undef ;
}


=head1 FUNCTION Sched::Cond::eval_script

Cette methode fait un eval sur $node->{attr}

=head2 USAGE

      node : noeud XML::Mini ayant un attribut attr=''
      attr : nom de l'attribut de script

=head2 RETURN

    eval_script($node, 'cond') 

=cut

use Safe ;

# ce package inclus des fonctions utilisables dans un eval_script
# - mail, notification nagios etc...
use Sched::Script ;

sub eval_script($$)
{
    our ($node, $attr) = @_ ;

    my $expr = MIME::Base64::decode_base64($node->attribute($attr)) ;

    sub err { if (exists $result{$_[0]}){ $result{$_[0]} != 0}else{ undef }} ;
    sub ok  { if (exists $result{$_[0]}){ $result{$_[0]} == 0}else{ undef }} ;
    sub finish{ exists $result{$_[0]} } ;

    sub job_end { my ($result, $info) = @_ ;
		  
		  Sched::NS::register_id('=sigterm-status=', $result) 
		      if ((defined $result) and ($result =~ /^\d+$/)) ;
		  if (defined $info) {
		      $info =~ s/['"]/ /g ; #']
                      Sched::NS::register_id('=sigterm-info=', $info) 
		  }

		  kill('TERM', $$) ; 

		  $node->attribute('status', $result) ;

		  # en cond, le noeud n'est pas execute
		  $node->attribute('state', 'failed') 
		      if ($attr eq 'cond') ;

		  $node->attribute('info', "Arret du graph") ;

		  return 1 ;
		 } ;

    # si tous les resultats sont pas encore la, on fait pas
    if (! have_all_result($node, $attr)) {
	$Sched::log->write("E : $attr miss result") ;
	$node->attribute('info', "E : $attr miss result") ;
	return undef ;
    }

    # remplacement des variables
    Sched::Var::add_var('ID', $node->attribute('id')) ;
    $expr = Sched::Var::expand($expr) ; 
    Sched::Var::del_var('ID') ;

    my $jail = new Safe() ;
    $jail->permit(qw/time/);
    $jail->share(qw /&err &ok &job_end %result &finish/) ;

    # pour ajouter une fonction mail(), etc...
    $jail->share_from('Sched::Script', $Sched::Script::export_funcs);

    my $ret = $jail->reval($expr) ;
    if ($@) {
    	$Sched::log->write("E : impossible d'evaluer $expr ($@)") ;
	$node->attribute('info', "E : $attr return ($@)") ;
	$ret = undef ;
    }
    return $ret ;
}


=head1 FUNCTION Sched::Cond::check_all_result

Cette methode pose un callback (Sched::Callback) sur
un noeud par rapport aux resultats en attente.

=head2 USAGE

      node : noeud XML::Mini ayant un attribut cond=''
      cb  : &fonction callback
      arg : arguments passes au callback 

=cut

sub check_all_result
{
    my ($node, $cb, $arg) = @_ ;

    return 1 if (! defined $node->attribute('cond') 
		 || $node->attribute('cond') eq '') ; 

    my @res = depend_of($node) ;

    # validation de tous les resultats
    for my $r (@res) {
	next if (exists $result{$r}) ;

	Sched::Callback::add_cb("CHLD ${Sched::sid}:$r", $cb, $arg) ;
    }
}


=head1 FUNCTION Sched::Cond::is_ok

    Test si la condition est ok pour un noeud possedant un attribut cond.
    La procedure recupere les resultats attendus.

    Si la condition n'existe pas, on regarde le status de la tache precedente.

=head2 USAGE

      node : noeud XML::Mini ayant un attribut cond=''

=head2 RETURN

    1 => OK
    0 => ERR
    
    undef => manque de resultat

=cut

sub is_ok($)
{
    my ($node) = @_ ;
    my $ret = undef ;

    # par defaut, cond $result{parent} == 0
    if (   (not defined $node->attribute('cond')) 
	|| ($node->attribute('cond') eq '')) {
	my $parent = $node->parent() ;

	if (!$parent) {
	    $Sched::log->write("E : this node don't have parent") ;
	    return 0 ;
	}

	# ok si on est le task racine
	return 1 if ($parent->name() eq 'job') ;
	
	# ok si le status du precedent
	if ((defined $parent->attribute('status')) 
	    and 
	    ($parent->attribute('status') eq '0'))
	{
	    return 1 ;
	} else {
	    return 0 ;		# en erreur
	}
    }

    $ret = eval_script($node, 'cond') ; 

    return $ret ;
}

sub test
{

}

1;

__END__

=head1 AUTHOR

(C) 2004-2005 Eric Bollengier

You may reach me through the contact info at eric@eb.homelinux.org

=head1 LICENSE

    Part of the network scheduling system (Sched)
    Copyright (C) 2004-2005 Eric Bollengier
        All rights reserved.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

=cut

