package Sched::Net ;
use strict ;

# en attendant de savoir faire un flush sur la socket...
use Time::HiRes qw( usleep );

=head1 USAGE

    %proto = ( 
        'PING' => {	# pas de gestion des erreurs
		    arg => undef,
		    fct => sub {
		        $Sched::Net::master->send('OK') ;
		    },
		  },
    ) ;

    $Sched::init('job', '/path/to/cfg') ;

    $Sched::set_job_file('/path/to/jobfile') ; # if type eq job

    $Sched::Net::set_proto(%proto) ;

    Sched::Net::master_connect() ;

=head2 CONFIG

    [job]
    master_ip = localhost
    master_port = 5544
    master_passwd = mdp

    master_retry = 10
    master_ping = 110

    master_read_retry = 60
    master_encrypt = 1
    master_compress = 0


=cut

use Tie::File ;
use Event ;
use Sched ;
use Net::EasyTCP ;

our $master ;

=head2 FUNCTION Sched::Net::set_proto

   my %proto = (
   	     'PING' => {	  # name
   		 arg => '^\w+$',  # first arg
   		 fct => sub {	  # function to call fct($host, $arg)
   		     my ($host, $arg) = @_ ;
   		     Sched::Net::send_to_master('OK') ;
   		 },
   	     },
   	     );
   
=cut


my %proto ;

sub set_proto
{
    %proto = @_ ;
}

=head2 FUNCTION Sched::Net::master_connect

    do a

    if(Sched::Net::connect()) {
	&Sched::Net::master_listen() ;
    } else {
	&Sched::Net::reconnect() ;
    }

=cut

sub master_connect
{
    if(Sched::Net::connect()) {
	&Sched::Net::master_listen() ;
    } else {
	&Sched::Net::reconnect() ;
    }
}


=head2 FUNCTION Sched::Net::reconnect

    Add a timer to reconnect master

=cut

sub reconnect
{
    my $t = Event->timer(after => Sched::cfg('master_retry'),
			 interval => Sched::cfg('master_retry'),
                         desc => "Reconnect to master",
                         cb => sub {
			     my $self = shift ;
			     $Sched::log->write("D : reconnexion", 15) ;
			     if (Sched::Net::connect()) {
				 master_listen() ;
				 $self->w->cancel() ;
			     } else {
				 $self->w->again() ;
			     }
			 }) ;
    $t->start() ;
}

=head2 VARIABLE @wait_to_send

    un array tie sur disque des commandes en attente

=cut

my $_tie_ok ;
my @wait_to_send ;
my $tie_file ;

sub _init_tie
{
    $tie_file = "/master" . (($Sched::type eq 'job')?$Sched::sid:'') ;
    $tie_file =~ y/:/_/ ;
    $tie_file = Sched::cfg('db_dir') . $tie_file ;
    $_tie_ok = tie @wait_to_send, 'Tie::File', $tie_file, recsep => '' ;
}


sub connect
{
    my $cmd ;
    $master = new Net::EasyTCP(mode =>      'client',
			       host => Sched::cfg('master_ip'),
			       port => Sched::cfg('master_port'),
			       password => Sched::cfg('master_passwd'),
			       donotcompress => ! Sched::cfg('master_compress'),
			       donotencrypt  => ! Sched::cfg('master_encrypt'),
			       donotcompresswith => [],
			       donotencryptwith => [],
			       );

    if (!$master) {
	$Sched::log->write("E : erreur de connexion sur le master $@") ;
	return 0 ;
    }

    if ($Sched::type eq 'slave') {
	$cmd = "HELO $Sched::hostname" ;
    } elsif ($Sched::type eq 'job') {
	my $md5 = Sched::Job::get_md5($Sched::job_file) ;
	$cmd = "REGISTER $Sched::sid $md5" ;
    }

    $master->send($cmd) ;

    my $rep = $master->receive(5) ;

    if ($rep eq 'OK') {
	$Sched::log->write("I : connexion reussi") ;
    } else {
	$Sched::log->write("E : connexion refusee $@") ;
	$master->close() ;
	return 0 ;
    }

    if (! $_tie_ok) {
	_init_tie() ;
    }

    while (my $what = shift @wait_to_send) {
	$Sched::log->write("POP ==> $what", 20) ;
	if (!$master->send($what)) {
	    unshift @wait_to_send,$what ;
	    $master->close() ;
	    Sched::Net::reconnect() ;
	    return 0 ;
	}
    } 

    if ($_tie_ok) {
	$_tie_ok = undef ;
	untie @wait_to_send ;
	unlink ($tie_file) ;
    }

    return 1 ;
}

=head1 FUNCTION &Sched::Net::send_to_master

   envoi une commande au master ou bien on la spool

=head2 USAGE

   &Sched::Net::send_to_master($what [,$method [, $arg]]) ;

   o $what (string) : what you want to send

   o $method (string) : 
	- keep (default) : send later if network unavailable
	- keep_last  : (UPDATE, UPDATE) => (UPDATE)
        - drop : don't send later
        - cancel : delete $arg and return or push $what

   &Sched::Net::send_to_master("CMD arg") ;

   or 

   &Sched::Net::send_to_master('CANCEL arg', 'cancel', 'CMD arg') ;

   or 

   &Sched::Net::send_to_master('PING', 'drop') ;

=cut

sub send_to_master
{
    my $what = shift ;
    my $mode = shift || 'keep' ;
    
    if ((!defined $master) or (!$master->send($what))) {

	if (!$_tie_ok) {
	    _init_tie() ;
	}
	
	if ($mode eq 'drop') {
	    # on ne fait rien
	    return ;

	} elsif ($mode eq 'keep_last') {

	    # on ne prend que la derniere commande
	    # STATUS, UPDATE, WATCH...

	    my ($cmd, undef) = split(/\s+/, $what, 2) ;

	    # on parcourt la liste (a l'envers) et suppression
	    # des commandes d'avant

	    for (my $i = $#wait_to_send ; $i >= 0; $i--) {

		if (($wait_to_send[$i]) && ($wait_to_send[$i] =~ /^$cmd\b/)) {
		    splice(@wait_to_send, $i, 1) ;
		}
	    }

	    push @wait_to_send, $what ;

	} elsif ($mode eq 'keep') {

	    # on garde la commande
	    push @wait_to_send, $what ;

	} elsif ($mode eq 'cancel') {

	    # on efface toutes les commandes $arg
	    # et si on trouve rien, on ajoute la nouvelle
	    # commande a la fin

	    my $push = 1 ;
	    my $arg = shift || die 'E : send_to_master : cancel need arg' ;
	    
	    for (my $i = $#wait_to_send ; $i >= 0; $i--) {
		if ($wait_to_send[$i] eq $arg) {
		    splice(@wait_to_send, $i, 1) ; ;
		    $push = 0 ;
		}
	    }

	    if ($push) {
		push @wait_to_send, $what ;
	    }
	} else {

	    die "E : send_to_master : mode $mode is unknow" ;

	}

    } else {
	# visiblement le flush n'arrive pas toujours...
	# donc il faut envoyer un packet apres chaque commande
	usleep(10) ;
	$master->send('NOP') ;
    }
}


# hash contenant des evt
my %event ;

sub _got_error
{
    $Sched::log->write("E : erreur sur master") ;
    $event{socket_read}->cancel() ;
    $event{socket_error}->cancel() ;
    $event{ping}->cancel() ;
    $master->close() ;
    
    Sched::Net::reconnect() ;
}

sub _got_data
{
    while (my $data = $master->receive(1)) {
	$Sched::log->write("D : ==> $data", 15) ;

	my ($cmd, $arg) = split(/\s+/, $data, 2) ;
	
	if ($cmd and defined $proto{$cmd}) {
	    if ((!$proto{cmd}->{arg}) || ($arg =~ $proto{$cmd}->{arg})) {
		$proto{$cmd}->{fct}($arg) ;
	    } else {
		$Sched::Net::master->send("ERR") ;
	    }
	} else {
	    # erreur de protocole => deconnexion
	    _got_error() ; 
	}
    }
}

my %conf_default = ( master_port       => 5544,
		     master_retry      => 60,
		     master_ping       => 110,
		     master_read_retry => 30,
		     master_encrypt    => 1,
		     master_compress   => 1,
		     db_dir   => '/tmp',
		     );

sub valid_conf
{
    my $verbose = shift ;
    my $ret = 1 ;
    # user group

   for my $k qw(master_ip master_passwd)
    {
	if (!Sched::cfg($k)) {
	    $Sched::log->write("W : [$Sched::type]/$k is not defined");
	    $ret = 0 ;
	} else {
	    $Sched::log->write("D : $k = " . 
			       (($k eq 'master_passwd')?'xxx':Sched::cfg($k)),
			       15) ;
	}
    }

    for my $k qw(db_dir)
    {
	if (!Sched::cfg($k)) {
	    $Sched::log->write("E : [$Sched::type]/$k is not defined");
	    $ret = 0 ;
	} elsif(! -d Sched::cfg($k)) {
	    $Sched::log->write("E : [$Sched::type]/$k is not a directory (" . Sched::cfg($k) . ")");
	    $ret = 0 ;
	} else {
	    $Sched::log->write("D : $k = " . Sched::cfg($k), 15) ;
	}
    }
	
    for my $k (keys %conf_default)
    {
	if (!Sched::cfg($k)) {
	    $Sched::log->write("W : [$Sched::type]/$k is not defined");
	    $Sched::log->write("I : $k = $conf_default{$k}") ;
	    $Sched::cfg->newval($Sched::type, $k, $conf_default{$k}) ;
	} else {
	    $Sched::log->write("D : $k = " . Sched::cfg($k), 15) ;
	}
    }

    return $ret ;
}

sub master_close
{
    my $now = shift ;
    
    if ($master) {
	if (!$now) {
	    $master->send("PING") ;
	}
	$master->close() ;
    }
}

#use Socket ;

sub master_listen
{
    my $sock = $master->socket() ;

    # to prevent message not be sent
    my $oldfh = select($sock); $| = 1; select($oldfh);

#    ca marche pas...
#    setsockopt($sock, Socket::IPPROTO_TCP(), Socket::TCP_NODELAY(), 1) ;
#    require "sys/socket.ph";    # for &TCP_NODELAY
#    Socket::setsockopt($sock, SOL_SOCKET, &TCP_NODELAY, 1);

    my $t = Event->io(fd => $sock, 
		      poll => 'r', 
		      timeout => Sched::cfg('net_read_retry'),
		      cb => \&_got_data) ;
    $t->start() ;
    $event{socket_read} = $t ;

    $t = Event->io(fd => $master->socket(), 
		   poll => 'e', 
		   cb => \&_got_error,
		   );

    $t->start() ;
    $event{socket_error} = $t ;

    $t = Event->timer(after => Sched::cfg('master_ping'),
		      interval => Sched::cfg('master_ping'),
		      desc => "ping master",
		      cb => sub {
			  my $self = shift ;
			  $master->send("PING") ;
			  $self->w->again() ;
		      }) ;


    $event{ping} = $t ;
}

1;

__END__

=head1 AUTHOR

(C) 2004-2005 Eric Bollengier

You may reach me through the contact info at eric@eb.homelinux.org

=head1 LICENSE

    Part of the network scheduling system (Sched)
    Copyright (C) 2004-2005 Eric Bollengier
        All rights reserved.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

=cut
