package Sched::Cmd ;

=head1 NAME Sched::Cmd

Ce module est utilise pour lancer des commandes et recuperer leur
sortie.

Version $Id: Cmd.pm,v 1.1 2005/04/05 20:01:49 mcgregor Exp $

=head2 USAGE

    Sched::init('job', "/path/to/config") ;
    Sched::set_serial() ;
    Sched::set_sid() ;

    new Sched::Cmd(serial => $serial, 
		   xml => $xml, | data => $data) ;

=cut


use strict ;
use IO::Pipe ;
use MIME::Base64 ;
use XML::Mini::Document ;

use Sched::Var ;
use Sched::Net ;
use Sched::NS ;

our %objects_pid ;	# pid -> Sched::Cmd

=head1 FUNCTION Sched::Cmd->new

Cette methode cree une nouvelle commande.

=head2 USAGE

    * self     => 

      data     => <task id='test'..../>

      xml      => XML::Mini::Element
      serial   => identifiant
      cmdline  => ligne de commande
      stdin    => fichier entree standard
      stdout   => fichier de sortie standard
      stderr   => fichier de sortie d'erreur
      user     => utilisateur
      id       => identifiant de la commande
      group    => "group1 group2"

=head2 ATTRIBUT

      state    => new|running|failed|finish
      status   => 0..255
      sid      => serial:host:job:id

=head2 RETURN

    $obj : OK

=cut

sub new
{
    my $class = shift ;
    my (%arg) = @_ ;

    return undef unless (exists $arg{serial}) ;

    if (exists $arg{data}) {
	my $xml = new XML::Mini::Document($arg{data}) ;
	$arg{xml} = $xml->getRoot()->getElement('task') if ($xml) ;
    }

    my $self = bless {}, $class ;

    $self->{xml} = $arg{xml} ;

    my $id   = $self->attribute('id') ;

    if (! defined $self->{xml} or ! defined $id) {
	$Sched::log->write("E : erreur d'initialisation (pas d'id)") ;
	return undef ;
    }
 
    my $root = Sched::NS::aa('=xml-root=') ;
       
    if ($root and $self->attribute('use')) {
	# dans le sched_master, on a pas acces au xml-root
	my $use_t = $self->attribute('use') ;


	for my $t (@{ $root->getAllChildren('template') }) {
	    if ($t->attribute('id') eq $use_t) {

		for my $attr (qw/stdin stdout stderr host env after_cmd
			         user group cmdline maxtime cond/)
		{
		    if (defined $t->attribute($attr) 
			and 
			not defined $self->{xml}->attribute($attr))
		    {
			$self->{xml}->attribute($attr, $t->attribute($attr)) ;
		    }
		}

		last ;
	    }
	}
    }

    my %default = ( 'stdin' => '/dev/null',
		    'stdout' => "${id}.stdout",
		    'stderr' => "${id}.stderr") ;

    for my $f (keys %default) {
	$arg{$f} =  $arg{f} || $self->attribute($f) || $default{$f} ;
    }
    
    for my $attr ('serial', 'user', 'group', 'id', 'stdin', 'stderr',
		  'stdout', 'maxtime', 'host', 'sid', 'cmdline')
    {
	$self->attribute($attr, $arg{$attr}) if (defined $arg{$attr}) ;
    }

    $self->attribute('state', 'new') 
	if (not defined $self->attribute('state')) ;

    $self->attribute('info', '')  
	if (not defined $self->attribute('info')) ;

    # pour affichage des commentaires
    $self->attribute('sid', $Sched::sid . ":" . $id) 
	if (not defined $self->attribute('sid')) ;

    Sched::NS::register_id($self->attribute('sid'), $self) ;

    Sched::Var::add_var('ID', $id) ;

    for my $a (qw/stdout stdin stderr host cmdline cond env/) 
    {
	$self->attribute($a, Sched::Var::expand($self->attribute($a))) 
	    if (defined $self->attribute($a))  ; 
    }

    Sched::Var::del_var('ID') ;

    return bless ($self, $class) ;
}

# TODO : transparence code/decode
my %attr_base64 = map { $_ => 1 } qw/comment cond cmdline env/ ;

sub attribute
{
    my $self = shift ,
    my ($attr, $val) = @_ ;

    if ($attr_base64{$attr}) {
	if (defined $val) {
	    return $self->{xml}->attribute($attr, 
					 MIME::Base64::encode_base64($val,''));
	} else {
	    my $v = $self->{xml}->attribute($attr) ;

	    if ($v) {
		$v = MIME::Base64::decode_base64($v) ;
	    } 
	    return $v ;
	}
    } else {
	return $self->{xml}->attribute(@_) ;
    }
}

=head1 FUNCTION Sched::Cmd::update_from_xml

Permet de mettre a jour les variables
depuis un flux xml

=head2 RETURN

    0 => ERR (sid different)
    1 => OK

=cut

sub update_from_xml($$)
{
    my ($self, $xml) = @_;

    return 0 unless (defined $self->attribute('sid') 
		     and
		     $xml->attribute('sid')) ;

    return 0 unless ($self->attribute('sid') eq $xml->attribute('sid')) ;

    for my $a qw(state info status pid end_date start_date) {
	$self->attribute($a, $xml->attribute($a)) 
	    if (defined $xml->attribute($a)) ;
    }

    if (($self->attribute('state') eq "finish") or
	($self->attribute('state') eq "failed")) {
	Sched::NS::unregister_id($self->attribute('sid')) ;
    }

    return 1 ;
}


=head1 FUNCTION Sched::Cmd::update_from_data

Permet de mettre a jour les variables
depuis un string xml (<task ...>)

=head2 RETURN

    0 => ERR (sid different)
    1 => OK

=cut

sub update_from_data($$)
{
    my ($self, $data) = @_ ;

    my $xml = new XML::Mini::Document($data) ;
    $xml = $xml->getRoot()->getElement('task') ;
    return $self->update_from_xml($xml) ;

}


=head1 FUNCTION Sched::Cmd::get_from_sid

Permet de recuperer la reference d'un Cmd
avec le serial et l'id.

=cut

sub get_from_sid($)
{
    my ($sid) = @_ ;

    return Sched::NS::aa($sid) ;
}


=head1 FUNCTION Sched::Cmd::print

Permet d'afficher un objet Cmd

=head2 USAGE

    $obj->print() ;

=head2 RETURN

=cut

sub print( $ )
{
    my $self = shift ;
    print $self->{xml}->toString() ;
}

=head1 FUNCTION Sched::Cmd::get_env

    Recuperation de job->env et cmd->env

=head2 USAGE

    Sched::Cmd::get_env(xmlnode) ;

    
=head2 RETURN

    %env = (VAR => val)

=cut


sub get_env
{
    my $xml = shift ;

    my $str_env = $xml->attribute('env') 
	or return () ;

    $str_env = MIME::Base64::decode_base64($str_env) ;
    
    # env doit  etre clean...
    my %env = split(/\s*=\s*['"]?|['"]?\s*\n\s*|['"]?\s*$/, $str_env) ;

#    my %env ;
#    for my $l (split(/\n/, $str_env)) {
#	if ($l =~ /\s*([\S=]+)\s*=\s*(['"])?([^\2]+)\2?$/) { 
#	    $env{$1} = $3 ;
#	}
#    }

    return %env ;
}

=head1 FUNCTION Sched::Cmd::_touch

    Creation d'un fichier (stdout, stderr). Le fichier sera cree uniquement
    si il est dans le work_dir et si il n'existe pas deja.

=head2 USAGE

    Sched::Cmd::_touch('file', uid, gid) ;

=head2 DEBUG
    
    utilise 25 en niveau d'erreur
    
=head2 RETURN
    
    0 : ERR
    1 : OK

=cut

use File::Basename ;
use Cwd ;

sub _touch
{
    my ($file, $uid, $gid) = @_ ;
    
    # on ne peut pas creer un fichier ""
    unless ($file) {
	$Sched::log->write("E : error with file ($file)", 25) ;
	return 0 ;
    }

    # on ne touche pas un fichier existant
    if (stat($file)) {
	$Sched::log->write("E : file already exists ($file)", 25) ;
	return 0 ;
    }

    # pas de / dans le nom du fichier
    if ($file =~ m!/!) {
	$Sched::log->write("E : path file not in pwd ($file))", 25) ;
	return 0 ;
    }

    # creation du fichier
    unless (open(FP, ">$file")) {
	$Sched::log->write("E : error openning $file", 25) ;
	return 0 ;
    }
    close(FP) ;

    # chmod si $uid et $group sont definis
    if (defined $uid and defined $gid) {
	$Sched::log->write("D : touching $file ($uid, $gid)", 25) ;
	chown $uid, $gid, $file ;
    }

    return 1 ;
}

=head1 FUNCTION Sched::Cmd::run

Lancement de la commande

=head2 USAGE

    $obj->run() ;

=head2 RETURN

    0 : ERR
    1 : OK

=cut

sub run ( $ )
{
    # la fonction die est annul par la boucle Event...
    sub my_die {
	my $txt = shift ;
	print STDERR $txt, "\n" ;
	$Sched::log->write($txt) ;
	exit (1) ;
    }

    my ($self) = @_ ;

    my $sid = $self->attribute('sid') ;

    my $cmd = $self->attribute('cmdline');

    $Sched::log->write("D : $sid : run $cmd", 15) ; 

    $self->attribute('state', 'running') ;
    $self->attribute('start_date', time()) ;

    if (defined $self->attribute('host') 
	and 
	$self->attribute('host') ne $Sched::hostname)
    {
	&Sched::Net::send_to_master("CMD " . $self->attribute('id')) ;
	return 1 ;
    }

    # preparation de l'environnement
    # les variables TASK overwrite celle du job (si y'en a un)
    my $job = Sched::NS::aa('=xml-root=') || $self->{xml} ;
    my %env = (get_env($job), get_env($self->{xml})) ;

    my $pipe = new IO::Pipe ;

    my $pid = fork() ;

    if ($pid > 0) {
	$Sched::log->write("I : [$sid] SPAWN $pid", 10);

	$self->attribute('pid', $pid) ;

	# mise en place d'un index
	$objects_pid{$pid} = $self ;

	# sync with child
	$pipe->writer() ;
	$pipe->autoflush(1) ;
	print $pipe "OK\n" ;
	$pipe->close() ;

    } elsif($pid == 0) {

	# sync with father
	$pipe->reader() ;
	$pipe->autoflush(1) ;
	my $tmp = <$pipe> ;
	$pipe->close() ;

	# trap die croak
	$SIG{__DIE__} = \&my_die ;

	my $user =  $self->attribute('user') || 'nobody' ;
	my $uid = getpwnam($user) ;

	my $group = $self->attribute('group') || 'nogroup' ;
	my $gid = getgrnam($group) ;
	
	for my $f (qw/stdout stderr/) {	    
	    _touch($self->attribute($f),  $uid,  $gid) ;
	}

	if ($> == 0) {
	    if ( not defined $uid)
	    {
	      my_die "E : [$sid] Impossible d'utiliser $user ($!)";
	    }

	    $> = $< =  $uid ;
	}

	$( = $) =  $gid if (defined $gid) ;

	# mise en place des variables ENV
	Sched::Var::add_var('ID', $self->attribute('id')) ;
	map { $ENV{$_} = Sched::Var::expand($env{$_}) } keys %env ;

	# TODO : Bug ici...
	# as t'on le droit d'ecrire le fichier ici...
	# si on ecrit dans le work_dir/serial le user
	# du task a peu de chance de pouvoir ecrire
	#
	# il faut ouvrir le fichier avant
	#  - si le fichier est dans work_dir
	#	ouverture (+ chown)
	#       attention aux liens... (passwd -> /etc/passwd) quand je l'ouvre
	#       avec root kaikai
	#  - si le fichier n'est pas dans work_dir
	#	ouverture avec le user choisi

	close(STDIN) ; 
	open(STDIN, $self->attribute('stdin')) or 
 my_die "E : [$sid] ouverture stdin [" . $self->attribute('stdin') . "] ($!)" ;

	close(STDOUT) ;
	open(STDOUT, ">>" . $self->attribute('stdout')) or 
 my_die "E : [$sid] ouverture stdout [" . $self->attribute('stdout') ."] ($!)";

	close(STDERR) ;
	open(STDERR, ">>" . $self->attribute('stderr')) or 
 my_die "E : [$sid] ouverture stderr [" . $self->attribute('stderr') ."] ($!)";

	Sched::Net::master_close('now') ;

	exec ($cmd) if ($cmd) ;

	my_die "E : [$sid] execution ($!) de " . $cmd ;

    } else {
	$self->attribute('state', 'failed') ;
	$self->attribute('status', '1') ;
	$self->attribute('info', "$?") ;
	return 0 ;
    }

    return 1 ;
}



=head1 FUNCTION Sched::Cmd::wait

Recupere et met a jour la sortie d'un Cmd

=head2 USAGE

    $obj = Sched::Cmd::wait() ;

=head2 RETURN

    $obj  : OK
    undef : ERR

=cut

use POSIX qw(:sys_wait_h) ;

sub wait
{
    my ($self) = @_ ;
    my $pid_wait = ($self?$self->attribute('pid'):-1) ;

    my $pid = POSIX::waitpid( $pid_wait, POSIX::WNOHANG );

    if ($pid > 0) {
	my $p = $objects_pid{$pid} or 
	    die "E : processus inattendu" ;
	delete $objects_pid{$pid} ;
	Sched::NS::unregister_id($p->attribute('sid')) ;

	$Sched::log->write("I : wait ". $p->attribute('sid'), 10) ;
	$p->attribute('status', sprintf("%d", $?)) ;
	$p->attribute('state',"finish") ;
	$p->attribute('end_date', time()) ;
	return $p ;
    }
    return undef ;
}

=head1 FUNCTION Sched::Cmd::cancel

Envoi un signal a un Cmd. Au premier signal, le status
du Cmd passe a 'kill'.
Si un signal a deja ete envoye, le processus est tue.

Le status du Cmd passe a kill, il ne peut plus etre lance

=head2 USAGE

    $obj->cancel($signal)
	or
    $obj->cancel() ;

=cut


sub cancel
{
    my ($self, $signal) = @_ ;
    
    if (!$signal) {
	$signal = 15 ;

	if ($self->attribute('state') eq "kill") {
	    $signal = 9 ;
	}
    }

    if ($signal == 15) {
	$self->attribute('state', 'kill') ;
    }

    $self->attribute('status',127) ;	# status provisoir

    $self->attribute('info', "Canceled") ;

    # on ne peut killer qu'un processus en vie (sur la machine)
    if (defined $self->attribute('host') 
	and 
	$self->attribute('host') ne $Sched::hostname)
    {
	&Sched::Net::send_to_master("CANCEL " . $self->attribute('id'),
				    'cancel', "CMD " . $self->attribute('id'));
    } else {
	if (defined $self->attribute('pid') 
	    and 
	    ($self->attribute('state') =~ /kill|running/)) 
	{
	    kill($signal, $self->attribute('pid')) ;
	}
    }
}

=head1 FUNCTION &Sched::Cmd::to_xml

  Cette fonction exporte un Sched::Cmd en XML

=head2 USAGE

  $string = $obj->to_xml() ;

=cut

sub to_xml
{
    my ($self) = @_ ;
    

    my $ret = "<task " ;

    for my $k (keys %{ $self->{xml}->{'_attributes'} }) {
    	if (($k eq 'cmdline') # la conversion a deja ete faite 
	    and (   (! $self->attribute('state'))   # pas de state
		    or ($self->attribute('state') eq "new") # pas lancee
		)
	   )
	{
	  my $cmd = $self->attribute('cmdline');
	  $cmd = Sched::Var::expand($cmd) ; 
	  $cmd = MIME::Base64::encode_base64($cmd, ''); chomp($cmd) ;
	  $ret .= " $k='$cmd' " ;
	} else {
	  $ret .= " $k='" . $self->{xml}->attribute($k) . "' " 
	      if ($self->{xml}->attribute($k) ne '') ;
	}
    }

    $ret .= " />" ;

    return $ret ;
}


=head1 FUNCTION Sched::Cmd::reset_task

    Reset de toutes les sous taches

=head2 USAGE

    $t = Sched::Cmd::find_task_by_id($root, $id) ;
    Sched::Cmd::reset_task($xml_node, $stop) ;

    Stop est optionnel, il permet de ne pas etre recursif

=cut

sub reset_task
{
    my $root = shift ;
    my $stop = shift ;

    return unless ($root) ;

    for my $a (qw/state status pid start_date end_date info serial sid/) {
	if (defined $root->{_attributes}->{$a}) {
	    delete $root->{_attributes}->{$a} ;
	}
    }

    return if ($stop) ;

    for my $t (@{ $root->getAllChildren('task') }) {
	reset_task($t) ;
    }
}



=head1 FUNCTION &Sched::Cmd::find_task_by_id

  Parcours un arbre xml et trouve une task par son id
    (a deplacer...)

=cut

sub find_task_by_id
{
    my ($root, $id) = @_ ;
    
    my $ret ;

    for my $t (@{ $root->getAllChildren('task') }) {
	return $t if ($id eq $t->attribute('id')) ;
	$ret = find_task_by_id($t, $id) ;
	return $ret if (defined $ret) ;
    }
    return undef ;
}

=head1 FUNCTION Sched::Cmd::dispose

  Permet de relacher cet objet

=cut

sub dispose
{
    my ($self) = @_ ;
    Sched::NS::unregister_id($self->attribute('sid')) ;
}
   
sub test
{

}
1;

__END__

=head1 AUTHOR

(C) 2004-2005 Eric Bollengier

You may reach me through the contact info at eric@eb.homelinux.org

=head1 LICENSE

    Part of the network scheduling system (Sched)
    Copyright (C) 2004-2005 Eric Bollengier
        All rights reserved.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

=cut

