package SchedUI::ItemEdit ;
use strict ;

=head1 PACKAGE SchedUI::ItemEdit

    Ce package sert de base pour l'edition des objets (Item, Job...)

=head2 USAGE

    Il faut remplir les listes (entry, text, spin, encode)

    Les noms des widgets sont de la forme entry_item1, text_spin1...
    pour savoir si il faut indiquer dans le champs encode les bons items

    La boite de dialog Glade porte le nom dlg_(node)
    <node1 /> ==> dlg_node1

    Apres modification des parametres, le Sched::Callback CHANGE uid
    permet de prendre en compte les modifications. (sur les champs XML)

=head2 ATTRIBUTE

    entry (rl) = [qw /item1 item2/] 
    text (rl) = [qw /item1 item2/]
    spin (rl) = [qw /item1 item2/]
    encode (rl) = [qw /item1 item2/]
    
    uid (s) : to submit a event
    xml (r) : XML::Mini node
    
    glade (r) : XML::Mini

=cut

our $version = '$Id: ItemEdit.pm,v 1.1 2005/04/05 19:45:18 mcgregor Exp $' ;

use Sched ;
use SchedUI::Msg ;
use Sched::Callback ;
use Gtk2::GladeXML ;

sub new
{
    my ($class, $uid, $xml) = @_ ;

    my $self ;
    $self->{entry} = ['id'] ;
    $self->{text} = [] ;
    $self->{spin} = [] ;
    $self->{encode} = [] ;

    $self->{verify} = { 
	id    => '^[\w\d\.-]+$',
	user  => '^[${}\w\d\.-]*$',
	group => '^[${}\w\d\.-]*$',
	host  => '^[${}\w\d\.-]*$',
	stdin => '^[${}/\w\d\.,-]*$',
	stdout => '^[${}/\w\d\.,-]*$',
	stderr => '^[${}/\w\d\.,-]*$',
	'use'  =>  '^([\w\d\.-]+)?(,[\w\d\.-]+)*$'
    } ;

    $self->{uid} = $uid ;

    $self->{xml} = $xml ;

    my $what = $xml->name ;

    $self->{glade} = Gtk2::GladeXML->new($SchedUI::glade_file, "dlg_$what") ;

    bless($self) ;
    return $self ;
}

# set_default(ItemEdit [,XML::Mini])
sub set_default
{
    my ($self, $xml) = @_ ;

    if (not defined $xml) {
	$xml = $self->{xml} ;
    }

    for my $i (@{$self->{entry}}) {
	my $w = $self->get_widget("entry_$i") ;
	$w->set_text($self->decode($i ,$xml->attribute($i))) 
	    if (defined $xml->attribute($i)) ;
    }

    for my $i (@{$self->{text}}) {
	my $w = $self->get_widget("text_$i") ;
	my $b = new Gtk2::TextBuffer() ;
	
	$b->set_text($self->decode($i, $xml->attribute($i)))
	    if (defined $xml->attribute($i)) ;
	$w->set_buffer($b) ;
    }

    for my $i (@{$self->{spin}}) {
	my $w = $self->get_widget("spin_$i") ;
	$w->set_value($xml->attribute($i)) if (defined $xml->attribute($i)) ;
    }
}

sub encode
{
    my ($self, $i, $str) = @_ ;
    return '' if (!$str) ;

    if (grep(/^$i$/, @{$self->{encode}})) {
	return MIME::Base64::encode_base64($str, '') ;
    } else {
	return $str ;
    }
}


sub decode
{
    my ($self, $i, $str) = @_ ;

    if (grep(/^$i$/, @{$self->{encode}})) {
	return MIME::Base64::decode_base64($str) ;
    } else {
	return $str ;
    }
}

sub on_cancel_clicked
{
    my $self = shift ;
    my $t = $self->{xml}->name ;
    my $w = $self->get_widget("dlg_$t") ;
    $w->destroy() ;
    delete $self->{glade} ;
    delete $self->{xml} ;
}
sub on_ok_clicked
{
    my $self = shift ;
    my $xml = $self->{xml} ;

    my $ok_to_apply = 1 ;
    my $err_msg = '' ;
    my %apply ;

    # we must verify user input

    for my $i (@{$self->{entry}}) {
	my $w = $self->get_widget("entry_$i") ;
	my $t =  $w->get_text() ;

	if ($self->{verify}{$i}) {
	    my $re = $self->{verify}{$i} ;
	    if ($t !~ $re) {
		$ok_to_apply = 0 ;
		$err_msg .= "Value $i ($t) don't match $re\n" ; 
	    }
	}

	$t = $self->encode($i, $t) ;
	if ($t) {
	    $apply{$i} = $t ;
	} elsif (defined $xml->attribute($i)) {
	    delete $xml->{_attributes}->{$i} ;
	}
    }

    for my $i (@{$self->{text}}) {
	my $w = $self->get_widget("text_$i") ;
	my $b = $w->get_buffer() ;
	my $s = $b->get_iter_at_offset(0) ;
	my $e = $b->get_iter_at_offset(-1) ;

	my $t = $b->get_text($s, $e, 1) ;

	if ($self->{verify}{$i}) {
	    my $re = $self->{verify}{$i} ;
	    if ($t !~ $re) {
		$ok_to_apply = 0 ;
		$err_msg .= "Value $i ($t) don't match $re\n" ; 
	    }
	}

	$t = $self->encode($i, $t) ;
	if ($t) {
	    $apply{$i} = $t ;
	} elsif (defined $xml->attribute($i)) {
	    delete $xml->{_attributes}->{$i} ;
	}
    }

    for my $i (@{$self->{spin}}) {
	my $w = $self->get_widget("spin_$i") ;
	my $v = $w->get_value() ;
	$apply{$i} = $w->get_value() if ($v) ;
    }

    if ($ok_to_apply) {
	map { $xml->attribute($_, $apply{$_}) } keys %apply ;

	Sched::Callback::recall_cb("CHANGE " . $self->{uid}) 
	    if ($self->{uid}) ;

	$self->on_cancel_clicked() ;

    } else {
	SchedUI::Msg::display_warning($err_msg) ;
    }
}
sub get_widget
{
    my ($self, $n) = @_ ;
    return $self->{glade}->get_widget($n) ;
}

sub show
{
   my ($self) = @_ ;
   my $t = $self->{xml}->name ;
   my $w = $self->{glade}->get_widget("dlg_$t") ;
   $w->show() ;
}

sub init
{
    my ($self) = @_ ;
    $self->{glade}->signal_autoconnect_from_package($self) ;
    $self->set_default() ;
    $self->show() ;
}


sub test
{

}

1;

__END__

=head1 AUTHOR

(C) 2004-2005 Eric Bollengier

You may reach me through the contact info at eric@eb.homelinux.org

=head1 LICENSE

    Part of the network scheduling system (Sched)
    Copyright (C) 2004-2005 Eric Bollengier All rights reserved.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

=cut

