/*

    ScatterJn, version 1.0
    ImageJ plugin for the evaluation of analytical microscopy data

    Copyright (C) 2015 Fabian Zeitvogel, University of Tuebingen

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/


/*

This plugin is based on the plugin ScatterJ (http://download.savannah.gnu.org/releases/scatterj/), including part of the source code

*/


import java.io.*;
import ij.*;
import ij.io.*;
import ij.gui.*;
import ij.plugin.*;
import ij.process.*;
import java.util.*;
import ij.plugin.frame.*;
import java.awt.*;
import java.awt.event.*;
import java.math.*;
import ij.text.*; 


public class ScatterJn_ extends PlugInFrame implements ActionListener {

//========== global variables ===========

String str_title="ScatterJn";
String str_version="version 1.0";

int w,h,d,n,dim=3,n_px;
int x,y,z,i;
int n_roi_sp=0;
int n_roi_map=0;
ImagePlus sp_img,map_img;
ij.measure.Calibration img_calibration;

short[][][][] data;
boolean[][][] mask;
boolean[][][] mask_projection;	// projection of mask into spatial domain
boolean[][][] mask_spatial;		// the spatial-domain mask created form a mask image
int[][] origins;		// positions of the origins of the scatterplots in the matrix image
int[][][] sp;			// the scatterplots
int[][] sp_number;		// index number of scatterplot in depndence on the coordinates in the scatterplot matrix
int components[][];		// translates scatterplot index back into indices of components

Roi[] ROIs;		// stores ROIs drawn in scatterplot matrix image
Roi map_ROI;	// stores ROI drawn in map image
int map_ROI_start=0,map_ROI_end=0;

int sp_length=256;
int img_in_map=0;
int ROIcolour=0x0000FF;
int px_colour=0xFF0000;
String[] img_titles=new String[1];
boolean nb_in_sp=false,titles_in_sp=true;
boolean input_selected=false;
int reduction=1;


// labels of buttons in main window:

String str_select="Select input images";
String str_settings="Settings";

String str_acceptSP="Accept ROI";
String str_removeSP="Remove ROI";
String str_angular="Angular distance map";
String str_resetSP="Reset";

String str_acceptMap="Accept ROI (map)";
String str_mask="Apply mask";
String str_includeall="Remove ROI (map)";

String str_info="Info & manual";

int[] colourscale_default_greyscale	=	{0x111111,0x333333,0x555555,0x777777,0x999999,0xaaaaaa,0xbbbbbb,0xcccccc,0xdddddd};
int[] colourscale_default_RGB	=   	{0x200080,0x0000ff,0x00e0ff,0x008080,0x00e000,0xffff00,0xff8000,0xff0000,0x800000};
int[] thresholds_default	=	{1,2,4,8,16,32,64,128,256};
int[] colourscale			=	{0x111111,0x333333,0x555555,0x777777,0x999999,0xaaaaaa,0xbbbbbb,0xcccccc,0xdddddd};
int[] thresholds			=	{1,2,4,8,16,32,64,128,256};

Panel p1;
TextArea ta1;
Label label_nb_px;

static int _X=0,_Y=1;
int xoffset=50,yoffset=50;

//=======================================



public ScatterJn_() {

// build up main window

	super("ScatterJn");
	GridBagLayout gbl=new GridBagLayout();
	GridBagConstraints gbc=new GridBagConstraints();
	
	p1=new Panel(); p1.setLayout(gbl);
	
	Label l1=new Label(str_title);
	p1.add(l1);
	gbc.gridwidth=GridBagConstraints.REMAINDER;
	gbc.fill=GridBagConstraints.HORIZONTAL;
	gbc.anchor=GridBagConstraints.LINE_START;
	gbl.setConstraints(l1,gbc);

	Label l1b=new Label(str_version);
	p1.add(l1b);
	gbc.anchor=GridBagConstraints.LINE_START;
	gbl.setConstraints(l1b,gbc);

	Button bselect=new Button(str_select);
	bselect.addActionListener(this);
	p1.add(bselect);
	gbl.setConstraints(bselect,gbc);

	Button bsettings=new Button(str_settings);
	bsettings.addActionListener(this);
	p1.add(bsettings);
	gbl.setConstraints(bsettings,gbc);

	ta1=new TextArea("Selected input images\n\nnone",5,30,TextArea.SCROLLBARS_BOTH);
	ta1.setEditable(false);
	p1.add(ta1);
	gbl.setConstraints(ta1,gbc);

	Label l2=new Label("Scatterplot matrix:");
	p1.add(l2);
	gbl.setConstraints(l2,gbc);

	Button bacceptSP=new Button(str_acceptSP);
	bacceptSP.addActionListener(this);
	p1.add(bacceptSP);
	gbl.setConstraints(bacceptSP,gbc);

	Button bremoveSP=new Button(str_removeSP);
	bremoveSP.addActionListener(this);
	p1.add(bremoveSP);
	gbl.setConstraints(bremoveSP,gbc);

	Button bangular=new Button(str_angular);
	bangular.addActionListener(this);
	p1.add(bangular);
	gbl.setConstraints(bangular,gbc);

	Button breset=new Button(str_resetSP);
	breset.addActionListener(this);
	p1.add(breset);
	gbl.setConstraints(breset,gbc);

	Label l3=new Label("Spatial-domain map:");
	p1.add(l3);
	gbl.setConstraints(l3,gbc);

	Button bincludeROI=new Button(str_acceptMap);
	bincludeROI.addActionListener(this);
	p1.add(bincludeROI);
	gbl.setConstraints(bincludeROI,gbc);

	Button bmask=new Button(str_mask);
	bmask.addActionListener(this);
	p1.add(bmask);
	gbl.setConstraints(bmask,gbc);

	Button bincludeall=new Button(str_includeall);
	bincludeall.addActionListener(this);
	p1.add(bincludeall);
	gbl.setConstraints(bincludeall,gbc);

	label_nb_px=new Label("0 datapoints in selection");
	p1.add(label_nb_px);
	gbl.setConstraints(label_nb_px,gbc);

	Button binfo=new Button(str_info);
	binfo.addActionListener(this);
	p1.add(binfo);
	gbl.setConstraints(binfo,gbc);

	add(p1);

	pack();
	GUI.center(this);
	show();

} // ScatterJ_2



public void actionPerformed(ActionEvent e) {

	String action=e.getActionCommand();

	
	if (action.equals(str_select)) { 
		select_input();
	}

	if (action.equals(str_settings)) { 
		settings();
	}

	if (action.equals(str_acceptSP)) { 
		accept_ROI();
	}

	if (action.equals(str_removeSP)) { 
		remove_ROI();
	}

	if (action.equals(str_angular)) { 
		angular_distance_map();
	}

	if (action.equals(str_resetSP)) { 
		reset_sp();
	}

	if (action.equals("Accept ROI (map)")) { 
		accept_ROI_spatial();
	}

	if (action.equals("Apply mask")) { 
		apply_mask();
	}

	if (action.equals("Remove ROI (map)")) { 
		remove_ROI_spatial();
	}


	if (action.equals(str_info)) { 
		show_info(); 
	}


} // actionPerformed


public void run(String arg) {

} // run


public void select_input(){

// select input images using a series of two dialogs

	int i,w1,h1,d1;
	short a;
	int[] img_index;
	ImagePlus[] imgs;
	ImageProcessor ip;
	ImageStack st;
	String s;

	int dim_old;
	boolean keep_titles;
	String[] old_titles;

	dim_old=dim;

// display dialog to define number of input images
	
	GenericDialog gd=new GenericDialog("Select input images");
	gd.addNumericField("Number of images:",dim,0);
	gd.showDialog();
	if (gd.wasCanceled()) return;

	dim=(int)gd.getNextNumber();
	if (dim<2) {
		IJ.showMessage(str_title,"Please select at least 2 images");
		return;
	} // if

	keep_titles=((dim_old==dim)&&(img_titles.length==dim));

	int[] wl=WindowManager.getIDList();
	if ((wl==null)||(wl.length<dim)) { 
		IJ.showMessage(str_title,"Could not find "+dim+" open images.");
		return;
	} //if

// display dialog to select input images
	
	String[] titles=new String[wl.length];
	for (i=0; i<wl.length; i++) {
		ImagePlus img=WindowManager.getImage(wl[i]);
		if (img!=null) titles[i]=img.getTitle();
		else titles[i]="";
	} //for

	gd=new GenericDialog("Select input images");
	for (i=0;i<dim;i++){
		gd.addChoice("Image "+(i+1)+":",titles,titles[i]);
	}
	gd.addSlider("Histogram binning:",1,4,reduction);
	if (keep_titles) gd.addCheckbox("Keep previous image titles",false);
	gd.showDialog();
	if (gd.wasCanceled()) return;

	if (keep_titles) keep_titles=(gd.getNextBoolean());
	old_titles=new String[dim];
	if (keep_titles){
			for (i=0;i<dim;i++) old_titles[i]=img_titles[i];
	} // if keep_titles

	imgs=new ImagePlus[dim];
	img_titles=new String[dim];

	s="Input images:\n";
	img_index=new int[dim];
	for (i=0;i<dim;i++){

		img_index[i]=gd.getNextChoiceIndex();
		imgs[i]=WindowManager.getImage(wl[img_index[i]]);
		img_titles[i]=(keep_titles)?(old_titles[i]):(titles[img_index[i]]);
		s+="\n"+(i+1)+": "+titles[img_index[i]];
	} // for

	reduction=(int)gd.getNextNumber();
	if (reduction<1) reduction=1;
	if (reduction>255) reduction=255;
	sp_length=(int)255/reduction+1;

// check if images are suitable	
	
	for (i=0;i<dim;i++){

		ip=imgs[i].getProcessor();
		w1=ip.getWidth();
		h1=ip.getHeight();
		d1=imgs[i].getStack().getSize();

		if (i==0) {
			w=w1; h=h1; d=d1;
		} // if i==0
		else {
			if ((w1!=w)||(h1!=h)||(d1!=d)) {
				IJ.showMessage("Image "+(i+1)+":\ndimensions don't fit");
				return;
			} // if (!=)
		} // else

		if (imgs[i].getType()!=ImagePlus.GRAY8)  {
			IJ.showMessage("Image "+(i+1)+" is not 8-bit greyscale");
			return;
		} // if

		if 	(imgs[i].getStack().isVirtual()) {
			IJ.showMessage("Image "+(i+1)+" is a virtual stack.\n \nScatterJ cannot use virtual stacks as input images.\n \nTo convert a virtual into a non-virtual stack,\n - duplicate it, or\n - save and re-load it.");
			return;
		} // if

	} // for

	ta1.setText(s);
	
// form datapoint from grey values of images
	
	data=new short[w][h][d][dim];
	mask_spatial=new boolean[w][h][d];

	for (i=0;i<dim;i++){
		st=imgs[i].getStack();

		for (x=0;x<w;x++){
		for (y=0;y<h;y++){
		for (z=0;z<d;z++){
			a=(short)st.getVoxel(x,y,z);
			a=(short)(a/reduction);		// binning of grey values by reduction factor
			data[x][y][z][i]=a;
			if (i==0) mask_spatial[x][y][z]=true;
		}}} // for x,y,z
		
		IJ.showStatus("Creating data matrix... "+(100*i/dim)+"%");	

	} // for i

	if (img_in_map>=dim) img_in_map=0;

	img_calibration=imgs[0].getCalibration();
	input_selected=true;

	calculate_scatterplots();
	draw_sp_matrix_blank(true);	// true: create new image
	draw_map_blank(true);		// ture: create new image
	reset_map_ROI(true);		// true: create new mask_spatial array

} // select_input


private void calculate_scatterplots(){

// calcualte scatterplots from datapoints

	int i,k,l;

	n=0;
	for (i=0;i<dim;i++) n+=i;

// create nwe scatterplot matrix
	
	sp=new int[n][sp_length][sp_length];

	for (i=0;i<n;i++){
	for (k=0;k<sp_length;k++){
	for (l=0;l<sp_length;l++){
		sp[i][k][l]=0;
	}}} // for i,k,l

// calculate index and position for each scatterplot
	
	origins=new int[n][2];
	sp_number=new int[dim][dim];
	components=new int[n][2];

	i=-1;
	for (x=0;x<dim;x++){
	for (y=x+1;y<dim;y++){
		i++;
		sp_number[x][y]=i;
		components[i][_X]=x;
		components[i][_Y]=y;
		origins[i][_X]=xoffset+x*(sp_length+1)+1;
		origins[i][_Y]=yoffset+y*(sp_length)+y*1;
	}} // for x,y
	
// calculate actual scatterplots
	
	for (x=0;x<w;x++){
	for (y=0;y<h;y++){
	for (z=0;z<d;z++){

		for (k=0;k<dim;k++){
		for (l=k+1;l<dim;l++){
			sp [sp_number[k][l]] [data[x][y][z][k]] [data[x][y][z][l]] ++;
		}} // for k,l

	}} // for y,z
	IJ.showStatus("Calcualting scatterplots... "+(100*x/w)+"%");
	} // for x

	clear_SP_ROIs();

} // calculate_scatterplots


private void draw_sp_matrix_blank(boolean newimg){

// draw the "blank" scatterplot matrix into the scatterplot matrix image
// i.e. the framework of axes, the titles, etc. without the datapoints
// newimage=ture enforces creation of a new image

	ImageProcessor ip,ip1;
	int a=(dim-1)*(sp_length+1)+1;
	int x0,y0,x1,x2,y1,y2,k;
	String s;
	ImagePlus dummy;
	java.awt.Polygon p;

	if (sp_img!=null){
		if (!sp_img.isVisible()) newimg=true;
	}

	if ((newimg)&&(sp_img!=null)) sp_img.close();
	if ((newimg)||(sp_img==null)){
		sp_img=NewImage.createRGBImage("Scatterplot matrix",a+2*xoffset,a+2*yoffset,1,NewImage.FILL_WHITE);
		sp_img.show();	
	}
	
// draw frames
	
	ip=sp_img.getProcessor();
	ip.setColor(0);
	ip.setLineWidth(1);

	for (k=1;k<dim;k++){
	
		x1=xoffset;
		x2=xoffset+k*(sp_length+1);
		y1=yoffset+(k-1)*(sp_length+1);
		y2=yoffset+k*(sp_length+1);

		ip.drawLine(x1,y1,x2,y1);
		ip.drawLine(x1,y2,x2,y2);
		
		y1=yoffset+(k-1)*(sp_length+1);
		y2=yoffset+(dim-1)*(sp_length+1);
		x1=xoffset+(k-1)*(sp_length+1);
		x2=xoffset+k*(sp_length+1);

		ip.drawLine(x1,y1,x1,y2);
		ip.drawLine(x2,y1,x2,y2);

	} // for k

	for (i=0;i<n;i++) draw_single_sp(i);

// delete old scatterplot titles from image by overwriting with white rectangle

	ip.setColor(0xFFFFFF);

	p=new Polygon();
	x1=0; y1=0; x2=xoffset; y2=ip.getHeight()-1;
	p.addPoint(x1,y1);
	p.addPoint(x1,y2);
	p.addPoint(x2,y2);
	p.addPoint(x2,y1);
	ip.fillPolygon(p);

	p=new Polygon();
	x1=0; y1=ip.getHeight()-1; x2=ip.getWidth()-1; y2=y1-yoffset+1;
	p.addPoint(x1,y1);
	p.addPoint(x1,y2);
	p.addPoint(x2,y2);
	p.addPoint(x2,y1);
	ip.fillPolygon(p);

	ip.setColor(0);

	if (titles_in_sp){

// draw horizontal titles directly

		for (i=0;i<dim-1;i++){
			x0=xoffset+i*(sp_length+1)+1;	// same as in calculation of origins
			y0=yoffset+(dim-1)*(sp_length+1)+1;
			s=img_titles[i];
			x1=x0+sp_length/2-ip.getStringWidth(s)/2;
			y1=y0+25;
			ip.drawString(s,x1,y1);
		} // for i

// vertical titles are first drawn into an invisibly dummy image
// and then roatated and copied into the scatterplot image

		for (i=1;i<dim;i++){
			s=img_titles[i];
			x0=0;
			x1=30;
			y0=yoffset+i*(sp_length+1)+1;
			y1=ip.getStringWidth(s);
			y2=y0-sp_length/2+y1/2;
			dummy=NewImage.createRGBImage("",y1,x1,1,NewImage.FILL_WHITE);
			ip1=dummy.getProcessor();
			ip1.setColor(0);
			ip1.drawString(s,0,x1-1);
			for (x=0;x<y1;x++){
			for (y=0;y<x1;y++){
				if (ip1.getPixel(x,y)==0) ip.putPixel(x0+y,y2-x,0);
			}} // for x,y
			dummy=null;
		} // for i

	} // if titles_in_sp

	sp_img.updateAndDraw();

} // draw sp_matrix_blank


private void draw_single_sp(int i){

// draws the datapoints of one single scatterplot into the scatterplot matrix image

	int x0,y0,c,a;
	ImageProcessor ip=sp_img.getProcessor();

	x0=origins[i][_X];
	y0=origins[i][_Y];

	for (x=0;x<sp_length;x++){
	for (y=0;y<sp_length;y++){
		a=sp[i][x][y];
		c=(a==0)?0xFFFFFF:get_datapoint_colour(sp[i][x][y]);
		if (sp[i][x][y]==0) c=0xFFFFFF;
		ip.putPixel(x0+x,y0-y-1,c);	
	}} // for x,y

	if (nb_in_sp) {
		ip.setColor(0);
		ip.drawString(""+(i+1),x0+10,y0-sp_length-1+20);

	} // if nb_in_sp

} // draw_single_sp


private int get_datapoint_colour(int a){

// returns the colour assigned to a certain value by the colourscale

	if (a==0) return 0xFFFFFF;
	int i=0;
	int l=thresholds.length;
	
	while (i<l){
		if (a<thresholds[i]) return colourscale[i-1];
		i++;
	} // while
	
	return colourscale[i-1];

} // get_datapoint_colour


private void accept_ROI(){

// update classification using a ROI drawn in one of the scatterplots in the scatterplot matrix image

	ImageProcessor ip=sp_img.getProcessor();
	Roi r=sp_img.getRoi();
	int x0,y0,x1,x2,y1,y2,i,i_sp=0;
	java.awt.Rectangle ra;
	boolean b;

// check if ROI exists in scatterplot image	

	if (r==null){
		IJ.showMessage("No ROI selected");
		return;
	} 

	ra=r.getBounds();
	boolean found=false;
	i=0;

// determine scatterplot that contains the ROI	
	
	while ((i<n)&&(!found)){
		x1=origins[i][_X];
	//	x2=x1+sp_length-1;
		x2=x1+sp_length;	
		y2=origins[i][_Y];
		y1=y2-sp_length;
		
		if ((ra.x>=x1)&&(ra.x<=x2)&&(ra.y>=y1)&&(ra.y<=y2)) found=true;
		
		if (found){

		// check if ROI exceeds the boundaries of the scatterplot

			if ((ra.x+ra.width>x2)||(ra.y+ra.height>y2)) { // i.e. if rectangle exceeds scatterplot
				IJ.showMessage("Please make sure the ROI lies completely\nwithin one of the scatterplots");
				return;
			} // if

			i_sp=i;

		} // if found

		i++;
	} // while
		
	if (!found){
		IJ.showMessage("Please make sure the ROI lies completely\nwithin one of the scatterplots");
		return;
	} 

// copy ROI into mask
	
	if (ROIs[i_sp]!=null) delete_SP_ROI_from_matrix(i_sp);

	x0=origins[i_sp][_X];
	y0=origins[i_sp][_Y];

	for (x=0;x<sp_length;x++){
	for (y=0;y<sp_length;y++){
		x1=x0+x;
		y1=y0-y-1;
		mask[i_sp][x][y]=(r.contains(x1,y1));
	}} // for x,y

	redraw_SP_matrix();
	draw_map_blank(false);

	ROIs[i_sp]=r;

	sp_img.setRoi(r);
	n_roi_sp++;

	update_mask_projection();

}	// accept_ROI


private void delete_SP_ROI_from_matrix(int i){

// clears the roi in SP number i
// this does not affect the display of the SP matrix etc.
// nor does it recalculate mask_projection

	for (x=0;x<sp_length;x++){
	for (y=0;y<sp_length;y++){

		mask[i][x][y]=true;

	}} // for i,x,y


	for (x=0;x<w;x++){
	for (y=0;y<h;y++){
	for (z=0;z<d;z++){

	 	mask_projection[x][y][z]=true;

	}}} // for x,y,z

	ROIs[i]=null;
	n_roi_sp--;

} // clear_single_roi_sp


private void update_mask_projection(){

// updates the variable mask_projection 
// after selections in the scatterplot matrix have changed 
// and after selections in the spatial-domain image have changed
// mask_projection holds the projection of the selection in the scatterplot matrix 
// into spatial domain combined with the selection made directly in the spatial-domain image

	ImageProcessor 	ip=sp_img.getProcessor();
	ImageStack 		st=map_img.getStack();
	
	int x1,y1;	
	boolean b;

	n_px=0; 	// n_px hold the number of datapoints in the selection
	label_nb_px.setText(""+n_px+" datapoints in selection");

	if ((n_roi_sp==0)&&(n_roi_map==0)) return;


	for (x=0;x<w;x++){
	for (y=0;y<h;y++){
	for (z=0;z<d;z++){

		// for each pixel/voxel in the spatial-domain map,
		// check if (x,y,z) within all SP-ROIs:

		b=((mask_projection[x][y][z]) && (mask_spatial[x][y][z]));
			// mask_projection is all "true" if a ROI has been removed
			// if a ROI has been added, it contains the former projection
			// thus this step should save some time

		i=0;
		while ((b)&&(i<n)){
			b=( (b) && (mask [ i ]  [ data[x][y][z][components[i][_X]] ] [ data[x][y][z][components[i][_Y]] ]) );
			i++;
		} // while	
		
		mask_projection[x][y][z]=b;

		// if so, draw pixel in all SPs:

		if (b){
			for (i=0;i<n;i++){
				x1=origins[i][_X]+data[x][y][z][components[i][_X]];
				y1=origins[i][_Y]-data[x][y][z][components[i][_Y]]-1;
			 	ip.putPixel(x1,y1,px_colour);	// draw pixel (x1,y1) into SP i
			} // for i
			st.setVoxel(x,y,z,px_colour);		// draw voxel (x,y,z) into spatial-domain map 
			n_px++;
		} // if b

	}}} // for x,y,z

	draw_SP_ROI_outlines();
	draw_map_ROI_outline();
	label_nb_px.setText(""+n_px+" datapoints in selection");

	sp_img.updateAndDraw();
	map_img.updateAndDraw();

} // update_mask_projection


private void draw_SP_ROI_outlines(){

// draws the outlines of the ROIs stored in the variable ROIs
// into the scatterplots in the scatterplot matrix image


	ImageProcessor ip=sp_img.getProcessor();
	Roi r;

	ip.setColor(ROIcolour);
	for (i=0;i<n;i++) {
		
		r=ROIs[i];
		if (r!=null) r.drawPixels(ip);
		
	} // for i

} // draw_SP_ROI_outlines



private void clear_SP_ROIs(){

// removes all selections made in the scatterplot matrix

	mask=new boolean[n][sp_length][sp_length];
	mask_projection=new boolean[w][h][d];

	for (i=0;i<n;i++){
	for (x=0;x<sp_length;x++){
	for (y=0;y<sp_length;y++){

		mask[i][x][y]=true;

	}}} // for i,x,y

	for (x=0;x<w;x++){
	for (y=0;y<h;y++){
	for (z=0;z<d;z++){

		mask_projection[x][y][z]=true;	

	}}} // for x,y,z

	ROIs=new Roi[n];
	for (i=0;i<n;i++) ROIs[i]=null;

	n_roi_sp=0;

} // clear_SP_ROIs



private void redraw_SP_matrix(){

// draws the scatterplot matrix into the scatterplot matrix image

	draw_sp_matrix_blank(false);
	for (int i=0; i<n; i++) draw_single_sp(i);

} // redraw_SP_matrix



private void reset_sp(){

// removes all selections made in the scatterplot matrix
// and redraws scatterplot matrix image and spatial-domain image

	clear_SP_ROIs();
	redraw_SP_matrix();
	
	draw_map_blank(false); 		// false: do not create new image
	update_mask_projection(); 	// to account for the spatial mask, which might still hold a selection

	n_px=0;
	label_nb_px.setText(""+n_px+" datapoints in selection");

} // reset_sp


public void remove_ROI(){

// remove a single ROI defined in the scatterplot matrix
// the ROI to be removed is selected by the user in a dialog

	int i;
	String[] roi_names=new String[n];
	boolean[] roi_exists=new boolean[n];
	String s;

// determine scatterplots in which ROIs have been defined
	
	for (i=0;i<n;i++){
		s=""+(i+1)+": ";

		if (ROIs[i]!=null){
			s=s+img_titles[components[i][_Y]]+" / "+img_titles[components[i][_X]];
			roi_exists[i]=true;
		} // if
		else {
			s=s+"(none)";
			roi_exists[i]=false;
		} // else

		roi_names[i]=s;

	} // for i

// show diolog to select ROI to be removed	
	
	GenericDialog gd=new GenericDialog("Remove ROI in SP matrix");
	gd.addChoice("ROI to remove:",roi_names,roi_names[0]);
	gd.showDialog();
	if (gd.wasCanceled()) return;

	i=gd.getNextChoiceIndex();

	if (!roi_exists[i]){
		IJ.showMessage("No ROI selected in scatterplot "+(i+1));
		return;
	} // if

// apply changes

	delete_SP_ROI_from_matrix(i);
	redraw_SP_matrix();
	draw_map_blank(false);	// draw spatial-domain map; false --> not in a new window
	update_mask_projection();
	
} // remove_ROI



private void draw_map_blank(boolean newimg){

// draws the spatial-domain map in the spatial-domain image
//	newimage=ture enforces creation of a new image

	ImageStack st;
	int c;

	if (map_img!=null){
		if (!map_img.isVisible()) newimg=true;
	}

	if ((newimg)&&(map_img!=null)) map_img.close();
	if ((newimg)||(map_img==null)){
		map_img=NewImage.createRGBImage(str_title+" spatial-domain map",w,h,d,NewImage.FILL_WHITE);
		map_img.setCalibration(img_calibration);
		map_img.show();	
	}

	st=map_img.getStack();
	
	for (x=0;x<w;x++){
	for (y=0;y<h;y++){
	for (z=0;z<d;z++){
	
		if (img_in_map>=0) {
	
			// if one of the input images is used a background for the spatial-domain image:
			// get grey value of pixel/voxel (x,y,z) of the input image number image_in_stack
			// as stored in the variable data
	
			c=data[x][y][z][img_in_map]*reduction;
				// multiply with reduction factor as grey values are stroed in data in "binned" form
				// this causes a discontinuous histogram,
				// which should be barely noticeable at least for small reduction factors
				
			if (c>255) c=255; 	// just to be on the safe side
			c=0x010101*c; 		// convert 8-bit greyscale into RGB
			
		} // if img_in_map>=0
		else {
		
			// black background if none of the input images is used 
			c=0;
		
		} // else
		st.setVoxel(x,y,z,c);
	}}} // for x,y,z
	
	map_img.updateAndDraw();

} // draw_map_blank


private void reset_map_ROI(boolean newmask){

// delete selection made in spatial domain

	if ((newmask)||(mask_spatial==null)){
		mask_spatial=new boolean[x][y][z];
		map_ROI_start=0;
		map_ROI_end=d-1;
	} // if

	for (x=0;x<w;x++){
	for (y=0;y<h;y++){
	for (z=0;z<d;z++){
		mask_spatial[x][y][z]=true;
		mask_projection[x][y][z]=true;
	}}} // for x,y,z

	map_ROI=null;
	n_roi_map=0;

} // reset_map_ROI



private void draw_map_ROI_outline(){

// draws the ROI stored in map_ROI into the spatial-domain image

	ImageStack st=map_img.getStack();
	ImageProcessor ip_map;
	Roi r;

	if (map_ROI!=null) {

		map_img.deleteRoi();	// this seems to be necessary to avoid that the active ROI gets drawn

		for (z=map_ROI_start;z<=map_ROI_end;z++){
			ip_map=st.getProcessor(z+1);
			ip_map.setColor(ROIcolour);
			map_ROI.drawPixels(ip_map);
		} // for z

	} // if

} // draw_map_ROI_outline



private void accept_ROI_spatial(){

// update classification of datapoints using a ROI defined in the spatial-domain image

	ImageStack st=map_img.getStack();
	ImageProcessor ip;
	Roi r=map_img.getRoi();
	boolean b;
	
	if (r==null){
		IJ.showMessage("No ROI selected in spatial-domain map");
		return;
	} // if


	if (d>1){

		GenericDialog gd=new GenericDialog("Include spatial ROI");

		gd.addMessage("Slices to include:");
		gd.addSlider("First slice",1,d,map_ROI_start+1);
		gd.addSlider("Last slice",1,d,map_ROI_end+1);
		gd.showDialog();
		if (gd.wasCanceled()) return;
		
		map_ROI_start=(int)gd.getNextNumber()-1;
		map_ROI_end=(int)gd.getNextNumber()-1;

	} // if

	reset_map_ROI(false);

	for (x=0;x<w;x++){
	for (y=0;y<h;y++){
	for (z=0;z<d;z++){

			b=((z>=map_ROI_start)&&(z<=map_ROI_end));
			if (b){
				b=r.contains(x,y);
			} // if b
			mask_spatial[x][y][z]=b;

	}}} // for x,y,z

	map_ROI=r;
	n_roi_map=1;

	draw_map_blank(false);
	redraw_SP_matrix();

	update_mask_projection();

} // accept_ROI_spatial



private void remove_ROI_spatial(){

// removes selection made in spatial domain and updates spatial-dmoain image and scatterplot matrix image

	reset_map_ROI(false); // false: do not open new window
	draw_map_blank(false);
	redraw_SP_matrix();
	update_mask_projection();

} // remove_ROI_spatial


private void angular_distance_map(){

// creates an angular distance map

	String[] sp_names=new String[n];
	String s;
	int i,i_sp;
	int x1,y1,x_sp,y_sp,b,c,i1;
	float[][] ad_matrix=new float[sp_length][sp_length];
	float[][][] ad_map=new float[w][h][d];	
	float a,a_min,a_max;
	ImageProcessor ip=sp_img.getProcessor();
	ImageStack st=map_img.getStack();
	int n_colours;
	boolean multiple_dp=false;
	int[] cs;	// colourscale for angular distance maps

	for (i=0;i<n;i++){
		s=""+(i+1)+" : "+img_titles[components[i][_Y]]+" / "+img_titles[components[i][_X]];
		sp_names[i]=s;
	} // for i

	GenericDialog gd=new GenericDialog("Angular distance map");
	gd.addChoice("Select scatterplot:", sp_names, sp_names[0]);
	gd.addCheckbox("Only draw pixels representing exactly one datapoint",!multiple_dp);
	gd.showDialog();
	if (gd.wasCanceled()) return;

	i_sp=gd.getNextChoiceIndex();
	multiple_dp=(!gd.getNextBoolean());

// calculate angular distances in selected scatterplot

	a_min=200;		// choose extreme values that will be replaced immediately 
	a_max=-200;
	a=0;

	for (x=0;x<sp_length;x++){
	for (y=0;y<sp_length;y++){

		if (sp[i_sp][x][y]>0)  {

			if (x==0){
				if (y>0) a=90;
				if (y<0) a=-90;
				if (y==0) a=0;
			} // if x==0
			else {
				a=(float)(Math.atan((double)y/x)*360/(2*Math.PI));
			} // else
			
			if (a<a_min) a_min=a;
			if (a>a_max) a_max=a;
			ad_matrix[x][y]=a;
			
		} // if (sp[i_sp][x][y]>0)

	}} // for x,y


// calculate spatial-domain map

	x_sp=components[i_sp][_X];
	y_sp=components[i_sp][_Y];
	
	for (x=0;x<w;x++){
	for (y=0;y<h;y++){
	for (z=0;z<d;z++){

		a=ad_matrix [data[x][y][z][x_sp]] [data[x][y][z][y_sp]] ;
		ad_map[x][y][z]=a;

	}}} // for x,y,z
		
	redraw_SP_matrix();
	draw_map_blank(false);

// calculate colourscale representing angular distances
	
	n_colours=0x300;
	cs=new int[n_colours];

	int R=0x010000,G=0x000100,B=0x000001;
	for (i=0; i<=0xff; i++) { 
		cs[i]		=	i*(G+B);	
		cs[0x100+i]	=	0xff*G+(0xff-i)*B+i*R; 			
		cs[0x200+i]	=	0xff*R+(0xff-i)*G;
	} // for i

// create images

	for (x=0;x<w;x++){
	for (y=0;y<h;y++){
	for (z=0;z<d;z++){

		i1=(int)Math.round((ad_map[x][y][z]-a_min)/(a_max-a_min)*(n_colours-1));
		if (i1<0) i1=0;
		if (i1>=n_colours) i1=n_colours-1;
		c=cs[i1];

		st.setVoxel(x,y,z,c);		
		for (i=0;i<n;i++){
		
			x1=data[x][y][z][components[i][_X]];
			y1=data[x][y][z][components[i][_Y]];
			b=sp[i][x1][y1];
		
			if (b>0){
		

				if (i==i_sp){
					ip.putPixel( origins[i][_X]+x1, origins[i][_Y]-y1-1, c);
				} // if i==i_sp

				else {
					if ((b==1)||(multiple_dp)) ip.putPixel( origins[i][_X]+x1, origins[i][_Y]-y1-1, c);
				} //else

			} // if >0

		} // for i
		
	}}} // for x,y,z

	sp_img.updateAndDraw();
	map_img.updateAndDraw();

} // angular_distance_map


private void settings() {

// displays a dialog allowing the user to adjust settings

	String s;
	boolean open_colourscale;

	GenericDialog gd=new GenericDialog("Settings");
	
	gd.addStringField("ROI outline colour:",int2rgb(ROIcolour));
	gd.addStringField("Colour of selected datapoints:",int2rgb(px_colour));
	gd.addNumericField("Background image in spatial-domain map:",(img_in_map+1),0);

	if (data!=null){
		for (i=0;i<dim;i++){
			gd.addStringField("Image "+(i+1)+" title:",img_titles[i],20);
		} // for i
	} // if

	gd.addCheckbox("Draw image titles in scatterplot matrix",titles_in_sp);
	gd.addCheckbox("Draw number in scatterplot",nb_in_sp);
	gd.addCheckbox("Edit scatterplot colour scale (new window)",false);

	gd.showDialog();
	if (gd.wasCanceled()) return;
	
	s=gd.getNextString();
		ROIcolour=rgb2int(s);
	s=gd.getNextString();
		px_colour=rgb2int(s);
	img_in_map=(int)gd.getNextNumber()-1;
		if (img_in_map>=dim) img_in_map=0;
		// 0 <= img_in_map < dim 	: use the respective input image 
		// img_in_map < 0 			: use black background
		
	if (data!=null){
		for (i=0;i<dim;i++){
			img_titles[i]=gd.getNextString();
		} // for i
	} // if

	titles_in_sp=(gd.getNextBoolean());
	nb_in_sp=(gd.getNextBoolean());
	open_colourscale=(gd.getNextBoolean());

	if (open_colourscale) edit_colourscale();

	if (input_selected){
		redraw_SP_matrix();
		draw_map_blank(false);
		update_mask_projection(); // to make changes active
	} // if	

} // settings


private String int2rgb(int a) {

// converts an integer representing an RGB value into a string
// consiting of the three values in decimal numbers seperated by commas and spaces

	String s="";
	s+=Integer.toString((0xff0000 & a)/0x010000)+", ";
	s+=Integer.toString((0x00ff00 & a)/0x000100)+", ";
	s+=Integer.toString( 0x0000ff & a);	
	return s;

} // int2rgb


private int rgb2int(String s) {

// converst a string representing an RGB vlue in dcimal numbers
// into an integer

	String s1="";
	int i;
	int[] colour={0,0,0};
	char c;
	int k=0;

	s+=" ";

	for (i=0;i<s.length();i++){
		c=s.charAt(i);
		if ((c>=0x30)&&(c<=0x39)) s1+=c;
		else {
			if ((s1.length()>0)&&(k<3)) {
				colour[k]=Integer.valueOf(s1);
				if ((colour[k]<0)||(colour[k]>255)) colour[k]=0;
				s1="";
				k++;
			} // if s1.length()>0
		} // else
	} // for i

	return ( 0x010000*colour[0] + 0x000100*colour[1] +  0x000001*colour[2] ); 

} // rgb2int


private void edit_colourscale(){

// opens a dialog allowing the user to edit the colourscale
// used to represent scatterplots

	int i,l,a,i_reset,max_dp;
	String s;
	boolean reset_colours,reset_thresholds;
	String[] choices={"-----","colours to default greyscale","colours to default RGB","thresholds to powers of 2"};

	max_dp=0;
	for (i=0;i<n;i++) {
	for (x=0;x<sp_length;x++){
	for (y=0;y<sp_length;y++){
		if ((x>0)||(y>0)){
			if (sp[i][x][y]>max_dp) max_dp=sp[i][x][y];
		} // if x,y>0
	}}} // for i,x,y

	GenericDialog gd=new GenericDialog("Edit scatterplot colour scale");

	gd.addMessage("Max. number of datapoints at the\nsame position in the scatterplots: "+max_dp+"\n(origins excluded)"); 	
	
	l=colourscale.length;
	for (i=0;i<l;i++) {
		gd.addStringField("Colour: ",int2rgb(colourscale[i]),15);
		gd.addNumericField("Threshold: ",thresholds[i],0);
	} // for i

	gd.addChoice("Reset...",choices,choices[0]);

	gd.showDialog();
	if (gd.wasCanceled()) return;

	for (i=0;i<l;i++) {
		s=gd.getNextString();
		colourscale[i]=rgb2int(s);
		a=(int)gd.getNextNumber();
		if (a<0) a=0;
		thresholds[i]=a;
	} // for i

// if selected, reset to pre-defined colour or threshold schemes:	
	
	i_reset=gd.getNextChoiceIndex();

	if (i_reset==1){
		for (i=0;i<l;i++) {
			colourscale[i]=colourscale_default_greyscale[i];
		} // for i
	} // if 

	if (i_reset==2){
		for (i=0;i<l;i++) {
			colourscale[i]=colourscale_default_RGB[i];
		} // for i
	} // if 

	if (i_reset==3){
		for (i=0;i<l;i++) {
			thresholds[i]=thresholds_default[i];
		} // for i
	} // if 

} // edit_colourscale



private void apply_mask(){

// select a suitable image as a mask to define the selection in spatial domain

	int i,k;
	int[] img_dim,img_index;
	ImagePlus img;
	ImageProcessor ip;
	ImageStack st;
	String[] titles,titles1;

// search for suitable images
// (i.e. same dimensions as input images, 8-bit greyscale type)
	
	int[] wl=WindowManager.getIDList();
	if ((wl==null)||(wl.length<1)) { 
		IJ.showMessage(str_title,"Could not find any open images");
		return;
	} //if

	titles1=new String[wl.length];
	img_index=new int[wl.length];
	k=0;

	for (i=0; i<wl.length; i++) {
		img=WindowManager.getImage(wl[i]);
		if (img!=null){

			img_dim=img.getDimensions();
			if ( (img_dim[0]==w) && (img_dim[1]==h) && (img_dim[3]==d) && (img.getType()==ImagePlus.GRAY8) ){
				k++;
				titles1[k-1]=img.getTitle();
				img_index[k-1]=wl[i];
			} // if dimensions match

		} // if img!=null
			
	} //for
	
// show a dialog offering the user a choice of suitbale mask images
	
	titles=new String[k];
	for (i=0; i<k; i++) {
		titles[i]=titles1[i];
	} //for

	GenericDialog gd=new GenericDialog("Select mask images");
	gd.addMessage("Select mask image");
	gd.addChoice("Mask:",titles,titles[0]);
	gd.showDialog();
	if (gd.wasCanceled()) return;

	i=gd.getNextChoiceIndex();

	img=WindowManager.getImage(img_index[i]);
	st=img.getStack();

	reset_map_ROI(false); // clear existing selection in spatial domain	
	
// apply mask: the mask is defind by all pixels/voxels 
// in the mask image with grey value 255	
	
	for (x=0;x<w;x++){
	for (y=0;y<h;y++){
	for (z=0;z<d;z++){
		mask_spatial[x][y][z]=((int)st.getVoxel(x,y,z)==255);
	}}} // for x,y,z

	n_roi_map=1; // although no ROI is stored in map_ROI; if n_roi_map remains zero, update_mask_projection won't act

// apply changes
	
	draw_map_blank(false); // false: do not create new image
	redraw_SP_matrix();
	update_mask_projection(); // to account for spatial mask

} // apply mask


private void show_info(){

// displays the info window

	GenericDialog gd=new GenericDialog(str_title+" info");
	TextArea ta=new TextArea(info_text,40,100,TextArea.SCROLLBARS_VERTICAL_ONLY);
	ta.setEditable(false);
	gd.add(ta);
	gd.addMessage("");
	gd.hideCancelButton();
	gd.setModal(false);
	gd.showDialog();

} // show_info

// string containing the text to be displayed in the info window:

String info_text=								""
					+"\n============================================"
					+"\n"
					+"\n"
					+"\n    ScatterJn, version 1.0"
					+"\n    ImageJ plugin for the evaluation of analytical microscopy data"
					+"\n"
					+"\n    Copyright (C) 2015 Fabian Zeitvogel, University of Tuebingen"
					+"\n"
					+"\n    This program is free software: you can redistribute it and/or modify"
					+"\n    it under the terms of the GNU General Public License as published by"
					+"\n    the Free Software Foundation, either version 3 of the License, or"
					+"\n    (at your option) any later version."
					+"\n"
					+"\n    This program is distributed in the hope that it will be useful,"
					+"\n    but WITHOUT ANY WARRANTY; without even the implied warranty of"
					+"\n    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the"
					+"\n    GNU General Public License for more details."
					+"\n"
					+"\n    You should have received a copy of the GNU General Public License"
					+"\n    along with this program.  If not, see <http://www.gnu.org/licenses/>."
					+"\n"
					+"\n"
					+"\n============================================"
					+"\n"
					+"\n"
					+"\nPurpose"
					+"\n"
					+"\nScatterJn is scatterplot tool that can handle more than two (i.e. \"n\") input images. It is partly based on the ScatterJ plugin (http://download.savannah.gnu.org/releases/scatterj/). It generates a scatterplot matrix and offers tools for interactive classification."
					+"\n"
					+"\nInput"
					+"\n"
					+"\nAs input, the plugin requires a set of 8-bit greyscale images (or a set of stacks). Input images must be of the same dimensions and need to be aligned with respect to each other so that they all represent the same area (or volume)."
					+"\n"
					+"\nProcessing"
					+"\n"
					+"\nThe input images are combined into datapoints. If all input images are aligned, pixels at the same position (spatial coordinates) in the different images correspond to the same spot on the sample. Each image is assumed to represent a different quality of the sample, such as the concentration of a certain chemical species. For each combination of spatial coordinates, a datapoint is created from the grey values of the corresponding pixels. If n input images are used, a datapoint will have n coordinates. Each pixel position in the spatial domain, defined by 2 or 3 spatial coordinates, is thus linked to one datapoint with n coordinates. The datapoints are plotted in a scatterplot matrix. The scatterplot matrix consists of a set of individual 2D scatterplots that represent all combinations of 2 input images. This visualizes the relationship of the different variables on each other. A spatial-domain map of the sample area, having the same dimensions as the input images, is shown in an additional image window."
					+"\n"
					+"\nInteractive functions"
					+"\n"
					+"\nUsing ScatterJn, it is possible to interactively classify datapoints according the their position in the different scatterplots and to the position of their origin in the input images. This is done by selection regions of interest (ROIs) in the scatterplot matrix and in the spatial-domain map. Datapoints that are within all the selected areas are automatically highlighted."
					+"\n"
					+"\nIn addition to this \"binary\" classification method, a function of for \"gradual\" classification is implemented. This function assigns a value to each datapoint that corresponds to the position of the datapoint in one of the scatterplots. This position is characterized by the angular distance of the projection of the datapoint in the selected scatterplot. The values assigned to each datapoint are represented using a colour scale. "
					+"\n"
					+"\n============================================"
					+"\n"
					+"\n"
					+"\nManual"
					+"\n"
					+"\nThe usage of the different functions of ScatterJ 2 are described in the following:"
					+"\n"
					+"\n1. Select input images"
					+"\n"
					+"\nClick the \"Select input images\" button to start the selection procedure. Input images must be loaded in ImageJ."
					+"\n"
					+"\n- Define number of images"
					+"\nIn the first dialogue window, define the number of input images to be used. A new window will open."
					+"\n"
					+"\n- Image selection"
					+"\nSelect the input images from dropdown menus."
					+"\n"
					+"\n- Histogram binning"
					+"\nFor larger numbers of input images, it may be useful to reduce the size of the scatterplots. This can be done by entering a histogram binning factor greater than 1. All grey values will be divided by that factor (integer division), reducing the size of the scatterplots."
					+"\n"
					+"\n- Keep image titles"
					+"\nIf the number of input images is the same as the number used before, it is possible to retain the titles defined in the Settings dialogue (see below). This can be useful when dealing with several similar datasets. Note that the titles are assigned to the image number. This means that, if image titles are kept, the images have to be selected in the same order as before. "
					+"\n"
					+"\n2. Settings"
					+"\n"
					+"\nThe \"Settings\" button will open a dialogue that allows the user to adjust display settings."
					+"\n"
					+"\n- ROI outline colour"
					+"\nDefines the colour in which the outlines of selected ROIs are drawn. Colours are defined by providing RGB values in decimal format, seperated by commas or spaces."
					+"\n"
					+"\n- Colour of selected datapoints"
					+"\nDefines the colour in which selected datapoints are highlighted."
					+"\n"
					+"\n- Background image in map"
					+"\nDefines the number of the input image that is used as a background in the map image. If a number <= 0 is provided, a black background will be used."
					+"\n"
					+"\n- Draw image titles in scatterplot matrix"
					+"\nIf activated, axes in the scatterplot matrix will be labelled with the image titles."
					+"\n"
					+"\n- Image titles"
					+"\nChange the titles used to refer to the input images e.g. for labelling of the axes."
					+"\n"
					+"\n- Draw number in scatterplot"
					+"\nIf activated, a running number will be drawn in the upper left corner of each scatterplot."
					+"\n"
					+"\n- Change scatterplot colour scale (new window)"
					+"\nThe scatterplots are plotted using a colour scale to indicate the number of datapoints that are projected onto the same pixel. If activated, a new dialogue window is opened (once the present dialogue is confirmed using the \"OK\" button) that permits the user to adjust that colour scale:"
					+"\n"
					+"\n- Colour/threshold "
					+"\nThe colour scale is defined by a set of up to nine colours and the threshold separating the intervals that are assigned a specific colour. If zero is entered as a threshold value, the respective colour will not be used in the scatterplot. "
					+"\n"
					+"\n- Reset"
					+"\nUsing the \"Reset...\" dropdown menu, colours can be reset to a default greyscale or RGB scheme, and thresholds can be reset to powers of 2."
					+"\n"
					+"\n"
					+"\n"
					+"\n2. Binary classification"
					+"\n"
					+"\n2.1. Accept ROI"
					+"\nTo define a ROI in one of the scatterplots for classification of datapoints, draw the ROI using one of ImageJ's ROI tools and click on the \"Accept ROI\" button. It is possible to define one ROI per scatterplot. Datapoints that are within the selected ROI are highlighted in all scatterplots and in the map image. If ROIs are defined in more than one scatterplot (or in the map), only datapoints that are within all the selected ROIs will be highlighted. Note that, in scatterplots where no ROI is defined, selected and non-selected datapoints may be projected onto the same pixel, which will in that case appear in the highlighting colour anyway."
					+"\n"
					+"\n2.2. Remove ROI"
					+"\nRemoves one of the ROIs defined in the scatterplot matrix. The respective scatterplot is selected from a list."
					+"\n"
					+"\n2.3. Reset"
					+"\nRemoves all ROIs from the scatterplot matrix."
					+"\n"
					+"\n2.4. Accept ROI (map)"
					+"\nIn analogy to defining ROIs in the scatterplot matrix, this function defines a previously drawn ROI in the map image. It is possible to define one ROI in the map image."
					+"\n"
					+"\n2.5. Apply mask"
					+"\nIn addition to drawing a ROI, the selected area in the map can be defined using a mask. The mask is generated from an 8-bit greyscale image with the same dimensions as the input images. Pixels with a grey value of 255 in the mask image will be selected in the map."
					+"\n"
					+"\n2.6. Remove ROI (map)"
					+"\nRemoves the ROI/selection from the map image."
					+"\n"
					+"\n3. Angular distance map"
					+"\n"
					+"\nThis function assigns a value to each datapoint according to its angular distance to the x-axis  in one of the scatterplots (i.e. the angle between the position vector of the datapoint's projection in the scatterplot and the respective x-axis), and displays this value using a colour scale. The scatterplot in which the angular distance should be calculated is chosen from a list. "
					+"\n"
					+"\n- Only draw pixels representing exactly one datapoint"
					+"\nIn the selected scatterplot as well as in the map image, all pixels can be unambigously assigned a colour representing an angular distance value. In the other scatterplots, several datapoints may be projected onto the same pixel. If the option \"Only draw pixels representing exactly one datapoint\" is activated (default setting), the angular-distance colour scale will not be applied to ambiguous pixels. If the option is deactivated, the angular-distance colour scale will be applied to all pixels. For ambiguous pixels, colours plotted at a later time will overwrite colours plotted earlier."
					+"\n"
					+"\n"
					+"\n4. Info & manual"
					+"\n"
					+"\nDisplays this window."
					+"\n"
					+"\n"
					+"\n"
					+"\n"
					+"\n"
					+"\n";

} // ScatterJn


