/*

    ScatterJ, version 1.04
    ImageJ plugin for the evaluation of analytical microscopy data

    Copyright (C) 2014 Fabian Zeitvogel, University of Tuebingen

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/



import java.io.*;
import ij.*;
import ij.io.*;
import ij.gui.*;
import ij.plugin.*;
import ij.process.*;
import java.util.*;
import ij.plugin.frame.*;
import java.awt.*;
import java.awt.event.*;
import java.math.*;
import ij.text.*; 


public class ScatterJ_ extends PlugInFrame implements ActionListener {


//global variables ========================================================================
//=========================================================================================
//=========================================================================================

String version="1.04";

int[][] sp=new int[256][256]; // the matrix containing the occurence frequency of pixels/voxels a certain position in the scatterplot

int xmax=255;		
int ymax=255;
int ymin=0;
int xmin=0;
int pixelsize=1;
float scale_x=1;	
float scale_y=1;
String filename_x="";
String filename_y="";

int[][][] voxels1;
int[][][] voxels2;

String sp_title="Scatterplot";
String title_x="";	
String title_y="";
String str_xtics="";	
String str_ytics="";

int h,w,n;

int yoffset=50;
int xoffset=90;

private static int i1;
private static int i2;
private static int n_windows;
private static int n_messages;

int legend_style=0;

ImagePlus sp_img; 	// image containing the scatterplot

ImagePlus img1; 	// �nput images
ImagePlus img2;
boolean[][][] img_selection;

ij.measure.Calibration img_calibration;

Panel p1,p11,p12,p111;
Label label_x,label_y;

int regression_stored=0;
double stored_slope,stored_yintc;

// default colourscales

int[] default_thresholds=	{	1,		2,		4,		8,		16,		32,		64,		128,		256};
int[] default_RGB=   	{	0x200080,	0x0000ff,	0x00e0ff, 	0x008080, 	0x00e000,	0xffff00,	0xff8000,	0xff0000,	0x800000};
int[] default_greyscale1	=   	{	0xdddddd,	0xcccccc, 	0xbbbbbb,	0xaaaaaa, 	0x999999,	0x777777,	0x555555,	0x333333,	0x111111};
int[] default_greyscale2	=   	{	0x111111,	0x333333,	0x555555,	0x777777,	0x999999,	0xaaaaaa,	0xbbbbbb, 	0xcccccc, 	0xdddddd};

int[] colours, thresholds;

TextWindow tw;
String str_title="ScatterJ";

//=========================================================================================
//=========================================================================================
//=========================================================================================

public ScatterJ_() {

	super("ScatterJ");

	check_ImageJ_version();
	
	p1=new Panel(); p1.setLayout(new GridLayout(2,1));
	p11=new Panel(); p11.setLayout(new GridLayout(4,1));
	p111=new Panel(); p111.setLayout(new GridLayout(1,2));
	p12=new Panel(); p12.setLayout(new GridLayout(4,2));

	p11.add(new Label(str_title+", version "+version));
	label_x=new Label(filename_x);
	p11.add(label_x);
	label_y=new Label(filename_y);
	p11.add(label_y);

	p12button("Create scatterplot");
	p12button("Replot");
	p12button("Axes");
	p12button("Colour scale");
	p12button("Backmapping");
	p12button("Statistics");
	p12button("Deviation map");
	p12button("Export data");
	p12button("Info");

	p11.add(p111);
	p1.add(p11);
	p1.add(p12);
	add(p1);

	init_colourscale();

	pack();
	GUI.center(this);
	show();

} // ScatterJ

// p12button: adds a button to panel p12

void p12button(String s){

	Button b=new Button(s);
	b.addActionListener(this);
	p12.add(b);

} // p12button


void init_colourscale(){

	int i;
	int n_colours=default_thresholds.length;
	colours=new int[n_colours];
	thresholds=new int[n_colours];
	for (i=0;i<n_colours;i++) colours[i]=default_RGB[i];
	for (i=0;i<n_colours;i++) thresholds[i]=default_thresholds[i];

} // init_colourscale;



public void actionPerformed(ActionEvent e) {

	String action=e.getActionCommand();

	
	if (action.equals("Replot")) { 
		plot_img();
	}

	if (action.equals("Create scatterplot")) { 
		new_scatterplot();
		update_frame();
	}

	if (action.equals("Deviation map")) { 
		deviation_map();
	}

	if (action.equals("Export data")) {
		export_data(); 
	}

	if (action.equals("Axes")) { 
		define_axes();
	}

	if (action.equals("Statistics")) { 
		calculate_statistics();
	}

	if (action.equals("Colour scale")) { 
		define_colours(); 
	}

	if (action.equals("Backmapping")) { 
		icp(); 
	}


	if (action.equals("Info")) { 
		show_info(); 
	}


} // actionPerformed


public void run(String arg) {

} // run


// calculate_statistics: calculate selected statistics for selected data

private void calculate_statistics() {

	// calculation of statistical parameters based on
	// - R.R. Sokal, F.J. Rohlf: Biometry: the principles and practice of statistics in biological research, third edition, W.H. Freeman and Company, New York, 1995
	// - J.E. Jackson: A user's guide to principal components, John Wiley & sons, inc., New York, 1991
	// - Jackson et al.: Principal components and factor analysis: part I-principal components, J Qual Technol 12, pp. 1-9, 1980
	// - Mueller et al.: Tafeln der mathematischen Statistik, 3rd ed., Fachbuchverlag, Leipzig, 1979

	boolean drawroi, calibrated_units, ignore_zeros;
	boolean general_show, ols_show, ols_draw, gm_show, gm_draw, pa_show, pa_draw_pa, pa_draw_ma, pa_draw_ellipse, pa_set_length, pearson_show;
	boolean do_ols, do_gm, do_pa, do_pearson, show_parameters;

	double ols_slope=0, ols_yintc=0, ols_r2=0, ols_rms=0, gm_slope=0, gm_yintc=0, gm_r2=0, gm_rms=0, pa_slope_pa=0, pa_yintc_pa=0, pa_slope_ma=0, pa_yintc_ma=0, pa_rms=0, pearson=0;
		// variables had to be assigned a value to avoid error messages of compiler

	int x,y,x1,x2,n1,n_total,i;
	int upperx,lowerx;
	double meanx,meany,varx,vary,stdx,stdy,sxy,l1=0,l2=0;
	double alpha=0,sin_alpha=0,cos_alpha=0,sin_negalpha=0,cos_negalpha=0,sin_90alpha=0,cos_90alpha=0,T05,hl_pa=0,hl_ma=0;
	double scalex,scaley;

	int _x=0, _y=1, _n=2;
	double[][] dp; // list of datapoints: x,y,n
	
	// show dialog	========================================================================

	GenericDialog gd=new GenericDialog("Calculate Statistics");
	gd.addMessage("The selected parameters will be calculated for the ROI\nor for the part of the data shown in the diagram");
	gd.addMessage("General settings");
	gd.addCheckbox("use calibrated units",true);
	gd.addCheckbox("ignore (0,0)",true);
	gd.addCheckbox("draw ROI outline",false);
	gd.addCheckbox("show general parameters",false);
	gd.addMessage("Ordinary least squares regression");
	gd.addCheckbox("show results",false);
	gd.addCheckbox("draw line",false);
	gd.addMessage("Reduced major axis");
	gd.addCheckbox("show results",false);
	gd.addCheckbox("draw line",false);
	gd.addMessage("Major axis");
	gd.addCheckbox("show results",false);
	gd.addCheckbox("draw major axis",false);
	gd.addCheckbox("draw minor axis",false);
	gd.addCheckbox("draw 95% confidence ellipse",false);
	gd.addCheckbox("set axis lengths to +/- 1.5 lengths of semiaxes",false);
	

	gd.addMessage("Pearson's coefficient");
	gd.addCheckbox("show",false);

	gd.showDialog();
	if (gd.wasCanceled()) return;

	// read settings ==================================================================

	calibrated_units=(gd.getNextBoolean());
	ignore_zeros=(gd.getNextBoolean());
	drawroi=(gd.getNextBoolean());
	general_show=(gd.getNextBoolean());

	ols_show=(gd.getNextBoolean());
	ols_draw=(gd.getNextBoolean());
	do_ols=(ols_show || ols_draw);
	
	gm_show=(gd.getNextBoolean());
	gm_draw=(gd.getNextBoolean());
	do_gm=(gm_show || gm_draw);

	pa_show=(gd.getNextBoolean());
	pa_draw_pa=(gd.getNextBoolean());
	pa_draw_ma=(gd.getNextBoolean());
	pa_draw_ellipse=(gd.getNextBoolean());
	pa_set_length=(gd.getNextBoolean());
	do_pa=(pa_show || pa_draw_pa || pa_draw_ma || pa_draw_ellipse);

	
	pearson_show=(gd.getNextBoolean());
	do_pearson=(pearson_show);

	show_parameters=(general_show || ols_show || gm_show || pa_show ||pearson_show);

	// prepare/initialize for calculation ============================================

	if (calibrated_units) {
		scalex=scale_x; scaley=scale_y;
	}
	else {
		scalex=1; scaley=1;
	} // if calibrated_units

	ImageProcessor ip=sp_img.getProcessor();
	
	Roi r=sp_img.getRoi();
	boolean[][] selection_contains=get_selection_content(r!=null);

	n_total=0; 	// total number of datapoints
	n1=0; 		// number of pixels in plot

	for (x=xmin; x<=xmax; x++) {  
	for (y=ymin; y<=ymax; y++) {  
	
		if ((selection_contains[x][y]) && ((!ignore_zeros)||(x+y>0)) ) { n_total+=sp[x][y]; n1++; }
	
	}} // for x,y

	dp=new double[n1][3];
	i=-1;
	
	for (x=xmin; x<=xmax; x++) {  
	for (y=ymin; y<=ymax; y++) {  
	
		if 	( (selection_contains[x][y]) && ((!ignore_zeros)||(x+y>0)) ){
			
			i++;
			dp[i][_x]=(double)x*scalex;
			dp[i][_y]=(double)y*scaley;
			dp[i][_n]=sp[x][y];

		} //if 

	}} // for x,y

	// calculate general statistical parameters ================================================================

	double sum;

	sum=0;
	for (i=0; i<n1; i++) sum+=dp[i][_x]*dp[i][_n];
	meanx=sum/n_total;
		
	sum=0;
	for (i=0; i<n1; i++) sum+=dp[i][_y]*dp[i][_n];
	meany=sum/n_total;

	sum=0;
	for (i=0; i<n1; i++) sum+=Math.pow(dp[i][_x]-meanx,2)*dp[i][_n];
	varx=sum/(n_total-1); 
	stdx=Math.sqrt(varx);

	sum=0;
	for (i=0; i<n1; i++) sum+=Math.pow(dp[i][_y]-meany,2)*dp[i][_n];
	vary=sum/(n_total-1); 
	stdy=Math.sqrt(vary);

	sum=0;
	for (i=0; i<n1; i++) sum+=(dp[i][_x]-meanx)*(dp[i][_y]-meany)*dp[i][_n];
	sxy=sum/(n_total-1);
	
	// calculate user-selected parameters ================================================================

	if ( do_pearson || do_gm || do_ols ) {
		pearson=sxy/(stdx*stdy);
	} // if do_pearson

	if (do_ols){

		ols_slope=sxy/varx;
		ols_yintc=meany-ols_slope*meanx;
		ols_r2=Math.pow(pearson,2);

		stored_slope=ols_slope*scalex/scaley;
		stored_yintc=ols_yintc/scaley;
		regression_stored=1;

	} //if do_ols

	if (do_gm) {

		gm_slope=pearson/Math.abs(pearson)*stdy/stdx; 	// sign(r) = r/abs(r);
		gm_yintc=meany-gm_slope*meanx; 
		
		stored_slope=gm_slope*scalex/scaley;
		stored_yintc=gm_yintc/scaley;
		regression_stored=2;

	} // if do_gm

	if (do_pa){

		double D=Math.sqrt(Math.pow(vary+varx,2)-4*(varx*vary-Math.pow(sxy,2)));
		l1=(varx+vary+D)/2;
		l2=(varx+vary-D)/2;

		pa_slope_pa=sxy/(l1-vary);
		pa_yintc_pa=meany-pa_slope_pa*meanx;

		pa_slope_ma=(double)(-1)/pa_slope_pa;
		pa_yintc_ma=meany-pa_slope_ma*meanx;

		double n2=n_total; n2=n2*(n2-2);

		stored_slope=pa_slope_pa*scalex/scaley;
		stored_yintc=pa_yintc_pa/scaley;
		regression_stored=3;

	} //if do_pa



	// calculate root mean square errors ========================================================

	for (i=0; i<n1; i++) { // calculate residuals

		if (do_ols) ols_rms+=Math.pow( dp[i][_y]-(dp[i][_x]*ols_slope+ols_yintc), 2 )*dp[i][_n];

		if (do_gm)  gm_rms+= 1/gm_slope * Math.pow( (dp[i][_y]-(dp[i][_x]*gm_slope+gm_yintc)), 2 )*dp[i][_n];

		if (do_pa)  pa_rms+= 1/(1+Math.pow(pa_slope_pa,2)) * Math.pow( (dp[i][_y]-(dp[i][_x]*pa_slope_pa+pa_yintc_pa)), 2 )*dp[i][_n];
	
	} // for i	

	ols_rms=Math.sqrt(ols_rms/n_total);	
	gm_rms=Math.sqrt(gm_rms/n_total);	
	pa_rms=Math.sqrt(pa_rms/n_total);	

	// draw selected structures ============================================================	

	ip.setColor(Toolbar.getForegroundColor());

	if ((drawroi)&&(r!=null)) {
		r.drawPixels(ip);
		sp_img.draw();
	} // if drawroi

	if (r==null) {
		lowerx=xmin;
		upperx=xmax;
	} // if r==null
	else {
		java.awt.Rectangle ra=r.getBounds();
		lowerx=(int)Math.round((ra.x-xoffset)/pixelsize)+xmin;
		upperx=lowerx+(int)Math.round(ra.width/pixelsize); 
		if (lowerx<xmin) lowerx=xmin;
		if (upperx>xmax) upperx=xmax;
	} // else

	if (ols_draw) {

		x1=check_limits(lowerx,ols_slope*scalex/scaley,ols_yintc/scaley,xmin,xmax,ymin,ymax);
		x2=check_limits(upperx,ols_slope*scalex/scaley,ols_yintc/scaley,xmin,xmax,ymin,ymax);

		int y1=(int)Math.round(ols_yintc/scaley+ols_slope*scalex/scaley*x1);
		int y2=(int)Math.round(ols_yintc/scaley+ols_slope*scalex/scaley*x2);

		ip.drawLine((x1-xmin)*pixelsize+xoffset,(ymax-y1)*pixelsize+yoffset,(x2-xmin)*pixelsize+xoffset,(ymax-y2)*pixelsize+yoffset);
		sp_img.draw();

	} // if ols_draw

	if (pa_draw_pa || pa_draw_ma || pa_draw_ellipse) {

		alpha=Math.atan(pa_slope_pa);
		sin_alpha=Math.sin(alpha);
		cos_alpha=Math.cos(alpha);
		sin_negalpha=Math.sin(-alpha);
		cos_negalpha=Math.cos(-alpha);
		T05=(double)2*(n_total-1)/(n_total-2)*F_05(n_total); // T^2_alpha from Jackson 1980; F_.05[2,n-2] from function F_05(n) (see below)

		hl_pa=Math.sqrt(l1*T05); // length of "semimajor axis" (Jackson 1991)
		hl_ma=Math.sqrt(l2*T05); // length of "semiminor axis" (Jackson 1991)

		sp_img.draw();

	} // if pa_draw_...

	if (pa_draw_pa) {

		x1=(pa_set_length) ? (int)Math.round((meanx-cos_alpha*1.5*hl_pa)/scalex) : lowerx;
		x2=(pa_set_length) ? (int)Math.round((meanx+cos_alpha*1.5*hl_pa)/scalex) : upperx;

		x1=check_limits(x1,pa_slope_pa*scalex/scaley,pa_yintc_pa/scaley,xmin,xmax,ymin,ymax);
		x2=check_limits(x2,pa_slope_pa*scalex/scaley,pa_yintc_pa/scaley,xmin,xmax,ymin,ymax);

		int y1=(int)Math.round(pa_yintc_pa/scaley+pa_slope_pa*scalex/scaley*x1);
		int y2=(int)Math.round(pa_yintc_pa/scaley+pa_slope_pa*scalex/scaley*x2);
		
		if (x1==x2) { y1=(pa_set_length) ? (int)Math.round(meany-1.5*hl_pa) : ymin; y2=(pa_set_length) ? (int)Math.round(meany+1.5*hl_pa) : ymax; } // in case line is vertical

		ip.drawLine((x1-xmin)*pixelsize+xoffset,(ymax-y1)*pixelsize+yoffset,(x2-xmin)*pixelsize+xoffset,(ymax-y2)*pixelsize+yoffset);

		sp_img.draw();

	} // if pa_draw_pa

	if (pa_draw_ma) {

		x1=(pa_set_length) ? (int)Math.round((meanx-sin_alpha*1.5*hl_ma)/scalex) : lowerx;
		x2=(pa_set_length) ? (int)Math.round((meanx+sin_alpha*1.5*hl_ma)/scalex) : upperx;
		
		x1=check_limits(x1,pa_slope_ma*scalex/scaley,pa_yintc_ma/scaley,xmin,xmax,ymin,ymax);
		x2=check_limits(x2,pa_slope_ma*scalex/scaley,pa_yintc_ma/scaley,xmin,xmax,ymin,ymax);

		int y1=(int)Math.round(pa_yintc_ma/scaley+pa_slope_ma*scalex/scaley*x1);
		int y2=(int)Math.round(pa_yintc_ma/scaley+pa_slope_ma*scalex/scaley*x2);

		if (x1==x2) { y1=(pa_set_length) ? (int)Math.round(meany-1.5*hl_ma) : ymin; y2=(pa_set_length) ? (int)Math.round(meany+1.5*hl_ma) : ymax; } // in case line is vertical

		ip.drawLine((x1-xmin)*pixelsize+xoffset,(ymax-y1)*pixelsize+yoffset,(x2-xmin)*pixelsize+xoffset,(ymax-y2)*pixelsize+yoffset);

		sp_img.draw();

	} // if ols_draw

	if (pa_draw_ellipse){

		int plotcolour=Toolbar.getForegroundColor().getRGB();

		int[] dx={0,0,1,-1};
		int[] dy={1,-1,0,0};
		double anglexy,min=0,d;
		int dir,xstart,ystart,icmax,ic;

		double scmin=(scalex<scaley)?scalex:scaley;
		int lmax=(int)Math.ceil(8*(hl_pa+hl_ma)/scmin*pixelsize); // max. number of pixels in ellipse line < frame (2*principal axis+2*minor axis); additional factor 2 because only directly neighbouring pixels count 
		lmax*=10; // to be on the safe side...
		
		int[] ellipsex=new int[lmax];
		int[] ellipsey=new int[lmax];
		boolean new_px,ellipse_closed=false;

		xstart=(int)Math.round((hl_pa*cos_alpha+meanx)/scalex*pixelsize);
		ystart=(int)Math.round((hl_pa*cos_alpha*pa_slope_pa+meany)/scaley*pixelsize);
		x=xstart; y=ystart;
		ellipsex[0]=x; ellipsey[0]=y;

		
		icmax=1;

		while (!ellipse_closed) {

			dir=4; 				// no direction chosen
			for (i=0; i<4; i++) {		// try all four directly neighbouring pixels

				new_px=true;
				for (ic=0;ic<icmax;ic++) { if ((ellipsex[ic]==x+dx[i])&&(ellipsey[ic]==y+dy[i])) new_px=false;}		// test if selected pixel has already been drawn
				if (new_px) {
					double xsc=(double)(x+dx[i])/pixelsize*scalex-meanx;
					double ysc=(double)(y+dy[i])/pixelsize*scaley-meany;
					d=Math.abs(Math.pow((cos_negalpha*xsc-sin_negalpha*ysc)/hl_pa,2)+Math.pow((sin_negalpha*xsc+cos_negalpha*ysc)/hl_ma,2)-1);	// calculate distance to true ellipse line

						if ((dir==4)||(d<min)) {  // i.e., if dealing with the first unmarked pixel or if pixel closer to true ellipse line than previous pixels
							min=d;
							dir=i;
						} // if

				} // if
			
			} // for i


			x=x+dx[dir]; 	// move forward
			y=y+dy[dir];
          
			ellipsex[icmax]=x; ellipsey[icmax]=y;	// add current pixel to list of already drawn pixels
 
			icmax+=1;	

			if (icmax>5) {ellipsex[0]=-1; ellipsey[0]=-1;};

			if ( (x/pixelsize<=xmax)&&(x/pixelsize>=xmin)&&(y<=ymax*pixelsize)&&(y>(ymin-1)*pixelsize))
					ip.drawDot(xoffset-xmin*pixelsize+x,yoffset+ymax*pixelsize-y);		// note: x,y are not in units of scatterplot pixels, but of canvas pixels, so they do not have to be multiplied with the pixelsize!
	
			//ellipse_closed=((x==xstart)&&(y==ystart));

			for (dir=0;dir<=3;dir++){
				if ( (x+dx[dir]==ellipsex[1]) && (y+dy[dir]==ellipsey[1]) && (icmax>10)) ellipse_closed=true;
			} // for 

		} //while	
	
	} // if draw_pa_ellipse


	if (gm_draw) {

		x1=check_limits(lowerx,gm_slope*scalex/scaley,gm_yintc/scaley,xmin,xmax,ymin,ymax);
		x2=check_limits(upperx,gm_slope*scalex/scaley,gm_yintc/scaley,xmin,xmax,ymin,ymax);

		int y1=(int)Math.round(gm_yintc/scaley+gm_slope*scalex/scaley*x1);
		int y2=(int)Math.round(gm_yintc/scaley+gm_slope*scalex/scaley*x2);

		ip.drawLine((x1-xmin)*pixelsize+xoffset,(ymax-y1)*pixelsize+yoffset,(x2-xmin)*pixelsize+xoffset,(ymax-y2)*pixelsize+yoffset);
		sp_img.updateAndDraw();

	} // if gm_draw

	sp_img.show();

	// show text output ================================================================	

	if (show_parameters) {
		
		String s="";

		if (general_show) {
			s=	s+"\n\nGeneral parameters:"
				+"\n  - mean x\t"+IJ.d2s(meanx,5)
				+"\n  - mean y\t"+IJ.d2s(meany,5)
				+"\n  - std x\t"+IJ.d2s(stdx,5)
				+"\n  - std y\t"+IJ.d2s(stdy,5)
				+"\n  - variance x\t"+IJ.d2s(varx,5)
				+"\n  - variance y\t"+IJ.d2s(vary,5)
				+"\n  - covariance\t"+IJ.d2s(sxy,5);
		} // if general_show

		if (ols_show) {
			s=	s+"\n\nOrdinary least squares regression:"
				+"\n  - slope\t"+IJ.d2s(ols_slope,5)
				+"\n  - y-intercept\t"+IJ.d2s(ols_yintc,5)
				+"\n  - root-mean-square error\t"+IJ.d2s(ols_rms,5);
		} // if ols_show

		if (gm_show) {
			s=	s+"\n\nReduced major axis regression:"
				+"\n  - slope\t"+IJ.d2s(gm_slope,5)
				+"\n  - y-intercept\t"+IJ.d2s(gm_yintc,5)
				+"\n  - root-mean-square error\t"+IJ.d2s(gm_rms,5);
		} // if gm_show

		if (pa_show) {
			s=	s+"\n\nMajor axis regression:"
				+"\n  Major axis"
				+"\n   - slope\t"+IJ.d2s(pa_slope_pa,5)
				+"\n   - y-intercept\t"+IJ.d2s(pa_yintc_pa,5)
				+"\n   - root-mean-square error\t"+IJ.d2s(pa_rms,5)
				+"\n  Minor axis"
				+"\n   - slope\t"+IJ.d2s(pa_slope_ma,5)
				+"\n   - y-intercept\t"+IJ.d2s(pa_yintc_ma,5);
		} // if pa_show

		if (pearson_show) {
			s=	s+"\n\nPearson's coefficient:"
				+"\n - r\t"+IJ.d2s(pearson,5);
		} // if pearson_show
		
		add_to_textwindow(s);

	} // if show_parameters

} // calculate_statistics


private double F_05(int n1){

	// will return a value for F_.05[2,n-2] based on
	// Mueller et al.: Tafeln der mathematischen Statistik, 3rd ed., Fachbuchverlag, Leipzig, 1979
	// assuming convergence to 3.00 for n>1000
	// and using linear interpolation if no value for n is given

	double[] F_n=		{1,	2,	3,	4,	5,	6,	7,	8,	9,	10,	11,	12,	13,	14,	15,	16,	17,	18,	19,	20,	21,	22,	23,	24,	25,	26,	27,	28,	29,	30,	32,	34,	36,	38,	40,	42,	44,	46,	48,	50,	55,	60,	70,	80,	90,	100,	125,	150,	200,	300,	500,	1000};
	double[] F_result=	{200,	19,	9.55,	6.94,	5.79,	5.14,	4.74,	4.46,	4.26,	4.1,	3.98,	3.89,	3.81,	3.74,	3.68,	3.63,	3.59,	3.55,	3.52,	3.49,	3.47,	3.44,	3.42,	3.4,	3.39,	3.37,	3.35,	3.34,	3.33,	3.32,	3.29,	3.28,	3.26,	3.24,	3.23,	3.22,	3.21,	3.2,	3.19,	3.18,	3.16,	3.15,	3.13,	3.11,	3.1,	3.09,	3.07,	3.06,	3.04,	3.03,	3.01,	3};


	int pos;
	int i;

	// convergence for large numbers
	if (n1>=1000) return 3.00;


	i=0;
	pos=-1;
	while (pos==-1) {
		// no interpolation if n in table
		if (F_n[i]==n1) return F_result[i];
		
		// determine position if n not in table
		if ((F_n[i]<n1)&&(F_n[i+1]>n1)) pos=i;
		i++;
	} // while

	// linear interpolation
	double slope=(double)(F_result[pos+1]-F_result[pos])/(F_n[pos+1]-F_n[pos]);
	double yintc=F_result[pos]-slope*F_n[pos];
	return (yintc+slope*n1);
} // F_05


// add_to_textwindow: appends a string to the textwindow ==============

private void add_to_textwindow(String s){

	if ( (tw==null) || (!(tw.isShowing())) ) 
		tw=new TextWindow(str_title+": results"," \t ","",500,600);
	n_messages++;
	s="\n\n== Output # "+n_messages+" =========\n"+s;
	tw.append(s);
	
} // add_to_textwindow


// icp: manual backmapping ("interactive correlation partitioning") ============================================

private void icp(){

	boolean mask=true;
	boolean drawroi;
	boolean[][] selection_content;

	if (n_windows==0) n_windows=1;
	String imgtitle="("+n_windows+") ROI map";

	String[] choices={"black & white mask","greyscale images"};
	
	ImageProcessor ip=sp_img.getProcessor();
	if (ip==null) return;

	Roi r=sp_img.getRoi();
	
		if (r==null) { // if no ROI is defined
			IJ.showMessage("Define ROI first");
			return;
		} // if 
	
	GenericDialog gd=new GenericDialog("Backmapping");
	gd.addMessage(	"This will create a map of the pixels contributing"
					+"\nto a selected region in the scatterplot");
	gd.addChoice("output style",choices,choices[0]);
	gd.addStringField("smage title",imgtitle,imgtitle.length());
	gd.addNumericField("set current number to:",n_windows,0);
	gd.addCheckbox("draw ROI outline",false);
	gd.addMessage("number of datapoints within ROI: "+sum_up_roi());

	gd.showDialog();
	if (gd.wasCanceled()) return;

	int choiceindex=gd.getNextChoiceIndex();
	mask=(choiceindex==0);
	imgtitle=gd.getNextString();
	n_windows=(int)gd.getNextNumber();
	drawroi=(gd.getNextBoolean());

	n_windows++;
	
	selection_content=get_selection_content(true);
	backwards_map(selection_content,imgtitle,mask);

	if (drawroi) {
		ip.setColor(Toolbar.getForegroundColor());
		r.drawPixels(ip);
		sp_img.draw();
	} // if drawroi

} // icp


//sum_up_roi: calculate the number of datapoints within a ROI ============================================================

private int sum_up_roi(){

	int n=0;
	String s;

	ImageProcessor ip=sp_img.getProcessor();
	if (ip==null) return 0;

	boolean[][] selection_contains=get_selection_content(true); 

	for (int x=xmin;x<=xmax;x++){
	for (int y=ymin;y<=ymax;y++){
				if (selection_contains[x][y]) n+=sp[x][y];
	}} // for x,y

	return n;

} // sum_up_roi


//backwards_map: create a map of pixels corresponding to datapoints within the ROI ==============================================================

private void backwards_map(boolean[][] selection_contains, String imgtitle, boolean mask) {

	int x,y,z;
	int gvx,gvy,a1,a2;

	ImagePlus newimg1, newimg2;
	String title1,title2;

	ImageProcessor ip=sp_img.getProcessor();
	if (ip==null) return;

	ImageStack stack1=img1.getStack();
	ImageStack stack2=img2.getStack();

	int n=stack1.getSize();
	int w=stack1.getWidth();
	int h=stack1.getHeight();

	voxels1=new int[w][h][n];
	voxels2=new int[w][h][n];

	for (x=0; x<w; x++) {
	for (y=0; y<h; y++) {
	for (z=0; z<n; z++) {
		voxels1[x][y][z]=((byte)stack1.getVoxel(x,y,z))&0xff;		// & 0xff to eliminate sign
		voxels2[x][y][z]=((byte)stack2.getVoxel(x,y,z))&0xff;
	}}} // for x,y,z

	title1=imgtitle;
	title2=imgtitle;
	if (!mask) {
		title1=title1+" (x: "+filename_x+")";
		title2=title2+" (y: "+filename_y+")";
	} // if !mask
	newimg1=NewImage.createByteImage(title1,w,h,n,NewImage.FILL_BLACK);
	newimg2=NewImage.createByteImage(title2,w,h,n,NewImage.FILL_BLACK);

	for (z=0; z<n; z++) {
	for (x=0; x<w;  x++) {
	for (y=0; y<h;  y++) {

		gvx=voxels1[x][y][z] ;
		gvy=voxels2[x][y][z] ;


		if ((selection_contains[gvx][gvy]) && (img_selection[x][y][z])) {

			if (mask) { 
				a1=0xff; 
				a2=0xff; 
			} 
			else  { 
				a1=voxels1[x][y][z];
				a2=voxels2[x][y][z];
			} 
			
			newimg1.getStack().setVoxel(x,y,z,a1);
			newimg2.getStack().setVoxel(x,y,z,a2);

		} // if

	}} // for x,y

	IJ.showStatus("Backmapping... "+100*z/n+" %");

	} // for z

	newimg1.setCalibration(img_calibration);
	newimg2.setCalibration(img_calibration);

	newimg1.show();
	if (!(mask)) newimg2.show();

} // backwards_map


// deviation_map: calculate and create a deviation map ====================================

float deviationmap_slope=0; // global variables that will save the settings for the next call of the function
float deviationmap_yintc=0;
int deviationmap_mode=0;

private void deviation_map() {

	boolean draw_line,roi_is_line,roi_only,calibrated_units,ignore_zeros=false,absolute_values;
	boolean use_line_parameters=false, use_regression_parameters=false;

	final int YDIST=0, AREA=1, PERPENDICULAR=2, ANGLE=3;
	int mode;
	int x,y,z,intgvx,intgvy;
	float gvx,gvy,dist=0;
	float slope=deviationmap_slope;
	float yintc=deviationmap_yintc;
	float dist_min=(xmax-xmin)*(ymax-ymin)*scale_x; // coarse upper approximation
	float dist_max=0;

	if (n_windows==0) n_windows=1;
	String imgtitle="("+n_windows+") Deviation map";

	ImagePlus newimg1;

	String[] choices={"vertical (y-)distance from reference line","product of horizontal and vertical distance (square root)","perpendicular distance from reference line","angle"};

	ImageProcessor ip=sp_img.getProcessor();
	if (ip==null) return;

	Roi r=sp_img.getRoi();
	roi_is_line=false;
	if (r!=null) roi_is_line=(r.isLine());

	if (roi_is_line) {

		Polygon hull=r.getConvexHull();
	
		double x1,x2,y1,y2,yy;
		
		if (hull.npoints>2){ 			// line width > 2

			x1=hull.xpoints[0]; y1=hull.ypoints[0]; 
			x2=hull.xpoints[3]; y2=hull.ypoints[3]; 

		} // if hull.npoints>2
		else {	

			x1=hull.xpoints[0]; y1=hull.ypoints[0]; 
			x2=hull.xpoints[1]; y2=hull.ypoints[1]; 

		} // else

//		changes version 1.03
//
//		previous (faulty) formulas:
//
//			x1=(x1-xoffset+xmin)/pixelsize;
//			x2=(x2-xoffset+xmin)/pixelsize;
//
//		changed to:

		x1=(x1-xoffset)/pixelsize+xmin;
		x2=(x2-xoffset)/pixelsize+xmin;		


		y1=(double)ymax-(y1-yoffset)/pixelsize;
		y2=(double)ymax-(y2-yoffset)/pixelsize;

		slope=(float)((y2-y1)/(x2-x1));
		yintc=(float)(y1-(slope*x1));
		slope=slope*scale_y/scale_x;
		yintc=yintc*scale_y;

		use_line_parameters=true;

	} // if roi_is_line

	if ((regression_stored>0)&&(!use_line_parameters)) {

		slope=(float)stored_slope*scale_y/scale_x;
		yintc=(float)stored_yintc*scale_y;
		use_regression_parameters=true;

	} // if regression_stored>0
	
	GenericDialog gd=new GenericDialog("Deviation map");
	gd.addMessage(	"This will create a map of the data representing the deviation of \n"+ 
					"datapoints in the scatterplot from a given reference line.\n"+
					"Positive/negative sign denotes position above/below the line");



	if ((use_line_parameters) || (use_regression_parameters)) {
		
		String[] str_regressiontypes={"","OLS","RMA","MA"};

		String s1="";
		if (use_line_parameters) 		s1="You have drawn a line corresponding to";
		if (use_regression_parameters)	s1="Result of last regression analysis ("+str_regressiontypes[regression_stored]+"):";
		
		gd.addMessage(	s1	+	"\n"+
						"       y="+IJ.d2s(slope*scale_x/scale_y,2)+"*x+"+IJ.d2s(yintc/scale_y,2)+" in pixels or\n"+
						"       y="+IJ.d2s(slope,2)+"*x+"+IJ.d2s(yintc,2)+" in calibrated units");

	} // if 

	gd.addNumericField("Slope of reference line:",slope,2);
	gd.addNumericField("y-intercept of reference line",yintc,2);
	gd.addCheckbox("draw reference line in scatterplot",false);
	gd.addChoice("calculation of distance",choices,choices[deviationmap_mode]);
	gd.addCheckbox("only for ROI", ( (!roi_is_line)&&(r!=null) ) ) ;
	gd.addCheckbox("use calibrated units", true);
	gd.addCheckbox("ignore (0,0)", true);
	gd.addMessage("");
	gd.addCheckbox("absolute values",false);

	gd.showDialog();
	if (gd.wasCanceled()) return;

	slope=(float)gd.getNextNumber(); 
	yintc=(float)gd.getNextNumber(); 
	draw_line=(gd.getNextBoolean());
	roi_only=(gd.getNextBoolean());
	calibrated_units=(gd.getNextBoolean());
	ignore_zeros=(gd.getNextBoolean());

	absolute_values=(gd.getNextBoolean());
	mode=gd.getNextChoiceIndex();


	ImageStack stack1=img1.getStack();
	ImageStack stack2=img2.getStack();

	int n=stack1.getSize();
	int w=stack1.getWidth();
	int h=stack1.getHeight();

	voxels1=new int[w][h][n];
	voxels2=new int[w][h][n];

	float[][][] cm=new float[w][h][n];

	boolean[][] selection_contains=get_selection_content(roi_only);

	IJ.showStatus("Preparing...");

	for (x=0; x<w; x++) {
	for (y=0; y<h; y++) {
	for (z=0; z<n; z++) {
		voxels1[x][y][z]=((byte)stack1.getVoxel(x,y,z))&0xff;
		voxels2[x][y][z]=((byte)stack2.getVoxel(x,y,z))&0xff;
	}}} // for x,y,z
	
	float alpha=-(float)Math.atan(slope); 		// negative sign (clockwise direction)
	float sin_alpha=(float)Math.sin(alpha); 	// calculate in advance to save some time later...
	float cos_alpha=(float)Math.cos(alpha);
	
	float x0=0;
	float y0=yintc; // y0=y(x0);


	if (mode==ANGLE){
		x0=(calibrated_units)?(float)xmin*scale_x:(float)xmin;
		y0=(float)slope*x0+yintc;
	} // if mode==ANGLE

	for (z=0; z<n; z++) {
	for (x=0; x<w;  x++) {
	for (y=0; y<h;  y++) {
		
		intgvx=voxels1[x][y][z] ;
		intgvy=voxels2[x][y][z] ;

		gvx=(calibrated_units)?(float)scale_x*intgvx:intgvx;
		gvy=(calibrated_units)?(float)scale_y*intgvy:intgvy;
			
		switch(mode) {
			case YDIST: 
				dist=gvy-(gvx*slope+yintc);			
				break;			
			case PERPENDICULAR: // perpendicular distance: rotate by alpha (clockwise) around y-intercept --> distance = y-coordinate
				dist=sin_alpha*(gvx-x0)+cos_alpha*(gvy-y0); 	
				break;			
			case AREA:
				dist= (float)Math.sqrt( Math.abs((gvy-(slope*gvx+yintc)) * (gvx-(gvy-yintc)/slope)) ); 
				float sign=(gvy>=(slope*gvx+yintc))?1:(-1); // positive if above line
				dist=sign*dist;
				break;
			case ANGLE:
				dist=(float)((Math.atan((gvy-y0)/(gvx-x0))-Math.atan(slope))*360/(2*Math.PI));
				break;
			default: dist=0;
		} // switch
	
		float sign=(dist>=0)?1:-1;
		
		if (absolute_values) dist=(float)Math.abs(dist);

		if ( ((!ignore_zeros) || (gvx+gvy>0)) && (selection_contains[intgvx][intgvy]) && (img_selection[x][y][z]) )	{

			if (dist<dist_min) dist_min=dist;
			if (dist>dist_max) dist_max=dist;

		} // if

		cm[x][y][z]=dist;
		
	}} // for x,y

	IJ.showStatus("Calculating deviation map... "+100*z/n+" %");

	} // for z

	String[] colourscaletype={"RGB 1","RGB 2","greyscale"}; 
	boolean cm_rgb=true;
	int n_colours=1;
	int[] colourscale=new int[1];
	int cstype=0;

	String str_ticks=""+(float)Math.round(dist_min*100)/100+", "+(float)Math.round((dist_min+dist_max)/2*100)/100+", "+(float)Math.round(dist_max*100)/100;

	gd=new GenericDialog("Plot options");
	gd.addMessage("min distance= "+dist_min);
	gd.addMessage("max distance= "+dist_max);
	gd.addNumericField("scale min: ", dist_min,2);
	gd.addNumericField("scale max: ",dist_max,2);
	gd.addChoice("Colour scale type",colourscaletype,colourscaletype[cstype]);
	gd.addStringField("Ticks at",str_ticks,str_ticks.length());
	gd.addStringField("Image title: ",imgtitle,imgtitle.length());
	gd.addNumericField("Set current number to:",n_windows,0);

	gd.showDialog();
	if (gd.wasCanceled()) return;

	dist_min=(float)gd.getNextNumber();
	dist_max=(float)gd.getNextNumber();

	cstype=gd.getNextChoiceIndex();
	cm_rgb=(!(cstype==1));

	str_ticks=gd.getNextString();
	imgtitle=gd.getNextString();
	n_windows=(int)gd.getNextNumber();

	n_windows++;


	int i;

	int R=0x010000, G=0x000100, B=0x000001;

	
	switch (cstype) {

		case 0: { // RGB 1

			n_colours=3*256;
			colourscale=new int[n_colours];

			for (i=0; i<=0xff; i++) { 	
				colourscale[i]=i*(G+B);
				colourscale[0x100+i]=0xff*G+(0xff-i)*B+i*R;
				colourscale[0x200+i]=0xff*R+(0xff-i)*G;

			} // for i
			break;

		} // case 0

		case 1: { // RGB 2: purple-yellow
		
			n_colours=3*256;
			colourscale=new int[n_colours];

			for (i=0; i<=0xff; i++) { 
				colourscale[i]		=	i/2*R		+	i*B;	
				colourscale[256+i]	=	(0x100+i)/2*R	+	(i/2)*G	+	(0xff-i)*B; 			
				colourscale[512+i]	=	0xff*R			+	(0x100+i)/2*G;		
			} // for i
			break;
			
		} // case 1

		case 2: { // greyscale
		
			int gv_start=0x10;
			int gv_end=0xe0;

			n_colours=gv_end-gv_start+1;
			colourscale=new int[n_colours];
		
			for (i=0; i<n_colours; i++) {colourscale[i]=gv_start+i;}
			break;
			
		} // case 2

	} // switch

	int w1=20;
	int w2=30;
	int w3=70;
	int h1=20;

	if (cstype!=2) newimg1=NewImage.createRGBImage(imgtitle,w+w1+w2+w3,h,n,NewImage.FILL_WHITE);
	else newimg1=NewImage.createByteImage(imgtitle,w+w1+w2+w3,h,n,NewImage.FILL_WHITE);

	int h_colourscale=h-50;
	int[] colourscale_for_plot=new int[h_colourscale+1];
	for (i=0; i<=h_colourscale; i++) {
		colourscale_for_plot[h_colourscale-i]=colourscale[(int)Math.round((float)i/h_colourscale*(n_colours-1) )];
	}
	
	String s=str_ticks+" ";

	for (z=0; z<n; z++){

		ip=newimg1.getStack().getProcessor(z+1);

		for (x=0; x<w; x++) {
		for (y=0; y<h; y++) {

			dist=cm[x][y][z];
			if (dist>dist_max) dist=dist_max;
			if (dist<dist_min) dist=dist_min;
			float a=(dist-dist_min)/(dist_max-dist_min)*(n_colours-1);
			int c=colourscale[(int)Math.round(a)];
			
			if ( 	((!ignore_zeros) || (voxels1[x][y][z]+voxels2[x][y][z]>0)) && (selection_contains[voxels1[x][y][z]][voxels2[x][y][z]]) && (img_selection[x][y][z]) ) 
				ip.putPixel( x, y, c );
	
		}} // for x,y

		for (i=0; i<=h_colourscale; i++) {
			int yy=h1+i;
			for (int xx=w+w1; xx<w+w1+w2; xx++){ 
					ip.putPixel( xx, yy, colourscale_for_plot[i] );
			} // for xx
		} // for i

		String nb="";

		for (int pos=0; pos<s.length(); pos++) {
			char c=s.charAt(pos);
			if ( (Character.isDigit(c)) || (c==0x2e) || (c==0x2d) ) nb=nb+c;
			else {
				if (!(nb.equals(""))) {
					ip.setColor(0x000000); // black
					float f=Float.valueOf(nb).floatValue();
					int yy=h1+h_colourscale-Math.round((f-dist_min)/(dist_max-dist_min)*h_colourscale);	
					ip.drawLine( w+w1+w2, yy, w+w1+w2+10, yy );
					ip.drawString( nb, w+w1+w2+15, yy+8 );

					nb="";
				} // if
			} // else
		} // for pos	
		
		IJ.showStatus("Plotting deviation map... "+100*z/n+" %");
	
	} // for z
	
	if (draw_line){ 

		double slope1=(calibrated_units)?slope*scale_x/scale_y:slope;
		double yintc1=(calibrated_units)?yintc/scale_y:yintc;

		int x1=check_limits(xmin,slope1,yintc1,xmin,xmax,ymin,ymax);
		int x2=check_limits(xmax,slope1,yintc1,xmin,xmax,ymin,ymax);

		int y1=(int)Math.round(yintc1+slope1*x1);
		int y2=(int)Math.round(yintc1+slope1*x2);

		ImageProcessor ip1=sp_img.getProcessor();
		ip1.setColor(Toolbar.getForegroundColor());
		ip1.drawLine((x1-xmin)*pixelsize+xoffset,(ymax-y1)*pixelsize+yoffset,(x2-xmin)*pixelsize+xoffset,(ymax-y2)*pixelsize+yoffset);

		sp_img.draw();

	} // if draw_line

	newimg1.setCalibration(img_calibration);
	
	newimg1.show();

	deviationmap_slope=slope;  // to keep setting for next turn
	deviationmap_yintc=yintc;
	deviationmap_mode=mode;


} // deviation_map


// define_colours: open window enabling the user to manipulate the colour scale ==============================

private void define_colours() {

	String[] default_choices={"-----","colours to default RGB","colours to default greyscale (1)","colours to default greyscale (2)","thresholds to powers of 2","to logarithmic thresholds"};
	String[] legend_options={"ribbon","boxes"};
	
	int n_colours=thresholds.length;
	int m=sp_max();
	
	GenericDialog gd=new GenericDialog("Define colours");
	
	int i;
	String s=(xmin+ymin==0)?" (origin excluded)":"";
	gd.addMessage("Define colours and tresholds\nMax. frequency within displayed plot"+s+": "+Integer.toString(m));

	for (i=0; i<n_colours; i++) {
		gd.addNumericField("Threshold",thresholds[i],0);
		gd.addStringField("Colour (RGB)",int2rgb(colours[i]),12);
	} // for
		
	gd.addChoice("Reset...",default_choices,"no");
	gd.addChoice("Legend style",legend_options,legend_options[legend_style]);
	
	gd.showDialog();
	if (gd.wasCanceled()) return;

	for (i=0; i<n_colours; i++) {
		thresholds[i]=(int)gd.getNextNumber();
		colours[i]=rgb2int(gd.getNextString());
	} // for

	int index_reset=gd.getNextChoiceIndex();
	legend_style=gd.getNextChoiceIndex();

	if (index_reset==1) { for (i=0;i<n_colours;i++) {colours[i]=default_RGB[i];} }
	if (index_reset==2) { for (i=0;i<n_colours;i++) {colours[i]=default_greyscale1[i];} }
	if (index_reset==3) { for (i=0;i<n_colours;i++) {colours[i]=default_greyscale2[i];} }
	if (index_reset==4) { for (i=0;i<n_colours;i++) {thresholds[i]=default_thresholds[i];} }
	if (index_reset==5) calculate_log_thresholds(m);

	plot_img();
	
} // define_colours


// calculate_log_thresholds: calculates the thresholds of a logarithmic colour scale

private void calculate_log_thresholds(int m){

	int i;
	
	GenericDialog gd=new GenericDialog("Logarithmic thresholds");
	gd.addSlider("Number of steps:",1,9,9);
	gd.addMessage("In the case of low frequencies, rounding may produce duplicate thresholds.");
	gd.showDialog();
	if (gd.wasCanceled()) return;

	int n_steps=(int)gd.getNextNumber();
	double a=Math.exp(Math.log(m)/n_steps);

	for (i=0;i<thresholds.length;i++){
		if (i<n_steps) thresholds[i]=(int)Math.round(Math.pow(a,i));
		else thresholds[i]=0;
	} // for i

} // calculate_log_thresholds


// plot_img: create a scatterplot... ========================================================

protected void plot_img() {

	if (sp_img!=null) sp_img.close();

	int ww=(xmax-xmin+1)*pixelsize;
	int hh=(ymax-ymin+1)*pixelsize;

	int wimg=xoffset+ww+150;
	int himg=yoffset+hh+100;

	int[][] px=new int[256][256];


	for (int x=0; x<=255; x++) {
	for (int y=0; y<=255; y++) {

		int colour=get_colour(sp[x][y]);
		px[x][y]=colour;

	}} // for x,y

	if (sp_img!=null) sp_img.close();

	sp_img=NewImage.createRGBImage(sp_title,wimg,himg,1,NewImage.FILL_WHITE);

	ImageProcessor ip=sp_img.getProcessor();

	// draw pixels =====================================

	for (int y=ymin; y<=ymax; y++) { 
	for (int x=xmin; x<=xmax; x++) {
	
	for (int xx=0; xx<pixelsize; xx++) {
	for (int yy=0; yy<pixelsize; yy++) {

		ip.setColor(px[x][y]);
		ip.drawPixel((x-xmin)*pixelsize+xx+xoffset,(ymax-y)*pixelsize+yy+yoffset);

	}} // for xx,yy

	}} // for x,y

	// label axes =====================================

	ip.setColor(0x000000);
	ip.setLineWidth(1);

	ip.drawRect(xoffset-1,yoffset-1,ww+2,hh+2);

	int i;
	String nb="";
	String s=str_xtics+" "; // instead of appending " " to str_xticks

	for (i=0; i<s.length(); i++) {

		char c=s.charAt(i);
		if ( (Character.isDigit(c)) || (c==0x2b) || (c==0x2d) || (c==0x2e) || (c==0x45) || (c==0x65) ) nb=nb+c; // if c in "0123456789+-.eE"
		else {
			if (!(nb.equals(""))) {
				draw_tick_x(ip,nb);
				nb="";
			} // if 
		} // else

	} // for i

	nb="";
	s=str_ytics+" ";

	for (i=0; i<s.length(); i++) {

		char c=s.charAt(i);
		if ( (Character.isDigit(c)) || (c==0x2b) || (c==0x2d) || (c==0x2e) || (c==0x45) || (c==0x65) ) nb=nb+c; // if c in "0123456789+-.eE"
		else {
			if (!(nb.equals(""))) {
				draw_tick_y(ip,nb);
				nb="";
			} // if 
		} // else

	} // for i

	draw_legend(ip, xoffset+ww+10, yoffset);

	ImagePlus imgdummy=NewImage.createByteImage("",himg,20,1,NewImage.FILL_WHITE);
	ImageProcessor ipdummy=imgdummy.getProcessor();
	ipdummy.setColor(0x000000);
	ipdummy.moveTo(1,15);
	ipdummy.drawString(title_y);
	int l_ytitle=ipdummy.getStringWidth(title_y);
	int h3=(int)Math.round( (float)(hh-l_ytitle)/2 );
	for (int xx=1; xx<=l_ytitle; xx++) {
	for (int yy=0; yy<20; yy++) {
		if (ipdummy.getPixel(xx,yy)==0) ip.putPixel(yy+10,yoffset+hh-h3-xx,0x000000);
	}} // for xx,yy

	ip.moveTo(xoffset + (int)Math.round( (float)(ww-ip.getStringWidth(title_x))/2 ), yoffset+hh+55);
	ip.drawString(title_x);

	sp_img.show();

} // plot_img


// new_scatterplot: select images/stacks to create a new scatterplot ===========================

private void new_scatterplot() {

	int[] wl=WindowManager.getIDList();
	int x,y,z;

	if ((wl==null)||(wl.length==1)) { 
		IJ.showMessage(str_title,"Two images/stacks needed as input");
		return;
	} //if

	String[] titles=new String[wl.length];

	for (int i=0; i<wl.length; i++) {
		ImagePlus img=WindowManager.getImage(wl[i]);
		if (img!=null)
			titles[i]=img.getTitle();
		else
			titles[i]="";
	} //for

	if (i1>=titles.length) i1=0;
	if (i2>=titles.length) i2=0;
	if ((i1==i2)&&((i2+1)<titles.length)) i2++; // damit zwei verschiedene stacks markiert werden

	GenericDialog gd= new GenericDialog("New scatterplot");
	gd.addMessage("Create a new scatterplot from the selected images/stacks");
	gd.addChoice("stack 1 (x): ",titles,titles[i1]);
	gd.addChoice("stack 2 (y): ",titles,titles[i2]);
	gd.addStringField("title",sp_title,sp_title.length());
	gd.addCheckbox("reset axes",true);

	gd.showDialog();
	if (gd.wasCanceled()) return;

	i1=gd.getNextChoiceIndex();
	i2=gd.getNextChoiceIndex();
	sp_title=gd.getNextString();
	boolean reset_axes=gd.getNextBoolean();

	img1=WindowManager.getImage(wl[i1]);
	img2=WindowManager.getImage(wl[i2]);
	
	if ( (img1.getType()!=ImagePlus.GRAY8) || (img1.getType()!=ImagePlus.GRAY8) ) {
		IJ.showMessage(str_title,"Images must be 8-bit greyscale");
		return;
	} // if
	
	if ( (img1.getStackSize()!=img2.getStackSize()) || (img1.getWidth()!=img2.getWidth()) || (img1.getHeight()!=img2.getHeight()) ) {
		IJ.showMessage(str_title,"Images must have the same dimensions");
		return;
	} // if

	img_calibration=img1.getCalibration();

	filename_x=titles[i1];
	filename_y=titles[i2];

	check_virtualstack(img1.getStack(),filename_x);
	check_virtualstack(img2.getStack(),filename_y);

	Roi roi1=img1.getRoi();
	Roi roi2=img2.getRoi();
	boolean r1=(roi1!=null);
	boolean r2=(roi2!=null);

	int roiindex=0; // no roi selected
	int n1=img1.getStack().getSize();
	int z1=0;
	int z2=n1-1; // so n will not be overwritten in case the ROI-selection dialog is canceled

	if (r1||r2) {

		
		String s="You have selected a ROI ";
		if (!r2) s+="in image "+filename_x;
		if (!r1) s+="in image "+filename_y;
		if (r1&&r2) s+="in both images";
		s+=".\nLimit analysis to " + ((r1&&r2) ? "one of the ROIs?" : "this ROI?");
		
		String[] choices1={"whole stack","ROI in "+filename_x,"ROI in "+filename_y};
		String[] choices2={"whole stack","ROI"};
		
		gd=new GenericDialog("ROI");
		gd.addMessage(s);
		if (r1&&r2) 	gd.addChoice("Confine dataset to",choices1,choices1[1]);
		else 			gd.addChoice("Confine dataset to",choices2,choices2[1]);
		if (n1>1) {
			gd.addMessage("Slices to include:");
			gd.addSlider("First slice",1,n1,z1+1);
			gd.addSlider("Last slice",1,n1,z2+1);
		} // if (n1>1)

		gd.showDialog();
		if (gd.wasCanceled()) return;
		
		roiindex=(int)gd.getNextChoiceIndex();
		if (n1>1) {
			z1=(int)gd.getNextNumber()-1;
			z2=(int)gd.getNextNumber()-1;
			if (z1>z2) {
				IJ.showMessage("Bad range");
				return;
			} // if (z1>z2)
		} // if (n1>1)
		
		if ((roiindex==1)&&(!r1)) roiindex=2;

	} // if (r1||r2)

	get_img_selection_content(roiindex,z1,z2);

	if (reset_axes) {
		title_x=filename_x; 
		title_y=filename_y;
		scale_x=1; scale_y=1;
		xmin=0; xmax=255;
		ymin=0; ymax=255;
		reset_ticks();
		pixelsize=1;
	} // if reset_axes

	regression_stored=0; // because no regression analysis has yet been performed on the new scatterplot

	calculate_scatterplot();

	plot_img();

} //new_scatterplot


// calculate_scatterplot: calculate the frequency matrix ========================

private void calculate_scatterplot() {

	int x,y,z;
	int gvx, gvy;

	for (x=0; x<256; x++) {
	for (y=0; y<256; y++) {
		sp[x][y]=0;
	}} // for x,y

	ImageStack stack1=img1.getStack();
	ImageStack stack2=img2.getStack();

	w=img1.getWidth();
	h=img1.getHeight();
	n=img1.getStack().getSize();

	voxels1=new int[w][h][n];
	voxels2=new int[w][h][n];

	IJ.showStatus("Preparing...");

	for (x=0; x<w; x++) {
	for (y=0; y<h; y++) {
	for (z=0; z<n; z++) {
		voxels1[x][y][z]=((byte)stack1.getVoxel(x,y,z))&0xff;		// & 0xff to eliminate sign
		voxels2[x][y][z]=((byte)stack2.getVoxel(x,y,z))&0xff;
	}}} // for x,y,z

	for (z=0; z<n; z++) {
	for (x=0; x<w; x++) {
	for (y=0; y<h; y++) {

		gvx=voxels1[x][y][z];
		gvy=voxels2[x][y][z];
		
		if (img_selection[x][y][z]) { sp[gvx][gvy]++;}

	}} // for x,y
	IJ.showStatus("Calculating... "+100*z/n+" %");
	} // for z

} // calculate_scatterplot


// define_axes: diaolog allowing the user to calibrate the axes of the scatterplot ======================

private void define_axes() {

	GenericDialog gd=new GenericDialog("Define axes");
	
	gd.addMessage("Plot limits:");
	gd.addNumericField("x min",xmin,0);
	gd.addNumericField("x max",xmax,0);
	gd.addNumericField("y min",ymin,0);
	gd.addNumericField("y max",ymax,0);
	gd.addNumericField("pixel size",pixelsize,0);

	gd.addMessage("Axis labeling:");
	int wstrfield=(title_x.length()>title_y.length())?title_x.length():title_y.length();
	gd.addStringField("title x",title_x,wstrfield);
	gd.addStringField("title y",title_y,wstrfield);
	gd.addNumericField("calibration factor x",scale_x,6);
	gd.addNumericField("calibration factor y",scale_y,6);

	wstrfield=(str_xtics.length()>str_ytics.length())?str_xtics.length():str_ytics.length();
	gd.addStringField("draw x-ticks at",str_xtics,wstrfield);
	gd.addStringField("draw y-ticks at",str_ytics,wstrfield);
	gd.addCheckbox("draw ticks at 0%, 50%, 100%", false);

	gd.showDialog();

	if (gd.wasCanceled()) return;

	xmin=(int)gd.getNextNumber();
		if (xmin<0) xmin=0;
	xmax=(int)gd.getNextNumber();
		if (xmax>255) xmax=255;
	ymin=(int)gd.getNextNumber();
		if (ymin<0) ymin=0;
	ymax=(int)gd.getNextNumber();
		if (ymax>255) ymax=255;
	pixelsize=(int)gd.getNextNumber();
		if (pixelsize<1) pixelsize=1;

	title_x=gd.getNextString();
	title_y=gd.getNextString();
	scale_x=(float)gd.getNextNumber();
	scale_y=(float)gd.getNextNumber();

	str_xtics=gd.getNextString();
	str_ytics=gd.getNextString();
	boolean resetticks=gd.getNextBoolean();	

	if (resetticks) reset_ticks();

	plot_img();

} // define_axes


// export_data: export data to an ASCII file =======================

private void export_data() {

	char _separator=0x09; // tab

	String s;
	String[] options={"Matrix","List"};
	int mode,i;
	boolean calibrated_units,ignore_zeros;
	
	GenericDialog gd=new GenericDialog("Export data...");
	gd.addMessage("Export data to ASCII file");
	gd.addChoice("Format:",options,options[0]);
	gd.addCheckbox("Use calibrated units",true);

	gd.showDialog();
	if (gd.wasCanceled()) return;
	mode=gd.getNextChoiceIndex();
	calibrated_units=gd.getNextBoolean();

	
	SaveDialog sd=new SaveDialog("Export data...",sp_title,".dat");
	String dir=sd.getDirectory();
	String fn=sd.getFileName();
	if (fn==null) return;

	try {

		BufferedWriter bw=new BufferedWriter(new FileWriter(dir+fn));
		
		switch(mode){   
			case 0:	// matrix
				break;		
			case 1:	// list 1
				s	=	"x"	+	_separator	+	"y"	+	_separator	+	"n"	+	"\r\n";
				bw.write(s);	
				break;	
			default: {}	
		} // switch mode
	
		for (int y=0; y<256; y++) {
		for (int x=0; x<256; x++) {
		
			switch(mode){
			
				case 0:	// matrix
					s	=	Integer.toString(sp[x][y])	+	_separator;
					if (x==255) {s=s+"\r\n";}
					bw.write(s);
					break;		

				case 1:	// list
					if (sp[x][y]>0) {
						if (calibrated_units){
							s	=	Float.toString((float)x*scale_x)	+	_separator	+	Float.toString((float)y*scale_y)	+	_separator	+	Integer.toString(sp[x][y])	+	"\r\n";
						} // if calibrated_units
						else {
							s	=	Integer.toString(x)	+	_separator	+	Integer.toString(y)	+	_separator	+	Integer.toString(sp[x][y])	+	"\r\n";
						} // else
						bw.write(s);
					} // if sp[x][y]>0
					break;

				default: {}
				
			} // switch mode
			
		}} // for x,y

		bw.close();

	}  // try
	catch (Exception e) {
		IJ.error(str_title, e.getMessage());
		return;
	} // catch

} // export_data


// smaller functions: =========================================================================================
// ============================================================================================================
// ============================================================================================================


private void reset_ticks(){
	str_xtics=double2str(scale_x*xmin,2,3)+", "+double2str(scale_x*(xmin+xmax)/2,2,3)+", "+double2str(scale_x*xmax,2,3);
	str_ytics=double2str(scale_y*ymin,2,3)+", "+double2str(scale_y*(ymin+ymax)/2,2,3)+", "+double2str(scale_y*ymax,2,3);
} // reset_ticks


private void update_frame() {

	label_x.setText("x :"+filename_x);
	label_y.setText("y: "+filename_y);

} //update_frame


protected int sp_max() {

	int m=0;

	for (int x=xmin; x<=xmax; x++) {
	for (int y=ymin; y<=ymax; y++) {
			if ( (sp[x][y]>m) && ( (x+y)>0 ) ) m=sp[x][y];	// sp_max = max. pixel value
	}} // for x,y
	
	return m;

} // sp_max


private void draw_tick_x(ImageProcessor ip, String s) {

	double d=Double.parseDouble(s)/scale_x;
	int y=yoffset+(ymax-ymin+1)*pixelsize;
	int x=xoffset+((int)Math.round(d)-xmin)*pixelsize;
	ip.drawLine(x,y,x,y+9);
	ip.moveTo(x-(int)Math.floor(ip.getStringWidth(s)/2),y+25);	
	ip.drawString(s);

} // draw_tick_x


private void draw_tick_y(ImageProcessor ip, String s) {

	double d=Double.parseDouble(s)/scale_y;
	int x=xoffset;
	int y=yoffset+(ymax-(int)Math.round(d)+1)*pixelsize-1;
	ip.drawLine(x-9,y,x-1,y);
	ip.moveTo(x-ip.getStringWidth(s)-15,y+5);
	ip.drawString(s);

} // draw_tick_y


private int get_colour(int a) {

	int c=0;
	int max_colours=thresholds.length;

	if (a==0) c=0xffffff;
	else { 	
		for (int i=0; i<max_colours; i++) { 
			if ((a>=thresholds[i])&&(thresholds[i]!=0)) c=colours[i]; }
	} // else 

	return c;

} // get_colour


private void draw_legend(ImageProcessor ip, int x, int y) {

	int n=thresholds.length;
	int i=-1;
	int y1,y2=y;
	int ii;
	Polygon p;

	if (legend_style==1){ 	// "boxes" option
	
		for (ii=0; ii<n; ii++) {
			if (thresholds[ii]!=0) {
				i++;
				ip.setColor(colours[i]);

				p=new Polygon();
					p.addPoint(x,y+i*15);
					p.addPoint(x+20,y+i*15);
					p.addPoint(x+20,y+i*15+10);
					p.addPoint(x,y+i*15+10);
	
				ip.fillPolygon(p);		// colour area
				ip.setColor(0x000000);
				ip.drawPolygon(p); 		// black outline
				ip.moveTo(x+25,y+i*15+13);
				String s=Integer.toString(thresholds[ii]);
				ip.drawString(s);
			} //if
		} // for ii

	} // if legend_style==1
	else { 	// "ribbon" option
		i=0;
		for (ii=0;ii<n;ii++){
			if (thresholds[ii]>0) {
				i++;
				y1=y+i*15;
				y2=y+(i+1)*15;
				p=new Polygon();
					p.addPoint(x,y1);
					p.addPoint(x+20,y1);
					p.addPoint(x+20,y2);
					p.addPoint(x,y2);
				ip.setColor(colours[ii]); ip.fillPolygon(p);
				ip.setColor(0); ip.drawLine(x,y1,x+25,y1);
				ip.drawString(Integer.toString(thresholds[ii]),x+30,y1+8);
			} // if thresholds>0
		} // for ii
		
		p=new Polygon();
			p.addPoint(x,y);
			p.addPoint(x+20,y);
			p.addPoint(x+20,y2);
			p.addPoint(x,y2);
		ip.drawPolygon(p); 


	} // else 

} // draw_legend


private boolean[][] get_selection_content(boolean roi_only){

	// returns a 256 x 256 boolean matrix 
	// "true" if respective pixel is within the roi selected in the scatterplot
	// "false" if outside roi or ouside the limeits of the plot 

	int x,y;
	boolean[][] roi_content=new boolean[256][256];
	Roi r=sp_img.getRoi();

	if ((r==null)||(!roi_only)) {	// select all pixels of the plot
		for (x=0;x<=255;x++){
		for (y=0;y<=255;y++){
			roi_content[x][y]=((x>=xmin)&&(x<=xmax)&&(y>=ymin)&&(y<=ymax)); // so all points will be part of the selection
		}} // for x,y
	} // if 
	else {
		for (x=0;x<=255;x++){		// select only roi
		for (y=0;y<=255;y++){
			roi_content[x][y]=(	(r.contains((x-xmin)*pixelsize+xoffset,(ymax-y)*pixelsize+yoffset))
						&& ( (x>=xmin)&&(x<=xmax)&&(y>=ymin)&&(y<=ymax) )	);
		}} // for x,y
	} // else

	return roi_content;

} // scan_roi


private String double2str(double d, int a, int b){

	if (d==0) return "0"; else return IJ.d2s(d,a,b);

} // double2str

private void get_img_selection_content(int roiindex, int z1, int z2){

	int x,y,z;
	boolean inroi;
	Roi r=(roiindex==1)?img1.getRoi():img2.getRoi();;

	w=img1.getWidth();
	h=img1.getHeight();
	n=img1.getStack().getSize();

	img_selection=new boolean[w][h][n];

	for (x=0; x<w; x++) {
	for (y=0; y<h; y++) {
	for (z=0; z<n; z++) {

		img_selection[x][y][z]=true;

	}}} // for x,y,z

	for (x=0; x<w; x++) {
	for (y=0; y<h; y++) {

		if (roiindex>0) { inroi=r.contains(x,y); }
		else { inroi=true; }
		for (z=0; z<n; z++) {
			img_selection[x][y][z] = ((z>=z1)&&(z<=z2)) ? inroi : false;
		} // for z

	}} // for x,y,z
	
} // get_img_selection_content


private boolean within_roi(int x, int y, boolean[][] roi_contains){

	boolean within=false;

	if ((x>=0)&&(x<256)&&(y>=0)&&(y<256)){
		within=(roi_contains[x][y]);
	} // if

	return (within);

} // within_ro


private int rgb2int(String s) {

	String s1="";
	int i;
	int[] colour={0,0,0};
	char c;
	int k=0;

	s+=" ";

	for (i=0;i<s.length();i++){
		c=s.charAt(i);
		if ((c>=0x30)&&(c<=0x39)) s1+=c;
		else {
			if ((s1.length()>0)&&(k<3)) {
				colour[k]=Integer.valueOf(s1);
				if ((colour[k]<0)||(colour[k]>255)) colour[k]=0;
				s1="";
				k++;
			} // if s1.length()>0
		} // else
	} // for i

	return ( 0x010000*colour[0] + 0x000100*colour[1] +  0x000001*colour[2] ); 

} // rgb2int


private String int2rgb(int a) {

	String s="";
	s+=Integer.toString((0xff0000 & a)/0x010000)+", ";
	s+=Integer.toString((0x00ff00 & a)/0x000100)+", ";
	s+=Integer.toString( 0x0000ff & a);	
	return s;

} // int2rgb


private int check_limits(int x, double slope, double yintc, int lowerx, int upperx, int lowery, int uppery){

	if (x<lowerx) x=lowerx;
	if (x>upperx) x=upperx;
	if (slope*x+yintc>uppery) x=(int)Math.round((float)(uppery-yintc)/slope);
	if (slope*x+yintc<lowery) x=(int)Math.round((float)(lowery-yintc)/slope);
	return x;
	
} // check_limits

// check_ImageJ_version (added for version 1.01; adjusted for version 1.04) ===========

private void check_ImageJ_version(){

	String v=IJ.getVersion();

// added for version 1.04: return if first digit > 1
	if (v.charAt(0)>0x31) return;

	double a= Double.parseDouble(v.substring(0,4));
	boolean show_error=false;

	if (a<1.46) show_error=true;

	if (a==1.46){
		if ((int)(v.charAt(4))<102) show_error=true; 	// i.e. if version < 1.46f
	} // if 

	if (show_error) IJ.showMessage("You are running ImageJ version "+v+".\nTo run properly, ScatterJ requires version 1.46f or higher.");
	

} // check_ImageJ_version

// check_virtualstack (added for version 1.02) ===========

private void check_virtualstack(ImageStack st, String s){


	if (st.isVirtual()) IJ.showMessage(s+" is a virtual stack.\n \nScatterJ cannot use virtual stacks as input images.\n \nTo convert a virtual into a non-virtual stack,\n - duplicate it, or\n - save and re-load it.");
	

} // check_virtualstack



// show_info ============================================================================================


private void show_info(){

		String s=										"=========================================="
					+"\n"
					+"\n  ScatterJ, version 1.04"
					+"\n  Copyright (C) 2014 Fabian Zeitvogel, University of Tuebingen"
					+"\n"
					+"\n  This program is free software: you can redistribute it and/or modify"
					+"\n  it under the terms of the GNU General Public License as published by"
					+"\n  the Free Software Foundation, either version 3 of the License, or"
					+"\n  (at your option) any later version."
					+"\n"
					+"\n  This program is distributed in the hope that it will be useful,"
					+"\n  but WITHOUT ANY WARRANTY; without even the implied warranty of"
					+"\n  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the"
					+"\n  GNU General Public License for more details."
					+"\n"
					+"\n  You should have received a copy of the GNU General Public License"
					+"\n  along with this program. If not, see <http://www.gnu.org/licenses/>."
					+"\n"
					+"\n"
					+"\n=========================================="
					+"\n"
					+"\nThe purpose of this plugin is to enable the comparison of two images and to extract statistical and spatial  information based on their pixel values. It is intended for use with analytical-microscopy data such as  chemical/species maps from various types of microscopy. The input data must consist of two equally sized and  aligned 8-bit greyscale images or stacks. Pixels at the same coordinates in the images are assumed to represent  the same position within the sample; grey values are assumed to represent quantitative or semi-quantitative  information of some kind. Input images must be loaded in ImageJ. The various functions of the plugin are  explained as follows:"
					+"\n"
					+"\n"
					+"\nCreate Scatterplot"
					+"\n"
					+"\nThis function allows the user to select the input images in order to create the actual scatterplot (also called  2-dimensional histogram). A scatterplot has to be created before using any other function. The data contained in  the two input images or image stacks are considered to form a set of 2D datapoints whose coordinates are defined  by the grey values of respective pixels in the two images. These datapoints are drawn in a coordinate system. As  the scale is discrete, in general, multiple datapoints will have the same coordinates. Such multiple occurrences  are indicated by a colour scheme in which the point is drawn. This plugin uses a discrete colour scale, i.e. the  same colour is used for a range of numbers. While this eliminates some of the details, it greatly enhances the  clarity of representation of the whole scatterplot."
					+"\nThe \"Reset axes\" option will reset all adjustments concerning the axes (see below) to the default values."
					+"\n"
					+"\nIt is possible to confine the analysis to selected parts of the image. Therefore, the area of interest has to be  highlighted using ImageJ's region of interest (ROI) tools. If both images feature a ROI, you will be asked to  select one. If a ROI is selected in a stack, you will additionally be asked to define the range of slices to  include. Any such selection has to be done before calculating the scatterplot, i.e. before pressing the [Create  Scatterplot] button. At any later point, selecting a ROI in one of the original images will not have any effect."
					+"\n"
					+"\n"
					+"\nReplot"
					+"\n"
					+"\nThis function will re-do the scatterplot using the current settings, but deleting any further manipulations such  as lines or ROIs drawn in the plot."
					+"\n"
					+"\n"
					+"\nColour scale "
					+"\n"
					+"\nBy default, the scatterplot is represented using an RGB colour scale. The default threshold values (the values  that separate the ranges of numbers to be displayed by the same colour) are at subsequent powers of 2. Both the  colours and the thresholds can be modified. The dropdown menu \"Reset...\" provides the possibility to reset the  colour scale to default RGB scale or greyscale settings. Likewise, thresholds can be set to powers of 2 or to a  logarithmic scale adjusted to the present distribution. Colours can also be manually adjusted by entering RGB  values. This can be used to standardize the appearance of scatterplots or to enhance particular features. A  colour scale legend will be drawn next to the scatterplot, which can be displayed as a ribbon or as discrete  boxes."
					+"\n"
					+"\n"
					+"\nAxes"
					+"\n"
					+"\nThe \"Axes\" function can be used to adjust the size and labeling of the scatterplot as well as to introduce a  calibration factor. In detail, it is possible to "
					+"\n"
					+"\n- define labels for the axes (e.g. \"Fe-Ka signal (a.u.)\");"
					+"\n- enter a calibration factor defining how grey values translate into meaningful units (e.g. of concentration);"
					+"\n- define the \"ticks\" at the axes; a sequence of values can be defined that will be displayed in exactly the same  way as they are entered (e.g. \"2.1e-1\", \"1.000\", \"200\");"
					+"\n- limit the lengths of the axes (in pixels); areas not displayed in the plot will not be considered by further  operations such as statistical calculations or backmapping (see below);"
					+"\n- choose a larger pixel size, which is particularly useful if only a small part of the data range is being  displayed."
					+"\n"
					+"\n"
					+"\nStatistics"
					+"\n"
					+"\nThe \"Statistics\" function is mainly intended for linear regression analysis. In addition, Pearson's coefficient  as well as a number of general parameters (arithmetic mean, standard deviation) can be calculated. Three types of  linear regression are implemented, i.e. ordinary least squares (OLS), major axis (MA), and reduced major axis  (RMA) regression. For major axis regression, it is possible to draw the principle/major axis, the minor axis, and  the 95% confidence ellipse of an assumed bivariate normal distribution. For a more detailed description of the  different methods please refer to literature."
					+"\n"
					+"\nIn general, these calculations will be limited to the displayed area of the scatterplot, or to a subregion  thereof that can be selected by drawing a region of interest (ROI) and activating the respective checkbox (\"only  for ROI\"). In many cases, pixels that are zero in both images are not considered to be part of the actual data -  these can be excluded from the calculations by activating the checkbox \"ignore (0,0)\". If \"use calibrated units\"  is activated, the statistics will be calculated in units of the grey values multiplied with the calibration  factor (see \"Axes\"); otherwise the numbers of the grey values themselves will be used. Note that this can affect  the outcome of major axis regression when the lines are drawn in the plot. For any type of regression analysis  selected, activating the checkbox \"show results\" will display results in a text window, while \"draw line\" will  draw the respective line in the scatterplot using the current line width and foreground colour. "
					+"\n"
					+"\nNote that the number of decimals displayed is in most cases far beyond what is justified considering the nature  of the data. "
					+"\n"
					+"\n"
					+"\nBackmapping"
					+"\n"
					+"\nBackmapping is a simple, but useful and efficient tool to locate the areas in the original images corresponding  to visually identified clusters in the scatterplot. A region of the scatterplot that appears to be of particular  interest, e.g. because of evidently forming a cluster, can be selected by drawing a ROI using one of ImageJ's ROI  selection tools. (This has to be done before clicking the button.) The plugin identifies those pixels in the  original images that correspond to the selected area in the scatterplot and displays them in a new image window. "
					+"\n"
					+"\nTwo options are available:"
					+"\n- \"black & white mask\": selected pixels will be displayed in white, non-selected ones in black;"
					+"\n- \"greyscale images\": instead of simply highlighting selected pixels in white, the original grey values will be  used; naturally, two images will be created, one corresponding to each original image."
					+"\n"
					+"\nWhen new images are created for ROI backmapping or deviation maps (see below), the default image title starts  with a running number. This is meant to help distinguish those images. The counter can be set to a user-defined  integer value."
					+"\n"
					+"\n"
					+"\nDeviation maps"
					+"\n"
					+"\nDeviation maps display, for each pixel position of the original images, the distance of its respective datapoint  in the scatterplot from a given reference line. The reference line is defined by the user by entering its slope  and y-intercept or by drawing it using ImageJ's line-ROI tool. It can be chosen arbitrarily e.g. in order to  separate interesting regions within the scatterplot; alternatively, the parameters of a previously calculated  regression line can be used. Distances of datapoints from the reference line are represented using a colour  scale. The \"distance\" to by shown can be defined as the vertical distance (OLS), orthogonal distance (MA), or  square root of the product of horizontal and vertical distances (RMA). These quantities correspond to the  quantities minimized by OLS, MA, and RMA regression, respectively. They convert into each other by a  multiplication factor depending on the slope of the reference line. An additional option can be used to map the  angle between the reference line and the position vector of each datapoint; for calculation of the angles, the  origin is set to the intercept point of the reference line and the currently displayed y-axis."
					+"\nAfter calculation, a second window allows the user to adjust the colour scale settings. If calculated values  exceed the user-defined limits, the respective pixel will be saturated, i.e. assigned the maximum or minimum  value of the colour scale. Three colour scales are offered; \"RGB 1\" and \"RGB 2\" are inspired by (but not  identical to) ImageJ's \"fire\" colour scale and a colour scale found in aXis2000 (a software package used for the  analysis of synchrotron-based X-ray microscopy data) respectively."
					+"\nAgain, only the region of the scatterplot that is within the axis limits is considered, which can be further  limited to a subregion by drawing a ROI and activating the respective checkbox. If the checkbox \"use calibrated  units\" is activated, the calculations will be done using the calibrated values (see above). Double-zero pixels  (see above) can be excluded by activating the \"ignore (0,0)\" checkbox. Excluded pixels will be displayed in  white, which is not part of any of the provided colour scales."
					+"\n"
					+"\n"
					+"\nExport data"
					+"\n"
					+"\nThe scatterplot data can be exported to ASCII files. Two formats are provided. In \"matrix\" format, the data will  be saved as a 256 x 256 matrix corresponding to the scatterplot; the \"list\" format will produce a table showing  x, y, and the number of occurrence. Naturally, the \"use calibrated units\" option only applies to the latter  format."
					+"\n"
					+"\n"
					+"\nInfo"
					+"\n"
					+"\nWill display this window."
					+"\n"
					+"\n"
					+"\n"
					+"\n";


		TextArea ta=new TextArea(s,50,100,TextArea.SCROLLBARS_VERTICAL_ONLY);
		ta.setEditable(false);
		GenericDialog gd=new GenericDialog(str_title+" info");
		gd.add(ta);
		gd.addMessage("");
		gd.hideCancelButton();
		gd.setModal(false);
		gd.showDialog();

} // show_info

} // main
