(defpackage :edu.unm.hpc.Scat.installer
  (:nicknames :Scat.installer :installer)
  (:use :cl
        :sb-mop
        :sb-ext
        :cl-user
        :Scat.Lib
        :Scat.cfg
        :cl-cli
        ))

(in-package :edu.unm.hpc.Scat.installer)

(require 'sb-bsd-sockets)

(setf *production-mode* NIL)

(defvar *logstream* ())
(defvar *start* ())

;; write a log to the screen and, optionally the logfile
(defmacro write-log (&rest fmtargs)
   `(progn
    (if *logstream*
     (format *logstream* ,@fmtargs))
    (format t ,@fmtargs)))

(defgeneric ensure-directory (directory &optional perm))

(defmethod ensure-directory ((directory string) &optional (perm #o755))
  (let ((pieces (remove-if (lambda (x) (equal x "")) 
                           (cl-ppcre::split "/" directory)))
        (current ""))
    (mapcar 
     (lambda (thing) 
       (setf current (format () "~A/~A" current thing))
       (unless (directory current)
         (SB-UNIX:UNIX-MKDIR current perm)))
     pieces)))

(defmethod ensure-directory ((directory pathname))
  (ensure-directory (namestring directory)))

(defun hostname->ip (name) 
  (let ((hostent (car (sb-bsd-sockets:host-ent-addresses 
                       (sb-bsd-sockets:get-host-by-name name)))))
    (format () "~A.~A.~A.~A" 
            (aref hostent 0) 
            (aref hostent 1) 
            (aref hostent 2) 
            (aref hostent 3))))

(defun ip->hostname (ip)
  (sb-bsd-sockets:host-ent-name 
   (sb-bsd-sockets:get-host-by-address 
    (car 
     (sb-bsd-sockets:host-ent-addresses 
      (sb-bsd-sockets:get-host-by-name ip))))))


(defun banner (stream message)
  (let ((foo (make-string 79 :initial-element #\#)))
    (format stream "~A~%# ~A~78T#~%~A~%" 
            foo message foo)))

;; this /could/ be an IP, or it could be a hostname
(defvar temp-hostname ())
;; resolve above to a hostname, then an IP
(defvar hostname ())
(defvar ip ())
(defvar *remote-logdir* ())
(defvar *local-logdir* ())
(defvar *victim-fs* ())
(defvar *logfile* ())
(defvar *third-octet* ())
(defvar *magic-third-octet-value* 250)
;; Do I need this?!
;; (defvar *primary-ethernet-device* "en1")
(defvar *mount-prog* 
  #+DARWIN
  "/sbin/mount"
  #+FREEBSD
  "/sbin/mount"
  #+LINUX
  "/bin/mount")

;; External Programs
(defvar *shutdown-prog* "/sbin/shutdown")
(defvar *ssh-prog* "/usr/bin/ssh")
(defvar *dd-prog* "/bin/dd")
(defvar *mkfs-prog* "/sbin/mkfs")
(defvar *mkswap-prog* "/sbin/mkswap")
(defvar *portmap-prog* 
  #+DARWIN
  "/usr/sbin/portmap"
  #+FREEBSD
  "/usr/sbin/portmap"
  #+LINUX
  "/sbin/portmap")

(defun new-node ()
  (let ((hardware-address
         (get-net-dev-hw-addr (find-dev-with-addr ip *network-devices*))))
    (write-log "this is a new node~%")
    (write-log "my ethernet device is: ~A~%" Scat.cfg::*ethernet-device*)
    (write-log "its address: ~A~%" hardware-address)
  (register-node hardware-address)
  (reboot)))

(defun register-node (mac)
  (write-log "registering mac: ~A~%" mac)
  (ssh *fileserver* 
       (format ()
               "/opt/netinstall/bin/addnode.sh ~A; /etc/init.d/dhcpd restart"  
               mac)))

(defmacro find-matching-strings (regexp target-string)
  (let ((matches (gensym))
        (sub-matches (gensym)))
    `(multiple-value-bind (,matches ,sub-matches)
         (cl-ppcre:scan-to-strings ,regexp ,target-string)
       (when ,matches
         ,sub-matches))))

(defmacro nth-submatch (n regexp target-string)
  (let ((string-arr (gensym)))
    `(let ((,string-arr (find-matching-strings ,regexp ,target-string)))
       (WHEN ,string-arr
         (aref ,string-arr ,n)))))
        
(defmacro list-matching-strings (regexp target-string)
  (let ((string-arr (gensym))
        (len (gensym))
        (i (gensym)))
    `(let* ((,string-arr (find-matching-strings ,regexp ,target-string))
            (,len (length ,string-arr)))
       (loop for ,i from 0 below ,len 
            collect (aref ,string-arr ,i)))))

(defun safe-ip->hostname (ip)
  (handler-case (ip->hostname ip)
    (SB-BSD-SOCKETS:HOST-NOT-FOUND-ERROR () "FAKE.prewett.org")))

(defun mount-filesystem (special &optional node)
  (write-log "mounting filesystem: ~A onto: ~A~%" special node)
  (when *production-mode*
    (or 
     (equal 0
            (process-exit-code
             (if node
                 (run-program *mount-prog* `(,special ,node))
                 (run-program *mount-prog* `(,special)))))
     (error "error mounting filesystem: ~A ~A" special node))))

(defun ssh (remote-host &rest args)
  (if *production-mode*
      (apply log-command-output *ssh-prog* remote-host args)
      (write-log "should run: ~A ~A ~{~A~}~%" 
                 *ssh-prog* remote-host args)))

(defun start-portmap ()
  (write-log "starting portmapper~%")
  (when *production-mode*
    (run-program *portmap-prog* '("-i" "127.0.0.1" "-v"))))

;; XXX shouldn't these mount options be SCAT options? :P
(defun mount-tmp ()
  (write-log "mounting tempfs based /tmp/~%")
  (when *production-mode*
    (run-program *mount-prog* 
                 '("-t" "tmpfs" "-o" "rw,size=20M" "tmpfs" "/tmp/"))))

(defun reboot ()
  (write-log "rebooting~%")
  (when *production-mode*
    (run-program *shutdown-prog* '("-r" "now") :wait t :output ()))
  (cl-user::quit))

(defun get-ethernet-info ()
  (run-program "/sbin/ifconfig" () :output :stream :wait t))

(defun find-net-dev-info ()
  "find information about network devices on this system"
  (loop 
     with output = (make-string-output-stream) 
     with results = ()
     with stream = 
       (process-output (run-program "/sbin/ifconfig" () 
                                    :OUTPUT :stream :wait t))
     as line =  (read-line stream () ()) 
     while line 
     do
       (if (cl-ppcre::scan "^\\S+:" line)
           (progn
             (let ((outstr (get-output-stream-string output)))
               (unless (equal "" outstr)
                 (setf results 
                       (append results `(,(subseq outstr 0 
                                          (- (length outstr) 1))))))
               (setf output (make-string-output-stream)))))
       (format output "~A~%" line) 
     finally
       (let ((outstr (get-output-stream-string output)))
         (unless (equal "" outstr)
           (setf results 
                 (append results `(,(subseq outstr 0 
                                            (- (length outstr) 1)))))))
       (return results)))

(defun find-dev-with-addr (addr device-list)
  (find-if (lambda (str)
             (cl-ppcre::scan
              #+DARWIN
              (format () "inet ~A" addr)
              #+LINUX
              (format () "inet addr:~A" addr)
              str)) device-list))

(defun get-net-dev-hw-addr (device-string)
  (multiple-value-bind (matches sub-matches)
      (cl-ppcre::scan-to-strings 
       #+DARWIN 
       "ether (\\S+)"
       #+LINUX
       "HWaddr (\\S+)"
       device-string)
    (when matches 
      (aref sub-matches 0))))

(defun snarf-file (file)
  (let ((out (make-string-output-stream)))
    (with-open-file (stream file :direction :input)
      (loop as line = (read-line stream () ())
           while line
           do
           (format out "~A~%" line)))
    (let ((outstr (get-output-stream-string out)))
      (subseq outstr 0 (- (length outstr) 1)))))

;; this does not work with 'dan' style sfdisk input files
(defun list-partitions-from-file (filename)
  (let* ((input (snarf-file filename))
         (s (make-string-input-stream input))
         (ret ()))
    (loop as line = (read-line s () ())
         while line
         do
         (let ((matching (find-matching-strings "^(/dev/\\S+)" line)))
           (when matching
             (setf ret 
                   (list* 
                    (aref matching 0) ret)))))
    (nreverse ret)))

(defun list-filesystems-from-file (filename)
  (let* ((input (snarf-file filename))
         (s (make-string-input-stream input))
         (ret ()))
    (loop as line = (read-line s () ())
         while line
         do
         (let ((matching 
                (find-matching-strings 
                 "^(/dev/\\S+)\\s+\\S+\\s+(\\S+)" line)))
           (when matching
             (setf ret 
                   (list* 
                    `(,(aref matching 0) ,(aref matching 1)) ret)))))
    (nreverse ret)))

(defun get-partition-description-files (image-dir)
  (directory 
   (format () "~A/*.sfdisk" image-dir)))

(defun run-sfdisk (partition description-file)
  (run-program *sfdisk-prog* `(,partition)))

(defun log-command-output (command &rest args)
  (let ((output
         (process-output
          (run-program command args :wait t :output :stream))))
       (loop as line = (read-line output () ())
          while line
          do
            (write-log "~A~%" line))))

(defun make-filesystem (type device)
  (if (equal type "swap")
      (if *production-mode*
          (let ((process 
                 (run-program *mkswap-prog* 
                              `(,device) :wait t :output :stream)))
            (if (zerop (process-exit-code process))
                (progn
                  (write-log "made swap on ~A~%" device)
                  t)
                (error "error making swap filesystem on device: ~A~%" device)))
          (write-log "should run ~A on ~A~%" *mkswap-prog* device))
      (if *production-mode*
          (let ((process
                 (run-program *mkfs-prog* 
                              `("-t" ,type ,device) :wait t :output :stream)))
            (if (zerop (process-exit-code process))
                t
                (error "error making filesystem type: ~A on device: ~A~%"
                       type device)))
          (write-log "should run ~A to make filesystem type: ~A on device: ~A~%"
                     *mkfs-prog* type device))))

(defun wipe-disk (disk)
  (if *production-mode* 
      (log-command-output 
       *dd-prog* "if=/dev/zero" (format () "of=~A" disk) "bs=512" "count=1")
      (write-log "should run: ~A ~A ~A ~A ~A~%" *dd-prog* "if=/dev/zero" 
                 (format () "of=~A" disk) "bs=512" "count=1")))


(defun build-filesystem (disk)
  (let ((fs-type 
         (cadr 
          (assoc disk 
                 (list-filesystems-from-file 
                  "/netinstall/NFSRoot/install_scripts/images/compute/fstab")
                 :test #'equal))))
    (if fs-type
        (progn
          (wipe-disk disk)
          (make-filesystem fs-type disk))
        (write-log "no filesystem for: ~A~%" disk))))

(defun toplevel () 
  (read-scat-config)
  (defparameter *start* (get-universal-time))

  (banner t "JIM'S MAGIC INSTALL LISP PROGRAM")

  ;; start up the portmapper
  (start-portmap)
  (mount-filesystem "/proc/")

  (banner t "information about your System:")
  
  (defparameter network-devices (find-net-dev-info))

  (defparameter temp-hostname (sb-unix:unix-gethostname))
  (defparameter hostname (safe-ip->hostname temp-hostname))
  (defparameter short-hostname (nth-submatch 0  "(\\w+)\\." hostname))
  (defparameter ip (hostname->ip temp-hostname))
  (defparameter *third-octet* 
    (read-from-string
     (or (nth-submatch 0 ".*\\..*\\.(.*)\\..*" ip) "NIL")))

  (write-log "third octet: ~A~%" *third-octet*)

  (if (> *third-octet* *magic-third-octet-value*)
      (new-node)
      (write-log "this node already registered~%"))

  (defparameter image-dir 
    (format () "~A/~A" Scat.cfg::*NODE-IMAGE-DIR* short-hostname))

  (defparameter partition-descriptions
    (get-partition-description-files image-dir))

  ;; mount the log filesystem
  (mount-filesystem           
   (format () "~A:~A" 
           scat.cfg::*fileserver*
           scat.cfg::*remote-log-dir*)
   scat.cfg::*log-mountpoint*)

  (let* ((log-dir (format () "/mnt/logs/~A/" ip))
         (log-filename (format () "~A~A" log-dir "log")))
    (ensure-directory log-dir)
    (setf 
     *logstream* 
     (open log-filename :direction :output 
           :if-does-not-exist :create :if-exists :rename)))

  (ssh Scat.cfg::*fileserver*  
       (format () "logger ~A installation start" hostname))
  (write-log "installaton began at: ~A~%" 
             (sb-int:format-universal-time NIL *start*))
  (write-log "Hostname: ~A~%" hostname)
  (write-log "Short Hostname: ~A~%" short-hostname)
  (write-log "IP: ~A~%" ip)
  
  (write-log "my fileserver is: ~A~%" Scat.cfg::*fileserver*)
  (write-log "my logdir is: ~A~%" Scat.cfg::*remote-log-dir*)
  (write-log "my log mountpoint is: ~A~%" Scat.cfg::*log-mountpoint*)
  (write-log "my victim device is: ~A~%" Scat.cfg::*victim-fs*)
  (write-log "top image directory is: ~A~%"  Scat.cfg::*NODE-IMAGE-DIR*)
  (write-log "my image directory is: ~A~%" 
             image-dir)
  (write-log "found partition image description files: ~%~{~T~A~%~}" 
             partition-descriptions)

  ;; this is fuct.
  ;; doesn't work given 'dan' style sfdisk files
  ;; why do we even do this?  Why don't we just get the partitions we need
  ;; from the fstab file instead.  who cares about these files.
  ;;
  ;; maybe someday we could do some sort of consistency check, but for now...
  (mapcar 
   (lambda (partition-description)
     (let ((disk (nth-submatch 0 ".*/(\\S+)\\.sfdisk" 
                               (namestring partition-description))))
       (write-log "making partition table for: ~A~%" disk)
       (write-log "from: ~A~%" partition-description)
       (let ((partitions (list-partitions-from-file partition-description)))
         (write-log "found partitions: ~{~A ~}~%" partitions)
         (mapcar 
          (lambda (partition)
            (build-filesystem partition))
          partitions)
          )))
   partition-descriptions)

  (mount-tmp)
  

     

  (close *logstream*)
  (cl-user::quit))