;;;; S-expr Cluster Administration Toolkit (SCAT)
;;;; Copyright (C) 2007 James Earl Prewett

;;;; This program is free software; you can redistribute it and/or
;;;; modify it under the terms of the GNU General Public License
;;;; as published by the Free Software Foundation; either version 2
;;;; of the License, or (at your option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful,
;;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;;; GNU General Public License for more details.
;;;;
;;;; You should have received a copy of the GNU General Public License
;;;; along with this program; if not, write to the Free Software
;;;; Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

;;;; this file contains SCAT library functions

(defpackage :edu.unm.hpc.Scat.Lib
  (:nicknames :Scat.Lib)
  (:use :cl
        #+allegro :clos
        #+cmu :pcl
        #+sbcl :sb-mop
        #+lispworks :hcl
        :cl-user
        :com.gigamonkeys.pathnames)
  (:export 
   read-scat-config
   node-full-path
   state-full-path
   mac-full-path
   node-existsp
   state-existsp
   node-state
   set-state
   name-mac
   set-node-states
   nonexistent-node
   nonexistent-state
   set-type
   ))

(in-package :edu.unm.hpc.Scat.Lib)

;;; Paul Graham, On Lisp, p191
(defmacro aif (test-form then-form &optional else-form)
  `(let ((it ,test-form))
    (if it ,then-form ,else-form)))

(defmacro symlink-in-dir (directory from to)
  (let ((olddir (gensym)))
    `(let ((,olddir (truename ".")))
       (and
        (sb-posix:chdir ,directory)
        (sb-posix:symlink ,from ,to))
       (sb-posix:chdir ,olddir))))
    

;;;; functions/methods

;;; configuration functions
(defun read-scat-config (&optional config-directory)
  (Scat.Cfg::read-configuration-file 
   (merge-pathnames 
    #p"etc/Scat.cfg"
    (if config-directory
        config-directory
        (aif (sb-posix:getenv "SCAT_ROOT")
             ;; scat root must end in /, it is a *directory*
             (if (eql #\/ (aref it (- (length it) 1)))
                 it
                 (format () "~A/" it))
             (error "SCAT_ROOT environment variable is unset"))))))

;;; node state query/manipulation functions

(defun node-full-path (name)
  "given the name of a node, return the path where we expect its boot control file to be"
  (merge-pathnames Scat.Cfg::*pxe-config-dir* name))

(defun state-full-path (name)
  "given the name of a node state, return the path where we expect its boot control file to be"
  (merge-pathnames Scat.Cfg::*pxe-config-dir* name))

(defun type-full-path (name)
  "given the name of a node type, return the path where we expect its boot control file to be"
  (merge-pathnames Scat.Cfg::*node-image-dir* name))

(defun mac-full-path (mac)
  "given the name of a node state, return the path where we expect its boot control file to be"
  (merge-pathnames Scat.Cfg::*pxe-config-dir* mac))

(defun node-existsp (name)
  "return the truename of the node file and its filename"
  (let ((full-name (node-full-path name)))
    (values
     (probe-file full-name)
     full-name)))

(defun state-existsp (name)
  "return the truename of the state file and its filename"
  (let ((full-name (state-full-path name)))
    (values
     (probe-file full-name)
     full-name)))

(defun type-existsp (type)
  "return the truename of the type file and its filename"
  (let ((full-name (type-full-path type)))
    (values
     (probe-file full-name)
     full-name)))

(defun node-state (name)
  "return the state of a node"
  (let ((node (or
               (node-existsp name)
               (error 'nonexistent-node :node-name name))))
    (pathname-name node)))

(defun set-state (name state &optional allow-nonexistent-nodes)
  "set the state of a node"
  (multiple-value-bind (node-state-file node-file)
      (node-existsp name)
    (format t "node-state-file: ~A node-file: ~A~%" node-state-file node-file)
    (unless (or node-state-file allow-nonexistent-nodes)
      (error 'nonexistent-node :node-name name))
    (let ((state-file (state-existsp state)))
      (unless state-file
        (error 'nonexistent-state :state-name state))
      (when node-state-file (sb-posix:unlink node-file))
      (symlink-in-dir SCAT.CFG::*PXE-CONFIG-DIR* state node-file))))

(defun set-type (name type)
  "set the type of a node"
  (multiple-value-bind (node-type-file node-type)
      (type-existsp type)
    (let ((node-file (merge-pathnames Scat.Cfg::*node-image-dir* name)))
      (symlink-in-dir Scat.Cfg::*node-image-dir* type name))))

(defun name-mac (mac node)
  "link a node name to a mac address"
  (symlink-in-dir SCAT.CFG::*PXE-CONFIG-DIR* node mac))

(defun set-node-states (nodes state)
  (mapcar
   (lambda (node)
     (set-state node state))
   nodes))

;;;; conditions

(define-condition nonexistent-node (error)
  ((node-name :initarg :node-name :reader node-name))
  (:documentation "The error that happens when a node doesn't exist"))

(define-condition nonexistent-state (error)
  ((state-name :initarg :state-name :reader state-name))
  (:documentation "The error that happens when a state doesn't exist"))

(defun list-files-in-dir (directory)
  (let ((dir (sb-posix:opendir directory)))
    (loop for foo = (sb-posix:readdir dir)
          while (not (sb-alien:null-alien foo))
          when (not (or
                     (equal "." (sb-posix:dirent-name foo))
                     (equal ".." (sb-posix:dirent-name foo))))
          collect (sb-posix:dirent-name foo))))

(defun list-symlinks-in-dir (directory)
  (let ((files (list-files-in-dir directory)))
    (loop as file in files
          when (not (equal (merge-pathnames file directory) (truename (merge-pathnames file directory))))
          collect file)))

(defun list-node-files ()
  (let ((symlinks 
         (list-symlinks-in-dir EDU.UNM.HPC.SCAT.CFG::*PXE-CONFIG-DIR*)))
    (loop as link in symlinks
          when (not (cl-ppcre::scan "^01-00" link))
          collect link)))

(defun split-name (name)
  (multiple-value-bind
        (matches sub-matches)
    (cl-ppcre::scan-to-strings
     "([^0-9]+)([0-9]+)"
     name)
    sub-matches))

(defun noderange (start &optional (end start))
  (if (string-equal "ALL" start)
      (Scat.lib::list-node-files)
      (let* ((start (split-name start))
             (start-name (aref start 0))
             (start-num (aref start 1))
             (start-num-len (length start-num))
             (end (split-name end))
             (end-name (aref end 0))
             (end-num (aref end 1))
             (end-num-len (length end-num)))
        ;; some error checking
        (when (not 
               (eql start-num-len
                    end-num-len))
          (error "numeric padding issue!"))
        (when (not 
               (string-equal
                start-name
                end-name))
          (error "name mis-match!"))
        (loop for count 
              from (read-from-string start-num)
              to (read-from-string end-num)
              collect
              (format () 
                      "~A~v,'0d"
                      start-name
                      start-num-len
                      count)))))

(defun print-noderange (stream nodes)
  (format stream "~{~A~%~}" nodes))

(defun expand-nodelist (nodelist)
  (if (string-equal "ALL" nodelist)
      (Scat.lib::list-node-files)
      
      (remove-duplicates
       (mapcan
        (lambda (thing)
          (if (cl-ppcre::scan "-" thing)
              (apply #'noderange (cl-ppcre::split "-" thing))
              (list thing)))
        (cl-ppcre::split "," nodelist))
       :test #'equal)))

(defun find-dir-files (dir)
           (let ((found ())
                 (dir-files (list-directory dir)))
             (mapcar
              (lambda (file)
                (if
                 (not (directory-p file))
                 (setf found (cons file found))))
              dir-files)
             found))

(setf *macro-table* ())

(defun replace-text-macros (target-string start end match-start match-end reg-starts reg-ends)
  (let* ((thing (subseq target-string (aref reg-starts 0) (aref reg-ends 0)))
         (found (assoc thing *macro-table* :test #'equal)))
    (if found
        (format NIL "~A" (cadr found))
        (format NIL "$$~A$$" thing))))

(defun replace-file (orig new)
  (with-open-file (orig-file orig)
    (with-open-file (new-file new :direction :output :if-exists :supersede)
      (loop for line = (read-line orig-file () ())
         while line
         do 
           (format new-file "~A~%" 
                   (cl-ppcre::regex-replace-all
                    "\\$\\$(.*?)\\$\\$" line #'replace-text-macros))))))

(defun find-dir-dirs (dir)
  (let ((found ())
        (dir-files (list-directory dir)))
    (mapcar
     (lambda (file)
       (if
        (directory-p file)
        (setf found (cons file found))))
     dir-files)
    found))

(defun copy-directory-tree (orig new)
  (UNLESS (equal orig new)
    (multiple-value-bind (exist-p device inode mode links uid gid rdev size atime mtime ctime) (sb-unix:unix-stat orig)
      (when (not (directory-p new))
        (sb-posix:mkdir new mode))
      (when (not (equal (sb-posix:getuid) uid))
        (if (equal 0 (sb-posix:geteuid))
            (SB-POSIX:CHOWN new uid gid)
            (warn "cannot set file ownership: ~A" new)))
      (let ((subdirs (find-dir-dirs orig)))
        (mapcar
         (lambda (subdir)
           (let ((basename (subseq (directory-namestring subdir) 
                                   (length (directory-namestring orig)))))
             (copy-directory-tree
            (directory-namestring subdir)
            (format () "~A~A" (directory-namestring new) basename))))
         subdirs))
      (let ((files (find-dir-files orig)))
        (mapcar
         (lambda (file)
           (let ((basename (file-namestring file)))
             (replace-file
              file
              (format () "~A~A" (directory-namestring new) basename))))
         files)))))
