;;;; S-expr Cluster Administration Toolkit (SCAT)
;;;; Copyright (C) 2007 James Earl Prewett

;;;; This program is free software; you can redistribute it and/or
;;;; modify it under the terms of the GNU General Public License
;;;; as published by the Free Software Foundation; either version 2
;;;; of the License, or (at your option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful,
;;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;;; GNU General Public License for more details.
;;;;
;;;; You should have received a copy of the GNU General Public License
;;;; along with this program; if not, write to the Free Software
;;;; Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

;;;; this file contains SCAT library functions

(defpackage :edu.unm.hpc.Scat.Lib
  (:nicknames :Scat.Lib)
  (:use :cl
        #+allegro :clos
        #+cmu :pcl
        #+sbcl :sb-mop
        #+lispworks :hcl
        :cl-user)
  (:export 
   read-scat-config
   node-full-path
   state-full-path
   mac-full-path
   node-existsp
   state-existsp
   node-state
   set-state
   name-mac
   set-node-states
   nonexistent-node
   nonexistent-state
   ))

(in-package :edu.unm.hpc.Scat.Lib)

;;; Paul Graham, On Lisp, p191
(defmacro aif (test-form then-form &optional else-form)
  `(let ((it ,test-form))
    (if it ,then-form ,else-form)))

;;;; functions/methods

;;; configuration functions
(defun read-scat-config (&optional config-directory)
  (Scat.Cfg::read-configuration-file 
   (merge-pathnames 
    #p"etc/Scat.cfg"
    (if config-directory
        config-directory
        (aif (sb-posix:getenv "SCAT_ROOT")
             ;; scat root must end in /, it is a *directory*
             (if (eql #\/ (aref it (- (length it) 1)))
                 it
                 (format () "~A/" it))
             (error "SCAT_ROOT environment variable is unset"))))))

;;; node state query/manipulation functions

(defun node-full-path (name)
  "given the name of a node, return the path where we expect its boot control file to be"
  (merge-pathnames Scat.Cfg::*pxe-config-dir* name))

(defun state-full-path (name)
  "given the name of a node state, return the path where we expect its boot control file to be"
  (merge-pathnames Scat.Cfg::*pxe-config-dir* name))

(defun mac-full-path (mac)
  "given the name of a node state, return the path where we expect its boot control file to be"
  (merge-pathnames Scat.Cfg::*pxe-config-dir* mac))

(defun node-existsp (name)
  "return the truename of the node file and its filename"
  (let ((full-name (node-full-path name)))
    (values
     (probe-file full-name)
     full-name)))

(defun state-existsp (name)
  "return the truename of the state file and its filename"
  (let ((full-name (state-full-path name)))
    (values
     (probe-file full-name)
     full-name)))

(defun node-state (name)
  "return the state of a node"
  (let ((node (or
               (node-existsp name)
               (error 'nonexistent-node :node-name name))))
    (pathname-name node)))

(defun set-state (name state &optional allow-nonexistent-nodes)
  "set the state of a node"
  (multiple-value-bind (node-state-file node-file)
      (node-existsp name)
    (unless (or node-state-file allow-nonexistent-nodes)
      (error 'nonexistent-node :node-name name))
    (let ((state-file (state-existsp state)))
      (unless state-file
        (error 'nonexistent-state :state-name state))

      (when node-state-file (sb-posix:unlink node-file))
      (sb-posix:symlink state-file node-file))))

(defun name-mac (mac node)
  "link a node name to a mac address"
  (let ((node-full-path (node-full-path node))
        (mac-full-path (mac-full-path mac)))
    (sb-posix:symlink node-full-path mac-full-path)))

(defun set-node-states (nodes state)
  (mapcar
   (lambda (node)
     (set-state node state))
   nodes))

;;;; conditions

(define-condition nonexistent-node (error)
  ((node-name :initarg :node-name :reader node-name))
  (:documentation "The error that happens when a node doesn't exist"))

(define-condition nonexistent-state (error)
  ((state-name :initarg :state-name :reader state-name))
  (:documentation "The error that happens when a state doesn't exist"))

(defun list-files-in-dir (directory)
  (let ((dir (sb-posix:opendir directory)))
    (loop for foo = (sb-posix:readdir dir)
          while (not (sb-alien:null-alien foo))
          when (not (or
                     (equal "." (sb-posix:dirent-name foo))
                     (equal ".." (sb-posix:dirent-name foo))))
          collect (sb-posix:dirent-name foo))))

(defun list-symlinks-in-dir (directory)
  (let ((files (list-files-in-dir directory)))
    (loop as file in files
          when (not (equal (merge-pathnames file directory) (truename (merge-pathnames file directory))))
          collect file)))

(defun list-node-files ()
  (let ((symlinks 
         (list-symlinks-in-dir EDU.UNM.HPC.SCAT.CFG::*PXE-CONFIG-DIR*)))
    (loop as link in symlinks
          when (not (cl-ppcre::scan "^01-00" link))
          collect link)))

(defun split-name (name)
  (multiple-value-bind
        (matches sub-matches)
    (cl-ppcre::scan-to-strings
     "([^0-9]+)([0-9]+)"
     name)
    sub-matches))

(defun noderange (start &optional (end start))
  (if (string-equal "ALL" start)
      (Scat.lib::list-node-files)
      (let* ((start (split-name start))
             (start-name (aref start 0))
             (start-num (aref start 1))
             (start-num-len (length start-num))
             (end (split-name end))
             (end-name (aref end 0))
             (end-num (aref end 1))
             (end-num-len (length end-num)))
        (format t "generating range~%")
        ;; some error checking
        (when (not 
               (eql start-num-len
                    end-num-len))
          (error "numeric padding issue!"))
        (when (not 
               (string-equal
                start-name
                end-name))
          (error "name mis-match!"))
        (loop for count 
              from (read-from-string start-num)
              to (read-from-string end-num)
              collect
              (format () 
                      "~A~v,'0d"
                      start-name
                      start-num-len
                      count)))))

(defun print-noderange (stream nodes)
  (format stream "~{~A~%~}" nodes))

(defun expand-nodelist (nodelist)
  (if (string-equal "ALL" nodelist)
      (Scat.lib::list-node-files)
      
      (remove-duplicates
       (mapcan
        (lambda (thing)
          (if (cl-ppcre::scan "-" thing)
              (apply #'noderange (cl-ppcre::split "-" thing))
              (list thing)))
        (cl-ppcre::split "," nodelist))
       :test #'equal)))
