;;;; S-expr Cluster Administration Toolkit (SCAT)
;;;; Copyright (C) 2007 James Earl Prewett

;;;; This program is free software; you can redistribute it and/or
;;;; modify it under the terms of the GNU General Public License
;;;; as published by the Free Software Foundation; either version 2
;;;; of the License, or (at your option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful,
;;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;;; GNU General Public License for more details.
;;;;
;;;; You should have received a copy of the GNU General Public License
;;;; along with this program; if not, write to the Free Software
;;;; Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

;;;; this file contains SCAT library functions

(defpackage :edu.unm.hpc.Scat.Lib
  (:nicknames :Scat.Lib)
  (:use :cl
        #+allegro :clos
        #+cmu :pcl
        #+sbcl :sb-mop
        #+lispworks :hcl
        :cl-user
        :Scat.noderange)
  (:export 
   read-scat-config
   node-full-path
   state-full-path
   mac-full-path
   node-existsp
   state-existsp
   node-state
   set-state
   name-mac
   set-node-states
   nonexistent-node
   nonexistent-state
   ))

(in-package :edu.unm.hpc.Scat.Lib)

;;; Paul Graham, On Lisp, p191
(defmacro aif (test-form then-form &optional else-form)
  `(let ((it ,test-form))
    (if it ,then-form ,else-form)))

;;;; functions/methods

;;; configuration functions
(defun read-scat-config (&optional config-directory)
  (Scat.Cfg::read-configuration-file 
   (merge-pathnames 
    #p"etc/Scat.cfg"
    (if config-directory
        config-directory
        (aif (sb-posix:getenv "SCAT_ROOT")
             ;; scat root must end in /, it is a *directory*
             (if (eql #\/ (aref it (- (length it) 1)))
                 it
                 (format () "~A/" it))
             (error "SCAT_ROOT environment variable is unset"))))))

;;; node state query/manipulation functions

(defun node-full-path (name)
  "given the name of a node, return the path where we expect its boot control file to be"
  (merge-pathnames Scat.Cfg::*pxe-config-dir* name))

(defun state-full-path (name)
  "given the name of a node state, return the path where we expect its boot control file to be"
  (merge-pathnames Scat.Cfg::*pxe-config-dir* name))

(defun mac-full-path (mac)
  "given the name of a node state, return the path where we expect its boot control file to be"
  (merge-pathnames Scat.Cfg::*pxe-config-dir* mac))

(defun node-existsp (name)
  "return the truename of the node file and its filename"
  (let ((full-name (node-full-path name)))
    (values
     (probe-file full-name)
     full-name)))

(defun state-existsp (name)
  "return the truename of the state file and its filename"
  (let ((full-name (state-full-path name)))
    (values
     (probe-file full-name)
     full-name)))

(defun node-state (name)
  "return the state of a node"
  (let ((node (or
               (node-existsp name)
               (error 'nonexistent-node :node-name name))))
    (pathname-name node)))

(defun set-state (name state)
  "set the state of a node"
  (multiple-value-bind (node-state-file node-file)
      (node-existsp name)
    (unless node-state-file
      (error 'nonexistent-node :node-name name))
    (let ((state-file (state-existsp state)))
      (unless state-file
        (error 'nonexistent-state :state-name state))
      (sb-posix:unlink node-file)
      (sb-posix:symlink state-file node-file))))

(defun name-mac (mac node)
  "link a node name to a mac address"
  (let ((node-full-path (node-full-path node))
        (mac-full-path (mac-full-path mac)))
    (sb-posix:symlink node-full-path mac-full-path)))

(defun set-node-states (nodes state)
  (mapcar
   (lambda (node)
     (set-state node state))
   nodes))

;;;; conditions

(define-condition nonexistent-node (error)
  ((node-name :initarg :node-name :reader node-name))
  (:documentation "The error that happens when a node doesn't exist"))

(define-condition nonexistent-state (error)
  ((state-name :initarg :state-name :reader state-name))
  (:documentation "The error that happens when a state doesn't exist"))
