#!/usr/bin/perl
# This file is part of the Savane project
# <http://gna.org/projects/savane/>
#
# $Id: User.pm,v 1.34 2005/02/17 20:45:08 beuc Exp $
#
#  Copyright 2003-2004 (c) Mathieu Roy <yeupou--gnu.org>
#                          Sylvain Beucler <beuc--beuc.net>
#                          Free Software Foundation, Inc.
# 
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#
#

##
## Desc: any subs related to users.
##

use strict "vars";
require Exporter;

use Savannah::Util;

# Exports
our @ISA = qw(Exporter);
our @EXPORT = qw(GetUserSettings PrintUserSettings SetUserSettings GetUserGroups GetUserList GetUserName GetUserHome GetUserSSHKeyReal DeleteUser PrintAliasesList UserAddSSHKey UserAddGPGKey );
our $version = 1;

# Imports (needed for strict).
our $dbd;
our $qqch;
our $sys_homedir;
our $sys_homedir_subdirs;


#######################################################################
##
## Return infos, basic stuff
##
#######################################################################


#####
##### From the database
##### 

## Get any settings for any users.
# arg0 : which user name
# arg1 : which setting (none for all)
sub GetUserSettings { 
    GetDBSettings("user", "user_name='".SQLStringEscape($_[0])."'", $_[1]);
}

## Get any pref for any user
# arg0 : which user name
# arg1 : which setting (none for all)
sub GetUserPrefs { 
    GetDBSettings("user_preferences", "user_id='".GetUserSettings($_[0], "user_id")."' AND preference_name='".SQLStringEscape($_[1])."'");
}

## Show in a convenient way settings for a user
# arg1 : which user name
sub PrintUserSettings {
    print join " | ", GetUserSettings($_[0]);
}

sub PrintUserPrefs {
    print join " | ", GetUserPrefs($_[0]);
}


## Update in a convenient way settings for a user
# arg1 : which user name
# arg2 : which field
# arg3 : new value
sub SetUserSettings {
    SetDBSettings("user", "user_name='".SQLStringEscape($_[0])."'",
		  "$_[1]='".SQLStringEscape($_[2])."'");
}

## Get any pref for any user
# arg1 : which user name
# arg2 : preference name
# arg3 : new value
sub SetUserPrefs { 
    SetDBSettings("user_preferences", "user_id='".GetUserSettings($_[0], "user_id")."' AND preference_name='".SQLStringEscape($_[1])."'", "preference_value='".SQLStringEscape($_[2])."'");
}


## Get list of groups for a user (contrary of GetGroupUsers
# arg1 : which user name
sub GetUserGroups {
    return GetDBList("user_group", "user_id='".GetUserSettings($_[0], "user_id")."'", "group_id");
}


## Get a list of users.
# arg0 : which criterion
# arg1 : which field to be returned
sub GetUserList { 
    return GetDBList("user", $_[0], $_[1]);
}


## In many case, we have to deal with user_id but I find
## easier to deal directly with user_name
# arg1 : which user id
sub GetUserName {
    return $dbd->selectrow_array("SELECT user_name FROM user WHERE user_id='".$_[0]."'");
}

#####
##### From the system
#####

## Frequently we need the theorical home directory for a user,
## according to the configuration.
## Normally, sv_users should make sure that users got their home
## according to the theorical setting.
# arg1
sub GetUserHome {
    # Get the level of subdir for home
    #   0. Home is like /home/user
    #   1. Home is like /home/u/user
    #   2. Home is like /home/us/user
    my $user = $_[0];
    my $ret = $sys_homedir;
    
    if ($sys_homedir_subdirs && $sys_homedir_subdirs ne '0') {
	$ret .= "/".substr($user, 0, 1);
	if ($sys_homedir_subdirs eq '2') {
	    $ret .= "/".substr($user, 0, 2);
	} 
    }
    # Always in lowercase, even if the database information accepted
    # uppercase.
    return lc($ret."/".$user);
}

## This command will return the content of ~/.ssh/authorized_keys
## for a user.
## It will return nothing if the file is not found.
## Line breaks will be replaced by ###, to conform with the database
## way to store these data.
# arg1 : which user name
sub GetUserSSHKeyReal {
    my $file = GetUserHome($_[0])."/.ssh/authorized_keys";
    my $ret;
    if (-e $file) {
        open(SSH_KEY, "< $file");
        while (<SSH_KEY>) {
            s/\n/###/g; #'
            $ret .= $_;
        }
        close(SSH_KEY);
        # No return? Return a numeric false
        $ret = 0 unless $ret;
        return $ret;
    }
}


#######################################################################
##
## Do a specific task
##
#######################################################################

## Delete a userp account. This function should be used carrefully.
# arg1 : which user name, arg2 : boolean for database deletion
sub DeleteUser {
    # If it exists on the system, delete the account
    if (getpwnam($_[0])) {
	`userdel -r $_[0]`;
    }
    # Remove from the database, if arg 2 = 1    
    $dbd->do("DELETE FROM user WHERE user_name='".SQLStringEscape($_[0])."'") if ($_[0] && $_[1]);

    return 1;
}

## Delete several users accounts. This function should be used carrefully.
# arg1 : a criterion
sub DeleteUsers {    
    return $dbd->do("DELETE FROM user WHERE ".$_[0]) if $_[0];
}


## It's many times usefull to get a list of username:mail for
## usual users, for instance for /etc/aliases
# arg1 : HANDLE
sub PrintAliasesList {
    my $handle = STDOUT;
    $handle = $_[0] if $_[0];

    foreach my $line (GetDB("user", "status='A'", "user_name,email")) {
	my @line = split(",", $line);
	print $handle join(": ", map {defined $_ ? $_ : "(null)"} @line), "\n";
    }
}


## (Over)Write SSH public key for a user
## Checks of the content of $authorized_key must be done before.
## This function will do what you asked.
# arg1 : username
# arg2 : content
# return the number of keys registered.
sub UserAddSSHKey {
    my $user = $_[0];
    my $home = GetUserHome($user);
    my $authorized_keys = $_[1];
    my $ssh_keys_registered = 0;

    open(SSH_KEY, "> $home/.ssh/authorized_keys");
    # In the database, linebreak are ###
    $ssh_keys_registered = ($authorized_keys =~ s/###/\n/g); #' count keys
    print SSH_KEY $authorized_keys; 
    close(SSH_KEY);

    # Store the information in the database, an interface the frontend could
    # (but does not currently) use to tell the user if all this keys are 
    # registered on the system of pending registration
    SetUserSettings($user, "authorized_keys_count", $ssh_keys_registered);

    return $ssh_keys_registered;
}


## Register a GPG  public key for a user
## Checks of the content of $authorized_key must be done before.
## This function will do what you asked.
# arg1 : username
# arg2 : content
# return the number of keys registered.
sub UserAddGPGKey {
    my ($user, $key) = @_;

    my $home = GetUserHome($user);

    unlink("$home/.gnupg/pubring.gpg");

    return 1 unless ($key);

    my @gpg_args = ('/usr/bin/gpg',
		    '--quiet',
                    '--no-default-keyring',
                    '--keyring',
                    "$home/.gnupg/pubring.gpg",
                    '--import',
                    '-');

    my $pid = open (GPG, "|-");

    if ($pid) {                   # parent
        print GPG $key;
        close (GPG);
        my $ret = $?;
        if ($ret) {
            SetUserSettings($user, "gpg_key_count", 0);
        } else {
            SetUserSettings($user, "gpg_key_count", 1);
        }
        return $ret;
    } else {                      # child
        exec (@gpg_args) || return 1;
    }

    return 1;
}

return 1;
