#!/usr/bin/perl
# This file is part of the Savane project
# <http://gna.org/projects/savane/>
#
# $Id: sv_update_conf.pl,v 1.24 2004/08/24 06:21:37 yeupou Exp $
#
#
#
#  Copyright 2002-2003 (c) Mathieu Roy <yeupou@gnu.org>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#
#
#
#

##
## This script permit to create a configuration file. As a
## correct configuration is required in other to runs Savannah,
## this file is designed to be working completely alone.
##

use Getopt::Long;
use Term::ANSIColor qw(:constants);
use POSIX qw(strftime);
use File::Basename;

my $getopt;
my $debug; 
my $help;
my $conffile = "/etc/savannah/savannah.conf";
my $update;
my $recreate;
my $question_count = 0;
my $handle = STDOUT;
my $mode = "0750"; # 0750/-rwxr-x---
my $http_user;
my $non_interactive;


my ($topdir, $url_topdir, $incdir);

# get options
eval {
    $getopt = GetOptions("conffile=s" => \$conffile,
			 "http-user=s" => \$http_user,
			 "non-interactive" => \$non_interactive,
			 "update" => \$update,
			 "debug" => \$debug,
			 "recreate" => \$recreate,
			 "help" => \$help,
			 "default-domain=s" => \$default_domain,
			 "topdir=s" => \$topdir,
			 "url-topdir=s" => \$url_topdir,
			 "incdir=s" => \$incdir,
			 "dbhost=s" => \$dbhost,
			 "dbname=s" => \$dbname,
			 "dbuser=s" => \$dbuser,
			 "dbpasswd=s" => \$dbpasswd);
};

if($help || !$getopt) {
    print STDERR <<EOF;
Usage: $0 [--conffile=<conffile>] [--help] 

Generate or update a configuration file for your local
Savannah installation. This file is called savannah.conf and
is usually located in /etc/savannah 
With this script, we write savannah.conf.pl, and we use it to
generate a PHP version.

  General:
      --conffile=<conffile>  File name and path of your conf
                             as /etc/savannah/savannah.conf
      --http-user=<user>     User of the webserver who should be owner
	                     of the configuration file.
			     (www-data is usual with Debian+Apache)
			     (apache is usual with RedHat+Apache)
			     Avoid to point to symlinks using this option
      --recreate             (re)Create local.inc.pl
      --non-interactive      Run in non-interactive mode
                             (apply only with --recreate)
      --update               Update local.inc by reading local.inc.pl

      --help                 Show this help and exit
      --debug                Print to STDOUT the savannah.conf.pl that
                             would be created (debug option)

  Settings Modification:
      --default-domain=<dom> Default hostname
      --topdir=<dir>         Top directory of the installation
                             (the top directory of the Savannah package)
      --url-topdir=<dir>     Suffix appended to the default domain
      --incdir=<dir>         Local dir of the site-specific content
      --dbhost=<dom>         Database server hostname
      --dbname=<name>        Database name
      --dbuser=<user>        Database server user
      --dbpasswd=<passwd>    Database server password

Author: yeupou\@gnu.org
EOF
exit(1);
}

# If conffile already exists, we can use Savannah.pm
# Otherwise, we need to create conffile from scratch.
# In fact, we will simply execute the conffile picked.
if (-e "$conffile.pl") {
    # Make sure it is in the appropriate mode to be ran
    `chmod $mode $conffile.pl`;
    # Actually test to run it
    do "$conffile.pl" or print "\nUnable to run $conffile.pl\n", RED,"Most commonly, it's a privilege issue.",RESET,"\nThe current $conffile.pl will be ignored, and saved\nas $conffile.pl.bak\n\n";
    print "$conffile.pl exists\n" if $debug; 
} else { 
    $recreate = 1;
    print "no $conffile.pl - create it\n" if $debug;
}

# Some settings the user may want to override by command line
$sys_default_domain = $default_domain if $default_domain;
$sys_https_host = $https_host if $https_host;
$sys_https_port = $https_port if $https_port;

$sys_topdir= $topdir if $topdir;
$sys_url_topdir= $url_topdir if $url_topdir;
$sys_incdir= $incdir if $incdir;

$sys_dbhost= $dbhost if $dbhost;
$sys_dbname= $dbname if $dbname;
$sys_dbuser= $dbuser if $dbuser;
$sys_dbpasswd= $dbpasswd if $dbpasswd;

# A function to add easily new questions
sub AskSetting {    
    $question_count = $question_count+1;
    $item = $_[0]; 
    $item_name = $_[1];
    $item_help = $_[2];
    $item_example = $_[3];
    if ($_[4]) {
	$item_previous_answer = $_[4];
    } else {
	$item_previous_answer = "OUT";
    }
    if ($_[5]) {
	$item_quotes = "'";
    } else {
	$item_quotes = "\"";
    }
    
    # Get user input if in interactive mode
    if (!$non_interactive) {
	print "\n$question_count) $item_name\n($item_help)\nEx: ";
	print RED, $item_example, RESET;
	print "\n\n";
	print "[$item_previous_answer]: ";
	chomp($$item = <STDIN>);
	if ($$item eq "" && $item_previous_answer ne "OUT") { $$item = $item_previous_answer; }
	print "\n\n";
    } else {
	$$item = $item_previous_answer;
    }

    # Now, we set $$item to how we want to show it in the configuration
    # file.
    if ($$item eq "" || $$item eq "OUT") {
	$$item = "# our \$$item=$item_quotes$item_quotes\;";
    } else {
	# Escape quotes
	$$item =~ s/$item_quotes/\\$item_quotes/g;
	# Add the entry
	$$item = "our \$$item=$item_quotes$$item$item_quotes\;"; 	
    }
}


############################################### QUESTIONS #####

if ($recreate) {
    
    unless ($debug) {
	# We create the conf directory if missing
	$dirname = dirname($conffile);
	`mkdir -p $dirname` if ! -e $dirname;
	
        # We make a backup of the configuration file
	rename("$conffile.pl", "$conffile.pl.bak") && `chmod 0600 $conffile.pl.bak` if -e "$conffile.pl";

        # We check whether we can write in the configuration file
	die "Cannot writing in $dirname" unless -w $dirname;
    }

    print CYAN, "\n\t***** $conffile.pl (re)creation ******\n\n", RESET;
    print "If you want a value to be commented out, type \"OUT\"\n\n";
    print "If you do not understand what a question is about, it is probably not vital to you and you can comment it out by typing \"OUT\"\n\n";

    # ask the user to set the correct values.

    ## SERVER(S) ##
   
      AskSetting("sys_default_domain", "Default hostname", "It must be the naked form of the domain", "savannah.gnu.org", $sys_default_domain);

      AskSetting("sys_https_host", "HTTPS hostname", "It must be the naked form of the domain. If you do not have https server, comment out", "savannah.gnu.org, \$sys_default_domain", $sys_https_host);
      AskSetting("sys_https_port", "HTTPS port", "Port number. If you do not have https server, comment out", "443", $sys_https_port);

      AskSetting("sys_brother_domain", "Brother hostname", "You can run Savannah with two different domain names. You'll be able to write a different configuration for each one. The two brother/companion sites will share the same database. /etc/savannah/local.inc.pl is far more important since the default access to the database is done using this one. Here you can let your savannah installation aware of the existence of a brother/companion site, so while people login, it will be allowed to them to login on both site in one click. If you do not have brother/companion site, comment out. If you do not understand what is it about, you probably do not need that feature, comment out.", "savannah.nongnu.org", $sys_brother_domain);

    
      AskSetting("sys_dbhost", "SQL database hostname", "--", "localhost", $sys_dbhost);
      AskSetting("sys_dbname", "SQL database name", "--", "savannah", $sys_dbname);
      AskSetting("sys_dbuser", "SQL database user", "--", "mysqluser", $sys_dbuser);
      AskSetting("sys_dbpasswd", "SQL database password", "--", "mysqlpasswd", $sys_dbpasswd);


    ## INSTALLATION PATHS ##
    
      AskSetting("sys_topdir", "Local dir of the installation of Savannah, in which is the PHP frontend, in subdirectory ./frontend/php", "IT MUST BE AN ABSOLUTE PATH NAME", "/usr/local/sv/src/savannah", $sys_topdir);
      AskSetting("sys_url_topdir", "Default web directory", "Suffix appended to the default domain", "/", $sys_url_topdir);
      AskSetting("sys_incdir", "Local dir of the site-specific content", "--", "/etc/savannah/savannah-content", $sys_incdir);
      AskSetting("sys_miscdir", "Misc directory", "Local dir where dumps will be stored etc", "/subversions/sv", $sys_miscdir);


    ## GUI ##

      AskSetting("sys_name", "Platform name", "Name shown on public pages for the whole service", "Savannah", $sys_name, 1);
      AskSetting("sys_unix_group_name", "Server administration project unix name", "Unix group name of the meta-project used for administration", "savannah", $sys_unix_group_name);

      AskSetting("sys_default_locale", "Default locale", "It must be a valid locale name", "french", $sys_default_locale);
      AskSetting("sys_datefmt", "Date format", "Date formatting. If you want to use the default locale formating, comment out (it's usually a good idea)", "Y-M-d H:i", $sys_datefmt);

      AskSetting("sys_themedefault", "Default theme", "--", "savannah", $sys_themedefault);
      AskSetting("sys_logo_name", "Logo name", "The engine will search for a file like savannah/frontend/php/images/\$theme.theme/\$sys_logo_name", "floating.png", $sys_logo_name);
      AskSetting("sys_logo_name_width", "Logo width", "In pixel", "150", $sys_logo_name_width);
      AskSetting("sys_logo_name_height", "Logo heigth", "In pixel", "130", $sys_logo_name_height);


    ## MAIL, MAILING-LIST ##

      AskSetting("sys_mail_domain", "Mail domain", "--", "gnu.org", $sys_nglists_domain);
      AskSetting("sys_mail_admin", "Admin mail address", "The mail domain we'll be added to this username", "savannah-hackers", $sys_admin_list);
      AskSetting("sys_mail_replyto", "Default reply-to address", "--", "nobody", $sys_replyto);
      AskSetting("sys_mail_list", "[BACKEND SPECIFIC] List of emails", "If you do not want such file to be updated by the backend, comment out", "/etc/email-addresses", $sys_mail_list);
      AskSetting("sys_mail_aliases", "[BACKEND SPECIFIC] List of emails aliases", "If you do not want such file to be updated by the backend, comment out", "/etc/aliases", $sys_mail_aliases);


    ## USERS ACCOUNTS ##
    
    AskSetting("sys_use_pamauth", "PAM support", "AFS, Kerberos (...) authentication can be made via PAM", "no", $sys_use_pamauth);
    AskSetting("sys_use_krb5", "Kerberos 5", "If you do not know what it is about, you surely don't have to deal with a kerberos server, say no here.", "no", $sys_use_krb5);
    
    AskSetting("sys_homedir", "[BACKEND SPECIFIC] User home directory", "Usually /home. You can uncomment if you do not plan to provide accounts", "/home", $sys_homedir);
    AskSetting("sys_homedir_subdirs", "[BACKEND SPECIFIC] User home directory subdirs", "Users home is by default /home/user. If you set this to 1, you'll get /home/u/user, and if you set it to 2, you'll get /home/u/us/user. It may be very convenient if you have plenty of users.", "2", $sys_homedir_subdirs);
    AskSetting("sys_shell", "[BACKEND SPECIFIC] User default shell", "cvssh is a limited shell, choose /bin/bash if you want to provide full access to your users", "/usr/local/bin/cvssh", $sys_shell);
    
    AskSetting("sys_userx_prefix", "[BACKEND SPECIFIC] Prefix for user* binaries", "If you do not want to use useradd/usermod/userdel that are in the usual PATH but specific ones, you can type here their prefix. Otherwise, comment out", "/usr/local/savannah/bin", $sys_userx_prefix); 
    
    ## CRONJOBS ##
    
    AskSetting("sys_cron_users", "[BACKEND SPECIFIC] Cron job: related to users", "If you do not want your system to be synchronized with database automatically regarding to users infos (/home/$user), comment out", "yes", $sys_cron_users);
    AskSetting("sys_cron_groups", "[BACKEND SPECIFIC] Cron job: related to groups/projects", "If you do not want your system to be synchronized with database automatically regarding to groups infos, comment out", "yes", $sys_cron_groups);
  
    AskSetting("sys_cron_viewcvs_forbidden", "[BACKEND SPECIFIC] Cron job: related to viewcvs ignore list", "If you do not want your system to be synchronized with database automatically regarding to viewcvs forbidden list, comment out", "yes", $sys_cron_viewcvs_forbidden);  
  
  AskSetting("sys_cron_mail", "[BACKEND SPECIFIC] Cron job: related to mails", "If you do not want your system to be synchronized with database automatically regarding to mail infos (/etc/aliases...), comment out", "yes", $sys_cron_mail);  
    AskSetting("sys_cron_cleaner", "[BACKEND SPECIFIC] Cron job: database cleaning", "A special backend script will clean regularly the database. If you do not want that cleaning to be done, comment out. It is recommended to use it, even if your installation use no other backend tool", "yes", $sys_cron_cleaner);  


    ## EXTRA FEATURES ##

      AskSetting("sys_viewcvs_conffile", "[BACKEND SPECIFIC] Viewcvs configuration file", "Path to the viewcvs conffile. If you do not use viewcvs or if you do not want the fordibben setting of this configuration file to be edit by Savannah, comment out", "/etc/viewcvs/viewcvs.conf", $sys_viewcvs_conffile);
      AskSetting("sys_webalizer_dir", "Statistics: webalizer dir", "Where webalizer store its output. If you do not have webalizer, comment out", "/subversions/sv/src/savannah/www/webalizer", $sys_webalizer_dir);
      AskSetting("sys_webalizer_url", "Statistics: webalizer url", "http url for webalizer output. If you do not have webalizer, comment out", "http://savannah.gnu.org/webalizer", $sys_webalizer_url);
      AskSetting("sys_mrtg_dir", "Statistics: mrtg dir", "Where mrtg store its output. If you do not have mrtg, comment out", "/home/www/html/mrtg", $sys_mrtg_dir);
      AskSetting("sys_mrtg_url", "Statistics: mrtg url", "http url for webalizer output. If you do not have mrtg, comment out", "http://subversions.gnu.org/mrtg", $sys_mrtg_url);
      AskSetting("sys_use_google", "Google search", "Add a search via google option to the search module. If you do not want this search facility, comment out", "", $sys_use_google);

      AskSetting("sys_localdoc_file", "Local Administration Documentation File", "Will make available the content of a specific file to site admins. Give the path to the file or comment out", "", $sys_localdoc_file);

    ## SPECIFIC and (usually) DEPRECATED


############################################### OUTPUT CREATION #####

    unless ($debug) {
	open(CONFFILE, "> $conffile.pl") or die "Cannot open $conffile.pl for writing";
	$handle = CONFFILE;
    }


    # print the conffile
    print $handle <<EOF;
#!/usr/bin/perl
# This file is part of the Savane project
# <http://gna.org/projects/savane/>
#
# $Id: sv_update_conf.pl,v 1.24 2004/08/24 06:21:37 yeupou Exp $
#
#
# Savannah Site Configuration:
# Configuration file for a Savannah server
#/projects/savannah
#
# This file has been generated by sv_update_conf
# If you modify it, run sv_update_conf --update
# to update the PHP version of this file.
# You can recreate this file using sv_update_conf --recreate
# with ease.
# Ex: sv_update_conf --conffile=$conffile --update
#     sv_update_conf --conffile=$conffile --recreate
#
# IMPORTANT SECURITY NOTE.
# Configuration files should be rwx-only for root and apache's
# user (usually "www-data" on Debian GNU/Linux).
# So the owner should be apache's user.
#      - there is a mysql password
#      - these files are executed
# While it offers the incredible advantage to avoid reinventing a boring
# config parser and to permit hackers to create very efficient config
# by using perl/php, if anybody can edit it, anybody can push savannah to 
# execute malicious code when looking for it's configuration.
# (But it's like a /etc/profile ... /etc is for admins).
#
# By default:
#      Access: (0750/-rwxr-x---)
#      Uid: (    0/root)
#      Gid: (   33/www-data)
    
use strict;

    ## SERVER(S) ##

      # Default HTTP domain
      # It must be the naked form of the domain
      # Ex: "savannah.gnu.org"
      $sys_default_domain

      # Default HTTPS domain
      # It must be the naked form of the domain
      # Ex: "savannah.gnu.org", "443"
      $sys_https_host
      $sys_https_port
      
      # Brother HTTP(s) domain:
      # You can run Savannah with two different domain names. You'll be able 
      # to write a different configuration for each one.
      # The two brother/companion sites will share the same database.
      # /etc/savannah/savannah.conf.pl is far more important since the default
      # access to the database is done using this one.
      # Here you can let your savannah installation aware of the existence
      # of a brother/companion site, so while people login, it will be allowed 
      # to them to login on both site in one click.
      # If you do not have brother/companion site, comment out.
      $sys_brother_domain

      # SQL server:
      # Ex: "localhost", "savannah", "mysqluser", "mysqlpasswd"
      $sys_dbhost
      $sys_dbname
      $sys_dbuser
      $sys_dbpasswd


    ## INSTALLATION PATHS ##

      # Local dir of the installation of the PHP frontend
      # IT MUST BE AN ABSOLUTE PATH NAME
      # Ex: "/usr/local/sv/src/savannah/frontend/php"
      $sys_topdir
      
      # Default web directory - suffix appended to the default domain 
      # Ex: "/", "/savannah"
      $sys_url_topdir
    
      # Local dir of the site-specific content
      # Ex: "/etc/savannah/savannah-content"
      $sys_incdir

      # [BACKEND ONLY]
      # Local dir where dumps will be stored etc
      # Ex: "/subversions/sv"
      $sys_miscdir


    ## GUI ##
      
      # Platform name, server administration project:
      # Name of the platform running and unix_group_name of the project 
      # dedicated of the server administration
      # Ex: "Savannah", "savannah", 
      $sys_name
      $sys_unix_group_name
      
      # Default locale and date formatting.
      # Ex: "fr_FR", "Y-M-d H:i"
      $sys_default_locale
      # If you want to use the default locale formating, comment out
      $sys_datefmt

      # Frontend look:
      # Default theme, logo name and dimensions.
      # The engine will search for a file like 
      # savannah/www/images/\$theme.theme/$sys_logo_name
      # Ex: "savannah", "floating.png", "150", "130"
      $sys_themedefault
      $sys_logo_name
      $sys_logo_name_width
      $sys_logo_name_height
      

    ## MAIL, MAILING-LIST ##

      # Mail domain, admin mail address, default replyto, mail address list,
      # mail aliases:
      # Ex: "gnu.org", "savannah-hackers", "nobody", "/etc/email-addresses,"
      # "/etc/aliases"
      $sys_mail_domain
      $sys_mail_admin
      $sys_mail_replyto
      $sys_mail_list
      $sys_mail_aliases

    
    ## USERS ACCOUNTS ##
      
      # [BACKEND ONLY]
      # user* binaries path
      # If you do not want to use useradd/usermod/userdel that are in the
      # usual PATH but specific ones, you can type here their prefix. 
      # Otherwise, comment out.
      # If you do not know that this setting is about, comment out.
      $sys_userx_prefix

      # Provide login through PAM
      # (it can be AFS, Kerberos...)
      # Ex: "yes", "no"
      $sys_use_pamauth
      
      # Kerberos:
      # If you do not know what it is about, you surely don't have to
      # deal with a kerberos server, say no here.
      # IMPORTANT: this part will be removed, replaced by PAM
      # Ex: "no"
      $sys_use_krb5

      # [BACKEND ONLY]
      # User home directory
      # You can safely uncomment all these settings if you do not use
      # the savannah backend.
      # Ex: "/home"
      $sys_homedir

      # [BACKEND ONLY]
      # User home directory subdirs
      # Ex: "2" means users will have $sys_homedir/u/us/user as home.
      $sys_homedir_subdirs

      # [BACKEND ONLY]
      # User default shell.
      # Ex: "/usr/local/bin/cvssh"
      $sys_shell


    ## CRONJOBS ##
    
      # [BACKEND ONLY]
      # Cron jobs
      # If you do not want system to be synchronized with database
      # automatically
      # on the following topics, by the backend, comment out.
      $sys_cron_users
      $sys_cron_groups
      $sys_cron_viewcvs_forbidden
      $sys_cron_mail
      $sys_cron_cleaner

    
    ## EXTRA FEATURES ##
      
      # [BACKEND ONLY]
      # viewcvs ignore list
      # If you do not use viewcvs, if you do not want the backend to
      # edit that list comment out
      $sys_viewcvs_conffile 

      # Statistics
      # If you do not have webalizer or mrtg, if you do not want comment out.
      $sys_webalizer_dir
      $sys_webalizer_url
      $sys_mrtg_dir
      $sys_mrtg_url
      # search via google added to the search module
      $sys_use_google 
      # local admin doc: path of this path will be made available to site
      # admins
      $sys_localdoc_file

    ## DEVELOPMENT ##

      # Developers of Savannah should probably set this setting on their
      # test machine
      #our \$sys_debug_on=\"1\"\;

    
    ## WORKAROUNDS ##

      # DEPRECATED:
      # settings from the original SF code that does not seems usable nor 
      # usefull. They are needed in order for the PHP frontend to run for now.$s
      our \$sys_urlroot=\"\$sys_topdir/frontend/php\"\;
      our \$sys_default_dir=\"\$sys_url_topdir\"\;
      our \$sys_nglists_domain=\"\$sys_mail_domain\"\;
      our \$sys_replyto=\"\$sys_mail_replyto\"\;
      our \$sys_admin_list=\"\$sys_mail_admin\"\;
      our \$sys_shell_host=\"\$sys_default_domain\"\;
      our \$sys_users_host=\"\$sys_default_domain\"\;
      our \$sys_docs_host=\"\$sys_default_domain\"\;    
      our \$sys_dns1_host=\"\$sys_default_domain\"\;
      our \$sys_dns2_host=\"\$sys_default_domain\"\;
      our \$sys_lists_host=\"\$sys_default_domain\"\;
      our \$sys_lists_domain=\"\$sys_nglists_domain\"\;
      our \$sys_email_adress=\"\$sys_mail_admin\\\@\$sys_mail_domain\"\;
      our \$sys_server=\"\$sys_default_domain\"\;
      # where are themes (original theme system). Shouldnt be modified
      our \$sys_themeroot=\"\$sys_urlroot/themes/\"\;

# END
EOF

    unless ($debug) {
	
	# close the conffile, set it properly
	close(CONFFILE);
	`chmod $mode $conffile.pl`;
	print CYAN, "\n\t** $conffile.pl (re)created **\n\n", RESET;
	    
	}
}


# now update/create the local/inc.pl file
# we ask user to use --update but for now
# update is done in any cases

open(CONFFILEPL, "< $conffile.pl") or die "Cannot open $conffile.pl for reading";
open(CONFFILEPHP, "> $conffile.php") or die "Cannot open $conffile.php for writing";

print CONFFILEPHP "<?php\n";
print CONFFILEPHP "// THIS FILE WAS GENERATED (", strftime "%c)\n", localtime;
print CONFFILEPHP "// DO NOT MODIFY THIS FILE
// Modify only $conffile.pl and run sv_update_conf
// Ex: sv_update_conf --conffile=$conffile --update
//
// IMPORTANT SECURITY NOTE.
// Configuration files should be rwx-only for root and apache's
// user (usually \"www-data\" with Debian GNU/Linux).
// So the owner should be apache's user.
//      - there is a mysql password
//      - these files are executed
// While it offers the incredible advantage to avoid reinventing a boring
// config parser and to permit hackers to create very efficient config
// by using perl/php, if anybody can edit it, anybody can push savannah to 
// execute malicious code when looking for it's configuration.
// (But it's like a /etc/profile ... /etc is for admins).
//
// By default:
//     Access: (0750/-rwxr-x---)
//     Uid: (    0/root)
//     Gid: (   33/www-data)
";

while (<CONFFILEPL>) {
    s/our\ //;
    s/use\ .*$//;
    s/\\\@/\@/;
    next if /^#/;
    next if /^\s*(#|$)/;
		  print CONFFILEPHP "$_";
}
    
print CONFFILEPHP "// NOTHING SHOULD REMAIN BEYOND THE PHP CLOSING TAG\n";
print CONFFILEPHP "?>";
   
close(CONFFILEPHP);
close(CONFFILEPL);

`chmod $mode $conffile.php`;
`chown root.$http_user $conffile.php $conffile.pl` if $http_user; 
    
print "\nYou asked this script to run in non-interactive mode. You should \nprobably edit the file $conffile.pl now\n" if $non_interactive;


# EOF
