#!/usr/bin/perl
# This file is part of the Savane project
# <http://gna.org/projects/savane/>
#
# $Id: sv_users.pl,v 1.35 2004/02/05 12:02:17 yeupou Exp $
#
#  Copyright 2001      (c) Loic Dachary <loic@gnu.org> (sv_cvs.pl)
#            2003-2004 (c) Mathieu Roy <yeupou@gnu.org> 
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

##
## This script should be used via a cronjob to update the system
## by reading the database about users.
##
## It will add create/update an account for each user that belongs
## to a group.
##
## Users will all belong to the group svusers. Note that sv_groups should
## have checked if that group exists.
##
## Note that each group on savannah got two groups: $group and web$group.
## The first one to manage the source repository, the second one to
## manage the web repository. 
##
##
## WARNING: sv_groups should run first.
##

use strict;
use Savannah;
use Getopt::Long;
use Term::ANSIColor qw(:constants);
use POSIX qw(strftime);

# Import
our $sys_shell;
our $sys_cron_users;
our $sys_userx_prefix;

my $script = "sv_users";
my $logfile = "/var/log/sv_database2system.log";
my $lockfile = "/var/run/sv_database2system.lock";
my $getopt;
my $help;
my $debug;
my $cron;

my $useradd = "useradd";
my $usermod = "usermod";
my $userdel = "userdel";
my $userx_prefix;

my $one_group = 0;

my $svusersgroup = "svusers"; # this could be configurable, however it
                              # does not seems very important right now.

my $subversions = AreWeOnSubversions();

# get options
eval {
    $getopt = GetOptions("help" => \$help,
			 "debug" => \$debug,
			 "userx-prefix=s" => \$userx_prefix,
			 "cron" => \$cron,
			 "useradd=s" => \$useradd,
			 "usermod=s" => \$usermod,
			 "userdel=s" => \$userdel,
			 "one-group" => \$one_group);
};

if($help) {
    print STDERR <<EOF;
Usage: $0 [project] [OPTIONS] 

Update the system to reflect the database, about users.
Normally, sv_groups should run just before.

Note that users are associated with the group svusers.

  -h, --help                   Show this help and exit
  -d, --debug                  Do nothing, print everything
      --cron                   Option to set when including this script
                               in a crontab

      --userx-prefix=[prefix]  Specify a prefix for user* bin
                               For instance, for useradd.
      --useradd=[useradd]      Speficy useradd binary
      --usermod=[usermod]      Speficy usermod binary
      --userdel=[userdel]      Speficy userdel binary
  
Authors: Loic Dachary <loic\@gnu.org>, Mathieu Roy <yeupou\@gnu.org>
EOF
exit(1);
}

if ($userx_prefix) {
    $useradd = "$userx_prefix/$useradd";
    $usermod = "$userx_prefix/$usermod";
    $userdel = "$userx_prefix/$userdel";
} elsif ($sys_userx_prefix) {
    $useradd = "$sys_userx_prefix/$useradd";
    $usermod = "$sys_userx_prefix/$usermod";
    $userdel = "$sys_userx_prefix/$userdel";
}

# Test if we should run, according to conffile
exit if ($cron && ! $sys_cron_users);

# Log: Starting logging
open (LOG, ">>$logfile");
print LOG strftime "[$script] %c - starting\n", localtime;


# Locks: There are several sv_db2sys scripts but they should not run
#        concurrently.  So we add a lock
if (-e $lockfile) {
    print LOG "[$script] There's a lock ($lockfile), exiting\n";
    print LOG "[$script] ------------------------------------------------------\n";
    die "There's a lock ($lockfile), exiting";
}
`touch $lockfile`;


#######################################################################
##
## Grabbing database informations.
## 
## - db_user* items
## - db_user_group items
##
##
#######################################################################

# db_user:
#    Create an hash that contains users infos from the table user,
#    as lists for each user
#    ( @{$db_user{$user}} )
#    Additionally, create a list of users.
#
#    To limit the number of request, we use only one very long SQL request. 
my %db_user;
my @db_users;
foreach my $line (GetDB("user", 
			"status='A' OR status='D'",
			"user_name,email,realname,authorized_keys,gpg_key,status,unix_status, use_cvsadmin")) {
    chomp($line);
    my ($user, $email, $realname, $authorized_keys, $gpg_keys, $status, $unix_status, $use_cvsadmin) = split(",", $line);
    print "DBG db: get $user <$email> from database\n" if $debug;
    $realname =~ s/\://g;
    $db_user{$user} = [ ($user, $email, $realname, $authorized_keys, $gpg_keys, $status, $unix_status, $use_cvsadmin) ];
    push (@db_users, $user);
}

# db_user_group:
#    Create an hash that contains users groups infos (which user belongs to 
#    which group) from the table user_groups, as lists for each user
#    ( @{$db_user_group{$user}} )
#    We only consider active groups.
#
#    To limit the number of request, we use only one very long SQL request.
my @www_groups;
if ($subversions) { 
    # sv.gnu.org specific, related to the special group www 
    my @type1 = GetGroupList("type='1' AND status='A'", "unix_group_name");
    my @type3 = GetGroupList("type='3' AND status='A'", "unix_group_name");
    for (@type1, @type3) {
	push(@www_groups, "web".$_);
    }
}
my %db_user_group;
foreach my $line (GetDB("user_group,groups,user", 
			"groups.group_id=user_group.group_id AND user.user_id=user_group.user_id AND groups.status='A' AND user_group.admin_flags<>'P'",
			"user_name,unix_group_name")) {
    chomp($line);
    my ($user, $group) = split(",", $line);

    print "DBG db: $user is member of $group\n" if $debug;

    my @groups;

    # If the user picked use_cvsadmin, we add him in the anoncvs group
    # to allow him to edit files in CVSROOT. 
    # (yes, this group name is no longer pertinent)
    my ($u,$e,$r,$a,$g,$s,$un,$use_cvsadmin) = @{$db_user{$user}};

# yeupou@gnu.org 2004-01-10
# It is insecure to allow writing in CVSROOT
#    if ($use_cvsadmin) {
#	push(@groups, "anoncvs");
#   }

    unless ($group eq 'www' && $subversions) {
	push(@groups, $group);
	push(@groups, "web".$group) unless $one_group;
    } else {
	# www is a special group normally used to give
	# a user write access on the whole CVSROOT of 
 	# web area for any project of the group type 'www'
	# This stuff is gnu.org specific. If it really need to
	# be configurable (if someone want to use that), we'll
	# think about make it configurable.
	# For instance, we can imagine to select a box in the
	# group type form.
	
	# Need to understand if we really want www + gnu types
	# or only www.
	@groups = (@groups, "www", @www_groups);
	print "DBG db: $user is member of www !\n" if $debug;		
    }

    push(@{$db_user_group{$user}}, @groups);
}

# db_groups:
#
#    We need to able to determine whether a group is related to
#    Savannah or not.
# 
my @db_groups = GetGroupList(0, "unix_group_name");

print LOG strftime "[$script] %c - database infos grabbed\n", localtime;

#######################################################################
##
## Grabbing system informations.
## 
## - etc_password* items
## - etc_group* items
##
#######################################################################

# /etc/passwd (/etc/shadow...):
#    Create an hash that contains users infos from these files,
#    as lists for each user.
#    ( @{$etc_password{$user}} )
#    Find what is the maximum id number known.
#    Additionally, create a list of users.
my %etc_password;
my @etc_users;
my $etc_password_maxid = -1;
while (my @entry = getpwent()) {
    push(@etc_users, $entry[0]);
    $etc_password_maxid = $entry[2] > $etc_password_maxid ? $entry[2] : $etc_password_maxid;
    next if($entry[0] eq 'anoncvs' || $entry[0] eq 'webcvs' || $entry[0] eq 'nobody');
    $etc_password{$entry[0]} = [ @entry ];
    print "DBG etc: user $entry[0]\t\t maxid $etc_password_maxid\n" if $debug;
}
$etc_password_maxid++;

# /etc/group:
#    Create an hash that contains users infos about groups,
#    as lists for each group.
#    ( @{$etc_group_bygroup{$group}} )
#    Create an hash that contains users infos about groups,
#    as lists for each user (which to which groups belongs a user).
#    ( @{$etc_group{$user}} )
#    Find what is the maximum id number known.
my %etc_group_bygroup;
my %etc_group;
my $etc_group_maxid = -1;
while(my @entry = getgrent()) {
    $etc_group_bygroup{$entry[0]} = [ @entry ];  
    
    foreach my $user (split ' ', $entry[3]) {
	if ($user) {
	    print "DBG etc: user $user belongs to group $entry[0]\n" if $debug;
	    push(@{$etc_group{$user}}, $entry[0]);
	}
    }
    
    if($entry[0] ne 'nogroup') {
	$etc_group_maxid = $entry[2] > $etc_group_maxid ? $entry[2] : $etc_group_maxid;
    }
    print "DBG etc: group $entry[0]\t\t maxid $etc_group_maxid\n" if $debug;
}
$etc_group_maxid++; 

print LOG strftime "[$script] %c - system infos grabbed\n", localtime;

#######################################################################
##
## Doing comparisons
##
## - @only_in_db: users missing on the system
## - @well_known: users on the system and in the database
## - @to_be_remove: users marked as D, canditates for deletion
##
##   IMPORTANT: a user may be in @well_known but not associated to
##          any project!
##
#######################################################################

# Find out users only in database.
my %seen_in_etc;
my @only_in_db;
my @well_known;
foreach my $user (@etc_users) { 
    $seen_in_etc{$user} = 1;
}

foreach my $user (@db_users) {
    next if $user eq "None";
    unless ($seen_in_etc{$user}) {
	push(@only_in_db, $user);
	print "DBG compare: $user is seen only in database\n" if $debug;
    } else {
	push(@well_known, $user);
	print "DBG compare: $user is known by the database and the system\n" if $debug;
    }
}


# Find out users that should be really removed: they are marked as D
# in the database or they are not member of any group but are
# in the database and on the system.
# These users must be in both the database and the system.
my @to_be_removed;
foreach my $user (@well_known) {
    # First test: looks for the typical case when someone is no
    # longer member of any project: we just list
    # Second test: looks for the marked as D users
    if (!exists($db_user_group{$user}) || $db_user{$user}->[5] eq 'D') {
	push(@to_be_removed, $user);
    }
}

print LOG strftime "[$script] %c - comparison done\n", localtime;

#######################################################################
##
## Finally, update the system
##
#######################################################################

# Add users only in database, missing on the system
foreach my $user (@only_in_db){

    # We only create an account for project's members
    if (exists($db_user_group{$user})) {
	
	my ($user_name, $email, $realname, $authorized_keys, $gpg_keys, $status, $unix_status, $use_cvsadmin) = @{$db_user{$user}};
	my $home = GetUserHome($user);
	my $groups = join ",", @{$db_user_group{$user}};
	my $ssh_keys_count = 0;
	my $gpg_keys_count = 0;


	print "DBG create: $user belongs to $groups\n" if $debug;
	unless ($debug) {

	    # Make a backup if a home user already exists
	    `rm -fr $home.old` if -e "$home.old";
	    `mv $home $home.old` if -d "$home";
	        
	    # Build the home dir
	    `mkdir -p $home`; 
	    mkdir("$home/.ssh"); 
	    mkdir("$home/.gnupg"); 
	    `touch $home/.savannah $home/.ssh/authorized_keys`;

	    # Create a proper account
	    # (this should be done after the building of the home dir,
	    # because some versions of useradd tries foolishly to create
	    # the user home, and fail due to missing directories in the path)
	    `$useradd -p '*' -u $etc_password_maxid -d $home -c \"$realname\" -s $sys_shell -g svusers -G $groups $user`;
	    
	    # Add SSH public key
	    if ($authorized_keys ne "NULL") {
		$ssh_keys_count = UserAddSSHKey($user, $authorized_keys);
	    }


	    # Add GPG key
	    if ($gpg_keys ne "NULL") {
		# For gpg, I have to think about. Would we accept only gpg
		# signature complete, or fingerprints?
		$gpg_keys_count = UserAddGPGKey($user, $gpg_keys);
	    }
	    

	    # Fix modes and ownership
	    `chmod 2755 $home`;
	    `chmod 755 $home/.ssh  $home/.gnupg`;
	    `chmod 600 $home/.ssh/authorized_keys`;
	    `chown -R $user.svusers $home`;
	    # The following was done in sv_cvs
	    # `chown -R $user.anoncvs $home/.ssh $home/.savannah`;
	    
	}
	
	$etc_password_maxid++;
	print LOG strftime "[$script] %c ---- $useradd $user ($etc_password_maxid, $email, $home, $ssh_keys_count ssh keys, $gpg_keys_count gpg) $groups\n", localtime;
    }

}

print LOG strftime "[$script] %c - account creation done\n", localtime;


# Remove users marked as D in the database.
# This is the safest way to remove users quitting savannah.
foreach my $user (@to_be_removed) {

    print LOG strftime "[$script] %c ---- delete $user account\n", localtime;
    print "DBG delete: $user is marked as D or no longer member of any project\n" if $debug;
    DeleteUser($user) unless $debug;    

}

print LOG strftime "[$script] %c - account deletion done\n", localtime;

# Update existing users.
# These users are in the database and on the system.
foreach my $user (@well_known) {

    # Get usual infos
    my ($user_name,
	$email,
	$realname,
	$authorized_keys,
	$gpg_keys,
	$status,
	$unix_status,
	$use_cvsadmin) = @{$db_user{$user}};


    # Update groups: check if the system knows all groups the user
    # is member of.
    # Build a list of every groups, sys + database, and compare
    # to the sys reality. Update if different.
    my @groups_list;
    my @groups_list_etc = 0;
    @groups_list = @{$db_user_group{$user}} if
	(exists($db_user_group{$user}));
    @groups_list = (@groups_list, @{$etc_group{$user}}) if
	(exists($etc_group{$user}));
    @groups_list_etc =  @{$etc_group{$user}} if
	(exists($etc_group{$user}));
    
    my %seen_before = (); # remove duplicates
    @groups_list = grep { ! $seen_before{$_} ++ } @groups_list;
    

    if (@groups_list ne @groups_list_etc) {
	my $groups = join(",", @groups_list);
	`$usermod -G $groups $user` unless $debug;
	print LOG strftime "[$script] %c ---- update $user groups\n", localtime;
	print "DBG update: $user belongs to \t[db+sys]\t".join(", ", @groups_list)."\t[sys]\t".join(", ", @groups_list_etc)."\n" if $debug;
    } else {
	print "DBG update: NO UPDATE for user $user that belongs to \t[db+sys]\t".join(", ", @groups_list)."\t[sys]\t".join(", ", @groups_list_etc)."\n" if $debug;
    }

    # Update groups: check if the user is removed from a group in the
    # database.


    # Update name (name, email) if not accurate.
    my $password_realname = $etc_password{$user}->[6];
    my $expected_realname = "$realname";
    if ($password_realname ne $expected_realname) {
	`$usermod -c \"$expected_realname\" $user` unless $debug;
	print LOG strftime "[$script] %c ---- update $user /etc/passwd realname\n", localtime;
    }


    # Make sure that the homedirectory is correct according to the
    # configuration.
    my $password_home = $etc_password{$user}->[7];
    my $expected_home = GetUserHome($user);
    if ($password_home ne $expected_home) {
	`$usermod -d \"$expected_home\" $user` unless $debug;
	`mkdir -p $expected_home && rm -rf $expected_home` unless $debug;
	`mv -f $password_home $expected_home` unless $debug;
	print LOG strftime "[$script] %c ---- update $user /etc/passwd homedir\n", localtime;
    }


    # Update SSH public keys if not accurate and only if the user
    # got the default shell.
    # In other cases, we do not mess with the way login rights are 
    # managed. Example: a user who got a /bin/bash should change his
    # key by bash login, not by web interface.
    my $ssh_keys_count = 0;
    if ($etc_password{$user}->[8] eq $sys_shell) {
	print "DBG update: $user ssh key managed by sv\n" if $debug;  
	if ($authorized_keys ne GetUserSSHKeyReal($user)) {
	    unless ($debug) {
		$ssh_keys_count = UserAddSSHKey($user, $authorized_keys);
	    }
	    print LOG strftime "[$script] %c ---- update $ssh_keys_count $user ssh keys\n", localtime;
	} 
    } 
    
    # GPG DEACTIVATED FOR NOW.
    # Will be reactivated with the creation of a mail interface.
    if (0) {
	# Update GPG key if not accurate
	my $gpg_keys_count = 0;
	if ($gpg_keys && $gpg_keys ne "NULL") {
	    # For gpg, I have to think about. Would we accept only gpg
	    # signature complete, or fingerprints?
	    unless ($debug) {
		$gpg_keys_count = UserAddGPGKey($user, $gpg_keys);
	    }
	    print LOG strftime "[$script] %c ---- update $gpg_keys_count $user gpg key\n", localtime;
	}
    }
	   


}

print LOG strftime "[$script] %c - account updates done\n", localtime;

# Final exit
print LOG strftime "[$script] %c - work finished\n", localtime;
print LOG "[$script] ------------------------------------------------------\n";
unlink($lockfile);

# END
