-- Samizdat Database Creation
--
--   Copyright (c) 2002-2004  Dmitry Borodaenko <angdraug@debian.org>
--
--   This program is free software.
--   You can distribute/modify this program under the terms of
--   the GNU General Public License version 2 or later.
--

-- RDF Data Model
CREATE TABLE Resource (
	id SERIAL PRIMARY KEY,
	published_date TIMESTAMP WITH TIME ZONE -- received date with site tz
		DEFAULT CURRENT_TIMESTAMP NOT NULL,
	literal BOOLEAN DEFAULT false,
	uriref BOOLEAN DEFAULT false,
	label TEXT); -- literal value | external uriref | internal class name
	-- optimize: store external uriref hash in numeric field

CREATE INDEX Resource_label_idx ON Resource (label);

CREATE TABLE Statement (
	id INT PRIMARY KEY REFERENCES Resource,
	subject INT NOT NULL REFERENCES Resource,
	predicate INT NOT NULL REFERENCES Resource,
	object INT NOT NULL REFERENCES Resource,
	rating NUMERIC(4,2)); -- computed from Vote

CREATE INDEX Statement_subject_idx ON Statement (subject);

-- Members and Sessions
CREATE TABLE Member (
	id INT PRIMARY KEY REFERENCES Resource,
	login TEXT UNIQUE NOT NULL,
	full_name TEXT NOT NULL,
	email TEXT UNIQUE NOT NULL,
	passwd TEXT,
	prefs TEXT,
	confirm TEXT UNIQUE,
	session TEXT UNIQUE,
	login_time TIMESTAMP WITH TIME ZONE,
	last_time TIMESTAMP WITH TIME ZONE);

-- Messages and Threads
CREATE TABLE Message (
	id INT PRIMARY KEY REFERENCES Resource,
	parent INT REFERENCES Message,   -- In-Reply-To:
	description INT REFERENCES Message,   -- abstract or toc or thumbnail
	version_of INT REFERENCES Message,   -- current version
	open BOOLEAN DEFAULT false,   -- editing open for all members
	creator INT REFERENCES Member,   -- From:
	title TEXT NOT NULL,   -- Subject:
	format TEXT,   -- MIME type
	content TEXT);

CREATE INDEX Message_parent_idx ON Message (parent);

CREATE INDEX Message_version_of_idx ON Message (version_of);

-- Voting Data
CREATE TABLE Vote (
	id INT PRIMARY KEY REFERENCES Resource,
	proposition INT REFERENCES Statement,
	member INT REFERENCES Member,
	UNIQUE (proposition, member),
	rating NUMERIC(2));

-- Moderation Actions Log
CREATE TABLE Moderation (
	action_date TIMESTAMP WITH TIME ZONE
		DEFAULT CURRENT_TIMESTAMP PRIMARY KEY,
	moderator INT REFERENCES Member,
	resource INT REFERENCES Resource);

-- Material Items Sharing
CREATE TABLE Item (
	id INT PRIMARY KEY REFERENCES Resource,
	description INT REFERENCES Message,
	contributor INT REFERENCES Member,
	possessor INT REFERENCES Member);

CREATE TABLE Possession (
	id INT PRIMARY KEY REFERENCES Resource,
	item INT REFERENCES Item,
	taken_from INT REFERENCES Member,
	given_to INT REFERENCES Member);

-- grant access in case we run this under another user
GRANT INSERT, UPDATE, SELECT ON Resource, Resource_id_seq, Statement, Vote,
	Member, Message, Item, Possession
TO samizdat;
GRANT INSERT, SELECT ON Moderation TO samizdat;
