# Samizdat plugin management
#
#   Copyright (c) 2002-2009  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 3 or later.
#
# vim: et sw=2 sts=2 ts=8 tw=0

require 'samizdat/engine'

# plugin superclass
#
class Plugin
  def initialize(options)
    @options = (options or {})
  end

  def api
    nil
  end

  def default?
    false
  end

  def match?(*params)
    false
  end
end

# singleton Hash mapping plugin names to classes
#
class PluginClasses < Hash
  include Singleton
end

# plugins configuration for a Samizdat site
#
class Plugins
  def initialize(config)
    @plugins = {}
    @defaults = {}

    config.each do |api, plugins|
      next if 'options' == api
      @plugins[api] = []

      plugins.each do |name|
        plugin_class = load_plugin(name)
        next if plugin_class.nil?

        plugin = plugin_class.new(config['options'][name])
        if plugin.default?
          @defaults[api] = plugin
        else
          @plugins[api].push(plugin)
        end
      end
    end
  end

  # list of non-default plugins for the _api_
  #
  def [](api)
    @plugins[api]
  end

  # default plugin for the _api_
  #
  def default(api)
    @defaults[api]
  end

  # find an _api_-compatible plugin for the _params_
  #
  def find(api, *params)
    plugin = nil

    @plugins[api].each do |p|
      if p.match?(*params)
        plugin = p
        break
      end
    end

    plugin or @defaults[api]
  end

  def find_all(api, *params)
    plugins = []

    @plugins[api].each do |p|
      if p.match?(*params)
        plugins.push(p)
      end
    end

    plugins.push(@defaults[api]) if @defaults[api].kind_of?(Plugin)

    plugins
  end

  private

  PLUGIN_NAME_PATTERN = Regexp.new(/\A[[:alnum:]_]+\z/).freeze

  # assumes that plugin initialization code will store its class in the
  # PluginClasses singleton hash, and returns class object of the loaded plugin
  #
  def load_plugin(name)
    return nil unless name =~ PLUGIN_NAME_PATTERN
    require %{samizdat/plugins/#{name}}
    PluginClasses.instance[name]
  end
end
