#!/usr/bin/env ruby
#
# Samizdat template tests
# 
#   Copyright (c) 2002-2003 Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 2 or later.
#

require 'test/unit'
require 'rexml/document'
require 'test/util'
require 'samizdat'

class TestSession
    def initialize
        @id = 1
        @login = 'test'
        @full_name = 'Test'
        @options = {}
    end

    attr_accessor :id, :login, :full_name, :options
end

class TC_Template < Test::Unit::TestCase
    def setup
        # create a dummy session
        @s = TestSession.new
        @t = Template.new(@s)
    end

    def teardown
        @t = nil
        @s = nil
    end

    def test_box
        root = parse(@t.box('Title', 'Content.'))
        assert_equal 'box', root.attributes['class']
        assert_equal ['Title'], elements(root, 'div[@class="box-title"]')
        assert_equal ['Content.'], elements(root, 'div[@class="box-content"]')

        root = parse(@t.box(nil, 'Content.'))
        assert_equal [], elements(root, 'div[@class="box-title"]')
        assert_equal ['Content.'], elements(root, 'div[@class="box-content"]')
    end

    def test_form_field_label
        root = parse(@t.form_field(:label, 'some_field', 'Label'))
        assert_equal 'label', root.attributes['class']
        assert_equal nil, root.elements['label'].attributes['for']
        assert_equal ['Label'], elements(root, 'label')
    end

    def test_form_field_textarea
        root = parse(@t.form_field(:textarea, 't_a', "Some\nText."))
        assert_equal 't_a', root.attributes['name']
        assert_equal "Some\nText.", root.text.strip
    end

    def test_form_field_select
        root = parse(@t.form_field(:select, 't_select',
            ['a1', ['a2', 'Label 2']]))
        assert_equal 't_select', root.attributes['name']
        assert_equal ['a1'], elements(root, 'option[@value="a1"]')
        assert_equal ['Label 2'], elements(root, 'option[@value="a2"]')
    end

    def test_form_field_submit
        root = parse(@t.form_field(:submit, 't_submit', 'Click'))
        assert_equal 't_submit', root.attributes['name']
        assert_equal 'Click', root.attributes['value']
        assert_equal 'submit', root.attributes['type']
        assert_equal 'submit', root.attributes['class']
    end

    def test_form_field_input
        root = parse(@t.form_field(:password, 'passwd', 'secret'))
        assert_equal 'passwd', root.attributes['name']
        assert_equal 'secret', root.attributes['value']
        assert_equal 'password', root.attributes['type']
    end

    def test_form
        root = parse(@t.form('login.rb', [:password, 'passwd'], [:submit]))
        assert_equal 'login.rb', root.attributes['action']
        assert_equal 'post', root.attributes['method']
        assert_equal ['password', 'submit'], elements(root, 'div/input', 'type')

        root = parse(@t.form('message.rb', [:file, 't_file']))
        assert_equal 'message.rb', root.attributes['action']
        assert_equal 'post', root.attributes['method']
        assert_equal 'multipart/form-data', root.attributes['enctype']
        assert_equal ['file'], elements(root, 'div/input', 'type')
    end

    def test_member_box
        root = parse(@t.member_box)
        assert_equal [@s.id.to_s, 'logout.rb', 'member.rb'],
            elements(root, '//a', 'href')
    end

    def test_login_form
        @s.id = nil
        @t = Template.new(@s)
        root = parse(@t.member_box)
        assert_equal ['Login'], elements(root, 'div[@class="box-title"]')
        assert_equal ['login.rb'],
            elements(root, 'div[@class="box-content"]/form', 'action')
        assert_equal ['member.rb'],
            elements(root, 'div[@class="box-content"]/p/a', 'href')
    end

    def test_simple_page
        root = parse(@t.page('Test Title', 'Some text.', 'Some sidebar.'))
        assert_equal [@s.login, 'Test Title'],
            elements(root, '//div[@class="box-title"]')
        assert_equal 'Some sidebar.',
            text(root, '/html/body/div[@id="sidebar"]')
    end

    def test_multi_page
        root = parse( @t.page('Title',
            [ ['Title 1', 'Body 1.'], ['Title 2', 'Body 2.']],
            'Some sidebar.'))
        assert_equal ['Samizdat: Title'], elements(root, '/html/head/title')
        assert_equal 'Some sidebar.',
            text(root, '/html/body/div[@id="sidebar"]')
        main = root.elements['/html/body/div[@id="main"]']
        assert_equal ['Title 1', 'Title 2'],
            elements(main, 'div/div[@class="box-title"]')
        assert_equal ['Body 1.', 'Body 2.'],
            elements(main, 'div/div[@class="box-content"]')
    end
end
