#!/usr/bin/env ruby
#
# Samizdat RDF storage tests
# 
#   Copyright (c) 2002-2003 Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 2 or later.
#

require 'test/unit'
require 'samizdat'

class TestSession
    def base
        'http://localhost/' + config['site']['base'] + '/'
    end
end

class TC_Storage < Test::Unit::TestCase
    def setup
        @session = TestSession.new
        @rdf = SamizdatRDF.new(@session.base)
    end

    def teardown
        @rdf, @session = nil
    end

    def test_query_select
        # initialize
        query_text = %{
SELECT ?msg, ?title, ?name, ?date, ?rating
WHERE (dc::title ?msg ?title)
      (dc::creator ?msg ?creator)
      (s::fullName ?creator ?name)
      (dc::date ?msg ?date)
      (rdf::subject ?stmt ?msg)
      (rdf::predicate ?stmt dc::relation)
      (rdf::object ?stmt s::Quality)
      (s::rating ?stmt ?rating)
LITERAL ?rating >= -1
ORDER BY ?rating DESC
USING PRESET NS}
        begin
            query = SquishQuery.new(@rdf, query_text)
        rescue
            assert false, "SquishQuery initialization raised #{$!.class}: #{$!}"
        end

        # query parser
        assert_equal 'SELECT', query.key
        assert_equal :select, query.type
        assert_equal %w[?msg ?title ?name ?date ?rating], query.nodes
        assert query.pattern.include?(["#{ns['dc']}title", "?msg", "?title"])
        assert_equal '?rating >= -1', query.literal
        assert_equal '?rating', query.order
        assert_equal 'DESC', query.order_dir
        assert_equal ns['s'], query.ns['s']

        # transform_pattern
        assert_equal config['map']['dc::title'], Hash[*query.c[0][1]]
        assert_equal [[0, 1]], query.pm['?title']
        # todo: check jc
        assert query.b.include?('?title')

        # query result
        begin 
            sql1 = @rdf.select(query)
        rescue
            assert false, "select with pre-parsed query raised #{$!.class}: #{$!}"
        end
        begin
            sql2 = @rdf.select(query_text)
        rescue
            assert false, "select with query text raised #{$!.class}: #{$!}"
        end
        assert sql1 == sql2
    end

    def test_query_merge
        # initialize
        query_text = %{
INSERT ?msg
UPDATE ?title = 'Test Message', ?content = 'Some text.'
WHERE (dc::creator ?msg base::1)
      (dc::title ?msg ?title)
      (s::content ?msg ?content)
      (s::thread ?msg ?msg)
USING dc FOR #{ns['dc']}
      s FOR #{ns['s']}
      base FOR #{@session.base}}
        begin
            query = SquishQuery.new(@rdf, query_text)
        rescue
            assert false, "SquishQuery initialization raised #{$!.class}: #{$!}"
        end

        # query parser
        assert_equal 'INSERT', query.key
        assert_equal :merge, query.type
        assert_equal [['?msg'], {'?title' => '${0}', '?content' => '${1}'}],
            query.nodes
        assert query.pattern.include?(["#{ns['dc']}title", "?msg", "?title"])
        assert_equal ns['s'], query.ns['s']
        assert_equal "'Test Message'", query.unstring('${0}')
        assert_equal "'Some text.'", query.unstring('${1}')

        # transform_pattern
        assert_equal config['map']['dc::creator'], Hash[*query.c[0][1]]
        assert_equal [[0, 1]], query.pm["#{@session.base}1"]
        # todo: check jc
        assert query.b.include?('?title')

        # todo: check merge
    end

    def test_dangling_blank_node
        # initialize
        query_text = %{
SELECT ?msg
WHERE (s::inReplyTo ?msg ?parent)
USING s FOR #{ns['s']}}
        begin
            query = SquishQuery.new(@rdf, query_text)
        rescue
            assert false, "SquishQuery initialization raised #{$!.class}: #{$!}"
        end

        # query parser
        assert_equal 'SELECT', query.key
        assert_equal :select, query.type
        assert_equal %w[?msg], query.nodes
        assert query.pattern.include?(["#{ns['s']}inReplyTo", "?msg", "?parent"])
        assert_equal ns['s'], query.ns['s']

        # transform_pattern
        assert_equal config['map']['s::inReplyTo'], Hash[*query.c[0][1]]
        assert_equal [[0, 1]], query.pm['?parent']
        # todo: check jc
        assert query.b.include?('?parent')

        # query result
        begin 
            sql1 = @rdf.select(query)
        rescue
            assert false, "select with pre-parsed query raised #{$!.class}: #{$!}"
        end
        begin
            sql2 = @rdf.select(query_text)
        rescue
            assert false, "select with query text raised #{$!.class}: #{$!}"
        end
        assert sql1 == sql2

        # dangling blank node
        assert query.jc.grep(/\A#{query.b['?parent']} IS NOT NULL\z/).size > 0
    end

    #def test_internal_resource
    #end

    #def test_external_subject_internal_property
    #end
end
