# Samizdat HTML templates
# 
#   Copyright (c) 2002-2003 Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 2 or later.
#

require 'uri/common'

class Template
    def initialize(session)
        @session = session
    end

    # HTML header, title, and style settings
    #
    def head(title='')
        pingback = %{
    <link rel="pingback" href="#{@session.options['x-pingback']}" />} if
            @session.options['x-pingback']
%{<?xml version="1.0" encoding="UTF-8"?>
<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Strict//EN"
    "http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd">
<html xmlns="http://www.w3.org/1999/xhtml" xml:lang="en">
<head>
    <title>#{title}</title>
    <meta name="generator" content="Samizdat #{SAMIZDAT_VERSION}"/>
    <link rel="stylesheet" type="text/css" href="css/default.css"/>#{pingback}
</head>
<body>\n}
    end

    # close HTML
    #
    def foot
        %{</body>\n</html>}
    end

    # site head and subhead
    #
    def site_head(title='')
%{<div id="head">
    <div id="head-left"><a href="http://www.nongnu.org/samizdat/">Samizdat</a>
        / <a href="#{@session.base}">#{config['site']['name']}</a> / #{title}</div>
    <div id="head-right"><a href="message.rb">Publish</a>
        :: <a href="item.rb">Share</a>
        :: <a href="query.rb">Search</a></div>
</div>
<div id="subhead">
    <div id="subhead-right">Co-op Engine: Collaborative Open Publishing</div>
</div>\n}
    end

    # site foot (attributions)
    #
    def main_foot
%{<div id="foot">
Powered by <a href="http://www.fsf.org/">Free Software</a>, including
<a href="http://www.apache.org/">Apache</a> web server,
<a href="http://www.ruby-lang.org/">Ruby</a> programming language,
and <a href="http://www.postgresql.org/">PostgreSQL</a> database.<br/>
<a href="http://www.nongnu.org/samizdat/">Samizdat</a> engine is free
software; you can distribute/modify it under the terms of the GNU
<a href="COPYING">General Public License</a> version 2 or later.
</div>}
    end

    # wrap content in <div id="sidebar"> if non-empty
    #
    def sidebar(content)
        if content and content != '' then
%{<div id="sidebar">
#{content}
</div>\n}
        else '' end
    end

    # wrap title and content into a CSS-rendered box
    #
    def box(title, content)
        box_title = %{<div class="box-title">#{title}</div>} if title
%{<div class="box">
    #{box_title}<div class="box-content">
#{content}
    </div>
</div>\n}
    end

    # navigation link to a given page number
    #
    def nav(skip, script=File.basename(@session.script_name)+'?', name='skip')
%{<div class="nav"><a href="#{script}#{name}=#{skip}">Next page</a></div>\n}
    end

    # resource list with navigation link
    #
    def list(list, nav, foot='')
        even = 1
        %{<ul>\n} +
        list.collect {|li|
            even = 1 - even
            %{<li#{' class="even"' if even == 1}>#{li}</li>\n}
        }.join + %{</ul>\n<div class="foot">\n} +
        foot + ((list.size < limit_page)? '' : nav) + %{</div>\n}
    end

    # resource table with navigation link
    #
    def table(table, nav, foot='')
        even = 1
        %{<table>\n<thead><tr>\n} +
        table.shift.collect {|th| "<th>#{th}</th>\n" }.join +
        %{</tr></thead>\n<tbody>\n} +
        table.collect {|row|
            even = 1 - even   # todo: a CSS-only way to do this
            %{<tr#{' class="even"' if even == 1}>\n} + row.collect {|td|
                "<td>#{td}</td>\n"
            }.join + "</tr>\n"
        }.join +
        %{</tbody></table>\n<div class="foot">\n} +
        foot + ((table.size < limit_page)? '' : nav) + %{</div>\n}
    end

    # type can be any of the following:
    #
    # [:label]
    #   wrap label _value_ in a <div> tag and associate it with _name_ field
    # [:textarea] fixed text area 50x10 with _value_
    # [:select] _value_ is an array of options or pairs of [option, label]
    # [:submit] _value_ is a button label
    # [standard HTML input type] copy _type_ as is into <input> tag
    #
    def form_field(type, name=nil, value=nil, default=nil)
        value = CGI.escapeHTML(value) if value.class == String
        name = %{ name="#{name}"} if name
        case type
        when :label   # todo: generate field idref for label for attribute
%{<div class="label"><label>#{value}</label></div>\n}
        when :textarea
%{<textarea#{name} cols="50" rows="10">#{value}</textarea>\n}
        when :select
%{<select#{name}>\n} + value.collect {|option|
    v, l = (option.class == Array)? option : [option, option]
    selected = (v == default)? ' selected="selected"' : ''
    %{    <option#{selected} value="#{v}">#{l}</option>\n}
}.join + "</select>\n"
        when :submit
            value = 'Submit' if value.nil?
%{<input#{name} type="submit" value="#{value}" class="submit"/>\n}
        else
%{<input#{name} type="#{type}" value="#{value}"/>\n}
        end
    end

    # wrap a list of form fields into a form (see form_field)
    #
    # automatically detects if multipart/form-data is necessary
    #
    def form(action, *fields)
        if fields.assoc(:file) then
            enctype = ' enctype="multipart/form-data"'
        end
        %{<form action="#{action}" method="post"#{enctype}><div>\n} +
        fields.collect {|param| form_field(*param) }.join + "</div></form>\n"
    end

    # member options when logged on, login form otherwise
    #
    def member_box
        if @session.id then
            box @session.login,
%{<p><a href="#{@session.id}">#{CGI.escapeHTML(@session.full_name)}</a></p>
<p>
<a href="logout.rb">Logout</a><br/>
<a href="member.rb">Change</a>
</p>}
        else
            box('Login', "<p>Use existing account:</p>" + form( 'login.rb',
                [:label, 'login', 'Login'],
                [:text, 'login'],
                [:label, 'passwd', 'Password'],
                [:password, 'passwd'],
                [:label],
                [:submit]
            ) + '<p><a href="member.rb">Create New Account</a></p>')
        end
    end

    # wrap focus name and rating in <span> tags
    #
    def focus_info(id, focus)
        return nil if focus.uriref == focus.resource.uriref
        name = focus.name
        name = %{<a href="#{focus.uriref}">#{name}</a>} if focus.id.to_i > 0
%{<span class="focus-name">#{name}:</span> <span class="rating"><a href="resource.rb?id=#{id}&amp;related=#{CGI.escape(focus.id.to_s)}">#{focus}</a></span>}
    end

    # list supplied focuses using focus_info
    #
    def focus_box(id, focuses)
        return '' unless focuses.class == Array
        box('Related Focuses', focuses.sort {|a, b|
            b.sort_index <=> a.sort_index
        }.collect {|focus|
            %{<div class="focus">#{focus_info(id, focus)}</div>\n}
        }.join)
    end

    # focus rating vote form
    #
    def vote_box(focus)
        box( 'Vote',
            form( 'resource.rb', 
                [:hidden, 'id', focus.resource.id],
                [:hidden, 'related', focus.id],
                [:label, 'rating', %{#{focus.name} rating}],
                [:select, 'rating', [
                    [-2, '-2 (No)'], 
                    [-1, '-1 (Not Likely)'],
                    [0, '0 (Uncertain)'],
                    [1, '1 (Likely)'],
                    [2, '2 (Yes)'] ], 0],
                [:label], [:submit, nil, 'Vote']
            )
        )
    end

    # render resource description for resource listing or message header
    #
    def resource(id, title, info)
%{<div class="resource">
<div class="title"><a href="#{id}">#{CGI.escapeHTML(title)}</a></div>
<div class="info">#{info}</div>
</div>\n}
    end

    # render full message if content provided, and message info line otherwise
    #
    def message(id, date, creator, full_name, focus, title,
            format=nil, content=nil, parent=nil)
        date = date.to_time if date.class != Time
        date = date.strftime '%Y-%m-%d %H:%M %Z'
        parent = %{<a href="#{parent}">Parent Message</a>} if parent
        rating = focus_info(id, focus) if focus
        info = [ %{by <a href="#{creator}">#{CGI.escapeHTML(full_name)}</a> on #{date}}, parent, rating ].compact.join("\n :: ")
        if content then   # render content
            content = content.to_s
            content =
                case format
                when nil   # default text rendering
CGI.escapeHTML(content).split(/^\s*$/).collect {|p| '<p>' + p + "</p>\n" }.join
                when 'text/plain'   # inline verbatim text
"<pre>#{CGI.escapeHTML(content)}</pre>"
                when 'text/uri-list'   # RFC2483 commented URI list
'<p>' + content.split("\n").collect {|line|
    case line
    when /\A#\s*(.*?)\z/
        CGI.escapeHTML($1) + '<br/>'
    when URI::URI_REF
        line = CGI.escapeHTML(line)
        %{<a href="#{line}">#{line}</a><br/>}
    end
}.join + '</p>'
                when 'application/x-squish'   # inline query form
form('query.rb',
    [:textarea, 'query', content], [:label], [:submit, 'run', 'Run'])
                when /^image\//   # <img/> reference
%{<img alt="#{CGI.escapeHTML(title)}" src="#{content}"/>}
                else   # <a/> reference
%{<p><a href="#{content}">#{format} content</a></p>}
                end
            reply = id ? (%{<div class="foot">\n} + form( 'message.rb',
                [:hidden, 'title', title],
                [:hidden, 'parent', id],
                [:submit, 'reply', 'Reply']
            ) + %{</div>\n}) : nil
%{<div class="message">
<div class="title"><a href="#{id}">#{CGI.escapeHTML(title)}</a></div>
<div class="info">#{info}</div>
<div class="content">#{content}</div>
#{reply}
</div>\n}
        else   # message listing mode if no content is supplied
            info
        end
    end

    # wrap page title and body with heads, foots, and sidebar with member_box
    #
    # body can be String or Array of pairs [title, body]; in latter case, title
    # can be defaulted to the title of the first pair
    #
    def page(title, body, side='')
        body = [[title, body]] unless body.class == Array
        title = body[0][0] if title.nil?
        main = body.collect {|t, b| box(CGI.escapeHTML(t), b) }.join
        head('Samizdat: ' + title.to_s) + site_head(CGI.escapeHTML(title)) +
            sidebar(member_box + side) +
            %{\n<div id="main">\n} + main + main_foot + "\n</div>\n" + foot
    end
end
