# Samizdat session management
# 
#   Copyright (c) 2002-2003 Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 2 or later.
#

require 'delegate'
require 'digest/md5'

# session management and CGI parameter handling
#
class Session < SimpleDelegator
    @@cookie_name = config['session']['cookie']
    @@login_timeout = config['session']['login_timeout']
    @@last_timeout = config['session']['last_timeout']

    # create session cookie for this session
    #
    def cookie
        CGI::Cookie.new({
            'name' => @@cookie_name,
            'value' => @session,
            'expires' => Time.now + @@last_timeout
        })
    end

    # create expired session cookie
    #
    def logout_cookie
        CGI::Cookie.new({
            'name' => @@cookie_name,
            'expires' => Time.now
        })
    end

    # translate location to a real file name (uses Apache/mod_ruby specifics)
    #
    def filename(location)
        Apache.request.lookup_uri(location).filename
    end

    # set default CGI options (set charset to UTF-8)
    #
    # set id and refresh session if session cookie is valid
    #
    def initialize
        @cgi = CGI.new
        @options = {'charset' => 'utf-8', 'cookie' => []}

        @base = ENV['HTTPS'] ? 'https' : 'http'
        port = {'http' => 80, 'https' => 443}
        port = (@cgi.server_port == port[@base]) ? '' : ":#{@cgi.server_port}"
        @base = @base + '://' + @cgi.host + port + config['site']['base'] + '/'

        @template = Template.new(self)
        @rdf = SamizdatRDF.new(@base)

        # check session
        @session = @cgi.cookies[@@cookie_name][0]
        if @session and @session != '' then
db.transaction do |db|
            @id, @login, @full_name, @email, login_time, last_time = db.select_one 'SELECT id, login, full_name, email FROM Member WHERE session = ?', @session
            if @id then
                if (login_time and login_time < Time::now - @@login_timeout) or
                (last_time and last_time < Time::now - @@last_timeout) then
                    close   # stale session
                else
                    # uncomment to regenerate session on each access:
                    #@session = generate_session
                    #db.do "UPDATE Member SET last_time = 'now', session = ?
                    #WHERE id = ?", @session, @id
                    @options['cookie'] = [cookie]
                end
            end
end
        end
        super @cgi
    end

    # CGI options
    attr :options

    # base URI of the site
    attr_reader :base

    # Template object aware of this session's options
    attr_reader :template

    # SamizdatRDF object aware of this session's options
    attr_reader :rdf

    attr_reader :id, :login, :full_name, :email

    # open new session on login
    #
    # redirect to referer on success
    #
    def open(login, passwd)
        db.transaction do |db|
            @id, = db.select_one 'SELECT id FROM Member m WHERE login = ?
            AND passwd = ?', login, Digest::MD5.new(passwd).hexdigest
            if @id then
                @session = generate_session
                db.do "UPDATE Member SET login_time = 'now', last_time = 'now',
                session = ? WHERE id = ?", @session, @id
                db.commit
                out({'status' => 'REDIRECT', 'location' => referer,
                    'cookie' => cookie})
            else
                out() { template.page('Login Failed', 
                    '<p>Wrong login name or password. Try again.</p>') }
            end
        end
    end

    # erase session from database
    #
    def close
        db.do "UPDATE Member SET session=NULL WHERE id = ?", @id
        db.commit
        @id = nil
        @options['cookie'] = [logout_cookie]
    end

    # return list of values of cgi parameters, tranform empty values to nils
    #
    # unlike CGI#params, Session#params can take array of parameter names
    #
    def params(keys)
        keys.collect do |key|
            value = self[key]
            raise UserError, "Input size exceeds content size limit" if
                value.methods.include? :size and
                value.size > config['limit']['content']
            case value
            when String then (value =~ /[^\s]/)? value : nil
            when StringIO, Tempfile then value.read
            else nil
            end
        end
    end

    # always imitate CGI#[] from Ruby 1.8
    # 
    def [](key)
        @cgi.params[key][0]
    end

    # add fake cgi parameters
    #
    def []=(key, value)
        @cgi.params[key] = [value]
    end

    # print header and optionally content, then clean-up and exit
    #
    # generate error page on RuntumeError exceptions
    #
    def out(options={})
        @options.update(options)
        if block_given? then
            page =
                begin
                    yield template
                rescue AuthError
                    @options['status'] = 'AUTH_REQUIRED'
                    template.page( 'Please Login', 
%{<p>#{$!}. Use the form in the sidebar to login, if you are a registered
member, or to create a new account, if you don't have an account on this site
yet.</p>})
                rescue UserError
                    template.page('User Error',
%{<p>#{$!}.</p><p>Press 'Back' button of your browser to return.</p>})
                rescue ResourceNotFoundError
                    @options['status'] = 'NOT_FOUND'
                    referer = %{ (looks like it was <a href="#{@cgi.referer}">#{@cgi.referer}</a>)} if @cgi.referer
                    template.page('Resource Not Found',
%{<p>The resource you requested was not found on this site. Please report this
error back to the site you came from#{referer}.</p>})
                rescue RuntimeError
                    template.page('Runtime Error',
%{<p>Runtime error has occured: #{$!}.</p>
<pre>#{caller.join("\n")}</pre>
<p>Please report this error to the site administrator.</p>})
                end
            @cgi.out(@options) { page }
        else
            @cgi.header(@options)
        end
        @rdf = nil
        @template = nil
        @cgi = nil
        GC.start
        exit
    end

private

    def generate_session
        Digest::MD5.new(@id.to_s + Time::now.to_s).hexdigest
    end
end
