# Samizdat resource representation
# 
#   Copyright (c) 2002-2003 Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 2 or later.
#

class Resource
    def initialize(session, id=nil)
        @session = session
        @template = session.template
        @rdf = session.rdf
        case id
        when Integer then @id = id
        else
            @id = id.to_i   # only numeric ids allowed
        end
        @uriref = session.base + id.to_s
    end

    attr_reader :session, :template, :rdf, :id, :uriref 

    # in :list mode (default) return rendered desc
    #
    # in :short mode return type, head, and unrendered desc
    #
    # in :full mode, add related resources to desc
    #
    def render(mode=:list)
        if self.class == Focus and id.to_i == 0 then
            label = id.to_s
            external = true
        else
            external, literal, label = db.select_one 'SELECT uriref, literal, label FROM Resource WHERE literal = false AND id = ?', id
        end
        if external then
            type = 'Uriref'
            head = SquishQuery.ns_shrink(label)
            head.gsub!(/\Afocus::/, '')   # don't display prefix for focuses
            desc = %{refers to <a href="#{label}">external uriref</a>}
            if mode == :full then
                # todo: select all statements with this subject
            end
        elsif literal then
            type = 'Literal'
            head = label
            desc =
%{<a href="#{uriref}">#{SquishQuery.ns_shrink(uriref)}</a> = #{label}}
        else   # internal resource
            type = label
            head, desc = case label
            when nil then raise ResourceNotFoundError, id.to_s
            when 'Member' then render_member mode
            when 'Message' then render_message mode
            when 'Statement' then render_statement mode
            when 'Vote' then render_vote mode
            when 'Item' then render_item mode
            else raise RuntimeError, "Unknown resource type '#{label}'"
            end
        end
        case mode
        when :list then return template.resource(id, head, desc)
        when :short, :full then return type, head, desc
        end
    end

    # multimedia message content location
    # 
    # if _format_ or _login_ are not specified, this Resource object should be
    # instantiated from a valid _id_ of an existing resource for this method to
    # work
    #
    def content_location(id=@id, format=nil, login=nil)
        if login.nil? or format.nil? then
            if id.kind_of? Integer and id > 0 then
                format, login = rdf.select_one %{
SELECT ?format, ?login
WHERE (dc::format #{uriref} ?format)
      (dc::creator #{uriref} ?creator)
      (s::login ?creator ?login)
USING PRESET NS}
            else
                raise RuntimeError,
                    "Can't derive message format and creator login name"
            end
        end
        ext = config['file_extension'][format]
        ext = format.sub(/\A.*\//, '') if ext.nil?
        config['site']['content'] + '/' + login + '/' + id.to_s + '.' + ext
    end

private

    def render_member(mode)
        login, head = rdf.select_one "
SELECT ?login, ?full_name
WHERE (s::login #{uriref} ?login)
      (s::fullName #{uriref} ?full_name)
USING s FOR #{ns['s']}"
        desc = "Login: #{login}"
        if mode == :full then
            skip, = session.params %w[skip]
            skip = skip.to_i
            messages = rdf.select_all( %{
SELECT ?msg
WHERE (dc::date ?msg ?date)
      (dc::creator ?msg #{uriref})
ORDER BY ?date DESC
USING PRESET NS}, limit_page, limit_page * skip
            ).collect {|msg,| Resource.new(session, msg).render }
            desc = template.box(nil, desc)
            desc << template.box( 'Latest Messages', template.list(messages,
                template.nav(skip + 1, %{resource.rb?id=#{id}&amp;}))
            ) if messages.size > 0
        end
        return head, desc
    end

    def render_message(mode)
        date, head, creator, name, login = rdf.select_one %{
SELECT ?date, ?title, ?creator, ?name, ?login
WHERE (dc::date #{uriref} ?date)
      (dc::title #{uriref} ?title)
      (dc::creator #{uriref} ?creator)
      (s::fullName ?creator ?name)
      (s::login ?creator ?login)
USING dc FOR #{ns['dc']}
      s FOR #{ns['s']}}
        if mode == :short or mode == :full then   # optional properties
            content, = rdf.select_one "
SELECT ?content
WHERE (s::content #{uriref} ?content)
USING s FOR #{ns['s']}"
            format, = rdf.select_one "
SELECT ?format
WHERE (dc::format #{uriref} ?format)
USING dc FOR #{ns['dc']}"
            if content.nil? and format and
            not config['format']['inline'].include? format then
                content = content_location(id, format, login)
            end
        end
        if mode == :full then   # don't have to refer to parent in :short mode
            parent, = rdf.select_one "
SELECT ?parent
WHERE (s::inReplyTo #{uriref} ?parent)
USING s FOR #{ns['s']}"
        end
        focus = Focus.new(self) unless self.class == Focus
        desc = template.message(id, date, creator, name, focus, head,
            format, content, parent)
        if mode == :full
            skip, = session.params %w[skip]
            skip = skip.to_i
            replies = rdf.select_all( %{
SELECT ?msg
WHERE (dc::date ?msg ?date)
      (s::inReplyTo ?msg #{uriref})
ORDER BY ?date
USING PRESET NS}, limit_page, limit_page * skip
            ).collect {|msg,| Resource.new(session, msg).render(:short)[2] }
            desc << template.box( 'Replies', template.list(replies,
                template.nav(skip + 1, %{resource.rb?id=#{id}&amp;}))
            ) if replies.size > 0
        end
        return head, desc
    end

    def render_statement(mode)
        head = "Statement #{id}"
        stmt = rdf.select_one %{
SELECT ?p, ?s, ?o
WHERE (rdf::predicate #{uriref} ?p)
      (rdf::subject #{uriref} ?s)
      (rdf::object #{uriref} ?o)
USING rdf FOR #{ns['rdf']}}
        n = ['Predicate', 'Subject', 'Object']
        if mode == :list then
            desc = '(' + stmt.collect {|id|
                %{<a href="#{id}">#{n.shift} #{id}</a>}
            }.join(', ') + ')'
        else
            desc = stmt.collect {|id|
                template.box(n.shift, Resource.new(session, id).render)
            }.join
        end
        if mode == :full then
            # todo: display votes
        end
        return head, desc
    end

    def render_vote(mode)
        head = "Vote #{id}"
        stmt, member, name, rating = rdf.select_one "
SELECT ?stmt, ?member, ?name, ?rating
WHERE (s::voteProposition #{uriref} ?stmt)
      (s::voteMember #{uriref} ?member)
      (s::fullName ?member ?name)
      (s::voteRating #{uriref} ?rating)
USING s FOR #{ns['s']}"
        desc = %'<a href="#{member}">#{name}</a> gives rating #{"%4.2f" % rating} to the <a href="#{stmt}">Statement #{stmt}</a>.'
        if mode == :full then
            desc = template.box(nil, desc) + template.box("Vote Proposition",
                Resource.new(session, stmt).render(:full)[2])
        end
        return head, desc
    end

    def render_item(mode)
        # todo: short mode
        msg, date, creator, name, head, format, content, parent,
            contributor, c_name, possessor, p_name = rdf.select_one "
SELECT ?msg, ?date, ?creator, ?name, ?title, ?content, ?parent,
       ?contributor, ?c_name, ?possessor, ?p_name
WHERE (s::description #{uriref} ?msg)
      (dc::date ?msg ?date)
      (dc::creator ?msg ?creator)
      (s::fullName ?creator ?name)
      (dc::title ?msg ?title)
      (dc::format ?msg ?format)
      (s::content ?msg ?content)
      (s::inReplyTo ?msg ?parent)
      (s::contributor #{uriref} ?contributor)
      (s::fullName ?contributor ?c_name)
      (s::possessor #{uriref} ?possessor)
      (s::fullName ?possessor ?p_name)
USING dc FOR http://purl.org/dc/elements/1.1/
      s FOR #{ns['s']}"
        desc = 
%'<p><b>Contributor:</b> <a href="#{contributor}">#{c_name}</a></p>
<p><b>Possessor:</b> <a href="#{possessor}">#{p_name}</a></p>
<p><b>Description:</b></p>' +
            template.message(msg, date, creator, name, nil, \
                head, format, content, parent)
        head += ' / ' + possessor
        if mode == :full then
            query = "
SELECT ?item, ?name
WHERE (s::description #{uriref} ?msg)
      (s::description ?item ?msg)
      (s::possessor ?item ?possessor)
      (s::fullName ?possessor ?name)
USING s FOR #{ns['s']}"
            desc += template.box('Similar Items',
                '<p><b>Possessed by:</b></p>' +
                db.execute(rdf.select(query)) {|sth|
                    sth.fetch_many(limit_page).collect {|item,name|
                        %'<a href="#{item}">#{name}</a><br>'
                    }.join })
        end
        return head, desc
    end
end
